/*
 * Decompiled with CFR 0.152.
 */
package org.wildfly.security.sasl.gs2;

import java.io.IOException;
import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.UnsupportedCallbackException;
import javax.security.sasl.AuthorizeCallback;
import javax.security.sasl.SaslException;
import org.ietf.jgss.ChannelBinding;
import org.ietf.jgss.GSSContext;
import org.ietf.jgss.GSSCredential;
import org.ietf.jgss.GSSException;
import org.ietf.jgss.GSSManager;
import org.ietf.jgss.GSSName;
import org.ietf.jgss.Oid;
import org.wildfly.common.Assert;
import org.wildfly.security._private.ElytronMessages;
import org.wildfly.security.asn1.ASN1Exception;
import org.wildfly.security.asn1.DEREncoder;
import org.wildfly.security.auth.callback.IdentityCredentialCallback;
import org.wildfly.security.auth.callback.ServerCredentialCallback;
import org.wildfly.security.credential.GSSKerberosCredential;
import org.wildfly.security.sasl.gs2.Gs2;
import org.wildfly.security.sasl.gs2.Gs2Util;
import org.wildfly.security.sasl.util.AbstractSaslServer;
import org.wildfly.security.util.ByteIterator;
import org.wildfly.security.util.ByteStringBuilder;
import org.wildfly.security.util.CodePointIterator;

final class Gs2SaslServer
extends AbstractSaslServer {
    private static final int ST_NO_MESSAGE = 1;
    private static final int ST_FIRST_MESSAGE = 2;
    private static final int ST_ACCEPTOR = 3;
    private final boolean plus;
    private final String bindingType;
    private final byte[] bindingData;
    private final Oid mechanism;
    private GSSContext gssContext;
    private String authorizationID;
    private String boundServerName;

    Gs2SaslServer(String mechanismName, String protocol, String serverName, CallbackHandler callbackHandler, GSSManager gssManager, boolean plus, String bindingType, byte[] bindingData) throws SaslException {
        super(mechanismName, protocol, serverName, callbackHandler, ElytronMessages.saslGs2);
        this.plus = plus;
        this.bindingType = bindingType;
        this.bindingData = bindingData;
        try {
            this.mechanism = Gs2.getMechanismForSaslName(gssManager, mechanismName);
        }
        catch (GSSException e) {
            throw ElytronMessages.saslGs2.mechMechanismToOidMappingFailed(e).toSaslException();
        }
        GSSCredential credential = null;
        ServerCredentialCallback credentialCallback = new ServerCredentialCallback(GSSKerberosCredential.class);
        try {
            ElytronMessages.saslGs2.trace("Obtaining GSSCredential for the service from callback handler");
            callbackHandler.handle(new Callback[]{credentialCallback});
            credential = credentialCallback.applyToCredential(GSSKerberosCredential.class, GSSKerberosCredential::getGssCredential);
        }
        catch (IOException e) {
            throw ElytronMessages.saslGs2.mechCallbackHandlerFailedForUnknownReason(e).toSaslException();
        }
        catch (UnsupportedCallbackException e) {
            ElytronMessages.saslGs2.trace("Unable to obtain GSSCredential from callback handler", e);
        }
        try {
            if (credential == null) {
                GSSName ourName;
                if (serverName != null) {
                    String localName = protocol + "@" + serverName;
                    ElytronMessages.saslGs2.tracef("Our name is '%s'", (Object)localName);
                    ourName = gssManager.createName(localName, GSSName.NT_HOSTBASED_SERVICE, this.mechanism);
                } else {
                    ElytronMessages.saslGs2.tracef("Our name is unbound", new Object[0]);
                    ourName = null;
                }
                credential = gssManager.createCredential(ourName, Integer.MAX_VALUE, this.mechanism, 2);
            }
            this.gssContext = gssManager.createContext(credential);
        }
        catch (GSSException e) {
            throw ElytronMessages.saslGs2.mechUnableToCreateGssContext(e).toSaslException();
        }
    }

    @Override
    public void init() {
        this.setNegotiationState(1);
    }

    @Override
    public String getAuthorizationID() {
        return this.authorizationID;
    }

    @Override
    protected byte[] evaluateMessage(int state, byte[] message) throws SaslException {
        switch (state) {
            case 1: {
                if (message == null || message.length == 0) {
                    this.setNegotiationState(3);
                    return NO_BYTES;
                }
            }
            case 2: {
                byte[] token;
                int gs2HeaderLength;
                int gs2HeaderStartIndex;
                assert (!this.gssContext.isEstablished());
                if (message == null || message.length == 0) {
                    throw ElytronMessages.saslGs2.mechClientRefusesToInitiateAuthentication().toSaslException();
                }
                ByteIterator bi = ByteIterator.ofBytes(message);
                ByteIterator di = bi.delimitedBy(44);
                CodePointIterator cpi = di.asUtf8String();
                boolean gs2CbFlagPUsed = false;
                boolean nonStd = false;
                int b = bi.next();
                if (b == 70) {
                    this.skipDelimiter(bi);
                    nonStd = true;
                    b = bi.next();
                }
                if (b == 112) {
                    gs2CbFlagPUsed = true;
                    if (!this.plus) {
                        throw ElytronMessages.saslGs2.mechChannelBindingNotSupported().toSaslException();
                    }
                    if (bi.next() != 61) {
                        throw ElytronMessages.saslGs2.mechInvalidMessageReceived().toSaslException();
                    }
                    assert (this.bindingType != null);
                    assert (this.bindingData != null);
                    if (!this.bindingType.equals(cpi.drainToString())) {
                        throw ElytronMessages.saslGs2.mechChannelBindingTypeMismatch().toSaslException();
                    }
                    this.skipDelimiter(bi);
                } else if (b == 121) {
                    if (this.plus || this.bindingType != null && this.bindingData != null) {
                        throw ElytronMessages.saslGs2.mechChannelBindingNotProvided().toSaslException();
                    }
                    this.skipDelimiter(bi);
                } else if (b == 110) {
                    if (this.plus) {
                        throw ElytronMessages.saslGs2.mechChannelBindingNotProvided().toSaslException();
                    }
                    this.skipDelimiter(bi);
                } else {
                    throw ElytronMessages.saslGs2.mechInvalidMessageReceived().toSaslException();
                }
                b = bi.next();
                if (b == 97) {
                    if (bi.next() != 61) {
                        throw ElytronMessages.saslGs2.mechInvalidMessageReceived().toSaslException();
                    }
                    this.authorizationID = cpi.drainToString();
                    this.skipDelimiter(bi);
                } else if (b != 44) {
                    throw ElytronMessages.saslGs2.mechInvalidMessageReceived().toSaslException();
                }
                if (nonStd) {
                    gs2HeaderStartIndex = 2;
                    gs2HeaderLength = bi.offset() - 2;
                    token = bi.drain();
                } else {
                    gs2HeaderStartIndex = 0;
                    gs2HeaderLength = bi.offset();
                    try {
                        token = this.restoreTokenHeader(bi.drain());
                    }
                    catch (ASN1Exception e) {
                        throw ElytronMessages.saslGs2.mechUnableToCreateResponseTokenWithCause(e).toSaslException();
                    }
                }
                ByteStringBuilder gs2HeaderExcludingNonStdFlag = new ByteStringBuilder();
                gs2HeaderExcludingNonStdFlag.append(message, gs2HeaderStartIndex, gs2HeaderLength);
                try {
                    ChannelBinding channelBinding = Gs2Util.createChannelBinding(gs2HeaderExcludingNonStdFlag.toArray(), gs2CbFlagPUsed, this.bindingData);
                    this.gssContext.setChannelBinding(channelBinding);
                }
                catch (GSSException e) {
                    throw ElytronMessages.saslGs2.mechUnableToSetChannelBinding(e).toSaslException();
                }
                try {
                    byte[] response = this.gssContext.acceptSecContext(token, 0, token.length);
                    if (this.gssContext.isEstablished()) {
                        Oid actualMechanism = this.gssContext.getMech();
                        if (!this.mechanism.equals(actualMechanism)) {
                            throw ElytronMessages.saslGs2.mechGssApiMechanismMismatch().toSaslException();
                        }
                        this.storeBoundServerName();
                        this.checkAuthorizationID();
                        this.storeDelegatedGSSCredential();
                        this.negotiationComplete();
                    } else {
                        this.setNegotiationState(3);
                    }
                    return response;
                }
                catch (GSSException e) {
                    throw ElytronMessages.saslGs2.mechUnableToAcceptClientMessage(e).toSaslException();
                }
            }
            case 3: {
                assert (!this.gssContext.isEstablished());
                try {
                    byte[] response = this.gssContext.acceptSecContext(message, 0, message.length);
                    if (this.gssContext.isEstablished()) {
                        Oid actualMechanism = this.gssContext.getMech();
                        if (!this.mechanism.equals(actualMechanism)) {
                            throw ElytronMessages.saslGs2.mechGssApiMechanismMismatch().toSaslException();
                        }
                        this.storeBoundServerName();
                        this.checkAuthorizationID();
                        this.storeDelegatedGSSCredential();
                        this.negotiationComplete();
                    }
                    return response;
                }
                catch (GSSException e) {
                    throw ElytronMessages.saslGs2.mechUnableToAcceptClientMessage(e).toSaslException();
                }
            }
            case 0: {
                if (message != null && message.length != 0) {
                    throw ElytronMessages.saslGs2.mechMessageAfterComplete().toSaslException();
                }
                return null;
            }
        }
        throw Assert.impossibleSwitchCase(state);
    }

    @Override
    public void dispose() throws SaslException {
        try {
            this.gssContext.dispose();
        }
        catch (GSSException e) {
            throw ElytronMessages.saslGs2.mechUnableToDisposeGssContext(e).toSaslException();
        }
        finally {
            this.gssContext = null;
        }
    }

    private byte[] restoreTokenHeader(byte[] token) throws ASN1Exception {
        DEREncoder encoder = new DEREncoder();
        encoder.encodeImplicit(64, 0);
        encoder.startSequence();
        try {
            encoder.writeEncoded(this.mechanism.getDER());
        }
        catch (GSSException e) {
            throw new ASN1Exception(e);
        }
        encoder.writeEncoded(token);
        encoder.endSequence();
        return encoder.getEncoded();
    }

    private void storeBoundServerName() throws SaslException {
        try {
            String targetName = this.gssContext.getTargName().toString();
            String[] targetNameParts = targetName.split("[/@]");
            this.boundServerName = targetNameParts.length > 1 ? targetNameParts[1] : targetName;
        }
        catch (GSSException e) {
            throw ElytronMessages.saslGs2.mechUnableToDetermineBoundServerName(e).toSaslException();
        }
    }

    private void checkAuthorizationID() throws SaslException {
        String authenticationID;
        try {
            authenticationID = this.gssContext.getSrcName().toString();
        }
        catch (GSSException e) {
            throw ElytronMessages.saslGs2.mechUnableToDeterminePeerName(e).toSaslException();
        }
        ElytronMessages.saslGs2.tracef("checking if [%s] is authorized to act as [%s]...", (Object)authenticationID, (Object)this.authorizationID);
        if (this.authorizationID == null || this.authorizationID.isEmpty()) {
            this.authorizationID = authenticationID;
        }
        AuthorizeCallback authorizeCallback = new AuthorizeCallback(authenticationID, this.authorizationID);
        this.handleCallbacks(authorizeCallback);
        if (!authorizeCallback.isAuthorized()) {
            throw ElytronMessages.saslGs2.mechAuthorizationFailed(authenticationID, this.authorizationID).toSaslException();
        }
        ElytronMessages.saslGs2.trace("authorization id check successful");
    }

    private void storeDelegatedGSSCredential() throws SaslException {
        try {
            GSSCredential gssCredential = this.gssContext.getDelegCred();
            if (gssCredential != null) {
                this.tryHandleCallbacks(new IdentityCredentialCallback(new GSSKerberosCredential(gssCredential), true));
            } else {
                ElytronMessages.saslGs2.trace("No GSSCredential delegated during authentication.");
            }
        }
        catch (UnsupportedCallbackException | GSSException exception) {
            // empty catch block
        }
    }

    private void skipDelimiter(ByteIterator bi) throws SaslException {
        if (bi.next() != 44) {
            throw ElytronMessages.saslGs2.mechInvalidMessageReceived().toSaslException();
        }
    }

    @Override
    public Object getNegotiatedProperty(String propName) {
        this.assertComplete();
        if ("javax.security.sasl.bound.server.name".equals(propName)) {
            return this.boundServerName;
        }
        return null;
    }
}

