#!/bin/sh

usage() {
	cat << __EOF__
Usage: $0 --scope=SCOPE
        SCOPE is one of 'hosts', 'engine', 'all'
__EOF__
	exit 1
}

script_path="$(readlink -f "$0")"
cd "$(dirname "$script_path")"

MY_BIN_DIR="$(dirname ${PWD})/bin"
. "${MY_BIN_DIR}"/config.sh
. "${ENGINE_DATA_BIN_DIR}"/engine-prolog.sh

CONFIG_FILE="${PKG_SYSCONF_DIR}/config.yml"

SCOPE=all

COLLECTD_SYSTEMD_PG_CONF=/etc/systemd/system/collectd.service.d/postgresql.conf
CREATE_PG_PASS="${MY_BIN_DIR}"/create_collectd_pg_pass.sh

setup_db_creds() {
	COLLECTD_SYSTEMD_DIR="$(dirname ${COLLECTD_SYSTEMD_PG_CONF})"
	mkdir -p "${COLLECTD_SYSTEMD_DIR}" || die "Failed to create ${COLLECTD_SYSTEMD_DIR}"

	local tmpconf="$(mktemp)"
	cat > "${tmpconf}" << __EOF__
# This file was automatically generated by ${script_path}, do not edit manually
[Service]
ExecStartPre=${CREATE_PG_PASS}
Environment=PGHOST=${ENGINE_DB_HOST}
Environment=PGPORT=${ENGINE_DB_PORT}
Environment=PGDATABASE=${ENGINE_DB_DATABASE}
Environment=PGUSER=${ENGINE_DB_USER}
Environment=PGPASSFILE=${COLLECTD_PGPASS}
__EOF__
	[ $? == 0 ] || die "Failed to create ${tmpconf}"

	if ! cmp -s "${COLLECTD_SYSTEMD_PG_CONF}" "${tmpconf}"; then
		cat "${tmpconf}" > "${COLLECTD_SYSTEMD_PG_CONF}" || die "Failed to write ${COLLECTD_SYSTEMD_PG_CONF}"
	fi
	rm -f "${tmpconf}"

	# Required for systemd to notice
	systemctl daemon-reload
}

while [ -n "$1" ]; do
	x="$1"
	v="${x#*=}"
	shift
	case "${x}" in
		--scope=*)
			SCOPE="${v}"
		;;
		--help)
			usage
		;;
		*)
			die "Invalid option '${x}'"
		;;
	esac
done

# Always create collectd conf. Should be harmless if not needed.
# We could check if $SCOPE is 'engine' or 'all', but then it will
# not work if user passes some other valid ansible pattern.
setup_db_creds

if [ -r "${CONFIG_FILE}" ]; then
	ansible-playbook \
		playbook.yml \
		-e @"${CONFIG_FILE}" \
		-e ansible_ssh_private_key_file="${ENGINE_PKI}/keys/engine_id_rsa" \
		-e pg_db_name="${ENGINE_DB_DATABASE}" \
		-e ovirt_engine_fqdn="${ENGINE_FQDN}" \
		-l "${SCOPE}"
else
	die "${CONFIG_FILE} is missing. Copy and amend /etc/ovirt-engine-metrics/config.yml.example"
fi
