/*
 * Copyright (c) 2001-2003 The Trustees of Indiana University.  
 *                         All rights reserved.
 * Copyright (c) 1998-2001 University of Notre Dame. 
 *                         All rights reserved.
 * Copyright (c) 1994-1998 The Ohio State University.  
 *                         All rights reserved.
 * 
 * This file is part of the XMPI software package.  For license
 * information, see the LICENSE file in the top level directory of the
 * XMPI source distribution.
 * 
 * $HEADER$
 *
 * $Id: xmpi_tr_dump.cc,v 1.6 2003/09/03 04:28:54 jsquyres Exp $
 *
 *	Function:	- dump traces into a trace file
 */

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <Xm/Xm.h>
#include <Xm/SelectioB.h>

#include "lam.h"
#include "xmpi.h"
#include "xmpi_tr_dump.h"
#include "xmpi_error.h"
#include "xmpi_misc.h"
#include "xmpi_ctl.h"
#include "xmpi_db_parse.h"


/*
 * local functions
 */
static void dump_cb(Widget, XtPointer, 
		    XmSelectionBoxCallbackStruct*);

static int dump(char*);

/*
 * extern variables
 */
extern int xmpi_app_nprocs;	       /* number of procs in appl. */

extern struct _gps *xmpi_app_procs;    /* appl. GPS array */

extern char tracefile[];	       /* name of trace file */

/*
 * local variables
 */
static char tempfile[XMPI_PATHMAX];

/*
 *	xmpi_tr_dump
 *
 *	Function:	- dump current application's traces into a file
 *	Accepts:	- parent widget
 */
void
xmpi_tr_dump(Widget parent)
{
  Widget dialog_w;		       /* file name prompt dialog */

  XmString str1, str2;		       /* various strings */

  Arg args[3];

  if (xmpi_app_procs == 0) {
    errno = 0;
    xmpi_error(parent,
	       (char*) "No application for which to dump traces.");
    return;
  }
  str1 = XmStringCreateSimple((char*) "Enter trace file name.");

  if (!tempfile[0]) {
    sprintf(tempfile, "/tmp/lam-xmpi%d.lamtr", (int) getpid());
  }
  str2 = XmStringCreateSimple(tempfile);

  XtSetArg(args[0], XmNselectionLabelString, str1);
  XtSetArg(args[1], XmNautoUnmanage, False);
  XtSetArg(args[2], XmNtextString, str2);

  dialog_w = XmCreatePromptDialog(parent, (char*) "prompt_pop", args, 3);

  XmStringFree(str1);
  XmStringFree(str2);

  XtUnmanageChild(XmSelectionBoxGetChild(dialog_w,
					 XmDIALOG_HELP_BUTTON));

  XtAddCallback(dialog_w, XmNokCallback,
		(XtCallbackProc) dump_cb, NULL);
  XtAddCallback(dialog_w, XmNcancelCallback,
		(XtCallbackProc) XtDestroyWidget, NULL);

  XtManageChild(dialog_w);
  XtPopup(XtParent(dialog_w), XtGrabNone);
}

/*
 *	dump_cb
 *
 *	Function:	- dumps traces (callback)
 *	Accepts:	- callback data
 */
static void
dump_cb(Widget w, XtPointer,
	XmSelectionBoxCallbackStruct *cbs)
{
  struct stat finfo;		       /* file information */

  char *fname;			       /* file name */

  int exists = 1;		       /* file exists? */

  XmStringGetLtoR(cbs->value, XmSTRING_DEFAULT_CHARSET, &fname);

  if (!(fname && *fname)) {
    errno = 0;
    xmpi_error(w, (char*) "No filename specified.");
    return;
  }
/*
 * Check if file already exists and if so ask user if they want to overwrite
 * it or not.
 */
  if (stat(fname, &finfo)) {
    if (errno != ENOENT) {
      xmpi_error(w, fname);
      XtFree(fname);
      return;
    }
    exists = 0;
  }
  if (exists && !xmpi_yesno(w, (char*) "File exists. Overwrite?")) {
    XtFree(fname);
    return;
  }
  (void) dump(fname);

  XtFree(fname);
  XtDestroyWidget(w);
}

/*
 *	dump
 *
 *	Function:	- dumps traces
 *	Accepts:	- file name
 *	Returns		- 0 else LAMERROR
 */
static int
dump(char *file)
{
  int fd;			       /* trace file descriptor */

/*
 * Open the file and dump the traces into it.
 */
  fd = open(file, O_WRONLY | O_CREAT | O_TRUNC, 0644);

  if (fd < 0) {
    xmpi_error(0, file);
    return (LAMERROR);
  }
  xmpi_busy();
  xmpi_ctl_setinfo((char*) "Collecting traces...");

  if (xmpi_sys_trace(fd, xmpi_app_procs, xmpi_app_nprocs)) {
    xmpi_ctl_resetinfo();
    xmpi_unbusy();
    xmpi_error(0, (char*) "Collecting traces");
    close(fd);
    return (LAMERROR);
  }
  xmpi_ctl_resetinfo();
  xmpi_unbusy();
  close(fd);
  return (0);
}

/*
 *	xmpi_tr_express
 *
 *	Function:	- dumps traces and views them
 *	Accepts:	- parent widget
 */
void
xmpi_tr_express(Widget)
{
  FILE *dbfs;			       /* trace file stream */

  if (!tempfile[0]) {
    sprintf(tempfile, "/tmp/lam-xmpi%d.lamtr", (int) getpid());
  }
  if (dump(tempfile))
    return;

  strncpy(tracefile, tempfile, XMPI_PATHMAX);
  dbfs = fopen(tracefile, "r");

  if (dbfs == 0) {
    xmpi_error(0, tracefile);
  } else {
    xmpi_parse_tracefile(dbfs);
  }
}

/*
 *	xmpi_tr_cleanup
 *
 *	Function:	- removes temporary trace file, if any
 *	Accepts:	- parent widget
 */
void
xmpi_tr_cleanup()
{
  if (tempfile[0]) {
    unlink(tempfile);
    tempfile[0] = 0;
  }
}
