/*
 *  nextpnr -- Next Generation Place and Route
 *
 *  Copyright (C) 2018  Claire Xenia Wolf <claire@yosyshq.com>
 *
 *  Permission to use, copy, modify, and/or distribute this software for any
 *  purpose with or without fee is hereby granted, provided that the above
 *  copyright notice and this permission notice appear in all copies.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 *  WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 *  ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 *  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 *  ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 *  OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <boost/algorithm/string.hpp>
#include <iostream>

#include "gfx.h"

NEXTPNR_NAMESPACE_BEGIN

// LUTs
const float lut_w = 0.6732 - 0.6386;
const float lut_h = 0.9392 - 0.9074;
const float lut_x = 0.6386;
const float lut_y[] = {1. - 0.9392, 1. - 0.8870, 1. - 0.7864, 1. - 0.7321,
                       1. - 0.6399, 1. - 0.5847, 1. - 0.5068, 1. - 0.4503};
const float dff_w = 0.0580;
const float dff_x = 0.6821;
const float grp_lut_w = 0.1399;
const float grp_lut_h = 0.0954;
const float grp_lut_x = 0.6284;
const float grp_lut_y[] = {1. - 0.9440, 1. - 0.7915, 1. - 0.6442, 1. - 0.5101};

// mux
const float mux_w = 0.8134 - 0.7899;
const float mux_f = 0.9450 - 0.9358;
const float mux_h = grp_lut_h;

const float mux2lut5_x = 0.7900;
const float mux2lut5_y[] = {grp_lut_y[0], grp_lut_y[1], grp_lut_y[2], grp_lut_y[3]};
const float mux2lut6_x = 0.8378;
const float mux2lut6_y[] = {1. - 0.9261, 1. - 0.6205};
const float mux2lut7_x = 0.8859;
const float mux2lut7_y = 1. - 0.7870;
const float mux2lut8_x = 0.9337;
const float mux2lut8_y = 1. - 0.8098;

// pip
enum CruSide
{
    Top,
    Bottom,
    Left,
    Right,
    Center
};
const float cru_x = 0.2568;
const float cru_y = 1. - 0.9783;
const float cru_w = 0.6010 - cru_x;
const float cru_h = 1. - cru_y - 0.3742;

const float lut_A_off = 1. - 0.9107 - lut_y[0];
const float lut_D_off = lut_h - lut_A_off;
const float lut_B_off = lut_A_off - (lut_h - lut_D_off) / 3.;
const float lut_C_off = lut_D_off + (lut_h - lut_D_off) / 3.;

const float right_wire_dist = (grp_lut_y[1] - grp_lut_y[0] - grp_lut_h) / 11.;
const float left_wire_dist = cru_h / 100.;
const float top_wire_dist = cru_w / 100.;
const float clk_ce_set_vdist = (lut_y[1] - lut_y[0] - lut_h) / 4.;

const float sn_dist = cru_x / 125.;
const float ew_dist = (1. - cru_y - cru_h) / 130.;
const float wrap_len = 0.02f;
const float spine_pip_off = 0.11f;

const float io_x = cru_x + cru_w + 0.1;
const float io_w = (1. - io_x) / 3.;
const float io_gap = 0.03;
const float io_h = (cru_h - 4. * io_gap) / 2.;
const float io_y = cru_y + io_gap;

const float ios_scl = 0.5;
const float ios_h = ios_scl * io_h;
const float ios_w = ios_scl * io_w;
const float ios_gap_y = io_gap;
const float ios_gap_x = io_gap * 1.4;
const float ios_x = io_x;
const float ios_y = ios_scl * io_y;

const dict<IdString, float> portPoint = {
        {id_O, 3. * io_h / 4.},
        {id_I, 2. * io_h / 4.},
        {id_OEN, 1. * io_h / 4.},
};

const dict<IdString, std::vector<std::tuple<float, float, float, float>>> portSign = {
        {id_O,
         {{io_h / 14. * 1.33, portPoint.at(id_O) + io_h / 14., io_h / 14. * 1.66, portPoint.at(id_O) + io_h / 14.},
          {io_h / 14. * 1.66, portPoint.at(id_O) + io_h / 14., io_h / 14. * 2., portPoint.at(id_O) + io_h / 14. * 0.6},
          {io_h / 14. * 2., portPoint.at(id_O) + io_h / 14. * 0.6, io_h / 14. * 2.,
           portPoint.at(id_O) - io_h / 14. * 0.6},
          {io_h / 14. * 2., portPoint.at(id_O) - io_h / 14. * 0.6, io_h / 14. * 1.66, portPoint.at(id_O) - io_h / 14.},
          {io_h / 14. * 1.66, portPoint.at(id_O) - io_h / 14., io_h / 14. * 1.33, portPoint.at(id_O) - io_h / 14.},
          {io_h / 14. * 1.33, portPoint.at(id_O) - io_h / 14., io_h / 14. * 1., portPoint.at(id_O) - io_h / 14. * 0.6},
          {io_h / 14. * 1., portPoint.at(id_O) - io_h / 14. * 0.6, io_h / 14. * 1.,
           portPoint.at(id_O) + io_h / 14. * 0.6},
          {io_h / 14. * 1., portPoint.at(id_O) + io_h / 14. * 0.6, io_h / 14. * 1.33,
           portPoint.at(id_O) + io_h / 14.}}},
        {id_I,
         {{io_h / 14., portPoint.at(id_I) + io_h / 14., 2. * io_h / 14., portPoint.at(id_I) + io_h / 14.},
          {io_h / 14. * 1.5, portPoint.at(id_I) + io_h / 14., 1. * io_h / 14. * 1.5, portPoint.at(id_I) - io_h / 14.},
          {io_h / 14., portPoint.at(id_I) - io_h / 14., 2. * io_h / 14., portPoint.at(id_I) - io_h / 14.}}},
        {id_OEN,
         {{io_h / 14. * 1.33, portPoint.at(id_OEN) + io_h / 14., io_h / 14. * 1.66, portPoint.at(id_OEN) + io_h / 14.},
          {io_h / 14. * 1.66, portPoint.at(id_OEN) + io_h / 14., io_h / 14. * 2.,
           portPoint.at(id_OEN) + io_h / 14. * 0.6},
          {io_h / 14. * 2., portPoint.at(id_OEN) + io_h / 14. * 0.6, io_h / 14. * 2.,
           portPoint.at(id_OEN) - io_h / 14. * 0.6},
          {io_h / 14. * 2., portPoint.at(id_OEN) - io_h / 14. * 0.6, io_h / 14. * 1.66,
           portPoint.at(id_OEN) - io_h / 14.},
          {io_h / 14. * 1.66, portPoint.at(id_OEN) - io_h / 14., io_h / 14. * 1.33, portPoint.at(id_OEN) - io_h / 14.},
          {io_h / 14. * 1.33, portPoint.at(id_OEN) - io_h / 14., io_h / 14. * 1.,
           portPoint.at(id_OEN) - io_h / 14. * 0.6},
          {io_h / 14. * 1., portPoint.at(id_OEN) - io_h / 14. * 0.6, io_h / 14. * 1.,
           portPoint.at(id_OEN) + io_h / 14. * 0.6},
          {io_h / 14. * 1., portPoint.at(id_OEN) + io_h / 14. * 0.6, io_h / 14. * 1.33,
           portPoint.at(id_OEN) + io_h / 14.},
          {io_h / 14. * 2.2, portPoint.at(id_OEN) + io_h / 14., io_h / 14. * 3.2, portPoint.at(id_OEN) + io_h / 14.},
          {io_h / 14. * 2.2, portPoint.at(id_OEN) + 0., io_h / 14. * 3.2, portPoint.at(id_OEN) + 0.},
          {io_h / 14. * 2.2, portPoint.at(id_OEN) - io_h / 14., io_h / 14. * 3.2, portPoint.at(id_OEN) - io_h / 14.},
          {io_h / 14. * 2.2, portPoint.at(id_OEN) + io_h / 14., io_h / 14. * 2.2, portPoint.at(id_OEN) - io_h / 14.}}},
};

const dict<IdString, float> spineY = {
        {id_SPINE0, 1. - 1. * ew_dist},  {id_SPINE1, 1. - 2. * ew_dist},  {id_SPINE2, 1. - 3. * ew_dist},
        {id_SPINE3, 1. - 4. * ew_dist},  {id_SPINE4, 1. - 5. * ew_dist},  {id_SPINE5, 1. - 6. * ew_dist},
        {id_SPINE6, 1. - 7. * ew_dist},  {id_SPINE7, 1. - 8. * ew_dist},  {id_SPINE8, 1. - 1. * ew_dist},
        {id_SPINE9, 1. - 2. * ew_dist},  {id_SPINE10, 1. - 3. * ew_dist}, {id_SPINE11, 1. - 4. * ew_dist},
        {id_SPINE12, 1. - 5. * ew_dist}, {id_SPINE13, 1. - 6. * ew_dist}, {id_SPINE14, 1. - 7. * ew_dist},
        {id_SPINE15, 1. - 8. * ew_dist}, {id_SPINE16, 1. - 1. * ew_dist}, {id_SPINE17, 1. - 2. * ew_dist},
        {id_SPINE18, 1. - 3. * ew_dist}, {id_SPINE19, 1. - 4. * ew_dist}, {id_SPINE20, 1. - 5. * ew_dist},
        {id_SPINE21, 1. - 6. * ew_dist}, {id_SPINE22, 1. - 7. * ew_dist}, {id_SPINE23, 1. - 8. * ew_dist},
        {id_SPINE24, 1. - 1. * ew_dist}, {id_SPINE25, 1. - 2. * ew_dist}, {id_SPINE26, 1. - 3. * ew_dist},
        {id_SPINE27, 1. - 4. * ew_dist}, {id_SPINE28, 1. - 5. * ew_dist}, {id_SPINE29, 1. - 6. * ew_dist},
        {id_SPINE30, 1. - 7. * ew_dist}, {id_SPINE31, 1. - 8. * ew_dist},
};

const dict<IdString, std::pair<CruSide, float>> pipPoint = {
        {id_X01, {Center, cru_y + 1. * cru_h / 9.}},
        {id_X02, {Center, cru_y + 2. * cru_h / 9.}},
        {id_X03, {Center, cru_y + 3. * cru_h / 9.}},
        {id_X04, {Center, cru_y + 4. * cru_h / 9.}},
        {id_X05, {Center, cru_y + 5. * cru_h / 9.}},
        {id_X06, {Center, cru_y + 6. * cru_h / 9.}},
        {id_X07, {Center, cru_y + 7. * cru_h / 9.}},
        {id_X08, {Center, cru_y + 8. * cru_h / 9.}},
        // LUT inputs
        {id_A0, {Right, lut_y[0] + lut_A_off}},
        {id_B0, {Right, lut_y[0] + lut_B_off}},
        {id_C0, {Right, lut_y[0] + lut_C_off}},
        {id_D0, {Right, lut_y[0] + lut_D_off}},
        {id_A1, {Right, lut_y[1] + lut_A_off}},
        {id_B1, {Right, lut_y[1] + lut_B_off}},
        {id_C1, {Right, lut_y[1] + lut_C_off}},
        {id_D1, {Right, lut_y[1] + lut_D_off}},
        {id_A2, {Right, lut_y[2] + lut_A_off}},
        {id_B2, {Right, lut_y[2] + lut_B_off}},
        {id_C2, {Right, lut_y[2] + lut_C_off}},
        {id_D2, {Right, lut_y[2] + lut_D_off}},
        {id_A3, {Right, lut_y[3] + lut_A_off}},
        {id_B3, {Right, lut_y[3] + lut_B_off}},
        {id_C3, {Right, lut_y[3] + lut_C_off}},
        {id_D3, {Right, lut_y[3] + lut_D_off}},
        {id_A4, {Right, lut_y[4] + lut_A_off}},
        {id_B4, {Right, lut_y[4] + lut_B_off}},
        {id_C4, {Right, lut_y[4] + lut_C_off}},
        {id_D4, {Right, lut_y[4] + lut_D_off}},
        {id_A5, {Right, lut_y[5] + lut_A_off}},
        {id_B5, {Right, lut_y[5] + lut_B_off}},
        {id_C5, {Right, lut_y[5] + lut_C_off}},
        {id_D5, {Right, lut_y[5] + lut_D_off}},
        {id_A6, {Right, lut_y[6] + lut_A_off}},
        {id_B6, {Right, lut_y[6] + lut_B_off}},
        {id_C6, {Right, lut_y[6] + lut_C_off}},
        {id_D6, {Right, lut_y[6] + lut_D_off}},
        {id_A7, {Right, lut_y[7] + lut_A_off}},
        {id_B7, {Right, lut_y[7] + lut_B_off}},
        {id_C7, {Right, lut_y[7] + lut_C_off}},
        {id_D7, {Right, lut_y[7] + lut_D_off}},
        // wires below LUT0
        {id_Q0, {Right, grp_lut_y[0] - right_wire_dist}},
        {id_F0, {Right, grp_lut_y[0] - 2. * right_wire_dist}},
        {id_OF3, {Right, grp_lut_y[0] - 3. * right_wire_dist}},
        // wires between LUT1 and LUT2
        {id_Q2, {Right, grp_lut_y[1] - right_wire_dist}},
        {id_F2, {Right, grp_lut_y[1] - 2. * right_wire_dist}},
        {id_OF2, {Right, grp_lut_y[1] - 3. * right_wire_dist}},
        {id_OF1, {Right, grp_lut_y[1] - 4. * right_wire_dist}},
        {id_OF0, {Right, grp_lut_y[1] - 5. * right_wire_dist}},
        {id_SEL1, {Right, grp_lut_y[1] - 6. * right_wire_dist}},
        {id_OF7, {Right, grp_lut_y[1] - 7. * right_wire_dist}},
        {id_SEL0, {Right, grp_lut_y[1] - 8. * right_wire_dist}},
        {id_F1, {Right, grp_lut_y[1] - 9. * right_wire_dist}},
        {id_Q1, {Right, grp_lut_y[1] - 10. * right_wire_dist}},
        // wires between LUT3 and LUT4
        {id_Q4, {Right, grp_lut_y[2] - right_wire_dist}},
        {id_F4, {Right, grp_lut_y[2] - 2. * right_wire_dist}},
        {id_OF4, {Right, grp_lut_y[2] - 3. * right_wire_dist}},
        {id_OF5, {Right, grp_lut_y[2] - 4. * right_wire_dist}},
        {id_SEL7, {Right, grp_lut_y[2] - 5. * right_wire_dist}},
        {id_SEL3, {Right, grp_lut_y[2] - 6. * right_wire_dist}},
        {id_SEL2, {Right, grp_lut_y[2] - 7. * right_wire_dist}},
        {id_F3, {Right, grp_lut_y[2] - 8. * right_wire_dist}},
        {id_Q3, {Right, grp_lut_y[2] - 9. * right_wire_dist}},
        // wires between LUT5 and LUT6
        {id_F6, {Right, grp_lut_y[3] - right_wire_dist}},
        {id_SEL5, {Right, grp_lut_y[3] - 2. * right_wire_dist}},
        {id_SEL4, {Right, grp_lut_y[3] - 4. * right_wire_dist}},
        {id_F5, {Right, grp_lut_y[3] - 5. * right_wire_dist}},
        {id_Q5, {Right, grp_lut_y[3] - 6. * right_wire_dist}},
        // Q6, Q7 --- IOB
        {id_Q6, {Right, grp_lut_y[3] + grp_lut_h * 0.33}},
        {id_Q7, {Right, grp_lut_y[3] + grp_lut_h * 0.66}},
        // wires above LUT7
        {id_F7, {Right, grp_lut_y[3] + grp_lut_h + right_wire_dist}},
        {id_SEL6, {Right, grp_lut_y[3] + grp_lut_h + 2. * right_wire_dist}},
        {id_OF6, {Right, grp_lut_y[3] + grp_lut_h + 3. * right_wire_dist}},
        // DI0-5
        {id_DI5, {Right, cru_y + cru_h - 0.5 * right_wire_dist}},
        {id_DI4, {Right, cru_y + cru_h - 1. * right_wire_dist}},
        {id_DI3, {Right, cru_y + cru_h - 1.5 * right_wire_dist}},
        {id_DI2, {Right, cru_y + cru_h - 2. * right_wire_dist}},
        {id_DI1, {Right, cru_y + cru_h - 2.5 * right_wire_dist}},
        {id_DI0, {Right, cru_y + cru_h - 3. * right_wire_dist}},
        // Q6
        // CLK, CE, SET-RESET
        {id_CLK0, {Right, lut_y[1] - clk_ce_set_vdist}},
        {id_CE0, {Right, lut_y[1] - 2. * clk_ce_set_vdist}},
        {id_LSR0, {Right, lut_y[1] - 3. * clk_ce_set_vdist}},
        {id_CLK1, {Right, lut_y[3] - clk_ce_set_vdist}},
        {id_CE1, {Right, lut_y[3] - 2. * clk_ce_set_vdist}},
        {id_LSR1, {Right, lut_y[3] - 3. * clk_ce_set_vdist}},
        {id_CLK2, {Right, lut_y[5] - clk_ce_set_vdist}},
        {id_CE2, {Right, lut_y[5] - 2. * clk_ce_set_vdist}},
        {id_LSR2, {Right, lut_y[5] - 3. * clk_ce_set_vdist}},
        // SN
        // 1 hop
        {id_S100, {Left, cru_y + 1. * left_wire_dist}},
        {id_S101, {Left, cru_y + 2. * left_wire_dist}},
        {id_S130, {Left, cru_y + 3. * left_wire_dist}},
        {id_S131, {Left, cru_y + 4. * left_wire_dist}},
        {id_N101, {Left, cru_y + 5. * left_wire_dist}},
        {id_N100, {Left, cru_y + 6. * left_wire_dist}},
        {id_N131, {Left, cru_y + 7. * left_wire_dist}},
        {id_N130, {Left, cru_y + 8. * left_wire_dist}},
        // 1 hop SN
        {id_N111, {Left, cru_y + 9. * left_wire_dist}},
        {id_SN10, {Left, cru_y + 10. * left_wire_dist}},
        {id_S111, {Left, cru_y + 11. * left_wire_dist}},
        {id_N121, {Left, cru_y + 12. * left_wire_dist}},
        {id_SN20, {Left, cru_y + 13. * left_wire_dist}},
        {id_S121, {Left, cru_y + 14. * left_wire_dist}},
        // 2 hop
        {id_S200, {Left, cru_y + 15. * left_wire_dist}},
        {id_S201, {Left, cru_y + 16. * left_wire_dist}},
        {id_N202, {Left, cru_y + 17. * left_wire_dist}},
        {id_S202, {Left, cru_y + 18. * left_wire_dist}},
        {id_N201, {Left, cru_y + 19. * left_wire_dist}},
        {id_N200, {Left, cru_y + 20. * left_wire_dist}},

        {id_S210, {Left, cru_y + 21. * left_wire_dist}},
        {id_S211, {Left, cru_y + 22. * left_wire_dist}},
        {id_N212, {Left, cru_y + 23. * left_wire_dist}},
        {id_S212, {Left, cru_y + 24. * left_wire_dist}},
        {id_N211, {Left, cru_y + 25. * left_wire_dist}},
        {id_N210, {Left, cru_y + 26. * left_wire_dist}},

        {id_S220, {Left, cru_y + 27. * left_wire_dist}},
        {id_S221, {Left, cru_y + 28. * left_wire_dist}},
        {id_N222, {Left, cru_y + 29. * left_wire_dist}},
        {id_S222, {Left, cru_y + 30. * left_wire_dist}},
        {id_N221, {Left, cru_y + 31. * left_wire_dist}},
        {id_N220, {Left, cru_y + 32. * left_wire_dist}},

        {id_S230, {Left, cru_y + 33. * left_wire_dist}},
        {id_S231, {Left, cru_y + 34. * left_wire_dist}},
        {id_N232, {Left, cru_y + 35. * left_wire_dist}},
        {id_S232, {Left, cru_y + 36. * left_wire_dist}},
        {id_N231, {Left, cru_y + 37. * left_wire_dist}},
        {id_N230, {Left, cru_y + 38. * left_wire_dist}},

        {id_S240, {Left, cru_y + 39. * left_wire_dist}},
        {id_S241, {Left, cru_y + 40. * left_wire_dist}},
        {id_N242, {Left, cru_y + 41. * left_wire_dist}},
        {id_S242, {Left, cru_y + 42. * left_wire_dist}},
        {id_N241, {Left, cru_y + 43. * left_wire_dist}},
        {id_N240, {Left, cru_y + 44. * left_wire_dist}},

        {id_S250, {Left, cru_y + 45. * left_wire_dist}},
        {id_S251, {Left, cru_y + 46. * left_wire_dist}},
        {id_N252, {Left, cru_y + 47. * left_wire_dist}},
        {id_S252, {Left, cru_y + 48. * left_wire_dist}},
        {id_N251, {Left, cru_y + 49. * left_wire_dist}},
        {id_N250, {Left, cru_y + 50. * left_wire_dist}},

        {id_S260, {Left, cru_y + 51. * left_wire_dist}},
        {id_S261, {Left, cru_y + 52. * left_wire_dist}},
        {id_N262, {Left, cru_y + 53. * left_wire_dist}},
        {id_S262, {Left, cru_y + 54. * left_wire_dist}},
        {id_N261, {Left, cru_y + 55. * left_wire_dist}},
        {id_N260, {Left, cru_y + 56. * left_wire_dist}},

        {id_S270, {Left, cru_y + 57. * left_wire_dist}},
        {id_S271, {Left, cru_y + 58. * left_wire_dist}},
        {id_N272, {Left, cru_y + 59. * left_wire_dist}},
        {id_S272, {Left, cru_y + 60. * left_wire_dist}},
        {id_N271, {Left, cru_y + 61. * left_wire_dist}},
        {id_N270, {Left, cru_y + 62. * left_wire_dist}},

        // Clocks
        {id_GT10, {Left, cru_y + 63. * left_wire_dist}},
        {id_GT00, {Left, cru_y + 68. * left_wire_dist}},

        // 4 hop
        {id_N808, {Left, cru_y + 73. * left_wire_dist}},
        {id_S800, {Left, cru_y + 74. * left_wire_dist}},
        {id_S804, {Left, cru_y + 75. * left_wire_dist}},
        {id_N804, {Left, cru_y + 76. * left_wire_dist}},
        {id_N800, {Left, cru_y + 77. * left_wire_dist}},
        {id_S808, {Left, cru_y + 78. * left_wire_dist}},

        {id_N818, {Left, cru_y + 79. * left_wire_dist}},
        {id_S810, {Left, cru_y + 80. * left_wire_dist}},
        {id_S814, {Left, cru_y + 81. * left_wire_dist}},
        {id_N814, {Left, cru_y + 82. * left_wire_dist}},
        {id_N810, {Left, cru_y + 83. * left_wire_dist}},
        {id_S818, {Left, cru_y + 84. * left_wire_dist}},

        {id_N828, {Left, cru_y + 85. * left_wire_dist}},
        {id_S820, {Left, cru_y + 86. * left_wire_dist}},
        {id_S824, {Left, cru_y + 87. * left_wire_dist}},
        {id_N824, {Left, cru_y + 88. * left_wire_dist}},
        {id_N820, {Left, cru_y + 89. * left_wire_dist}},
        {id_S828, {Left, cru_y + 90. * left_wire_dist}},

        {id_N838, {Left, cru_y + 91. * left_wire_dist}},
        {id_S830, {Left, cru_y + 92. * left_wire_dist}},
        {id_S834, {Left, cru_y + 93. * left_wire_dist}},
        {id_N834, {Left, cru_y + 94. * left_wire_dist}},
        {id_N830, {Left, cru_y + 95. * left_wire_dist}},
        {id_S838, {Left, cru_y + 96. * left_wire_dist}},

        // EW
        // 1 hop
        {id_E101, {Top, cru_x + 1. * top_wire_dist}},
        {id_E100, {Top, cru_x + 2. * top_wire_dist}},
        {id_E131, {Top, cru_x + 3. * top_wire_dist}},
        {id_E130, {Top, cru_x + 4. * top_wire_dist}},
        {id_W100, {Top, cru_x + 5. * top_wire_dist}},
        {id_W101, {Top, cru_x + 6. * top_wire_dist}},
        {id_W130, {Top, cru_x + 7. * top_wire_dist}},
        {id_W131, {Top, cru_x + 8. * top_wire_dist}},
        // 1 hop EW
        {id_E111, {Top, cru_x + 9. * top_wire_dist}},
        {id_EW10, {Top, cru_x + 10. * top_wire_dist}},
        {id_W111, {Top, cru_x + 11. * top_wire_dist}},
        {id_E121, {Top, cru_x + 12. * top_wire_dist}},
        {id_EW20, {Top, cru_x + 13. * top_wire_dist}},
        {id_W121, {Top, cru_x + 14. * top_wire_dist}},
        // 2 hop
        {id_E202, {Top, cru_x + 15. * top_wire_dist}},
        {id_E201, {Top, cru_x + 16. * top_wire_dist}},
        {id_W200, {Top, cru_x + 17. * top_wire_dist}},
        {id_E200, {Top, cru_x + 18. * top_wire_dist}},
        {id_W201, {Top, cru_x + 19. * top_wire_dist}},
        {id_W202, {Top, cru_x + 20. * top_wire_dist}},

        {id_E212, {Top, cru_x + 21. * top_wire_dist}},
        {id_E211, {Top, cru_x + 22. * top_wire_dist}},
        {id_W210, {Top, cru_x + 23. * top_wire_dist}},
        {id_E210, {Top, cru_x + 24. * top_wire_dist}},
        {id_W211, {Top, cru_x + 25. * top_wire_dist}},
        {id_W212, {Top, cru_x + 26. * top_wire_dist}},

        {id_E222, {Top, cru_x + 27. * top_wire_dist}},
        {id_E221, {Top, cru_x + 28. * top_wire_dist}},
        {id_W220, {Top, cru_x + 29. * top_wire_dist}},
        {id_E220, {Top, cru_x + 30. * top_wire_dist}},
        {id_W221, {Top, cru_x + 31. * top_wire_dist}},
        {id_W222, {Top, cru_x + 32. * top_wire_dist}},

        {id_E232, {Top, cru_x + 33. * top_wire_dist}},
        {id_E231, {Top, cru_x + 34. * top_wire_dist}},
        {id_W230, {Top, cru_x + 35. * top_wire_dist}},
        {id_E230, {Top, cru_x + 36. * top_wire_dist}},
        {id_W231, {Top, cru_x + 37. * top_wire_dist}},
        {id_W232, {Top, cru_x + 38. * top_wire_dist}},

        {id_E242, {Top, cru_x + 39. * top_wire_dist}},
        {id_E241, {Top, cru_x + 40. * top_wire_dist}},
        {id_W240, {Top, cru_x + 41. * top_wire_dist}},
        {id_E240, {Top, cru_x + 42. * top_wire_dist}},
        {id_W241, {Top, cru_x + 43. * top_wire_dist}},
        {id_W242, {Top, cru_x + 44. * top_wire_dist}},

        {id_E252, {Top, cru_x + 45. * top_wire_dist}},
        {id_E251, {Top, cru_x + 46. * top_wire_dist}},
        {id_W250, {Top, cru_x + 47. * top_wire_dist}},
        {id_E250, {Top, cru_x + 48. * top_wire_dist}},
        {id_W251, {Top, cru_x + 49. * top_wire_dist}},
        {id_W252, {Top, cru_x + 50. * top_wire_dist}},

        {id_E262, {Top, cru_x + 51. * top_wire_dist}},
        {id_E261, {Top, cru_x + 52. * top_wire_dist}},
        {id_W260, {Top, cru_x + 53. * top_wire_dist}},
        {id_E260, {Top, cru_x + 54. * top_wire_dist}},
        {id_W261, {Top, cru_x + 55. * top_wire_dist}},
        {id_W262, {Top, cru_x + 56. * top_wire_dist}},

        {id_E272, {Top, cru_x + 57. * top_wire_dist}},
        {id_E271, {Top, cru_x + 58. * top_wire_dist}},
        {id_W270, {Top, cru_x + 59. * top_wire_dist}},
        {id_E270, {Top, cru_x + 60. * top_wire_dist}},
        {id_W271, {Top, cru_x + 61. * top_wire_dist}},
        {id_W272, {Top, cru_x + 62. * top_wire_dist}},

        // Global taps -> bracnhes
        {id_GBO0, {Top, cru_x + 63. * top_wire_dist}},
        {id_GB00, {Top, cru_x + 64. * top_wire_dist}},
        {id_GB10, {Top, cru_x + 65. * top_wire_dist}},
        {id_GB20, {Top, cru_x + 66. * top_wire_dist}},
        {id_GB30, {Top, cru_x + 67. * top_wire_dist}},
        {id_GBO1, {Top, cru_x + 68. * top_wire_dist}},
        {id_GB40, {Top, cru_x + 68. * top_wire_dist}},
        {id_GB50, {Top, cru_x + 69. * top_wire_dist}},
        {id_GB60, {Top, cru_x + 70. * top_wire_dist}},
        {id_GB70, {Top, cru_x + 71. * top_wire_dist}},

        // 4 hop
        {id_E808, {Top, cru_x + 72. * top_wire_dist}},
        {id_W800, {Top, cru_x + 73. * top_wire_dist}},
        {id_W804, {Top, cru_x + 74. * top_wire_dist}},
        {id_E804, {Top, cru_x + 75. * top_wire_dist}},
        {id_E800, {Top, cru_x + 76. * top_wire_dist}},
        {id_W808, {Top, cru_x + 77. * top_wire_dist}},

        {id_E818, {Top, cru_x + 78. * top_wire_dist}},
        {id_W810, {Top, cru_x + 79. * top_wire_dist}},
        {id_W814, {Top, cru_x + 80. * top_wire_dist}},
        {id_E814, {Top, cru_x + 81. * top_wire_dist}},
        {id_E810, {Top, cru_x + 82. * top_wire_dist}},
        {id_W818, {Top, cru_x + 83. * top_wire_dist}},

        {id_E828, {Top, cru_x + 84. * top_wire_dist}},
        {id_W820, {Top, cru_x + 85. * top_wire_dist}},
        {id_W824, {Top, cru_x + 86. * top_wire_dist}},
        {id_E824, {Top, cru_x + 87. * top_wire_dist}},
        {id_E820, {Top, cru_x + 88. * top_wire_dist}},
        {id_W828, {Top, cru_x + 89. * top_wire_dist}},

        {id_E838, {Top, cru_x + 90. * top_wire_dist}},
        {id_W830, {Top, cru_x + 91. * top_wire_dist}},
        {id_W834, {Top, cru_x + 92. * top_wire_dist}},
        {id_E834, {Top, cru_x + 93. * top_wire_dist}},
        {id_E830, {Top, cru_x + 94. * top_wire_dist}},
        {id_W838, {Top, cru_x + 95. * top_wire_dist}},

};

// wire
const std::vector<IdString> decalless_wires = {id_X01, id_X02, id_X03, id_X04, id_X05, id_X06, id_X07};

const float clk_ce_set_hdist = dff_w / 4.;
const float dff_f_x = (grp_lut_x + grp_lut_w + dff_x + dff_w) / 2.;
const float mux5i_x = (grp_lut_x + grp_lut_w + mux2lut5_x) / 2.;

// id, {x1, y1, x2, y2}
const dict<IdString, std::vector<std::tuple<float, float, float, float>>> sliceLocalWires = {
        // dff
        {id_CLK0,
         {{cru_x + cru_w, pipPoint.at(id_CLK0).second, dff_x + clk_ce_set_hdist, pipPoint.at(id_CLK0).second},
          {dff_x + clk_ce_set_hdist, lut_y[1], dff_x + clk_ce_set_hdist, lut_y[0] + lut_h}}},
        {id_CE0,
         {{cru_x + cru_w, pipPoint.at(id_CE0).second, dff_x + 2. * clk_ce_set_hdist, pipPoint.at(id_CE0).second},
          {dff_x + 2 * clk_ce_set_hdist, lut_y[1], dff_x + 2. * clk_ce_set_hdist, lut_y[0] + lut_h}}},
        {id_LSR0,
         {{cru_x + cru_w, pipPoint.at(id_LSR0).second, dff_x + 3. * clk_ce_set_hdist, pipPoint.at(id_LSR0).second},
          {dff_x + 3 * clk_ce_set_hdist, lut_y[1], dff_x + 3. * clk_ce_set_hdist, lut_y[0] + lut_h}}},
        {id_CLK1,
         {{cru_x + cru_w, pipPoint.at(id_CLK1).second, dff_x + clk_ce_set_hdist, pipPoint.at(id_CLK1).second},
          {dff_x + clk_ce_set_hdist, lut_y[3], dff_x + clk_ce_set_hdist, lut_y[2] + lut_h}}},
        {id_CE1,
         {{cru_x + cru_w, pipPoint.at(id_CE1).second, dff_x + 2. * clk_ce_set_hdist, pipPoint.at(id_CE1).second},
          {dff_x + 2 * clk_ce_set_hdist, lut_y[3], dff_x + 2. * clk_ce_set_hdist, lut_y[2] + lut_h}}},
        {id_LSR1,
         {{cru_x + cru_w, pipPoint.at(id_LSR1).second, dff_x + 3. * clk_ce_set_hdist, pipPoint.at(id_LSR1).second},
          {dff_x + 3 * clk_ce_set_hdist, lut_y[3], dff_x + 3. * clk_ce_set_hdist, lut_y[2] + lut_h}}},
        {id_CLK2,
         {{cru_x + cru_w, pipPoint.at(id_CLK2).second, dff_x + clk_ce_set_hdist, pipPoint.at(id_CLK2).second},
          {dff_x + clk_ce_set_hdist, lut_y[5], dff_x + clk_ce_set_hdist, lut_y[4] + lut_h}}},
        {id_CE2,
         {{cru_x + cru_w, pipPoint.at(id_CE2).second, dff_x + 2. * clk_ce_set_hdist, pipPoint.at(id_CE2).second},
          {dff_x + 2 * clk_ce_set_hdist, lut_y[5], dff_x + 2. * clk_ce_set_hdist, lut_y[4] + lut_h}}},
        {id_LSR2,
         {{cru_x + cru_w, pipPoint.at(id_LSR2).second, dff_x + 3 * clk_ce_set_hdist, pipPoint.at(id_LSR2).second},
          {dff_x + 3 * clk_ce_set_hdist, lut_y[5], dff_x + 3. * clk_ce_set_hdist, lut_y[4] + lut_h}}},
        // lut
        {id_A0, {{cru_x + cru_w, lut_y[0] + lut_A_off, lut_x, lut_y[0] + lut_A_off}}},
        {id_B0, {{cru_x + cru_w, lut_y[0] + lut_B_off, lut_x, lut_y[0] + lut_B_off}}},
        {id_C0, {{cru_x + cru_w, lut_y[0] + lut_C_off, lut_x, lut_y[0] + lut_C_off}}},
        {id_D0, {{cru_x + cru_w, lut_y[0] + lut_D_off, lut_x, lut_y[0] + lut_D_off}}},
        {id_A1, {{cru_x + cru_w, lut_y[1] + lut_A_off, lut_x, lut_y[1] + lut_A_off}}},
        {id_B1, {{cru_x + cru_w, lut_y[1] + lut_B_off, lut_x, lut_y[1] + lut_B_off}}},
        {id_C1, {{cru_x + cru_w, lut_y[1] + lut_C_off, lut_x, lut_y[1] + lut_C_off}}},
        {id_D1, {{cru_x + cru_w, lut_y[1] + lut_D_off, lut_x, lut_y[1] + lut_D_off}}},
        {id_A2, {{cru_x + cru_w, lut_y[2] + lut_A_off, lut_x, lut_y[2] + lut_A_off}}},
        {id_B2, {{cru_x + cru_w, lut_y[2] + lut_B_off, lut_x, lut_y[2] + lut_B_off}}},
        {id_C2, {{cru_x + cru_w, lut_y[2] + lut_C_off, lut_x, lut_y[2] + lut_C_off}}},
        {id_D2, {{cru_x + cru_w, lut_y[2] + lut_D_off, lut_x, lut_y[2] + lut_D_off}}},
        {id_A3, {{cru_x + cru_w, lut_y[3] + lut_A_off, lut_x, lut_y[3] + lut_A_off}}},
        {id_B3, {{cru_x + cru_w, lut_y[3] + lut_B_off, lut_x, lut_y[3] + lut_B_off}}},
        {id_C3, {{cru_x + cru_w, lut_y[3] + lut_C_off, lut_x, lut_y[3] + lut_C_off}}},
        {id_D3, {{cru_x + cru_w, lut_y[3] + lut_D_off, lut_x, lut_y[3] + lut_D_off}}},
        {id_A4, {{cru_x + cru_w, lut_y[4] + lut_A_off, lut_x, lut_y[4] + lut_A_off}}},
        {id_B4, {{cru_x + cru_w, lut_y[4] + lut_B_off, lut_x, lut_y[4] + lut_B_off}}},
        {id_C4, {{cru_x + cru_w, lut_y[4] + lut_C_off, lut_x, lut_y[4] + lut_C_off}}},
        {id_D4, {{cru_x + cru_w, lut_y[4] + lut_D_off, lut_x, lut_y[4] + lut_D_off}}},
        {id_A5, {{cru_x + cru_w, lut_y[5] + lut_A_off, lut_x, lut_y[5] + lut_A_off}}},
        {id_B5, {{cru_x + cru_w, lut_y[5] + lut_B_off, lut_x, lut_y[5] + lut_B_off}}},
        {id_C5, {{cru_x + cru_w, lut_y[5] + lut_C_off, lut_x, lut_y[5] + lut_C_off}}},
        {id_D5, {{cru_x + cru_w, lut_y[5] + lut_D_off, lut_x, lut_y[5] + lut_D_off}}},
        {id_A6, {{cru_x + cru_w, lut_y[6] + lut_A_off, lut_x, lut_y[6] + lut_A_off}}},
        {id_B6, {{cru_x + cru_w, lut_y[6] + lut_B_off, lut_x, lut_y[6] + lut_B_off}}},
        {id_C6, {{cru_x + cru_w, lut_y[6] + lut_C_off, lut_x, lut_y[6] + lut_C_off}}},
        {id_D6, {{cru_x + cru_w, lut_y[6] + lut_D_off, lut_x, lut_y[6] + lut_D_off}}},
        {id_A7, {{cru_x + cru_w, lut_y[7] + lut_A_off, lut_x, lut_y[7] + lut_A_off}}},
        {id_B7, {{cru_x + cru_w, lut_y[7] + lut_B_off, lut_x, lut_y[7] + lut_B_off}}},
        {id_C7, {{cru_x + cru_w, lut_y[7] + lut_C_off, lut_x, lut_y[7] + lut_C_off}}},
        {id_D7, {{cru_x + cru_w, lut_y[7] + lut_D_off, lut_x, lut_y[7] + lut_D_off}}},
        // wires below LUT0
        {id_Q0,
         {{cru_x + cru_w, grp_lut_y[0] - right_wire_dist, dff_f_x, grp_lut_y[0] - right_wire_dist},
          {dff_f_x, grp_lut_y[0] - right_wire_dist, dff_f_x, lut_y[0] + lut_h / 2.},
          {dff_f_x, lut_y[0] + lut_h / 2., dff_x + dff_w, lut_y[0] + lut_h / 2.}}},
        {id_F0,
         {{cru_x + cru_w, grp_lut_y[0] - 2. * right_wire_dist, (lut_x + lut_w + dff_x) / 2.,
           grp_lut_y[0] - 2. * right_wire_dist},
          {(lut_x + lut_w + dff_x) / 2., grp_lut_y[0] - 2. * right_wire_dist, (lut_x + lut_w + dff_x) / 2.,
           lut_y[0] + lut_h / 2.},
          {lut_x + lut_w, lut_y[0] + lut_h / 2., dff_x, lut_y[0] + lut_h / 2.}}},
        {id_I0MUX0,
         {{(lut_x + lut_w + dff_x) / 2., grp_lut_y[0] - 2. * right_wire_dist, mux5i_x,
           grp_lut_y[0] - 2. * right_wire_dist},
          {mux5i_x, grp_lut_y[0] - 2. * right_wire_dist, mux5i_x, lut_y[0] + lut_h / 2.},
          {mux5i_x, lut_y[0] + lut_h / 2., mux2lut5_x, lut_y[0] + lut_h / 2.}}},
        {id_OF3,
         {{cru_x + cru_w, grp_lut_y[0] - 3. * right_wire_dist, mux2lut7_x + 4. / 3. * mux_w,
           grp_lut_y[0] - 3. * right_wire_dist},
          {mux2lut7_x + 4. / 3. * mux_w, grp_lut_y[0] - 3. * right_wire_dist, mux2lut7_x + 4. / 3. * mux_w,
           mux2lut7_y + mux_h / 2.},
          {mux2lut7_x + 4. / 3. * mux_w, mux2lut7_y + mux_h / 2., mux2lut7_x + mux_w, mux2lut7_y + mux_h / 2.}}},
        {id_I1MUX7, {{mux2lut7_x + 4. / 3. * mux_w, mux2lut7_y + mux_h / 2., mux2lut8_x, mux2lut7_y + mux_h / 2.}}},
        // wires between LUT1 and LUT2
        {id_Q1,
         {{cru_x + cru_w, grp_lut_y[1] - 10. * right_wire_dist, dff_f_x, grp_lut_y[1] - 10. * right_wire_dist},
          {dff_f_x, grp_lut_y[1] - 10. * right_wire_dist, dff_f_x, lut_y[1] + lut_h / 2.},
          {dff_f_x, lut_y[1] + lut_h / 2., dff_x + dff_w, lut_y[1] + lut_h / 2.}}},
        {id_F1,
         {{cru_x + cru_w, grp_lut_y[1] - 9. * right_wire_dist, (lut_x + lut_w + dff_x) / 2.,
           grp_lut_y[1] - 9. * right_wire_dist},
          {(lut_x + lut_w + dff_x) / 2., grp_lut_y[1] - 9. * right_wire_dist, (lut_x + lut_w + dff_x) / 2.,
           lut_y[1] + lut_h / 2.},
          {lut_x + lut_w, lut_y[1] + lut_h / 2., dff_x, lut_y[1] + lut_h / 2.}}},
        {id_I1MUX0,
         {{(lut_x + lut_w + dff_x) / 2., grp_lut_y[1] - 9. * right_wire_dist, mux5i_x,
           grp_lut_y[1] - 9. * right_wire_dist},
          {mux5i_x, grp_lut_y[1] - 9. * right_wire_dist, mux5i_x, lut_y[1] + lut_h / 2.},
          {mux5i_x, lut_y[1] + lut_h / 2., mux2lut5_x, lut_y[1] + lut_h / 2.}}},
        {id_SEL0,
         {{cru_x + cru_w, grp_lut_y[1] - 8. * right_wire_dist, mux2lut5_x + mux_w / 2.,
           grp_lut_y[1] - 8. * right_wire_dist},
          {mux2lut5_x + mux_w / 2., grp_lut_y[1] - 8. * right_wire_dist, mux2lut5_x + mux_w / 2.,
           mux2lut5_y[0] + mux_h - mux_f / 2.}}},
        {id_OF7,
         {{cru_x + cru_w, grp_lut_y[1] - 7. * right_wire_dist, mux2lut8_x + 4. / 3. * mux_w,
           grp_lut_y[1] - 7. * right_wire_dist},
          {mux2lut8_x + 4. / 3. * mux_w, grp_lut_y[1] - 7. * right_wire_dist, mux2lut8_x + 4. / 3. * mux_w,
           mux2lut8_y + mux_h / 2.},
          {mux2lut8_x + 4. / 3. * mux_w, mux2lut8_y + mux_h / 2., mux2lut8_x + mux_w, mux2lut8_y + mux_h / 2.}}},
        {id_SEL1,
         {{cru_x + cru_w, grp_lut_y[1] - 6. * right_wire_dist, mux2lut6_x + mux_w / 2.,
           grp_lut_y[1] - 6. * right_wire_dist},
          {mux2lut6_x + mux_w / 2., grp_lut_y[1] - 6. * right_wire_dist, mux2lut6_x + mux_w / 2.,
           mux2lut6_y[0] + mux_h - mux_f / 2.}}},
        {id_OF0,
         {{cru_x + cru_w, grp_lut_y[1] - 5. * right_wire_dist, mux2lut5_x + 4. / 3. * mux_w,
           grp_lut_y[1] - 5. * right_wire_dist},
          {mux2lut5_x + 4. / 3. * mux_w, grp_lut_y[1] - 5. * right_wire_dist, mux2lut5_x + 4. / 3. * mux_w,
           mux2lut5_y[0] + mux_h / 2.},
          {mux2lut5_x + 4. / 3. * mux_w, mux2lut5_y[0] + mux_h / 2., mux2lut5_x + mux_w, mux2lut5_y[0] + mux_h / 2.}}},
        {id_I1MUX1,
         {{mux2lut5_x + 4. / 3. * mux_w, mux2lut5_y[0] + mux_h / 2., mux2lut6_x, mux2lut5_y[0] + mux_h / 2.}}},
        {id_OF1,
         {{cru_x + cru_w, grp_lut_y[1] - 4. * right_wire_dist, mux2lut6_x + 4. / 3. * mux_w,
           grp_lut_y[1] - 4. * right_wire_dist},
          {mux2lut6_x + 4. / 3. * mux_w, grp_lut_y[1] - 4. * right_wire_dist, mux2lut6_x + 4. / 3. * mux_w,
           mux2lut6_y[0] + mux_h / 2.},
          {mux2lut6_x + 4. / 3. * mux_w, mux2lut6_y[0] + mux_h / 2., mux2lut6_x + mux_w, mux2lut6_y[0] + mux_h / 2.}}},
        {id_I1MUX3,
         {{mux2lut6_x + 4. / 3. * mux_w, grp_lut_y[1] - 4. * right_wire_dist, mux2lut6_x + 4. / 3. * mux_w,
           mux2lut7_y + mux_h * 1. / 4.},
          {mux2lut6_x + 4. / 3. * mux_w, mux2lut7_y + mux_h * 1. / 4., mux2lut7_x, mux2lut7_y + mux_h * 1. / 4.}}},
        {id_OF2,
         {{cru_x + cru_w, grp_lut_y[1] - 3. * right_wire_dist, mux2lut5_x + 5. / 3. * mux_w,
           grp_lut_y[1] - 3. * right_wire_dist},
          {mux2lut5_x + 5. / 3. * mux_w, grp_lut_y[1] - 3. * right_wire_dist, mux2lut5_x + 5. / 3. * mux_w,
           mux2lut5_y[1] + mux_h / 2.},
          {mux2lut5_x + 5. / 3. * mux_w, mux2lut5_y[1] + mux_h / 2., mux2lut5_x + mux_w, mux2lut5_y[1] + mux_h / 2.}}},
        {id_I0MUX1,
         {{mux2lut5_x + 5. / 3. * mux_w, grp_lut_y[1] - 3. * right_wire_dist, mux2lut5_x + 5. / 3. * mux_w,
           mux2lut6_y[0] + mux_h * 3. / 4.},
          {mux2lut5_x + 5. / 3. * mux_w, mux2lut6_y[0] + mux_h * 3. / 4., mux2lut6_x,
           mux2lut6_y[0] + mux_h * 3. / 4.}}},
        {id_F2,
         {{cru_x + cru_w, grp_lut_y[1] - 2. * right_wire_dist, (lut_x + lut_w + dff_x) / 2.,
           grp_lut_y[1] - 2. * right_wire_dist},
          {(lut_x + lut_w + dff_x) / 2., grp_lut_y[1] - 2. * right_wire_dist, (lut_x + lut_w + dff_x) / 2.,
           lut_y[2] + lut_h / 2.},
          {lut_x + lut_w, lut_y[2] + lut_h / 2., dff_x, lut_y[2] + lut_h / 2.}}},
        {id_I0MUX2,
         {{(lut_x + lut_w + dff_x) / 2., grp_lut_y[1] - 2. * right_wire_dist, mux5i_x,
           grp_lut_y[1] - 2. * right_wire_dist},
          {mux5i_x, grp_lut_y[1] - 2. * right_wire_dist, mux5i_x, lut_y[2] + lut_h / 2.},
          {mux5i_x, lut_y[2] + lut_h / 2., mux2lut5_x, lut_y[2] + lut_h / 2.}}},
        {id_Q2,
         {{cru_x + cru_w, grp_lut_y[1] - right_wire_dist, dff_f_x, grp_lut_y[1] - right_wire_dist},
          {dff_f_x, grp_lut_y[1] - right_wire_dist, dff_f_x, lut_y[2] + lut_h / 2.},
          {dff_f_x, lut_y[2] + lut_h / 2., dff_x + dff_w, lut_y[2] + lut_h / 2.}}},
        // wires between LUT3 and LUT4
        {id_Q3,
         {{cru_x + cru_w, grp_lut_y[2] - 9. * right_wire_dist, dff_f_x, grp_lut_y[2] - 9. * right_wire_dist},
          {dff_f_x, grp_lut_y[2] - 9. * right_wire_dist, dff_f_x, lut_y[3] + lut_h / 2.},
          {dff_f_x, lut_y[3] + lut_h / 2., dff_x + dff_w, lut_y[3] + lut_h / 2.}}},
        {id_F3,
         {{cru_x + cru_w, grp_lut_y[2] - 8. * right_wire_dist, (lut_x + lut_w + dff_x) / 2.,
           grp_lut_y[2] - 8. * right_wire_dist},
          {(lut_x + lut_w + dff_x) / 2., grp_lut_y[2] - 8. * right_wire_dist, (lut_x + lut_w + dff_x) / 2.,
           lut_y[3] + lut_h / 2.},
          {lut_x + lut_w, lut_y[3] + lut_h / 2., dff_x, lut_y[3] + lut_h / 2.}}},
        {id_I1MUX2,
         {{(lut_x + lut_w + dff_x) / 2., grp_lut_y[2] - 8. * right_wire_dist, mux5i_x,
           grp_lut_y[2] - 8. * right_wire_dist},
          {mux5i_x, grp_lut_y[2] - 8. * right_wire_dist, mux5i_x, lut_y[3] + lut_h / 2.},
          {mux5i_x, lut_y[3] + lut_h / 2., mux2lut5_x, lut_y[3] + lut_h / 2.}}},
        {id_SEL2,
         {{cru_x + cru_w, grp_lut_y[2] - 7. * right_wire_dist, mux2lut5_x + mux_w / 2.,
           grp_lut_y[2] - 7. * right_wire_dist},
          {mux2lut5_x + mux_w / 2., grp_lut_y[2] - 7. * right_wire_dist, mux2lut5_x + mux_w / 2.,
           mux2lut5_y[1] + mux_h - mux_f / 2.}}},
        {id_SEL3,
         {{cru_x + cru_w, grp_lut_y[2] - 6. * right_wire_dist, mux2lut7_x + mux_w / 2.,
           grp_lut_y[2] - 6. * right_wire_dist},
          {mux2lut7_x + mux_w / 2., grp_lut_y[2] - 6. * right_wire_dist, mux2lut7_x + mux_w / 2.,
           mux2lut7_y + mux_h - mux_f / 2.}}},
        {id_SEL7,
         {{cru_x + cru_w, grp_lut_y[2] - 5. * right_wire_dist, mux2lut8_x + mux_w / 2.,
           grp_lut_y[2] - 5. * right_wire_dist},
          {mux2lut8_x + mux_w / 2., grp_lut_y[2] - 5. * right_wire_dist, mux2lut8_x + mux_w / 2.,
           mux2lut8_y + mux_h - mux_f / 2.}}},
        {id_OF5,
         {{cru_x + cru_w, grp_lut_y[2] - 4. * right_wire_dist, mux2lut6_x + 4. / 3. * mux_w,
           grp_lut_y[2] - 4. * right_wire_dist},
          {mux2lut6_x + 4. / 3. * mux_w, grp_lut_y[2] - 4. * right_wire_dist, mux2lut6_x + 4. / 3. * mux_w,
           mux2lut6_y[1] + mux_h / 2.},
          {mux2lut6_x + 4. / 3. * mux_w, mux2lut6_y[1] + mux_h / 2., mux2lut6_x + mux_w, mux2lut6_y[1] + mux_h / 2.}}},
        {id_I0MUX3,
         {{mux2lut6_x + 4. / 3. * mux_w, grp_lut_y[2] - 4. * right_wire_dist, mux2lut6_x + 4. / 3. * mux_w,
           mux2lut7_y + mux_h * 3. / 4.},
          {mux2lut6_x + 4. / 3. * mux_w, mux2lut7_y + mux_h * 3. / 4., mux2lut7_x, mux2lut7_y + mux_h * 3. / 4.}}},
        {id_OF4,
         {{cru_x + cru_w, grp_lut_y[2] - 3. * right_wire_dist, mux2lut5_x + 4. / 3. * mux_w,
           grp_lut_y[2] - 3. * right_wire_dist},
          {mux2lut5_x + 4. / 3. * mux_w, grp_lut_y[2] - 3. * right_wire_dist, mux2lut5_x + 4. / 3. * mux_w,
           mux2lut5_y[2] + mux_h / 2.},
          {mux2lut5_x + 4. / 3. * mux_w, mux2lut5_y[2] + mux_h / 2., mux2lut5_x + mux_w, mux2lut5_y[2] + mux_h / 2.}}},
        {id_I1MUX5,
         {{mux2lut5_x + 4. / 3. * mux_w, mux2lut5_y[2] + mux_h / 2., mux2lut6_x, mux2lut5_y[2] + mux_h / 2.}}},
        {id_F4,
         {{cru_x + cru_w, grp_lut_y[2] - 2. * right_wire_dist, (lut_x + lut_w + dff_x) / 2.,
           grp_lut_y[2] - 2. * right_wire_dist},
          {(lut_x + lut_w + dff_x) / 2., grp_lut_y[2] - 2. * right_wire_dist, (lut_x + lut_w + dff_x) / 2.,
           lut_y[4] + lut_h / 2.},
          {lut_x + lut_w, lut_y[4] + lut_h / 2., dff_x, lut_y[4] + lut_h / 2.}}},
        {id_I0MUX4,
         {{(lut_x + lut_w + dff_x) / 2., grp_lut_y[2] - 2. * right_wire_dist, mux5i_x,
           grp_lut_y[2] - 2. * right_wire_dist},
          {mux5i_x, grp_lut_y[2] - 2. * right_wire_dist, mux5i_x, lut_y[4] + lut_h / 2.},
          {mux5i_x, lut_y[4] + lut_h / 2., mux2lut5_x, lut_y[4] + lut_h / 2.}}},
        {id_Q4,
         {{cru_x + cru_w, grp_lut_y[2] - right_wire_dist, dff_f_x, grp_lut_y[2] - right_wire_dist},
          {dff_f_x, grp_lut_y[2] - right_wire_dist, dff_f_x, lut_y[4] + lut_h / 2.},
          {dff_f_x, lut_y[4] + lut_h / 2., dff_x + dff_w, lut_y[4] + lut_h / 2.}}},
        // wires between LUT5 and LUT6
        {id_Q5,
         {{cru_x + cru_w, grp_lut_y[3] - 6. * right_wire_dist, dff_f_x, grp_lut_y[3] - 6. * right_wire_dist},
          {dff_f_x, grp_lut_y[3] - 6. * right_wire_dist, dff_f_x, lut_y[5] + lut_h / 2.},
          {dff_f_x, lut_y[5] + lut_h / 2., dff_x + dff_w, lut_y[5] + lut_h / 2.}}},
        {id_F5,
         {{cru_x + cru_w, grp_lut_y[3] - 5. * right_wire_dist, (lut_x + lut_w + dff_x) / 2.,
           grp_lut_y[3] - 5. * right_wire_dist},
          {(lut_x + lut_w + dff_x) / 2., grp_lut_y[3] - 5. * right_wire_dist, (lut_x + lut_w + dff_x) / 2.,
           lut_y[5] + lut_h / 2.},
          {lut_x + lut_w, lut_y[5] + lut_h / 2., dff_x, lut_y[5] + lut_h / 2.}}},
        {id_I1MUX4,
         {{(lut_x + lut_w + dff_x) / 2., grp_lut_y[3] - 5. * right_wire_dist, mux5i_x,
           grp_lut_y[3] - 5. * right_wire_dist},
          {mux5i_x, grp_lut_y[3] - 5. * right_wire_dist, mux5i_x, lut_y[5] + lut_h / 2.},
          {mux5i_x, lut_y[5] + lut_h / 2., mux2lut5_x, lut_y[5] + lut_h / 2.}}},
        {id_SEL4,
         {{cru_x + cru_w, grp_lut_y[3] - 4. * right_wire_dist, mux2lut5_x + mux_w / 2.,
           grp_lut_y[3] - 4. * right_wire_dist},
          {mux2lut5_x + mux_w / 2., grp_lut_y[3] - 4. * right_wire_dist, mux2lut5_x + mux_w / 2.,
           mux2lut5_y[2] + mux_h - mux_f / 2.}}},
        {id_SEL5,
         {{cru_x + cru_w, grp_lut_y[3] - 2. * right_wire_dist, mux2lut6_x + mux_w / 2.,
           grp_lut_y[3] - 2. * right_wire_dist},
          {mux2lut6_x + mux_w / 2., grp_lut_y[3] - 2. * right_wire_dist, mux2lut6_x + mux_w / 2.,
           mux2lut6_y[1] + mux_h - mux_f / 2.}}},
        {id_F6,
         {{cru_x + cru_w, grp_lut_y[3] - right_wire_dist, (lut_x + lut_w + dff_x) / 2., grp_lut_y[3] - right_wire_dist},
          {(lut_x + lut_w + dff_x) / 2., grp_lut_y[3] - right_wire_dist, (lut_x + lut_w + dff_x) / 2.,
           lut_y[6] + lut_h / 2.},
          {lut_x + lut_w, lut_y[6] + lut_h / 2., (lut_x + lut_w + dff_x) / 2., lut_y[6] + lut_h / 2.}}},
        {id_I0MUX6, {{(lut_x + lut_w + dff_x) / 2., lut_y[6] + lut_h / 2., mux2lut5_x, lut_y[6] + lut_h / 2.}}},
        // wires above LUT7
        {id_F7,
         {{cru_x + cru_w, grp_lut_y[3] + grp_lut_h + right_wire_dist, (lut_x + lut_w + dff_x) / 2.,
           grp_lut_y[3] + grp_lut_h + right_wire_dist},
          {(lut_x + lut_w + dff_x) / 2., grp_lut_y[3] + grp_lut_h + right_wire_dist, (lut_x + lut_w + dff_x) / 2.,
           lut_y[7] + lut_h / 2.},
          {lut_x + lut_w, lut_y[7] + lut_h / 2., (lut_x + lut_w + dff_x) / 2., lut_y[7] + lut_h / 2.}}},
        {id_I1MUX6, {{(lut_x + lut_w + dff_x) / 2., lut_y[7] + lut_h / 2., mux2lut5_x, lut_y[7] + lut_h / 2.}}},
        {id_SEL6,
         {{cru_x + cru_w, grp_lut_y[3] + grp_lut_h + 2. * right_wire_dist, mux2lut5_x + mux_w / 2.,
           grp_lut_y[3] + grp_lut_h + 2. * right_wire_dist},
          {mux2lut5_x + mux_w / 2., grp_lut_y[3] + grp_lut_h + 2. * right_wire_dist, mux2lut5_x + mux_w / 2.,
           mux2lut5_y[3] + mux_h - mux_f / 2.}}},
        {id_OF6,
         {{cru_x + cru_w, grp_lut_y[3] + grp_lut_h + 3. * right_wire_dist, mux2lut5_x + 4. / 3. * mux_w,
           grp_lut_y[3] + grp_lut_h + 3. * right_wire_dist},
          {mux2lut5_x + 4. / 3. * mux_w, grp_lut_y[3] + grp_lut_h + 3. * right_wire_dist, mux2lut5_x + 4. / 3. * mux_w,
           mux2lut5_y[3] + mux_h / 2.},
          {mux2lut5_x + 4. / 3. * mux_w, mux2lut5_y[3] + mux_h / 2., mux2lut5_x + mux_w, mux2lut5_y[3] + mux_h / 2.}}},
        {id_I0MUX5,
         {{mux2lut5_x + 4. / 3. * mux_w, mux2lut5_y[3] + mux_h / 2., mux2lut5_x + 4. / 3. * mux_w,
           mux2lut6_y[1] + mux_h * 3. / 4.},
          {mux2lut5_x + 4. / 3. * mux_w, mux2lut6_y[1] + mux_h * 3. / 4., mux2lut6_x,
           mux2lut6_y[1] + mux_h * 3. / 4.}}},
};

const dict<IdString, std::vector<std::tuple<float, float, float, float>>> globalSimpleWires = {
        {id_I0MUX7,
         {{mux2lut8_x, mux2lut8_y + mux_h / 4., mux2lut8_x - 1. / 3. * mux_w, mux2lut8_y + mux_h / 4.},
          {mux2lut8_x - 1. / 3. * mux_w, mux2lut8_y + mux_h / 4., mux2lut8_x - 1. / 3. * mux_w,
           cru_y - 2. * right_wire_dist},
          {mux2lut8_x - 1. / 3. * mux_w, cru_y - 2. * right_wire_dist, 1. + mux2lut7_x + 4. / 3. * mux_w,
           cru_y - 2. * right_wire_dist},
          {1. + mux2lut7_x + 4. / 3. * mux_w, cru_y - 2. * right_wire_dist, 1. + mux2lut7_x + 4. / 3. * mux_w,
           grp_lut_y[0] - 3. * right_wire_dist}}},
};

dict<IdString, std::vector<std::tuple<float, float, float, float>>> const globalWires = {
#define PIP_Y(pip_id) (pipPoint.at(pip_id).second)
#define WIRE_X(offset) (cru_x - ((float)offset) * sn_dist)
        // 1 hop
        {id_S10,
         {{WIRE_X(0), PIP_Y(id_S100), WIRE_X(1), PIP_Y(id_S100)},
          {WIRE_X(1), PIP_Y(id_S100), WIRE_X(1), -1. + PIP_Y(id_S101)},
          {WIRE_X(1), -1. + PIP_Y(id_S101), WIRE_X(0), -1. + PIP_Y(id_S101)}}},
        {id_N10,
         {{WIRE_X(0), PIP_Y(id_N100), WIRE_X(2), PIP_Y(id_N100)},
          {WIRE_X(2), PIP_Y(id_N100), WIRE_X(2), 1. + PIP_Y(id_N101)},
          {WIRE_X(2), 1. + PIP_Y(id_N101), WIRE_X(0), 1. + PIP_Y(id_N101)}}},
        {id_S10_loop0,
         {{WIRE_X(0), PIP_Y(id_S100), WIRE_X(1), PIP_Y(id_S100)},
          {WIRE_X(1), PIP_Y(id_S100), WIRE_X(1), -1. * wrap_len},
          {WIRE_X(1), -1. * wrap_len, WIRE_X(2), -1. * wrap_len},
          {WIRE_X(2), -1. * wrap_len, WIRE_X(2), PIP_Y(id_N101)},
          {WIRE_X(2), PIP_Y(id_N101), WIRE_X(0), PIP_Y(id_N101)}}},
        {id_N10_loop0,
         {{WIRE_X(0), PIP_Y(id_N100), WIRE_X(2), PIP_Y(id_N100)},
          {WIRE_X(2), PIP_Y(id_N100), WIRE_X(2), 1. + 1. * wrap_len},
          {WIRE_X(2), 1. + 1. * wrap_len, WIRE_X(1), 1. + 1. * wrap_len},
          {WIRE_X(1), 1. + 1. * wrap_len, WIRE_X(1), PIP_Y(id_S101)},
          {WIRE_X(1), PIP_Y(id_S101), WIRE_X(0), PIP_Y(id_S101)}}},
        {id_S13,
         {{WIRE_X(0), PIP_Y(id_S130), WIRE_X(3), PIP_Y(id_S130)},
          {WIRE_X(3), PIP_Y(id_S130), WIRE_X(3), -1. + PIP_Y(id_S131)},
          {WIRE_X(3), -1. + PIP_Y(id_S131), WIRE_X(0), -1. + PIP_Y(id_S131)}}},
        {id_N13,
         {{WIRE_X(0), PIP_Y(id_N130), WIRE_X(4), PIP_Y(id_N130)},
          {WIRE_X(4), PIP_Y(id_N130), WIRE_X(4), 1. + PIP_Y(id_N131)},
          {WIRE_X(4), 1. + PIP_Y(id_N131), WIRE_X(0), 1. + PIP_Y(id_N131)}}},
        {id_S13_loop0,
         {{WIRE_X(0), PIP_Y(id_S130), WIRE_X(3), PIP_Y(id_S130)},
          {WIRE_X(3), PIP_Y(id_S130), WIRE_X(3), -1. * wrap_len},
          {WIRE_X(3), -1. * wrap_len, WIRE_X(4), -1. * wrap_len},
          {WIRE_X(4), -1. * wrap_len, WIRE_X(4), PIP_Y(id_N131)},
          {WIRE_X(4), PIP_Y(id_N131), WIRE_X(0), PIP_Y(id_N131)}}},
        {id_N13_loop0,
         {{WIRE_X(0), PIP_Y(id_N130), WIRE_X(4), PIP_Y(id_N130)},
          {WIRE_X(4), PIP_Y(id_N130), WIRE_X(4), 1. + 1. * wrap_len},
          {WIRE_X(4), 1. + 1. * wrap_len, WIRE_X(3), 1. + 1. * wrap_len},
          {WIRE_X(3), 1. + 1. * wrap_len, WIRE_X(3), PIP_Y(id_S131)},
          {WIRE_X(3), PIP_Y(id_S131), WIRE_X(0), PIP_Y(id_S131)}}},
        // 1 hop SN
        {id_SN10,
         {{WIRE_X(0), PIP_Y(id_SN10), WIRE_X(6), PIP_Y(id_SN10)},
          {WIRE_X(6), PIP_Y(id_SN10), WIRE_X(6), 1. + PIP_Y(id_N111)},
          {WIRE_X(6), 1. + PIP_Y(id_N111), WIRE_X(0), 1. + PIP_Y(id_N111)},
          {WIRE_X(5), PIP_Y(id_SN10), WIRE_X(5), -1. + PIP_Y(id_S111)},
          {WIRE_X(5), -1. + PIP_Y(id_S111), WIRE_X(0), -1. + PIP_Y(id_S111)}}},
        {id_SN10_loop_n,
         {{WIRE_X(0), PIP_Y(id_SN10), WIRE_X(6), PIP_Y(id_SN10)},
          {WIRE_X(6), PIP_Y(id_SN10), WIRE_X(6), 1. + 1. * wrap_len},
          {WIRE_X(6), 1. + 1. * wrap_len, WIRE_X(5), 1. + 1. * wrap_len},
          {WIRE_X(5), 1. + 1. * wrap_len, WIRE_X(5), PIP_Y(id_SN10)},
          {WIRE_X(5), PIP_Y(id_SN10), WIRE_X(5), -1. + PIP_Y(id_S111)},
          {WIRE_X(5), -1. + PIP_Y(id_S111), WIRE_X(0), -1. + PIP_Y(id_S111)}}},
        {id_SN10_loop_s,
         {{WIRE_X(0), PIP_Y(id_SN10), WIRE_X(6), PIP_Y(id_SN10)},
          {WIRE_X(6), PIP_Y(id_SN10), WIRE_X(6), 1. + PIP_Y(id_N111)},
          {WIRE_X(6), 1. + PIP_Y(id_N111), WIRE_X(0), 1. + PIP_Y(id_N111)},
          {WIRE_X(5), PIP_Y(id_SN10), WIRE_X(5), -1. * wrap_len},
          {WIRE_X(5), -1. * wrap_len, WIRE_X(6), -1. * wrap_len},
          {WIRE_X(6), -1. * wrap_len, WIRE_X(6), PIP_Y(id_N111)},
          {WIRE_X(6), PIP_Y(id_N111), WIRE_X(0), PIP_Y(id_N111)}}},
        {id_SN20,
         {{WIRE_X(0), PIP_Y(id_SN20), WIRE_X(8), PIP_Y(id_SN20)},
          {WIRE_X(8), PIP_Y(id_SN20), WIRE_X(8), 1. + PIP_Y(id_N121)},
          {WIRE_X(8), 1. + PIP_Y(id_N121), WIRE_X(0), 1. + PIP_Y(id_N121)},
          {WIRE_X(7), PIP_Y(id_SN20), WIRE_X(7), -1. + PIP_Y(id_S121)},
          {WIRE_X(7), -1. + PIP_Y(id_S121), WIRE_X(0), -1. + PIP_Y(id_S121)}}},
        {id_SN20_loop_n,
         {{WIRE_X(0), PIP_Y(id_SN20), WIRE_X(8), PIP_Y(id_SN20)},
          {WIRE_X(8), PIP_Y(id_SN20), WIRE_X(8), 1. + 1. * wrap_len},
          {WIRE_X(8), 1. + 1. * wrap_len, WIRE_X(7), 1. + 1. * wrap_len},
          {WIRE_X(7), 1. + 1. * wrap_len, WIRE_X(7), PIP_Y(id_SN10)},
          {WIRE_X(7), PIP_Y(id_SN20), WIRE_X(7), -1. + PIP_Y(id_S121)},
          {WIRE_X(7), -1. + PIP_Y(id_S121), WIRE_X(0), -1. + PIP_Y(id_S121)}}},
        {id_SN20_loop_s,
         {{WIRE_X(0), PIP_Y(id_SN20), WIRE_X(8), PIP_Y(id_SN20)},
          {WIRE_X(8), PIP_Y(id_SN20), WIRE_X(8), 1. + PIP_Y(id_N121)},
          {WIRE_X(8), 1. + PIP_Y(id_N121), WIRE_X(0), 1. + PIP_Y(id_N121)},
          {WIRE_X(7), PIP_Y(id_SN20), WIRE_X(7), -1. * wrap_len},
          {WIRE_X(7), -1. * wrap_len, WIRE_X(8), -1. * wrap_len},
          {WIRE_X(8), -1. * wrap_len, WIRE_X(8), PIP_Y(id_N121)},
          {WIRE_X(8), PIP_Y(id_N121), WIRE_X(0), PIP_Y(id_N121)}}},
        // 2 hop
        {id_S20,
         {{WIRE_X(0), PIP_Y(id_S200), WIRE_X(11), PIP_Y(id_S200)},
          {WIRE_X(11), PIP_Y(id_S200), WIRE_X(11), -1. + PIP_Y(id_S201)},
          {WIRE_X(11), -1. + PIP_Y(id_S201), WIRE_X(0), -1. + PIP_Y(id_S201)},
          {WIRE_X(9), -1. + PIP_Y(id_S201), WIRE_X(9), -2. + PIP_Y(id_S202)},
          {WIRE_X(9), -2. + PIP_Y(id_S202), WIRE_X(0), -2. + PIP_Y(id_S202)}}},
        {id_N20,
         {{WIRE_X(0), PIP_Y(id_N200), WIRE_X(12), PIP_Y(id_N200)},
          {WIRE_X(12), PIP_Y(id_N200), WIRE_X(12), 1. + PIP_Y(id_N201)},
          {WIRE_X(12), 1. + PIP_Y(id_N201), WIRE_X(0), 1. + PIP_Y(id_N201)},
          {WIRE_X(10), 1. + PIP_Y(id_N201), WIRE_X(10), 2. + PIP_Y(id_N202)},
          {WIRE_X(10), 2. + PIP_Y(id_N202), WIRE_X(0), 2. + PIP_Y(id_N202)}}},
        {id_S20_loop0,
         {{WIRE_X(0), PIP_Y(id_S200), WIRE_X(11), PIP_Y(id_S200)},
          {WIRE_X(11), PIP_Y(id_S200), WIRE_X(11), -1. * wrap_len},
          {WIRE_X(11), -1. * wrap_len, WIRE_X(12), -1. * wrap_len},
          {WIRE_X(12), -1. * wrap_len, WIRE_X(12), PIP_Y(id_N201)},
          {WIRE_X(12), PIP_Y(id_N201), WIRE_X(0), PIP_Y(id_N201)},
          {WIRE_X(10), PIP_Y(id_N201), WIRE_X(10), 1. + PIP_Y(id_N202)},
          {WIRE_X(10), 1. + PIP_Y(id_N202), WIRE_X(0), 1. + PIP_Y(id_N202)}}},
        {id_N20_loop0,
         {{WIRE_X(0), PIP_Y(id_N200), WIRE_X(12), PIP_Y(id_N200)},
          {WIRE_X(12), PIP_Y(id_N200), WIRE_X(12), 1. + 1. * wrap_len},
          {WIRE_X(12), 1. + 1. * wrap_len, WIRE_X(11), 1. + 1. * wrap_len},
          {WIRE_X(11), 1. + 1. * wrap_len, WIRE_X(11), PIP_Y(id_S201)},
          {WIRE_X(11), PIP_Y(id_S201), WIRE_X(0), PIP_Y(id_S201)},
          {WIRE_X(9), PIP_Y(id_S201), WIRE_X(9), -1. + PIP_Y(id_S202)},
          {WIRE_X(9), -1. + PIP_Y(id_S202), WIRE_X(0), -1. + PIP_Y(id_S202)}}},
        {id_S20_loop1,
         {{WIRE_X(0), PIP_Y(id_S200), WIRE_X(11), PIP_Y(id_S200)},
          {WIRE_X(11), PIP_Y(id_S200), WIRE_X(11), -1. + PIP_Y(id_S201)},
          {WIRE_X(11), -1. + PIP_Y(id_S201), WIRE_X(0), -1. + PIP_Y(id_S201)},
          {WIRE_X(9), -1. + PIP_Y(id_S201), WIRE_X(9), -1. + -1. * wrap_len},
          {WIRE_X(9), -1. + -1. * wrap_len, WIRE_X(10), -1. + -1. * wrap_len},
          {WIRE_X(10), -1. + -1. * wrap_len, WIRE_X(10), -1. + PIP_Y(id_N202)},
          {WIRE_X(10), -1. + PIP_Y(id_N202), WIRE_X(0), -1. + PIP_Y(id_N202)}}},
        {id_N20_loop1,
         {{WIRE_X(0), PIP_Y(id_N200), WIRE_X(12), PIP_Y(id_N200)},
          {WIRE_X(12), PIP_Y(id_N200), WIRE_X(12), 1. + PIP_Y(id_N201)},
          {WIRE_X(12), 1. + PIP_Y(id_N201), WIRE_X(0), 1. + PIP_Y(id_N201)},
          {WIRE_X(10), 1. + PIP_Y(id_N201), WIRE_X(10), 2. + 1. * wrap_len},
          {WIRE_X(10), 2. + 1. * wrap_len, WIRE_X(9), 2. + 1. * wrap_len},
          {WIRE_X(9), 2. + 1. * wrap_len, WIRE_X(9), 1. + PIP_Y(id_S202)},
          {WIRE_X(9), 1. + PIP_Y(id_S202), WIRE_X(0), 1. + PIP_Y(id_S202)}}},
        {id_S21,
         {{WIRE_X(0), PIP_Y(id_S210), WIRE_X(15), PIP_Y(id_S210)},
          {WIRE_X(15), PIP_Y(id_S210), WIRE_X(15), -1. + PIP_Y(id_S211)},
          {WIRE_X(15), -1. + PIP_Y(id_S211), WIRE_X(0), -1. + PIP_Y(id_S211)},
          {WIRE_X(13), -1. + PIP_Y(id_S211), WIRE_X(13), -2. + PIP_Y(id_S212)},
          {WIRE_X(13), -2. + PIP_Y(id_S212), WIRE_X(0), -2. + PIP_Y(id_S212)}}},
        {id_N21,
         {{WIRE_X(0), PIP_Y(id_N210), WIRE_X(16), PIP_Y(id_N210)},
          {WIRE_X(16), PIP_Y(id_N210), WIRE_X(16), 1. + PIP_Y(id_N211)},
          {WIRE_X(16), 1. + PIP_Y(id_N211), WIRE_X(0), 1. + PIP_Y(id_N211)},
          {WIRE_X(14), 1. + PIP_Y(id_N211), WIRE_X(14), 2. + PIP_Y(id_N212)},
          {WIRE_X(14), 2. + PIP_Y(id_N212), WIRE_X(0), 2. + PIP_Y(id_N212)}}},
        {id_S21_loop0,
         {{WIRE_X(0), PIP_Y(id_S210), WIRE_X(15), PIP_Y(id_S210)},
          {WIRE_X(15), PIP_Y(id_S210), WIRE_X(15), -1. * wrap_len},
          {WIRE_X(15), -1. * wrap_len, WIRE_X(16), -1. * wrap_len},
          {WIRE_X(16), -1. * wrap_len, WIRE_X(16), PIP_Y(id_N211)},
          {WIRE_X(16), PIP_Y(id_N211), WIRE_X(0), PIP_Y(id_N211)},
          {WIRE_X(14), PIP_Y(id_N211), WIRE_X(14), 1. + PIP_Y(id_N212)},
          {WIRE_X(14), 1. + PIP_Y(id_N212), WIRE_X(0), 1. + PIP_Y(id_N212)}}},
        {id_N21_loop0,
         {{WIRE_X(0), PIP_Y(id_N210), WIRE_X(16), PIP_Y(id_N210)},
          {WIRE_X(16), PIP_Y(id_N210), WIRE_X(16), 1. + 1. * wrap_len},
          {WIRE_X(16), 1. + 1. * wrap_len, WIRE_X(15), 1. + 1. * wrap_len},
          {WIRE_X(15), 1. + 1. * wrap_len, WIRE_X(15), PIP_Y(id_S211)},
          {WIRE_X(15), PIP_Y(id_S211), WIRE_X(0), PIP_Y(id_S211)},
          {WIRE_X(13), PIP_Y(id_S211), WIRE_X(13), -1. + PIP_Y(id_S212)},
          {WIRE_X(13), -1. + PIP_Y(id_S212), WIRE_X(0), -1. + PIP_Y(id_S212)}}},
        {id_S21_loop1,
         {{WIRE_X(0), PIP_Y(id_S210), WIRE_X(15), PIP_Y(id_S210)},
          {WIRE_X(15), PIP_Y(id_S210), WIRE_X(15), -1. + PIP_Y(id_S211)},
          {WIRE_X(15), -1. + PIP_Y(id_S211), WIRE_X(0), -1. + PIP_Y(id_S211)},
          {WIRE_X(13), -1. + PIP_Y(id_S211), WIRE_X(13), -1. + -1. * wrap_len},
          {WIRE_X(13), -1. + -1. * wrap_len, WIRE_X(14), -1. + -1. * wrap_len},
          {WIRE_X(14), -1. + -1. * wrap_len, WIRE_X(14), -1. + PIP_Y(id_N212)},
          {WIRE_X(14), -1. + PIP_Y(id_N212), WIRE_X(0), -1. + PIP_Y(id_N212)}}},
        {id_N21_loop1,
         {{WIRE_X(0), PIP_Y(id_N210), WIRE_X(16), PIP_Y(id_N210)},
          {WIRE_X(16), PIP_Y(id_N210), WIRE_X(16), 1. + PIP_Y(id_N211)},
          {WIRE_X(16), 1. + PIP_Y(id_N211), WIRE_X(0), 1. + PIP_Y(id_N211)},
          {WIRE_X(14), 1. + PIP_Y(id_N211), WIRE_X(14), 2. + 1. * wrap_len},
          {WIRE_X(14), 2. + 1. * wrap_len, WIRE_X(13), 2. + 1. * wrap_len},
          {WIRE_X(13), 2. + 1. * wrap_len, WIRE_X(13), 1. + PIP_Y(id_S212)},
          {WIRE_X(13), 1. + PIP_Y(id_S212), WIRE_X(0), 1. + PIP_Y(id_S212)}}},
        {id_S22,
         {{WIRE_X(0), PIP_Y(id_S220), WIRE_X(19), PIP_Y(id_S220)},
          {WIRE_X(19), PIP_Y(id_S220), WIRE_X(19), -1. + PIP_Y(id_S221)},
          {WIRE_X(19), -1. + PIP_Y(id_S221), WIRE_X(0), -1. + PIP_Y(id_S221)},
          {WIRE_X(17), -1. + PIP_Y(id_S221), WIRE_X(17), -2. + PIP_Y(id_S222)},
          {WIRE_X(17), -2. + PIP_Y(id_S222), WIRE_X(0), -2. + PIP_Y(id_S222)}}},
        {id_N22,
         {{WIRE_X(0), PIP_Y(id_N220), WIRE_X(20), PIP_Y(id_N220)},
          {WIRE_X(20), PIP_Y(id_N220), WIRE_X(20), 1. + PIP_Y(id_N221)},
          {WIRE_X(20), 1. + PIP_Y(id_N221), WIRE_X(0), 1. + PIP_Y(id_N221)},
          {WIRE_X(18), 1. + PIP_Y(id_N221), WIRE_X(18), 2. + PIP_Y(id_N222)},
          {WIRE_X(18), 2. + PIP_Y(id_N222), WIRE_X(0), 2. + PIP_Y(id_N222)}}},
        {id_S22_loop0,
         {{WIRE_X(0), PIP_Y(id_S220), WIRE_X(19), PIP_Y(id_S220)},
          {WIRE_X(19), PIP_Y(id_S220), WIRE_X(19), -1. * wrap_len},
          {WIRE_X(19), -1. * wrap_len, WIRE_X(20), -1. * wrap_len},
          {WIRE_X(20), -1. * wrap_len, WIRE_X(20), PIP_Y(id_N221)},
          {WIRE_X(20), PIP_Y(id_N221), WIRE_X(0), PIP_Y(id_N221)},
          {WIRE_X(18), PIP_Y(id_N221), WIRE_X(18), 1. + PIP_Y(id_N222)},
          {WIRE_X(18), 1. + PIP_Y(id_N222), WIRE_X(0), 1. + PIP_Y(id_N222)}}},
        {id_N22_loop0,
         {{WIRE_X(0), PIP_Y(id_N220), WIRE_X(20), PIP_Y(id_N220)},
          {WIRE_X(20), PIP_Y(id_N220), WIRE_X(20), 1. + 1. * wrap_len},
          {WIRE_X(20), 1. + 1. * wrap_len, WIRE_X(19), 1. + 1. * wrap_len},
          {WIRE_X(19), 1. + 1. * wrap_len, WIRE_X(19), PIP_Y(id_S221)},
          {WIRE_X(19), PIP_Y(id_S221), WIRE_X(0), PIP_Y(id_S221)},
          {WIRE_X(17), PIP_Y(id_S221), WIRE_X(17), -1. + PIP_Y(id_S222)},
          {WIRE_X(17), -1. + PIP_Y(id_S222), WIRE_X(0), -1. + PIP_Y(id_S222)}}},
        {id_S22_loop1,
         {{WIRE_X(0), PIP_Y(id_S220), WIRE_X(19), PIP_Y(id_S220)},
          {WIRE_X(19), PIP_Y(id_S220), WIRE_X(19), -1. + PIP_Y(id_S221)},
          {WIRE_X(19), -1. + PIP_Y(id_S221), WIRE_X(0), -1. + PIP_Y(id_S221)},
          {WIRE_X(17), -1. + PIP_Y(id_S221), WIRE_X(17), -1. + -1. * wrap_len},
          {WIRE_X(17), -1. + -1. * wrap_len, WIRE_X(18), -1. + -1. * wrap_len},
          {WIRE_X(18), -1. + -1. * wrap_len, WIRE_X(18), -1. + PIP_Y(id_N222)},
          {WIRE_X(18), -1. + PIP_Y(id_N222), WIRE_X(0), -1. + PIP_Y(id_N222)}}},
        {id_N22_loop1,
         {{WIRE_X(0), PIP_Y(id_N220), WIRE_X(20), PIP_Y(id_N220)},
          {WIRE_X(20), PIP_Y(id_N220), WIRE_X(20), 1. + PIP_Y(id_N221)},
          {WIRE_X(20), 1. + PIP_Y(id_N221), WIRE_X(0), 1. + PIP_Y(id_N221)},
          {WIRE_X(18), 1. + PIP_Y(id_N221), WIRE_X(18), 2. + 1. * wrap_len},
          {WIRE_X(18), 2. + 1. * wrap_len, WIRE_X(17), 2. + 1. * wrap_len},
          {WIRE_X(17), 2. + 1. * wrap_len, WIRE_X(17), 1. + PIP_Y(id_S222)},
          {WIRE_X(17), 1. + PIP_Y(id_S222), WIRE_X(0), 1. + PIP_Y(id_S222)}}},
        {id_S23,
         {{WIRE_X(0), PIP_Y(id_S230), WIRE_X(23), PIP_Y(id_S230)},
          {WIRE_X(23), PIP_Y(id_S230), WIRE_X(23), -1. + PIP_Y(id_S231)},
          {WIRE_X(23), -1. + PIP_Y(id_S231), WIRE_X(0), -1. + PIP_Y(id_S231)},
          {WIRE_X(21), -1. + PIP_Y(id_S231), WIRE_X(21), -2. + PIP_Y(id_S232)},
          {WIRE_X(21), -2. + PIP_Y(id_S232), WIRE_X(0), -2. + PIP_Y(id_S232)}}},
        {id_N23,
         {{WIRE_X(0), PIP_Y(id_N230), WIRE_X(24), PIP_Y(id_N230)},
          {WIRE_X(24), PIP_Y(id_N230), WIRE_X(24), 1. + PIP_Y(id_N231)},
          {WIRE_X(24), 1. + PIP_Y(id_N231), WIRE_X(0), 1. + PIP_Y(id_N231)},
          {WIRE_X(22), 1. + PIP_Y(id_N231), WIRE_X(22), 2. + PIP_Y(id_N232)},
          {WIRE_X(22), 2. + PIP_Y(id_N232), WIRE_X(0), 2. + PIP_Y(id_N232)}}},
        {id_S23_loop0,
         {{WIRE_X(0), PIP_Y(id_S230), WIRE_X(23), PIP_Y(id_S230)},
          {WIRE_X(23), PIP_Y(id_S230), WIRE_X(23), -1. * wrap_len},
          {WIRE_X(23), -1. * wrap_len, WIRE_X(24), -1. * wrap_len},
          {WIRE_X(24), -1. * wrap_len, WIRE_X(24), PIP_Y(id_N231)},
          {WIRE_X(24), PIP_Y(id_N231), WIRE_X(0), PIP_Y(id_N231)},
          {WIRE_X(22), PIP_Y(id_N231), WIRE_X(22), 1. + PIP_Y(id_N232)},
          {WIRE_X(22), 1. + PIP_Y(id_N232), WIRE_X(0), 1. + PIP_Y(id_N232)}}},
        {id_N23_loop0,
         {{WIRE_X(0), PIP_Y(id_N230), WIRE_X(24), PIP_Y(id_N230)},
          {WIRE_X(24), PIP_Y(id_N230), WIRE_X(24), 1. + 1. * wrap_len},
          {WIRE_X(24), 1. + 1. * wrap_len, WIRE_X(23), 1. + 1. * wrap_len},
          {WIRE_X(23), 1. + 1. * wrap_len, WIRE_X(23), PIP_Y(id_S231)},
          {WIRE_X(23), PIP_Y(id_S231), WIRE_X(0), PIP_Y(id_S231)},
          {WIRE_X(21), PIP_Y(id_S231), WIRE_X(21), -1. + PIP_Y(id_S232)},
          {WIRE_X(21), -1. + PIP_Y(id_S232), WIRE_X(0), -1. + PIP_Y(id_S232)}}},
        {id_S23_loop1,
         {{WIRE_X(0), PIP_Y(id_S230), WIRE_X(23), PIP_Y(id_S230)},
          {WIRE_X(23), PIP_Y(id_S230), WIRE_X(23), -1. + PIP_Y(id_S231)},
          {WIRE_X(23), -1. + PIP_Y(id_S231), WIRE_X(0), -1. + PIP_Y(id_S231)},
          {WIRE_X(21), -1. + PIP_Y(id_S231), WIRE_X(21), -1. + -1. * wrap_len},
          {WIRE_X(21), -1. + -1. * wrap_len, WIRE_X(22), -1. + -1. * wrap_len},
          {WIRE_X(22), -1. + -1. * wrap_len, WIRE_X(22), -1. + PIP_Y(id_N232)},
          {WIRE_X(22), -1. + PIP_Y(id_N232), WIRE_X(0), -1. + PIP_Y(id_N232)}}},
        {id_N23_loop1,
         {{WIRE_X(0), PIP_Y(id_N230), WIRE_X(24), PIP_Y(id_N230)},
          {WIRE_X(24), PIP_Y(id_N230), WIRE_X(24), 1. + PIP_Y(id_N231)},
          {WIRE_X(24), 1. + PIP_Y(id_N231), WIRE_X(0), 1. + PIP_Y(id_N231)},
          {WIRE_X(22), 1. + PIP_Y(id_N231), WIRE_X(22), 2. + 1. * wrap_len},
          {WIRE_X(22), 2. + 1. * wrap_len, WIRE_X(21), 2. + 1. * wrap_len},
          {WIRE_X(21), 2. + 1. * wrap_len, WIRE_X(21), 1. + PIP_Y(id_S232)},
          {WIRE_X(21), 1. + PIP_Y(id_S232), WIRE_X(0), 1. + PIP_Y(id_S232)}}},
        {id_S24,
         {{WIRE_X(0), PIP_Y(id_S240), WIRE_X(27), PIP_Y(id_S240)},
          {WIRE_X(27), PIP_Y(id_S240), WIRE_X(27), -1. + PIP_Y(id_S241)},
          {WIRE_X(27), -1. + PIP_Y(id_S241), WIRE_X(0), -1. + PIP_Y(id_S241)},
          {WIRE_X(25), -1. + PIP_Y(id_S241), WIRE_X(25), -2. + PIP_Y(id_S242)},
          {WIRE_X(25), -2. + PIP_Y(id_S242), WIRE_X(0), -2. + PIP_Y(id_S242)}}},
        {id_N24,
         {{WIRE_X(0), PIP_Y(id_N240), WIRE_X(28), PIP_Y(id_N240)},
          {WIRE_X(28), PIP_Y(id_N240), WIRE_X(28), 1. + PIP_Y(id_N241)},
          {WIRE_X(28), 1. + PIP_Y(id_N241), WIRE_X(0), 1. + PIP_Y(id_N241)},
          {WIRE_X(26), 1. + PIP_Y(id_N241), WIRE_X(26), 2. + PIP_Y(id_N242)},
          {WIRE_X(26), 2. + PIP_Y(id_N242), WIRE_X(0), 2. + PIP_Y(id_N242)}}},
        {id_S24_loop0,
         {{WIRE_X(0), PIP_Y(id_S240), WIRE_X(27), PIP_Y(id_S240)},
          {WIRE_X(27), PIP_Y(id_S240), WIRE_X(27), -1. * wrap_len},
          {WIRE_X(27), -1. * wrap_len, WIRE_X(28), -1. * wrap_len},
          {WIRE_X(28), -1. * wrap_len, WIRE_X(28), PIP_Y(id_N241)},
          {WIRE_X(28), PIP_Y(id_N241), WIRE_X(0), PIP_Y(id_N241)},
          {WIRE_X(26), PIP_Y(id_N241), WIRE_X(26), 1. + PIP_Y(id_N242)},
          {WIRE_X(26), 1. + PIP_Y(id_N242), WIRE_X(0), 1. + PIP_Y(id_N242)}}},
        {id_N24_loop0,
         {{WIRE_X(0), PIP_Y(id_N240), WIRE_X(28), PIP_Y(id_N240)},
          {WIRE_X(28), PIP_Y(id_N240), WIRE_X(28), 1. + 1. * wrap_len},
          {WIRE_X(28), 1. + 1. * wrap_len, WIRE_X(27), 1. + 1. * wrap_len},
          {WIRE_X(27), 1. + 1. * wrap_len, WIRE_X(27), PIP_Y(id_S241)},
          {WIRE_X(27), PIP_Y(id_S241), WIRE_X(0), PIP_Y(id_S241)},
          {WIRE_X(25), PIP_Y(id_S241), WIRE_X(25), -1. + PIP_Y(id_S242)},
          {WIRE_X(25), -1. + PIP_Y(id_S242), WIRE_X(0), -1. + PIP_Y(id_S242)}}},
        {id_S24_loop1,
         {{WIRE_X(0), PIP_Y(id_S240), WIRE_X(27), PIP_Y(id_S240)},
          {WIRE_X(27), PIP_Y(id_S240), WIRE_X(27), -1. + PIP_Y(id_S241)},
          {WIRE_X(27), -1. + PIP_Y(id_S241), WIRE_X(0), -1. + PIP_Y(id_S241)},
          {WIRE_X(25), -1. + PIP_Y(id_S241), WIRE_X(25), -1. + -1. * wrap_len},
          {WIRE_X(25), -1. + -1. * wrap_len, WIRE_X(26), -1. + -1. * wrap_len},
          {WIRE_X(26), -1. + -1. * wrap_len, WIRE_X(26), -1. + PIP_Y(id_N242)},
          {WIRE_X(26), -1. + PIP_Y(id_N242), WIRE_X(0), -1. + PIP_Y(id_N242)}}},
        {id_N24_loop1,
         {{WIRE_X(0), PIP_Y(id_N240), WIRE_X(28), PIP_Y(id_N240)},
          {WIRE_X(28), PIP_Y(id_N240), WIRE_X(28), 1. + PIP_Y(id_N241)},
          {WIRE_X(28), 1. + PIP_Y(id_N241), WIRE_X(0), 1. + PIP_Y(id_N241)},
          {WIRE_X(26), 1. + PIP_Y(id_N241), WIRE_X(26), 2. + 1. * wrap_len},
          {WIRE_X(26), 2. + 1. * wrap_len, WIRE_X(25), 2. + 1. * wrap_len},
          {WIRE_X(25), 2. + 1. * wrap_len, WIRE_X(25), 1. + PIP_Y(id_S242)},
          {WIRE_X(25), 1. + PIP_Y(id_S242), WIRE_X(0), 1. + PIP_Y(id_S242)}}},
        {id_S25,
         {{WIRE_X(0), PIP_Y(id_S250), WIRE_X(31), PIP_Y(id_S250)},
          {WIRE_X(31), PIP_Y(id_S250), WIRE_X(31), -1. + PIP_Y(id_S251)},
          {WIRE_X(31), -1. + PIP_Y(id_S251), WIRE_X(0), -1. + PIP_Y(id_S251)},
          {WIRE_X(29), -1. + PIP_Y(id_S251), WIRE_X(29), -2. + PIP_Y(id_S252)},
          {WIRE_X(29), -2. + PIP_Y(id_S252), WIRE_X(0), -2. + PIP_Y(id_S252)}}},
        {id_N25,
         {{WIRE_X(0), PIP_Y(id_N250), WIRE_X(32), PIP_Y(id_N250)},
          {WIRE_X(32), PIP_Y(id_N250), WIRE_X(32), 1. + PIP_Y(id_N251)},
          {WIRE_X(32), 1. + PIP_Y(id_N251), WIRE_X(0), 1. + PIP_Y(id_N251)},
          {WIRE_X(30), 1. + PIP_Y(id_N251), WIRE_X(30), 2. + PIP_Y(id_N252)},
          {WIRE_X(30), 2. + PIP_Y(id_N252), WIRE_X(0), 2. + PIP_Y(id_N252)}}},
        {id_S25_loop0,
         {{WIRE_X(0), PIP_Y(id_S250), WIRE_X(31), PIP_Y(id_S250)},
          {WIRE_X(31), PIP_Y(id_S250), WIRE_X(31), -1. * wrap_len},
          {WIRE_X(31), -1. * wrap_len, WIRE_X(32), -1. * wrap_len},
          {WIRE_X(32), -1. * wrap_len, WIRE_X(32), PIP_Y(id_N251)},
          {WIRE_X(32), PIP_Y(id_N251), WIRE_X(0), PIP_Y(id_N251)},
          {WIRE_X(30), PIP_Y(id_N251), WIRE_X(30), 1. + PIP_Y(id_N252)},
          {WIRE_X(30), 1. + PIP_Y(id_N252), WIRE_X(0), 1. + PIP_Y(id_N252)}}},
        {id_N25_loop0,
         {{WIRE_X(0), PIP_Y(id_N250), WIRE_X(32), PIP_Y(id_N250)},
          {WIRE_X(32), PIP_Y(id_N250), WIRE_X(32), 1. + 1. * wrap_len},
          {WIRE_X(32), 1. + 1. * wrap_len, WIRE_X(31), 1. + 1. * wrap_len},
          {WIRE_X(31), 1. + 1. * wrap_len, WIRE_X(31), PIP_Y(id_S251)},
          {WIRE_X(31), PIP_Y(id_S251), WIRE_X(0), PIP_Y(id_S251)},
          {WIRE_X(29), PIP_Y(id_S251), WIRE_X(29), -1. + PIP_Y(id_S252)},
          {WIRE_X(29), -1. + PIP_Y(id_S252), WIRE_X(0), -1. + PIP_Y(id_S252)}}},
        {id_S25_loop1,
         {{WIRE_X(0), PIP_Y(id_S250), WIRE_X(31), PIP_Y(id_S250)},
          {WIRE_X(31), PIP_Y(id_S250), WIRE_X(31), -1. + PIP_Y(id_S251)},
          {WIRE_X(31), -1. + PIP_Y(id_S251), WIRE_X(0), -1. + PIP_Y(id_S251)},
          {WIRE_X(29), -1. + PIP_Y(id_S251), WIRE_X(29), -1. + -1. * wrap_len},
          {WIRE_X(29), -1. + -1. * wrap_len, WIRE_X(30), -1. + -1. * wrap_len},
          {WIRE_X(30), -1. + -1. * wrap_len, WIRE_X(30), -1. + PIP_Y(id_N252)},
          {WIRE_X(30), -1. + PIP_Y(id_N252), WIRE_X(0), -1. + PIP_Y(id_N252)}}},
        {id_N25_loop1,
         {{WIRE_X(0), PIP_Y(id_N250), WIRE_X(32), PIP_Y(id_N250)},
          {WIRE_X(32), PIP_Y(id_N250), WIRE_X(32), 1. + PIP_Y(id_N251)},
          {WIRE_X(32), 1. + PIP_Y(id_N251), WIRE_X(0), 1. + PIP_Y(id_N251)},
          {WIRE_X(30), 1. + PIP_Y(id_N251), WIRE_X(30), 2. + 1. * wrap_len},
          {WIRE_X(30), 2. + 1. * wrap_len, WIRE_X(29), 2. + 1. * wrap_len},
          {WIRE_X(29), 2. + 1. * wrap_len, WIRE_X(29), 1. + PIP_Y(id_S252)},
          {WIRE_X(29), 1. + PIP_Y(id_S252), WIRE_X(0), 1. + PIP_Y(id_S252)}}},
        {id_S26,
         {{WIRE_X(0), PIP_Y(id_S260), WIRE_X(35), PIP_Y(id_S260)},
          {WIRE_X(35), PIP_Y(id_S260), WIRE_X(35), -1. + PIP_Y(id_S261)},
          {WIRE_X(35), -1. + PIP_Y(id_S261), WIRE_X(0), -1. + PIP_Y(id_S261)},
          {WIRE_X(33), -1. + PIP_Y(id_S261), WIRE_X(33), -2. + PIP_Y(id_S262)},
          {WIRE_X(33), -2. + PIP_Y(id_S262), WIRE_X(0), -2. + PIP_Y(id_S262)}}},
        {id_N26,
         {{WIRE_X(0), PIP_Y(id_N260), WIRE_X(36), PIP_Y(id_N260)},
          {WIRE_X(36), PIP_Y(id_N260), WIRE_X(36), 1. + PIP_Y(id_N261)},
          {WIRE_X(36), 1. + PIP_Y(id_N261), WIRE_X(0), 1. + PIP_Y(id_N261)},
          {WIRE_X(34), 1. + PIP_Y(id_N261), WIRE_X(34), 2. + PIP_Y(id_N262)},
          {WIRE_X(34), 2. + PIP_Y(id_N262), WIRE_X(0), 2. + PIP_Y(id_N262)}}},
        {id_S26_loop0,
         {{WIRE_X(0), PIP_Y(id_S260), WIRE_X(35), PIP_Y(id_S260)},
          {WIRE_X(35), PIP_Y(id_S260), WIRE_X(35), -1. * wrap_len},
          {WIRE_X(35), -1. * wrap_len, WIRE_X(36), -1. * wrap_len},
          {WIRE_X(36), -1. * wrap_len, WIRE_X(36), PIP_Y(id_N261)},
          {WIRE_X(36), PIP_Y(id_N261), WIRE_X(0), PIP_Y(id_N261)},
          {WIRE_X(34), PIP_Y(id_N261), WIRE_X(34), 1. + PIP_Y(id_N262)},
          {WIRE_X(34), 1. + PIP_Y(id_N262), WIRE_X(0), 1. + PIP_Y(id_N262)}}},
        {id_N26_loop0,
         {{WIRE_X(0), PIP_Y(id_N260), WIRE_X(36), PIP_Y(id_N260)},
          {WIRE_X(36), PIP_Y(id_N260), WIRE_X(36), 1. + 1. * wrap_len},
          {WIRE_X(36), 1. + 1. * wrap_len, WIRE_X(35), 1. + 1. * wrap_len},
          {WIRE_X(35), 1. + 1. * wrap_len, WIRE_X(35), PIP_Y(id_S261)},
          {WIRE_X(35), PIP_Y(id_S261), WIRE_X(0), PIP_Y(id_S261)},
          {WIRE_X(33), PIP_Y(id_S261), WIRE_X(33), -1. + PIP_Y(id_S262)},
          {WIRE_X(33), -1. + PIP_Y(id_S262), WIRE_X(0), -1. + PIP_Y(id_S262)}}},
        {id_S26_loop1,
         {{WIRE_X(0), PIP_Y(id_S260), WIRE_X(35), PIP_Y(id_S260)},
          {WIRE_X(35), PIP_Y(id_S260), WIRE_X(35), -1. + PIP_Y(id_S261)},
          {WIRE_X(35), -1. + PIP_Y(id_S261), WIRE_X(0), -1. + PIP_Y(id_S261)},
          {WIRE_X(33), -1. + PIP_Y(id_S261), WIRE_X(33), -1. + -1. * wrap_len},
          {WIRE_X(33), -1. + -1. * wrap_len, WIRE_X(34), -1. + -1. * wrap_len},
          {WIRE_X(34), -1. + -1. * wrap_len, WIRE_X(34), -1. + PIP_Y(id_N262)},
          {WIRE_X(34), -1. + PIP_Y(id_N262), WIRE_X(0), -1. + PIP_Y(id_N262)}}},
        {id_N26_loop1,
         {{WIRE_X(0), PIP_Y(id_N260), WIRE_X(36), PIP_Y(id_N260)},
          {WIRE_X(36), PIP_Y(id_N260), WIRE_X(36), 1. + PIP_Y(id_N261)},
          {WIRE_X(36), 1. + PIP_Y(id_N261), WIRE_X(0), 1. + PIP_Y(id_N261)},
          {WIRE_X(34), 1. + PIP_Y(id_N261), WIRE_X(34), 2. + 1. * wrap_len},
          {WIRE_X(34), 2. + 1. * wrap_len, WIRE_X(33), 2. + 1. * wrap_len},
          {WIRE_X(33), 2. + 1. * wrap_len, WIRE_X(33), 1. + PIP_Y(id_S262)},
          {WIRE_X(33), 1. + PIP_Y(id_S262), WIRE_X(0), 1. + PIP_Y(id_S262)}}},
        {id_S27,
         {{WIRE_X(0), PIP_Y(id_S270), WIRE_X(39), PIP_Y(id_S270)},
          {WIRE_X(39), PIP_Y(id_S270), WIRE_X(39), -1. + PIP_Y(id_S271)},
          {WIRE_X(39), -1. + PIP_Y(id_S271), WIRE_X(0), -1. + PIP_Y(id_S271)},
          {WIRE_X(37), -1. + PIP_Y(id_S271), WIRE_X(37), -2. + PIP_Y(id_S272)},
          {WIRE_X(37), -2. + PIP_Y(id_S272), WIRE_X(0), -2. + PIP_Y(id_S272)}}},
        {id_N27,
         {{WIRE_X(0), PIP_Y(id_N270), WIRE_X(40), PIP_Y(id_N270)},
          {WIRE_X(40), PIP_Y(id_N270), WIRE_X(40), 1. + PIP_Y(id_N271)},
          {WIRE_X(40), 1. + PIP_Y(id_N271), WIRE_X(0), 1. + PIP_Y(id_N271)},
          {WIRE_X(38), 1. + PIP_Y(id_N271), WIRE_X(38), 2. + PIP_Y(id_N272)},
          {WIRE_X(38), 2. + PIP_Y(id_N272), WIRE_X(0), 2. + PIP_Y(id_N272)}}},
        {id_S27_loop0,
         {{WIRE_X(0), PIP_Y(id_S270), WIRE_X(39), PIP_Y(id_S270)},
          {WIRE_X(39), PIP_Y(id_S270), WIRE_X(39), -1. * wrap_len},
          {WIRE_X(39), -1. * wrap_len, WIRE_X(40), -1. * wrap_len},
          {WIRE_X(40), -1. * wrap_len, WIRE_X(40), PIP_Y(id_N271)},
          {WIRE_X(40), PIP_Y(id_N271), WIRE_X(0), PIP_Y(id_N271)},
          {WIRE_X(38), PIP_Y(id_N271), WIRE_X(38), 1. + PIP_Y(id_N272)},
          {WIRE_X(38), 1. + PIP_Y(id_N272), WIRE_X(0), 1. + PIP_Y(id_N272)}}},
        {id_N27_loop0,
         {{WIRE_X(0), PIP_Y(id_N270), WIRE_X(40), PIP_Y(id_N270)},
          {WIRE_X(40), PIP_Y(id_N270), WIRE_X(40), 1. + 1. * wrap_len},
          {WIRE_X(40), 1. + 1. * wrap_len, WIRE_X(39), 1. + 1. * wrap_len},
          {WIRE_X(39), 1. + 1. * wrap_len, WIRE_X(39), PIP_Y(id_S271)},
          {WIRE_X(39), PIP_Y(id_S271), WIRE_X(0), PIP_Y(id_S271)},
          {WIRE_X(37), PIP_Y(id_S271), WIRE_X(37), -1. + PIP_Y(id_S272)},
          {WIRE_X(37), -1. + PIP_Y(id_S272), WIRE_X(0), -1. + PIP_Y(id_S272)}}},
        {id_S27_loop1,
         {{WIRE_X(0), PIP_Y(id_S270), WIRE_X(39), PIP_Y(id_S270)},
          {WIRE_X(39), PIP_Y(id_S270), WIRE_X(39), -1. + PIP_Y(id_S271)},
          {WIRE_X(39), -1. + PIP_Y(id_S271), WIRE_X(0), -1. + PIP_Y(id_S271)},
          {WIRE_X(37), -1. + PIP_Y(id_S271), WIRE_X(37), -1. + -1. * wrap_len},
          {WIRE_X(37), -1. + -1. * wrap_len, WIRE_X(38), -1. + -1. * wrap_len},
          {WIRE_X(38), -1. + -1. * wrap_len, WIRE_X(38), -1. + PIP_Y(id_N272)},
          {WIRE_X(38), -1. + PIP_Y(id_N272), WIRE_X(0), -1. + PIP_Y(id_N272)}}},
        {id_N27_loop1,
         {{WIRE_X(0), PIP_Y(id_N270), WIRE_X(40), PIP_Y(id_N270)},
          {WIRE_X(40), PIP_Y(id_N270), WIRE_X(40), 1. + PIP_Y(id_N271)},
          {WIRE_X(40), 1. + PIP_Y(id_N271), WIRE_X(0), 1. + PIP_Y(id_N271)},
          {WIRE_X(38), 1. + PIP_Y(id_N271), WIRE_X(38), 2. + 1. * wrap_len},
          {WIRE_X(38), 2. + 1. * wrap_len, WIRE_X(37), 2. + 1. * wrap_len},
          {WIRE_X(37), 2. + 1. * wrap_len, WIRE_X(37), 1. + PIP_Y(id_S272)},
          {WIRE_X(37), 1. + PIP_Y(id_S272), WIRE_X(0), 1. + PIP_Y(id_S272)}}},
// clock taps
#define CLK_GT00_X 41.f
#define CLK_GT10_X 46.f
// 4 hop
#define HOP4X_START (CLK_GT00_X + 10.f)
#define HOP4X(offset) WIRE_X((float)offset + HOP4X_START)
        {id_S80,
         {{WIRE_X(0), PIP_Y(id_S800), HOP4X(16), PIP_Y(id_S800)},
          {HOP4X(16), PIP_Y(id_S800), HOP4X(16), PIP_Y(id_N808)},
          {HOP4X(16), PIP_Y(id_N808) - 0., HOP4X(14), PIP_Y(id_N808) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N808) - left_wire_dist - 0., HOP4X(14), PIP_Y(id_N808) - 1.},
          {HOP4X(14), PIP_Y(id_N808) - 1., HOP4X(12), PIP_Y(id_N808) - left_wire_dist - 1.},
          {HOP4X(12), PIP_Y(id_N808) - left_wire_dist - 1., HOP4X(12), PIP_Y(id_N808) - 2.},
          {HOP4X(12), PIP_Y(id_N808) - 2., HOP4X(10), PIP_Y(id_N808) - left_wire_dist - 2.},
          {HOP4X(10), PIP_Y(id_N808) - left_wire_dist - 2., HOP4X(10), PIP_Y(id_N808) - 3.},
          {HOP4X(10), PIP_Y(id_N808) - 3., HOP4X(8), PIP_Y(id_N808) - left_wire_dist - 3.},
          {HOP4X(8), PIP_Y(id_N808) - left_wire_dist - 3., HOP4X(8), PIP_Y(id_N808) - 4.},
          {HOP4X(8), PIP_Y(id_S804) - 4., WIRE_X(0), PIP_Y(id_S804) - 4.},
          {HOP4X(8), PIP_Y(id_N808) - 4., HOP4X(6), PIP_Y(id_N808) - left_wire_dist - 4.},
          {HOP4X(6), PIP_Y(id_N808) - left_wire_dist - 4., HOP4X(6), PIP_Y(id_N808) - 5.},
          {HOP4X(6), PIP_Y(id_N808) - 5., HOP4X(4), PIP_Y(id_N808) - left_wire_dist - 5.},
          {HOP4X(4), PIP_Y(id_N808) - left_wire_dist - 5., HOP4X(4), PIP_Y(id_N808) - 6.},
          {HOP4X(4), PIP_Y(id_N808) - 6., HOP4X(2), PIP_Y(id_N808) - left_wire_dist - 6.},
          {HOP4X(2), PIP_Y(id_N808) - left_wire_dist - 6., HOP4X(2), PIP_Y(id_N808) - 7.},
          {HOP4X(2), PIP_Y(id_N808) - 7., HOP4X(0), PIP_Y(id_N808) - left_wire_dist - 7.},
          {HOP4X(0), PIP_Y(id_N808) - left_wire_dist - 7., HOP4X(0), PIP_Y(id_S808) - 8.},
          {HOP4X(0), PIP_Y(id_S808) - 8., WIRE_X(0), PIP_Y(id_S808) - 8.}}},
        {id_N80,
         {{WIRE_X(0), PIP_Y(id_N800), HOP4X(17), PIP_Y(id_N800)},
          {HOP4X(17), PIP_Y(id_N800) + 0., HOP4X(15), PIP_Y(id_N800) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N800) + left_wire_dist + 0., HOP4X(15), PIP_Y(id_N800) + 1.},
          {HOP4X(15), PIP_Y(id_N800) + 1., HOP4X(13), PIP_Y(id_N800) + left_wire_dist + 1.},
          {HOP4X(13), PIP_Y(id_N800) + left_wire_dist + 1., HOP4X(13), PIP_Y(id_N800) + 2.},
          {HOP4X(13), PIP_Y(id_N800) + 2., HOP4X(11), PIP_Y(id_N800) + left_wire_dist + 2.},
          {HOP4X(11), PIP_Y(id_N800) + left_wire_dist + 2., HOP4X(11), PIP_Y(id_N800) + 3.},
          {HOP4X(11), PIP_Y(id_N800) + 3., HOP4X(9), PIP_Y(id_N800) + left_wire_dist + 3.},
          {HOP4X(9), PIP_Y(id_N800) + left_wire_dist + 3., HOP4X(9), PIP_Y(id_N800) + 4.},
          {HOP4X(9), PIP_Y(id_N804) + 4., WIRE_X(0), PIP_Y(id_N804) + 4.},
          {HOP4X(9), PIP_Y(id_N800) + 4., HOP4X(7), PIP_Y(id_N800) + left_wire_dist + 4.},
          {HOP4X(7), PIP_Y(id_N800) + left_wire_dist + 4., HOP4X(7), PIP_Y(id_N800) + 5.},
          {HOP4X(7), PIP_Y(id_N800) + 5., HOP4X(5), PIP_Y(id_N800) + left_wire_dist + 5.},
          {HOP4X(5), PIP_Y(id_N800) + left_wire_dist + 5., HOP4X(5), PIP_Y(id_N800) + 6.},
          {HOP4X(5), PIP_Y(id_N800) + 6., HOP4X(3), PIP_Y(id_N800) + left_wire_dist + 6.},
          {HOP4X(3), PIP_Y(id_N800) + left_wire_dist + 6., HOP4X(3), PIP_Y(id_N800) + 7.},
          {HOP4X(3), PIP_Y(id_N800) + 7., HOP4X(1), PIP_Y(id_N800) + left_wire_dist + 7.},
          {HOP4X(1), PIP_Y(id_N800) + left_wire_dist + 7., HOP4X(1), PIP_Y(id_N808) + 8.},
          {HOP4X(1), PIP_Y(id_N808) + 8., WIRE_X(0), PIP_Y(id_N808) + 8.}}},
        {id_S80_loop0,
         {{WIRE_X(0), PIP_Y(id_S800), HOP4X(16), PIP_Y(id_S800)},
          {HOP4X(16), PIP_Y(id_S800), HOP4X(16), PIP_Y(id_N808)},
          {HOP4X(16), PIP_Y(id_N808) - 0., HOP4X(14), PIP_Y(id_N808) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N808) - left_wire_dist - 0., HOP4X(14), -wrap_len - 0.},
          {HOP4X(14), -wrap_len - 0., HOP4X(15), -wrap_len - 0.},
          {HOP4X(15), -wrap_len - 0., HOP4X(15), PIP_Y(id_N800) - 0.},
          {HOP4X(15), PIP_Y(id_N800) - 0., HOP4X(13), PIP_Y(id_N800) + left_wire_dist - 0.},
          {HOP4X(13), PIP_Y(id_N800) + left_wire_dist - 0., HOP4X(13), PIP_Y(id_N800) + 1.},
          {HOP4X(13), PIP_Y(id_N800) + 1., HOP4X(11), PIP_Y(id_N800) + left_wire_dist + 1.},
          {HOP4X(11), PIP_Y(id_N800) + left_wire_dist + 1., HOP4X(11), PIP_Y(id_N800) + 2.},
          {HOP4X(11), PIP_Y(id_N800) + 2., HOP4X(9), PIP_Y(id_N800) + left_wire_dist + 2.},
          {HOP4X(9), PIP_Y(id_N800) + left_wire_dist + 2., HOP4X(9), PIP_Y(id_N800) + 3.},
          {HOP4X(9), PIP_Y(id_N800) + 3., HOP4X(7), PIP_Y(id_N800) + left_wire_dist + 3.},
          {HOP4X(9), PIP_Y(id_N804) + 3., WIRE_X(0), PIP_Y(id_N804) + 3.},
          {HOP4X(7), PIP_Y(id_N800) + left_wire_dist + 3., HOP4X(7), PIP_Y(id_N800) + 4.},
          {HOP4X(7), PIP_Y(id_N800) + 4., HOP4X(5), PIP_Y(id_N800) + left_wire_dist + 4.},
          {HOP4X(5), PIP_Y(id_N800) + left_wire_dist + 4., HOP4X(5), PIP_Y(id_N800) + 5.},
          {HOP4X(5), PIP_Y(id_N800) + 5., HOP4X(3), PIP_Y(id_N800) + left_wire_dist + 5.},
          {HOP4X(3), PIP_Y(id_N800) + left_wire_dist + 5., HOP4X(3), PIP_Y(id_N800) + 6.},
          {HOP4X(3), PIP_Y(id_N800) + 6., HOP4X(1), PIP_Y(id_N800) + left_wire_dist + 6.},
          {HOP4X(1), PIP_Y(id_N800) + left_wire_dist + 6., HOP4X(1), PIP_Y(id_N808) + 7.},
          {HOP4X(1), PIP_Y(id_N808) + 7., WIRE_X(0), PIP_Y(id_N808) + 7.}}},
        {id_S80_loop1,
         {{WIRE_X(0), PIP_Y(id_S800), HOP4X(16), PIP_Y(id_S800)},
          {HOP4X(16), PIP_Y(id_S800), HOP4X(16), PIP_Y(id_N808)},
          {HOP4X(16), PIP_Y(id_N808) - 0., HOP4X(14), PIP_Y(id_N808) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N808) - left_wire_dist - 0., HOP4X(14), PIP_Y(id_N808) - 1.},
          {HOP4X(14), PIP_Y(id_N808) - 1., HOP4X(12), PIP_Y(id_N808) - left_wire_dist - 1.},
          {HOP4X(12), PIP_Y(id_N808) - left_wire_dist - 1., HOP4X(12), -wrap_len - 1.},
          {HOP4X(12), -wrap_len - 1., HOP4X(13), -wrap_len - 1.},
          {HOP4X(13), -wrap_len - 1., HOP4X(13), PIP_Y(id_N800) - 1.},
          {HOP4X(13), PIP_Y(id_N800) - 1., HOP4X(11), PIP_Y(id_N800) + left_wire_dist - 1.},
          {HOP4X(11), PIP_Y(id_N800) + left_wire_dist - 1., HOP4X(11), PIP_Y(id_N800) - 0.},
          {HOP4X(11), PIP_Y(id_N800) - 0., HOP4X(9), PIP_Y(id_N800) + left_wire_dist - 0.},
          {HOP4X(9), PIP_Y(id_N800) + left_wire_dist - 0., HOP4X(9), PIP_Y(id_N800) + 1.},
          {HOP4X(9), PIP_Y(id_N800) + 1., HOP4X(7), PIP_Y(id_N800) + left_wire_dist + 1.},
          {HOP4X(9), PIP_Y(id_N804) + 1., WIRE_X(0), PIP_Y(id_N804) + 1.},
          {HOP4X(7), PIP_Y(id_N800) + left_wire_dist + 1., HOP4X(7), PIP_Y(id_N800) + 2.},
          {HOP4X(7), PIP_Y(id_N800) + 2., HOP4X(5), PIP_Y(id_N800) + left_wire_dist + 2.},
          {HOP4X(5), PIP_Y(id_N800) + left_wire_dist + 2., HOP4X(5), PIP_Y(id_N800) + 3.},
          {HOP4X(5), PIP_Y(id_N800) + 3., HOP4X(3), PIP_Y(id_N800) + left_wire_dist + 3.},
          {HOP4X(3), PIP_Y(id_N800) + left_wire_dist + 3., HOP4X(3), PIP_Y(id_N800) + 4.},
          {HOP4X(3), PIP_Y(id_N800) + 4., HOP4X(1), PIP_Y(id_N800) + left_wire_dist + 4.},
          {HOP4X(1), PIP_Y(id_N800) + left_wire_dist + 4., HOP4X(1), PIP_Y(id_N808) + 5.},
          {HOP4X(1), PIP_Y(id_N808) + 5., WIRE_X(0), PIP_Y(id_N808) + 5.}}},
        {id_S80_loop2,
         {{WIRE_X(0), PIP_Y(id_S800), HOP4X(16), PIP_Y(id_S800)},
          {HOP4X(16), PIP_Y(id_S800), HOP4X(16), PIP_Y(id_N808)},
          {HOP4X(16), PIP_Y(id_N808) - 0., HOP4X(14), PIP_Y(id_N808) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N808) - left_wire_dist - 0., HOP4X(14), PIP_Y(id_N808) - 1.},
          {HOP4X(14), PIP_Y(id_N808) - 1., HOP4X(12), PIP_Y(id_N808) - left_wire_dist - 1.},
          {HOP4X(12), PIP_Y(id_N808) - left_wire_dist - 1., HOP4X(12), PIP_Y(id_N808) - 2.},
          {HOP4X(12), PIP_Y(id_N808) - 2., HOP4X(10), PIP_Y(id_N808) - left_wire_dist - 2.},
          {HOP4X(10), PIP_Y(id_N808) - left_wire_dist - 2., HOP4X(10), -wrap_len - 2.},
          {HOP4X(10), -wrap_len - 2., HOP4X(11), -wrap_len - 2.},
          {HOP4X(11), -wrap_len - 2., HOP4X(11), PIP_Y(id_N800) - 2.},
          {HOP4X(11), PIP_Y(id_N800) - 2., HOP4X(9), PIP_Y(id_N800) + left_wire_dist - 2.},
          {HOP4X(9), PIP_Y(id_N800) + left_wire_dist - 2., HOP4X(9), PIP_Y(id_N800) - 1.},
          {HOP4X(9), PIP_Y(id_N800) - 1., HOP4X(7), PIP_Y(id_N800) + left_wire_dist - 1.},
          {HOP4X(9), PIP_Y(id_N804) - 1., WIRE_X(0), PIP_Y(id_N804) - 1.},
          {HOP4X(7), PIP_Y(id_N800) + left_wire_dist - 1., HOP4X(7), PIP_Y(id_N800) - 0.},
          {HOP4X(7), PIP_Y(id_N800) - 0., HOP4X(5), PIP_Y(id_N800) + left_wire_dist - 0.},
          {HOP4X(5), PIP_Y(id_N800) + left_wire_dist - 0., HOP4X(5), PIP_Y(id_N800) + 1.},
          {HOP4X(5), PIP_Y(id_N800) + 1., HOP4X(3), PIP_Y(id_N800) + left_wire_dist + 1.},
          {HOP4X(3), PIP_Y(id_N800) + left_wire_dist + 1., HOP4X(3), PIP_Y(id_N800) + 2.},
          {HOP4X(3), PIP_Y(id_N800) + 2., HOP4X(1), PIP_Y(id_N800) + left_wire_dist + 2.},
          {HOP4X(1), PIP_Y(id_N800) + left_wire_dist + 2., HOP4X(1), PIP_Y(id_N808) + 3.},
          {HOP4X(1), PIP_Y(id_N808) + 3., WIRE_X(0), PIP_Y(id_N808) + 3.}}},
        {id_S80_loop3,
         {{WIRE_X(0), PIP_Y(id_S800), HOP4X(16), PIP_Y(id_S800)},
          {HOP4X(16), PIP_Y(id_S800), HOP4X(16), PIP_Y(id_N808)},
          {HOP4X(16), PIP_Y(id_N808) - 0., HOP4X(14), PIP_Y(id_N808) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N808) - left_wire_dist - 0., HOP4X(14), PIP_Y(id_N808) - 1.},
          {HOP4X(14), PIP_Y(id_N808) - 1., HOP4X(12), PIP_Y(id_N808) - left_wire_dist - 1.},
          {HOP4X(12), PIP_Y(id_N808) - left_wire_dist - 1., HOP4X(12), PIP_Y(id_N808) - 2.},
          {HOP4X(12), PIP_Y(id_N808) - 2., HOP4X(10), PIP_Y(id_N808) - left_wire_dist - 2.},
          {HOP4X(10), PIP_Y(id_N808) - left_wire_dist - 2., HOP4X(10), PIP_Y(id_N808) - 3.},
          {HOP4X(10), PIP_Y(id_N808) - 3., HOP4X(8), PIP_Y(id_N808) - left_wire_dist - 3.},
          {HOP4X(8), PIP_Y(id_N808) - left_wire_dist - 3., HOP4X(8), -wrap_len - 3.},
          {HOP4X(8), -wrap_len - 3., HOP4X(9), -wrap_len - 3.},
          {HOP4X(9), -wrap_len - 3., HOP4X(9), PIP_Y(id_N800) - 3.},
          {HOP4X(9), PIP_Y(id_N804) - 3., WIRE_X(0), PIP_Y(id_N804) - 3.},
          {HOP4X(9), PIP_Y(id_N800) - 3., HOP4X(7), PIP_Y(id_N800) + left_wire_dist - 3.},
          {HOP4X(7), PIP_Y(id_N800) + left_wire_dist - 3., HOP4X(7), PIP_Y(id_N800) - 2.},
          {HOP4X(7), PIP_Y(id_N800) - 2., HOP4X(5), PIP_Y(id_N800) + left_wire_dist - 2.},
          {HOP4X(5), PIP_Y(id_N800) + left_wire_dist - 2., HOP4X(5), PIP_Y(id_N800) - 1.},
          {HOP4X(5), PIP_Y(id_N800) - 1., HOP4X(3), PIP_Y(id_N800) + left_wire_dist - 1.},
          {HOP4X(3), PIP_Y(id_N800) + left_wire_dist - 1., HOP4X(3), PIP_Y(id_N800) - 0.},
          {HOP4X(3), PIP_Y(id_N800) - 0., HOP4X(1), PIP_Y(id_N800) + left_wire_dist - 0.},
          {HOP4X(1), PIP_Y(id_N800) + left_wire_dist - 0., HOP4X(1), PIP_Y(id_N808) + 1.},
          {HOP4X(1), PIP_Y(id_N808) + 1., WIRE_X(0), PIP_Y(id_N808) + 1.}}},
        {id_S80_loop4,
         {{WIRE_X(0), PIP_Y(id_S800), HOP4X(16), PIP_Y(id_S800)},
          {HOP4X(16), PIP_Y(id_S800), HOP4X(16), PIP_Y(id_N808)},
          {HOP4X(16), PIP_Y(id_N808) - 0., HOP4X(14), PIP_Y(id_N808) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N808) - left_wire_dist - 0., HOP4X(14), PIP_Y(id_N808) - 1.},
          {HOP4X(14), PIP_Y(id_N808) - 1., HOP4X(12), PIP_Y(id_N808) - left_wire_dist - 1.},
          {HOP4X(12), PIP_Y(id_N808) - left_wire_dist - 1., HOP4X(12), PIP_Y(id_N808) - 2.},
          {HOP4X(12), PIP_Y(id_N808) - 2., HOP4X(10), PIP_Y(id_N808) - left_wire_dist - 2.},
          {HOP4X(10), PIP_Y(id_N808) - left_wire_dist - 2., HOP4X(10), PIP_Y(id_N808) - 3.},
          {HOP4X(10), PIP_Y(id_N808) - 3., HOP4X(8), PIP_Y(id_N808) - left_wire_dist - 3.},
          {HOP4X(8), PIP_Y(id_N808) - left_wire_dist - 3., HOP4X(8), PIP_Y(id_N808) - 4.},
          {HOP4X(8), PIP_Y(id_S804) - 4., WIRE_X(0), PIP_Y(id_S804) - 4.},
          {HOP4X(8), PIP_Y(id_N808) - 4., HOP4X(6), PIP_Y(id_N808) - left_wire_dist - 4.},
          {HOP4X(6), PIP_Y(id_N808) - left_wire_dist - 4., HOP4X(6), -wrap_len - 4.},
          {HOP4X(6), -wrap_len - 4., HOP4X(7), -wrap_len - 4.},
          {HOP4X(7), -wrap_len - 4., HOP4X(7), PIP_Y(id_N800) - 4.},
          {HOP4X(7), PIP_Y(id_N800) - 4., HOP4X(5), PIP_Y(id_N800) + left_wire_dist - 4.},
          {HOP4X(5), PIP_Y(id_N800) + left_wire_dist - 4., HOP4X(5), PIP_Y(id_N800) - 3.},
          {HOP4X(5), PIP_Y(id_N800) - 3., HOP4X(3), PIP_Y(id_N800) + left_wire_dist - 3.},
          {HOP4X(3), PIP_Y(id_N800) + left_wire_dist - 3., HOP4X(3), PIP_Y(id_N800) - 2.},
          {HOP4X(3), PIP_Y(id_N800) - 2., HOP4X(1), PIP_Y(id_N800) + left_wire_dist - 2.},
          {HOP4X(1), PIP_Y(id_N800) + left_wire_dist - 2., HOP4X(1), PIP_Y(id_N808) - 1.},
          {HOP4X(1), PIP_Y(id_N808) - 1., WIRE_X(0), PIP_Y(id_N808) - 1.}}},
        {id_S80_loop5,
         {{WIRE_X(0), PIP_Y(id_S800), HOP4X(16), PIP_Y(id_S800)},
          {HOP4X(16), PIP_Y(id_S800), HOP4X(16), PIP_Y(id_N808)},
          {HOP4X(16), PIP_Y(id_N808) - 0., HOP4X(14), PIP_Y(id_N808) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N808) - left_wire_dist - 0., HOP4X(14), PIP_Y(id_N808) - 1.},
          {HOP4X(14), PIP_Y(id_N808) - 1., HOP4X(12), PIP_Y(id_N808) - left_wire_dist - 1.},
          {HOP4X(12), PIP_Y(id_N808) - left_wire_dist - 1., HOP4X(12), PIP_Y(id_N808) - 2.},
          {HOP4X(12), PIP_Y(id_N808) - 2., HOP4X(10), PIP_Y(id_N808) - left_wire_dist - 2.},
          {HOP4X(10), PIP_Y(id_N808) - left_wire_dist - 2., HOP4X(10), PIP_Y(id_N808) - 3.},
          {HOP4X(10), PIP_Y(id_N808) - 3., HOP4X(8), PIP_Y(id_N808) - left_wire_dist - 3.},
          {HOP4X(8), PIP_Y(id_N808) - left_wire_dist - 3., HOP4X(8), PIP_Y(id_N808) - 4.},
          {HOP4X(8), PIP_Y(id_S804) - 4., WIRE_X(0), PIP_Y(id_S804) - 4.},
          {HOP4X(8), PIP_Y(id_N808) - 4., HOP4X(6), PIP_Y(id_N808) - left_wire_dist - 4.},
          {HOP4X(6), PIP_Y(id_N808) - left_wire_dist - 4., HOP4X(6), PIP_Y(id_N808) - 5.},
          {HOP4X(6), PIP_Y(id_N808) - 5., HOP4X(4), PIP_Y(id_N808) - left_wire_dist - 5.},
          {HOP4X(4), PIP_Y(id_N808) - left_wire_dist - 5., HOP4X(4), -wrap_len - 5.},
          {HOP4X(4), -wrap_len - 5., HOP4X(5), -wrap_len - 5.},
          {HOP4X(5), -wrap_len - 5., HOP4X(5), PIP_Y(id_N800) - 5.},
          {HOP4X(5), PIP_Y(id_N800) - 5., HOP4X(3), PIP_Y(id_N800) + left_wire_dist - 5.},
          {HOP4X(3), PIP_Y(id_N800) + left_wire_dist - 5., HOP4X(3), PIP_Y(id_N800) - 4.},
          {HOP4X(3), PIP_Y(id_N800) - 4., HOP4X(1), PIP_Y(id_N800) + left_wire_dist - 4.},
          {HOP4X(1), PIP_Y(id_N800) + left_wire_dist - 4., HOP4X(1), PIP_Y(id_N808) - 3.},
          {HOP4X(1), PIP_Y(id_N808) - 3., WIRE_X(0), PIP_Y(id_N808) - 3.}}},
        {id_S80_loop6,
         {{WIRE_X(0), PIP_Y(id_S800), HOP4X(16), PIP_Y(id_S800)},
          {HOP4X(16), PIP_Y(id_S800), HOP4X(16), PIP_Y(id_N808)},
          {HOP4X(16), PIP_Y(id_N808) - 0., HOP4X(14), PIP_Y(id_N808) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N808) - left_wire_dist - 0., HOP4X(14), PIP_Y(id_N808) - 1.},
          {HOP4X(14), PIP_Y(id_N808) - 1., HOP4X(12), PIP_Y(id_N808) - left_wire_dist - 1.},
          {HOP4X(12), PIP_Y(id_N808) - left_wire_dist - 1., HOP4X(12), PIP_Y(id_N808) - 2.},
          {HOP4X(12), PIP_Y(id_N808) - 2., HOP4X(10), PIP_Y(id_N808) - left_wire_dist - 2.},
          {HOP4X(10), PIP_Y(id_N808) - left_wire_dist - 2., HOP4X(10), PIP_Y(id_N808) - 3.},
          {HOP4X(10), PIP_Y(id_N808) - 3., HOP4X(8), PIP_Y(id_N808) - left_wire_dist - 3.},
          {HOP4X(8), PIP_Y(id_N808) - left_wire_dist - 3., HOP4X(8), PIP_Y(id_N808) - 4.},
          {HOP4X(8), PIP_Y(id_S804) - 4., WIRE_X(0), PIP_Y(id_S804) - 4.},
          {HOP4X(8), PIP_Y(id_N808) - 4., HOP4X(6), PIP_Y(id_N808) - left_wire_dist - 4.},
          {HOP4X(6), PIP_Y(id_N808) - left_wire_dist - 4., HOP4X(6), PIP_Y(id_N808) - 5.},
          {HOP4X(6), PIP_Y(id_N808) - 5., HOP4X(4), PIP_Y(id_N808) - left_wire_dist - 5.},
          {HOP4X(4), PIP_Y(id_N808) - left_wire_dist - 5., HOP4X(4), PIP_Y(id_N808) - 6.},
          {HOP4X(4), PIP_Y(id_N808) - 6., HOP4X(2), PIP_Y(id_N808) - left_wire_dist - 6.},
          {HOP4X(2), PIP_Y(id_N808) - left_wire_dist - 6., HOP4X(2), -wrap_len - 6.},
          {HOP4X(2), -wrap_len - 6., HOP4X(3), -wrap_len - 6.},
          {HOP4X(3), -wrap_len - 6., HOP4X(3), PIP_Y(id_N800) - 6.},
          {HOP4X(3), PIP_Y(id_N800) - 6., HOP4X(1), PIP_Y(id_N800) + left_wire_dist - 6.},
          {HOP4X(1), PIP_Y(id_N800) + left_wire_dist - 6., HOP4X(1), PIP_Y(id_N808) - 5.},
          {HOP4X(1), PIP_Y(id_N808) - 5., WIRE_X(0), PIP_Y(id_N808) - 5.}}},
        {id_S80_loop7,
         {{WIRE_X(0), PIP_Y(id_S800), HOP4X(16), PIP_Y(id_S800)},
          {HOP4X(16), PIP_Y(id_S800), HOP4X(16), PIP_Y(id_N808)},
          {HOP4X(16), PIP_Y(id_N808) - 0., HOP4X(14), PIP_Y(id_N808) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N808) - left_wire_dist - 0., HOP4X(14), PIP_Y(id_N808) - 1.},
          {HOP4X(14), PIP_Y(id_N808) - 1., HOP4X(12), PIP_Y(id_N808) - left_wire_dist - 1.},
          {HOP4X(12), PIP_Y(id_N808) - left_wire_dist - 1., HOP4X(12), PIP_Y(id_N808) - 2.},
          {HOP4X(12), PIP_Y(id_N808) - 2., HOP4X(10), PIP_Y(id_N808) - left_wire_dist - 2.},
          {HOP4X(10), PIP_Y(id_N808) - left_wire_dist - 2., HOP4X(10), PIP_Y(id_N808) - 3.},
          {HOP4X(10), PIP_Y(id_N808) - 3., HOP4X(8), PIP_Y(id_N808) - left_wire_dist - 3.},
          {HOP4X(8), PIP_Y(id_N808) - left_wire_dist - 3., HOP4X(8), PIP_Y(id_N808) - 4.},
          {HOP4X(8), PIP_Y(id_S804) - 4., WIRE_X(0), PIP_Y(id_S804) - 4.},
          {HOP4X(8), PIP_Y(id_N808) - 4., HOP4X(6), PIP_Y(id_N808) - left_wire_dist - 4.},
          {HOP4X(6), PIP_Y(id_N808) - left_wire_dist - 4., HOP4X(6), PIP_Y(id_N808) - 5.},
          {HOP4X(6), PIP_Y(id_N808) - 5., HOP4X(4), PIP_Y(id_N808) - left_wire_dist - 5.},
          {HOP4X(4), PIP_Y(id_N808) - left_wire_dist - 5., HOP4X(4), PIP_Y(id_N808) - 6.},
          {HOP4X(4), PIP_Y(id_N808) - 6., HOP4X(2), PIP_Y(id_N808) - left_wire_dist - 6.},
          {HOP4X(2), PIP_Y(id_N808) - left_wire_dist - 6., HOP4X(2), PIP_Y(id_N808) - 7.},
          {HOP4X(2), PIP_Y(id_N808) - 7., HOP4X(0), PIP_Y(id_N808) - left_wire_dist - 7.},
          {HOP4X(0), PIP_Y(id_N808) - left_wire_dist - 7., HOP4X(0), -wrap_len - 7.},
          {HOP4X(0), -wrap_len - 7., HOP4X(1), -wrap_len - 7.},
          {HOP4X(1), -wrap_len - 7., HOP4X(1), PIP_Y(id_N808) - 7.},
          {HOP4X(1), PIP_Y(id_N808) - 7., WIRE_X(0), PIP_Y(id_N808) - 7.}}},
        {id_N80_loop0,
         {{WIRE_X(0), PIP_Y(id_N800), HOP4X(17), PIP_Y(id_N800)},
          {HOP4X(17), PIP_Y(id_N800) + 0., HOP4X(15), PIP_Y(id_N800) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N800) + left_wire_dist + 0., HOP4X(15), wrap_len + 1.},
          {HOP4X(15), wrap_len + 1., HOP4X(14), wrap_len + 1.},
          {HOP4X(14), wrap_len + 1., HOP4X(14), PIP_Y(id_N808) + 0.},
          {HOP4X(14), PIP_Y(id_N808) + 0., HOP4X(12), PIP_Y(id_N808) - left_wire_dist + 0.},
          {HOP4X(12), PIP_Y(id_N808) - left_wire_dist + 0., HOP4X(12), PIP_Y(id_N808) - 1.},
          {HOP4X(12), PIP_Y(id_N808) - 1., HOP4X(10), PIP_Y(id_N808) - left_wire_dist - 1.},
          {HOP4X(10), PIP_Y(id_N808) - left_wire_dist - 1., HOP4X(10), PIP_Y(id_N808) - 2.},
          {HOP4X(10), PIP_Y(id_N808) - 2., HOP4X(8), PIP_Y(id_N808) - left_wire_dist - 2.},
          {HOP4X(8), PIP_Y(id_N808) - left_wire_dist - 2., HOP4X(8), PIP_Y(id_N808) - 3.},
          {HOP4X(8), PIP_Y(id_S804) - 3., WIRE_X(0), PIP_Y(id_S804) - 3.},
          {HOP4X(8), PIP_Y(id_N808) - 3., HOP4X(6), PIP_Y(id_N808) - left_wire_dist - 3.},
          {HOP4X(6), PIP_Y(id_N808) - left_wire_dist - 3., HOP4X(6), PIP_Y(id_N808) - 4.},
          {HOP4X(6), PIP_Y(id_N808) - 4., HOP4X(4), PIP_Y(id_N808) - left_wire_dist - 4.},
          {HOP4X(4), PIP_Y(id_N808) - left_wire_dist - 4., HOP4X(4), PIP_Y(id_N808) - 5.},
          {HOP4X(4), PIP_Y(id_N808) - 5., HOP4X(2), PIP_Y(id_N808) - left_wire_dist - 5.},
          {HOP4X(2), PIP_Y(id_N808) - left_wire_dist - 5., HOP4X(2), PIP_Y(id_N808) - 6.},
          {HOP4X(2), PIP_Y(id_N808) - 6., HOP4X(0), PIP_Y(id_N808) - left_wire_dist - 6.},
          {HOP4X(0), PIP_Y(id_N808) - left_wire_dist - 6., HOP4X(0), PIP_Y(id_S808) - 7.},
          {HOP4X(0), PIP_Y(id_S808) - 7., WIRE_X(0), PIP_Y(id_S808) - 7.}}},
        {id_N80_loop1,
         {{WIRE_X(0), PIP_Y(id_N800), HOP4X(17), PIP_Y(id_N800)},
          {HOP4X(17), PIP_Y(id_N800) + 0., HOP4X(15), PIP_Y(id_N800) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N800) + left_wire_dist + 0., HOP4X(15), PIP_Y(id_N800) + 1.},
          {HOP4X(15), PIP_Y(id_N800) + 1., HOP4X(13), PIP_Y(id_N800) + left_wire_dist + 1.},
          {HOP4X(13), PIP_Y(id_N800) + left_wire_dist + 1., HOP4X(13), wrap_len + 2.},
          {HOP4X(13), wrap_len + 2., HOP4X(12), wrap_len + 2.},
          {HOP4X(12), wrap_len + 2., HOP4X(12), PIP_Y(id_N808) + 1.},
          {HOP4X(12), PIP_Y(id_N808) + 1., HOP4X(10), PIP_Y(id_N808) - left_wire_dist + 1.},
          {HOP4X(10), PIP_Y(id_N808) - left_wire_dist + 1., HOP4X(10), PIP_Y(id_N808) + 0.},
          {HOP4X(10), PIP_Y(id_N808) + 0., HOP4X(8), PIP_Y(id_N808) - left_wire_dist + 0.},
          {HOP4X(8), PIP_Y(id_N808) - left_wire_dist + 0., HOP4X(8), PIP_Y(id_N808) - 1.},
          {HOP4X(8), PIP_Y(id_S804) - 1., WIRE_X(0), PIP_Y(id_S804) - 1.},
          {HOP4X(8), PIP_Y(id_N808) - 1., HOP4X(6), PIP_Y(id_N808) - left_wire_dist - 1.},
          {HOP4X(6), PIP_Y(id_N808) - left_wire_dist - 1., HOP4X(6), PIP_Y(id_N808) - 2.},
          {HOP4X(6), PIP_Y(id_N808) - 2., HOP4X(4), PIP_Y(id_N808) - left_wire_dist - 2.},
          {HOP4X(4), PIP_Y(id_N808) - left_wire_dist - 2., HOP4X(4), PIP_Y(id_N808) - 3.},
          {HOP4X(4), PIP_Y(id_N808) - 3., HOP4X(2), PIP_Y(id_N808) - left_wire_dist - 3.},
          {HOP4X(2), PIP_Y(id_N808) - left_wire_dist - 3., HOP4X(2), PIP_Y(id_N808) - 4.},
          {HOP4X(2), PIP_Y(id_N808) - 4., HOP4X(0), PIP_Y(id_N808) - left_wire_dist - 4.},
          {HOP4X(0), PIP_Y(id_N808) - left_wire_dist - 4., HOP4X(0), PIP_Y(id_S808) - 5.},
          {HOP4X(0), PIP_Y(id_S808) - 5., WIRE_X(0), PIP_Y(id_S808) - 5.}}},
        {id_N80_loop2,
         {{WIRE_X(0), PIP_Y(id_N800), HOP4X(17), PIP_Y(id_N800)},
          {HOP4X(17), PIP_Y(id_N800) + 0., HOP4X(15), PIP_Y(id_N800) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N800) + left_wire_dist + 0., HOP4X(15), PIP_Y(id_N800) + 1.},
          {HOP4X(15), PIP_Y(id_N800) + 1., HOP4X(13), PIP_Y(id_N800) + left_wire_dist + 1.},
          {HOP4X(13), PIP_Y(id_N800) + left_wire_dist + 1., HOP4X(13), PIP_Y(id_N800) + 2.},
          {HOP4X(13), PIP_Y(id_N800) + 2., HOP4X(11), PIP_Y(id_N800) + left_wire_dist + 2.},
          {HOP4X(11), PIP_Y(id_N800) + left_wire_dist + 2., HOP4X(11), wrap_len + 3.},
          {HOP4X(11), wrap_len + 3., HOP4X(10), wrap_len + 3.},
          {HOP4X(10), wrap_len + 3., HOP4X(10), PIP_Y(id_N808) + 2.},
          {HOP4X(10), PIP_Y(id_N808) + 2., HOP4X(8), PIP_Y(id_N808) - left_wire_dist + 2.},
          {HOP4X(8), PIP_Y(id_N808) - left_wire_dist + 2., HOP4X(8), PIP_Y(id_N808) + 1.},
          {HOP4X(8), PIP_Y(id_S804) + 1., WIRE_X(0), PIP_Y(id_S804) + 1.},
          {HOP4X(8), PIP_Y(id_N808) + 1., HOP4X(6), PIP_Y(id_N808) - left_wire_dist + 1.},
          {HOP4X(6), PIP_Y(id_N808) - left_wire_dist + 1., HOP4X(6), PIP_Y(id_N808) + 0.},
          {HOP4X(6), PIP_Y(id_N808) + 0., HOP4X(4), PIP_Y(id_N808) - left_wire_dist + 0.},
          {HOP4X(4), PIP_Y(id_N808) - left_wire_dist + 0., HOP4X(4), PIP_Y(id_N808) - 1.},
          {HOP4X(4), PIP_Y(id_N808) - 1., HOP4X(2), PIP_Y(id_N808) - left_wire_dist - 1.},
          {HOP4X(2), PIP_Y(id_N808) - left_wire_dist - 1., HOP4X(2), PIP_Y(id_N808) - 2.},
          {HOP4X(2), PIP_Y(id_N808) - 2., HOP4X(0), PIP_Y(id_N808) - left_wire_dist - 2.},
          {HOP4X(0), PIP_Y(id_N808) - left_wire_dist - 2., HOP4X(0), PIP_Y(id_S808) - 3.},
          {HOP4X(0), PIP_Y(id_S808) - 3., WIRE_X(0), PIP_Y(id_S808) - 3.}}},
        {id_N80_loop3,
         {{WIRE_X(0), PIP_Y(id_N800), HOP4X(17), PIP_Y(id_N800)},
          {HOP4X(17), PIP_Y(id_N800) + 0., HOP4X(15), PIP_Y(id_N800) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N800) + left_wire_dist + 0., HOP4X(15), PIP_Y(id_N800) + 1.},
          {HOP4X(15), PIP_Y(id_N800) + 1., HOP4X(13), PIP_Y(id_N800) + left_wire_dist + 1.},
          {HOP4X(13), PIP_Y(id_N800) + left_wire_dist + 1., HOP4X(13), PIP_Y(id_N800) + 2.},
          {HOP4X(13), PIP_Y(id_N800) + 2., HOP4X(11), PIP_Y(id_N800) + left_wire_dist + 2.},
          {HOP4X(11), PIP_Y(id_N800) + left_wire_dist + 2., HOP4X(11), PIP_Y(id_N800) + 3.},
          {HOP4X(11), PIP_Y(id_N800) + 3., HOP4X(9), PIP_Y(id_N800) + left_wire_dist + 3.},
          {HOP4X(9), PIP_Y(id_N800) + left_wire_dist + 3., HOP4X(9), wrap_len + 4.},
          {HOP4X(9), wrap_len + 4., HOP4X(8), wrap_len + 4.},
          {HOP4X(8), wrap_len + 4., HOP4X(8), PIP_Y(id_N808) + 3.},
          {HOP4X(8), PIP_Y(id_S804) + 3., WIRE_X(0), PIP_Y(id_S804) + 3.},
          {HOP4X(8), PIP_Y(id_N808) + 3., HOP4X(6), PIP_Y(id_N808) - left_wire_dist + 3.},
          {HOP4X(6), PIP_Y(id_N808) - left_wire_dist + 3., HOP4X(6), PIP_Y(id_N808) + 2.},
          {HOP4X(6), PIP_Y(id_N808) + 2., HOP4X(4), PIP_Y(id_N808) - left_wire_dist + 2.},
          {HOP4X(4), PIP_Y(id_N808) - left_wire_dist + 2., HOP4X(4), PIP_Y(id_N808) + 1.},
          {HOP4X(4), PIP_Y(id_N808) + 1., HOP4X(2), PIP_Y(id_N808) - left_wire_dist + 1.},
          {HOP4X(2), PIP_Y(id_N808) - left_wire_dist + 1., HOP4X(2), PIP_Y(id_N808) + 0.},
          {HOP4X(2), PIP_Y(id_N808) + 0., HOP4X(0), PIP_Y(id_N808) - left_wire_dist + 0.},
          {HOP4X(0), PIP_Y(id_N808) - left_wire_dist + 0., HOP4X(0), PIP_Y(id_S808) - 1.},
          {HOP4X(0), PIP_Y(id_S808) - 1., WIRE_X(0), PIP_Y(id_S808) - 1.}}},
        {id_N80_loop4,
         {{WIRE_X(0), PIP_Y(id_N800), HOP4X(17), PIP_Y(id_N800)},
          {HOP4X(17), PIP_Y(id_N800) + 0., HOP4X(15), PIP_Y(id_N800) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N800) + left_wire_dist + 0., HOP4X(15), PIP_Y(id_N800) + 1.},
          {HOP4X(15), PIP_Y(id_N800) + 1., HOP4X(13), PIP_Y(id_N800) + left_wire_dist + 1.},
          {HOP4X(13), PIP_Y(id_N800) + left_wire_dist + 1., HOP4X(13), PIP_Y(id_N800) + 2.},
          {HOP4X(13), PIP_Y(id_N800) + 2., HOP4X(11), PIP_Y(id_N800) + left_wire_dist + 2.},
          {HOP4X(11), PIP_Y(id_N800) + left_wire_dist + 2., HOP4X(11), PIP_Y(id_N800) + 3.},
          {HOP4X(11), PIP_Y(id_N800) + 3., HOP4X(9), PIP_Y(id_N800) + left_wire_dist + 3.},
          {HOP4X(9), PIP_Y(id_N800) + left_wire_dist + 3., HOP4X(9), PIP_Y(id_N800) + 4.},
          {HOP4X(9), PIP_Y(id_N804) + 4., WIRE_X(0), PIP_Y(id_N804) + 4.},
          {HOP4X(9), PIP_Y(id_N800) + 4., HOP4X(7), PIP_Y(id_N800) + left_wire_dist + 4.},
          {HOP4X(7), PIP_Y(id_N800) + left_wire_dist + 4., HOP4X(7), wrap_len + 5.},
          {HOP4X(7), wrap_len + 5., HOP4X(6), wrap_len + 5.},
          {HOP4X(6), wrap_len + 5., HOP4X(6), PIP_Y(id_N808) + 4.},
          {HOP4X(6), PIP_Y(id_N808) + 4., HOP4X(4), PIP_Y(id_N808) - left_wire_dist + 4.},
          {HOP4X(4), PIP_Y(id_N808) - left_wire_dist + 4., HOP4X(4), PIP_Y(id_N808) + 3.},
          {HOP4X(4), PIP_Y(id_N808) + 3., HOP4X(2), PIP_Y(id_N808) - left_wire_dist + 3.},
          {HOP4X(2), PIP_Y(id_N808) - left_wire_dist + 3., HOP4X(2), PIP_Y(id_N808) + 2.},
          {HOP4X(2), PIP_Y(id_N808) + 2., HOP4X(0), PIP_Y(id_N808) - left_wire_dist + 2.},
          {HOP4X(0), PIP_Y(id_N808) - left_wire_dist + 2., HOP4X(0), PIP_Y(id_S808) + 1.},
          {HOP4X(0), PIP_Y(id_S808) + 1., WIRE_X(0), PIP_Y(id_S808) + 1.}}},
        {id_N80_loop5,
         {{WIRE_X(0), PIP_Y(id_N800), HOP4X(17), PIP_Y(id_N800)},
          {HOP4X(17), PIP_Y(id_N800) + 0., HOP4X(15), PIP_Y(id_N800) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N800) + left_wire_dist + 0., HOP4X(15), PIP_Y(id_N800) + 1.},
          {HOP4X(15), PIP_Y(id_N800) + 1., HOP4X(13), PIP_Y(id_N800) + left_wire_dist + 1.},
          {HOP4X(13), PIP_Y(id_N800) + left_wire_dist + 1., HOP4X(13), PIP_Y(id_N800) + 2.},
          {HOP4X(13), PIP_Y(id_N800) + 2., HOP4X(11), PIP_Y(id_N800) + left_wire_dist + 2.},
          {HOP4X(11), PIP_Y(id_N800) + left_wire_dist + 2., HOP4X(11), PIP_Y(id_N800) + 3.},
          {HOP4X(11), PIP_Y(id_N800) + 3., HOP4X(9), PIP_Y(id_N800) + left_wire_dist + 3.},
          {HOP4X(9), PIP_Y(id_N800) + left_wire_dist + 3., HOP4X(9), PIP_Y(id_N800) + 4.},
          {HOP4X(9), PIP_Y(id_N804) + 4., WIRE_X(0), PIP_Y(id_N804) + 4.},
          {HOP4X(9), PIP_Y(id_N800) + 4., HOP4X(7), PIP_Y(id_N800) + left_wire_dist + 4.},
          {HOP4X(7), PIP_Y(id_N800) + left_wire_dist + 4., HOP4X(7), PIP_Y(id_N800) + 5.},
          {HOP4X(7), PIP_Y(id_N800) + 5., HOP4X(5), PIP_Y(id_N800) + left_wire_dist + 5.},
          {HOP4X(5), PIP_Y(id_N800) + left_wire_dist + 5., HOP4X(5), wrap_len + 6.},
          {HOP4X(5), wrap_len + 6., HOP4X(4), wrap_len + 6.},
          {HOP4X(4), wrap_len + 6., HOP4X(4), PIP_Y(id_N808) + 5.},
          {HOP4X(4), PIP_Y(id_N808) + 5., HOP4X(2), PIP_Y(id_N808) - left_wire_dist + 5.},
          {HOP4X(2), PIP_Y(id_N808) - left_wire_dist + 5., HOP4X(2), PIP_Y(id_N808) + 4.},
          {HOP4X(2), PIP_Y(id_N808) + 4., HOP4X(0), PIP_Y(id_N808) - left_wire_dist + 4.},
          {HOP4X(0), PIP_Y(id_N808) - left_wire_dist + 4., HOP4X(0), PIP_Y(id_S808) + 3.},
          {HOP4X(0), PIP_Y(id_S808) + 3., WIRE_X(0), PIP_Y(id_S808) + 3.}}},
        {id_N80_loop6,
         {{WIRE_X(0), PIP_Y(id_N800), HOP4X(17), PIP_Y(id_N800)},
          {HOP4X(17), PIP_Y(id_N800) + 0., HOP4X(15), PIP_Y(id_N800) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N800) + left_wire_dist + 0., HOP4X(15), PIP_Y(id_N800) + 1.},
          {HOP4X(15), PIP_Y(id_N800) + 1., HOP4X(13), PIP_Y(id_N800) + left_wire_dist + 1.},
          {HOP4X(13), PIP_Y(id_N800) + left_wire_dist + 1., HOP4X(13), PIP_Y(id_N800) + 2.},
          {HOP4X(13), PIP_Y(id_N800) + 2., HOP4X(11), PIP_Y(id_N800) + left_wire_dist + 2.},
          {HOP4X(11), PIP_Y(id_N800) + left_wire_dist + 2., HOP4X(11), PIP_Y(id_N800) + 3.},
          {HOP4X(11), PIP_Y(id_N800) + 3., HOP4X(9), PIP_Y(id_N800) + left_wire_dist + 3.},
          {HOP4X(9), PIP_Y(id_N800) + left_wire_dist + 3., HOP4X(9), PIP_Y(id_N800) + 4.},
          {HOP4X(9), PIP_Y(id_N804) + 4., WIRE_X(0), PIP_Y(id_N804) + 4.},
          {HOP4X(9), PIP_Y(id_N800) + 4., HOP4X(7), PIP_Y(id_N800) + left_wire_dist + 4.},
          {HOP4X(7), PIP_Y(id_N800) + left_wire_dist + 4., HOP4X(7), PIP_Y(id_N800) + 5.},
          {HOP4X(7), PIP_Y(id_N800) + 5., HOP4X(5), PIP_Y(id_N800) + left_wire_dist + 5.},
          {HOP4X(5), PIP_Y(id_N800) + left_wire_dist + 5., HOP4X(5), PIP_Y(id_N800) + 6.},
          {HOP4X(5), PIP_Y(id_N800) + 6., HOP4X(3), PIP_Y(id_N800) + left_wire_dist + 6.},
          {HOP4X(3), PIP_Y(id_N800) + left_wire_dist + 6., HOP4X(3), wrap_len + 7.},
          {HOP4X(3), wrap_len + 7., HOP4X(2), wrap_len + 7.},
          {HOP4X(2), wrap_len + 7., HOP4X(2), PIP_Y(id_N808) + 6.},
          {HOP4X(2), PIP_Y(id_N808) + 6., HOP4X(0), PIP_Y(id_N808) - left_wire_dist + 6.},
          {HOP4X(0), PIP_Y(id_N808) - left_wire_dist + 6., HOP4X(0), PIP_Y(id_S808) + 5.},
          {HOP4X(0), PIP_Y(id_S808) + 5., WIRE_X(0), PIP_Y(id_S808) + 5.}}},
        {id_N80_loop7,
         {{WIRE_X(0), PIP_Y(id_N800), HOP4X(17), PIP_Y(id_N800)},
          {HOP4X(17), PIP_Y(id_N800) + 0., HOP4X(15), PIP_Y(id_N800) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N800) + left_wire_dist + 0., HOP4X(15), PIP_Y(id_N800) + 1.},
          {HOP4X(15), PIP_Y(id_N800) + 1., HOP4X(13), PIP_Y(id_N800) + left_wire_dist + 1.},
          {HOP4X(13), PIP_Y(id_N800) + left_wire_dist + 1., HOP4X(13), PIP_Y(id_N800) + 2.},
          {HOP4X(13), PIP_Y(id_N800) + 2., HOP4X(11), PIP_Y(id_N800) + left_wire_dist + 2.},
          {HOP4X(11), PIP_Y(id_N800) + left_wire_dist + 2., HOP4X(11), PIP_Y(id_N800) + 3.},
          {HOP4X(11), PIP_Y(id_N800) + 3., HOP4X(9), PIP_Y(id_N800) + left_wire_dist + 3.},
          {HOP4X(9), PIP_Y(id_N800) + left_wire_dist + 3., HOP4X(9), PIP_Y(id_N800) + 4.},
          {HOP4X(9), PIP_Y(id_N804) + 4., WIRE_X(0), PIP_Y(id_N804) + 4.},
          {HOP4X(9), PIP_Y(id_N800) + 4., HOP4X(7), PIP_Y(id_N800) + left_wire_dist + 4.},
          {HOP4X(7), PIP_Y(id_N800) + left_wire_dist + 4., HOP4X(7), PIP_Y(id_N800) + 5.},
          {HOP4X(7), PIP_Y(id_N800) + 5., HOP4X(5), PIP_Y(id_N800) + left_wire_dist + 5.},
          {HOP4X(5), PIP_Y(id_N800) + left_wire_dist + 5., HOP4X(5), PIP_Y(id_N800) + 6.},
          {HOP4X(5), PIP_Y(id_N800) + 6., HOP4X(3), PIP_Y(id_N800) + left_wire_dist + 6.},
          {HOP4X(3), PIP_Y(id_N800) + left_wire_dist + 6., HOP4X(3), PIP_Y(id_N800) + 7.},
          {HOP4X(3), PIP_Y(id_N800) + 7., HOP4X(1), PIP_Y(id_N800) + left_wire_dist + 7.},
          {HOP4X(1), PIP_Y(id_N800) + left_wire_dist + 7., HOP4X(1), wrap_len + 8.},
          {HOP4X(1), wrap_len + 8., HOP4X(0), wrap_len + 8.},
          {HOP4X(0), wrap_len + 8., HOP4X(0), PIP_Y(id_S808) + 7.},
          {HOP4X(0), PIP_Y(id_S808) + 7., WIRE_X(0), PIP_Y(id_S808) + 7.}}},

#undef HOP4X
#define HOP4X(offset) WIRE_X((float)offset + HOP4X_START + 18.f)
        {id_S81,
         {{WIRE_X(0), PIP_Y(id_S810), HOP4X(16), PIP_Y(id_S810)},
          {HOP4X(16), PIP_Y(id_S810), HOP4X(16), PIP_Y(id_N818)},
          {HOP4X(16), PIP_Y(id_N818) - 0., HOP4X(14), PIP_Y(id_N818) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N818) - left_wire_dist - 0., HOP4X(14), PIP_Y(id_N818) - 1.},
          {HOP4X(14), PIP_Y(id_N818) - 1., HOP4X(12), PIP_Y(id_N818) - left_wire_dist - 1.},
          {HOP4X(12), PIP_Y(id_N818) - left_wire_dist - 1., HOP4X(12), PIP_Y(id_N818) - 2.},
          {HOP4X(12), PIP_Y(id_N818) - 2., HOP4X(10), PIP_Y(id_N818) - left_wire_dist - 2.},
          {HOP4X(10), PIP_Y(id_N818) - left_wire_dist - 2., HOP4X(10), PIP_Y(id_N818) - 3.},
          {HOP4X(10), PIP_Y(id_N818) - 3., HOP4X(8), PIP_Y(id_N818) - left_wire_dist - 3.},
          {HOP4X(8), PIP_Y(id_N818) - left_wire_dist - 3., HOP4X(8), PIP_Y(id_N818) - 4.},
          {HOP4X(8), PIP_Y(id_S814) - 4., WIRE_X(0), PIP_Y(id_S814) - 4.},
          {HOP4X(8), PIP_Y(id_N818) - 4., HOP4X(6), PIP_Y(id_N818) - left_wire_dist - 4.},
          {HOP4X(6), PIP_Y(id_N818) - left_wire_dist - 4., HOP4X(6), PIP_Y(id_N818) - 5.},
          {HOP4X(6), PIP_Y(id_N818) - 5., HOP4X(4), PIP_Y(id_N818) - left_wire_dist - 5.},
          {HOP4X(4), PIP_Y(id_N818) - left_wire_dist - 5., HOP4X(4), PIP_Y(id_N818) - 6.},
          {HOP4X(4), PIP_Y(id_N818) - 6., HOP4X(2), PIP_Y(id_N818) - left_wire_dist - 6.},
          {HOP4X(2), PIP_Y(id_N818) - left_wire_dist - 6., HOP4X(2), PIP_Y(id_N818) - 7.},
          {HOP4X(2), PIP_Y(id_N818) - 7., HOP4X(0), PIP_Y(id_N818) - left_wire_dist - 7.},
          {HOP4X(0), PIP_Y(id_N818) - left_wire_dist - 7., HOP4X(0), PIP_Y(id_S818) - 8.},
          {HOP4X(0), PIP_Y(id_S818) - 8., WIRE_X(0), PIP_Y(id_S818) - 8.}}},
        {id_N81,
         {{WIRE_X(0), PIP_Y(id_N810), HOP4X(17), PIP_Y(id_N810)},
          {HOP4X(17), PIP_Y(id_N810) + 0., HOP4X(15), PIP_Y(id_N810) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N810) + left_wire_dist + 0., HOP4X(15), PIP_Y(id_N810) + 1.},
          {HOP4X(15), PIP_Y(id_N810) + 1., HOP4X(13), PIP_Y(id_N810) + left_wire_dist + 1.},
          {HOP4X(13), PIP_Y(id_N810) + left_wire_dist + 1., HOP4X(13), PIP_Y(id_N810) + 2.},
          {HOP4X(13), PIP_Y(id_N810) + 2., HOP4X(11), PIP_Y(id_N810) + left_wire_dist + 2.},
          {HOP4X(11), PIP_Y(id_N810) + left_wire_dist + 2., HOP4X(11), PIP_Y(id_N810) + 3.},
          {HOP4X(11), PIP_Y(id_N810) + 3., HOP4X(9), PIP_Y(id_N810) + left_wire_dist + 3.},
          {HOP4X(9), PIP_Y(id_N810) + left_wire_dist + 3., HOP4X(9), PIP_Y(id_N810) + 4.},
          {HOP4X(9), PIP_Y(id_N814) + 4., WIRE_X(0), PIP_Y(id_N814) + 4.},
          {HOP4X(9), PIP_Y(id_N810) + 4., HOP4X(7), PIP_Y(id_N810) + left_wire_dist + 4.},
          {HOP4X(7), PIP_Y(id_N810) + left_wire_dist + 4., HOP4X(7), PIP_Y(id_N810) + 5.},
          {HOP4X(7), PIP_Y(id_N810) + 5., HOP4X(5), PIP_Y(id_N810) + left_wire_dist + 5.},
          {HOP4X(5), PIP_Y(id_N810) + left_wire_dist + 5., HOP4X(5), PIP_Y(id_N810) + 6.},
          {HOP4X(5), PIP_Y(id_N810) + 6., HOP4X(3), PIP_Y(id_N810) + left_wire_dist + 6.},
          {HOP4X(3), PIP_Y(id_N810) + left_wire_dist + 6., HOP4X(3), PIP_Y(id_N810) + 7.},
          {HOP4X(3), PIP_Y(id_N810) + 7., HOP4X(1), PIP_Y(id_N810) + left_wire_dist + 7.},
          {HOP4X(1), PIP_Y(id_N810) + left_wire_dist + 7., HOP4X(1), PIP_Y(id_N818) + 8.},
          {HOP4X(1), PIP_Y(id_N818) + 8., WIRE_X(0), PIP_Y(id_N818) + 8.}}},
        {id_S81_loop0,
         {{WIRE_X(0), PIP_Y(id_S810), HOP4X(16), PIP_Y(id_S810)},
          {HOP4X(16), PIP_Y(id_S810), HOP4X(16), PIP_Y(id_N818)},
          {HOP4X(16), PIP_Y(id_N818) - 0., HOP4X(14), PIP_Y(id_N818) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N818) - left_wire_dist - 0., HOP4X(14), -wrap_len - 0.},
          {HOP4X(14), -wrap_len - 0., HOP4X(15), -wrap_len - 0.},
          {HOP4X(15), -wrap_len - 0., HOP4X(15), PIP_Y(id_N810) - 0.},
          {HOP4X(15), PIP_Y(id_N810) - 0., HOP4X(13), PIP_Y(id_N810) + left_wire_dist - 0.},
          {HOP4X(13), PIP_Y(id_N810) + left_wire_dist - 0., HOP4X(13), PIP_Y(id_N810) + 1.},
          {HOP4X(13), PIP_Y(id_N810) + 1., HOP4X(11), PIP_Y(id_N810) + left_wire_dist + 1.},
          {HOP4X(11), PIP_Y(id_N810) + left_wire_dist + 1., HOP4X(11), PIP_Y(id_N810) + 2.},
          {HOP4X(11), PIP_Y(id_N810) + 2., HOP4X(9), PIP_Y(id_N810) + left_wire_dist + 2.},
          {HOP4X(9), PIP_Y(id_N810) + left_wire_dist + 2., HOP4X(9), PIP_Y(id_N810) + 3.},
          {HOP4X(9), PIP_Y(id_N810) + 3., HOP4X(7), PIP_Y(id_N810) + left_wire_dist + 3.},
          {HOP4X(9), PIP_Y(id_N814) + 3., WIRE_X(0), PIP_Y(id_N814) + 3.},
          {HOP4X(7), PIP_Y(id_N810) + left_wire_dist + 3., HOP4X(7), PIP_Y(id_N810) + 4.},
          {HOP4X(7), PIP_Y(id_N810) + 4., HOP4X(5), PIP_Y(id_N810) + left_wire_dist + 4.},
          {HOP4X(5), PIP_Y(id_N810) + left_wire_dist + 4., HOP4X(5), PIP_Y(id_N810) + 5.},
          {HOP4X(5), PIP_Y(id_N810) + 5., HOP4X(3), PIP_Y(id_N810) + left_wire_dist + 5.},
          {HOP4X(3), PIP_Y(id_N810) + left_wire_dist + 5., HOP4X(3), PIP_Y(id_N810) + 6.},
          {HOP4X(3), PIP_Y(id_N810) + 6., HOP4X(1), PIP_Y(id_N810) + left_wire_dist + 6.},
          {HOP4X(1), PIP_Y(id_N810) + left_wire_dist + 6., HOP4X(1), PIP_Y(id_N818) + 7.},
          {HOP4X(1), PIP_Y(id_N818) + 7., WIRE_X(0), PIP_Y(id_N818) + 7.}}},
        {id_S81_loop1,
         {{WIRE_X(0), PIP_Y(id_S810), HOP4X(16), PIP_Y(id_S810)},
          {HOP4X(16), PIP_Y(id_S810), HOP4X(16), PIP_Y(id_N818)},
          {HOP4X(16), PIP_Y(id_N818) - 0., HOP4X(14), PIP_Y(id_N818) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N818) - left_wire_dist - 0., HOP4X(14), PIP_Y(id_N818) - 1.},
          {HOP4X(14), PIP_Y(id_N818) - 1., HOP4X(12), PIP_Y(id_N818) - left_wire_dist - 1.},
          {HOP4X(12), PIP_Y(id_N818) - left_wire_dist - 1., HOP4X(12), -wrap_len - 1.},
          {HOP4X(12), -wrap_len - 1., HOP4X(13), -wrap_len - 1.},
          {HOP4X(13), -wrap_len - 1., HOP4X(13), PIP_Y(id_N810) - 1.},
          {HOP4X(13), PIP_Y(id_N810) - 1., HOP4X(11), PIP_Y(id_N810) + left_wire_dist - 1.},
          {HOP4X(11), PIP_Y(id_N810) + left_wire_dist - 1., HOP4X(11), PIP_Y(id_N810) - 0.},
          {HOP4X(11), PIP_Y(id_N810) - 0., HOP4X(9), PIP_Y(id_N810) + left_wire_dist - 0.},
          {HOP4X(9), PIP_Y(id_N810) + left_wire_dist - 0., HOP4X(9), PIP_Y(id_N810) + 1.},
          {HOP4X(9), PIP_Y(id_N810) + 1., HOP4X(7), PIP_Y(id_N810) + left_wire_dist + 1.},
          {HOP4X(9), PIP_Y(id_N814) + 1., WIRE_X(0), PIP_Y(id_N814) + 1.},
          {HOP4X(7), PIP_Y(id_N810) + left_wire_dist + 1., HOP4X(7), PIP_Y(id_N810) + 2.},
          {HOP4X(7), PIP_Y(id_N810) + 2., HOP4X(5), PIP_Y(id_N810) + left_wire_dist + 2.},
          {HOP4X(5), PIP_Y(id_N810) + left_wire_dist + 2., HOP4X(5), PIP_Y(id_N810) + 3.},
          {HOP4X(5), PIP_Y(id_N810) + 3., HOP4X(3), PIP_Y(id_N810) + left_wire_dist + 3.},
          {HOP4X(3), PIP_Y(id_N810) + left_wire_dist + 3., HOP4X(3), PIP_Y(id_N810) + 4.},
          {HOP4X(3), PIP_Y(id_N810) + 4., HOP4X(1), PIP_Y(id_N810) + left_wire_dist + 4.},
          {HOP4X(1), PIP_Y(id_N810) + left_wire_dist + 4., HOP4X(1), PIP_Y(id_N818) + 5.},
          {HOP4X(1), PIP_Y(id_N818) + 5., WIRE_X(0), PIP_Y(id_N818) + 5.}}},
        {id_S81_loop2,
         {{WIRE_X(0), PIP_Y(id_S810), HOP4X(16), PIP_Y(id_S810)},
          {HOP4X(16), PIP_Y(id_S810), HOP4X(16), PIP_Y(id_N818)},
          {HOP4X(16), PIP_Y(id_N818) - 0., HOP4X(14), PIP_Y(id_N818) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N818) - left_wire_dist - 0., HOP4X(14), PIP_Y(id_N818) - 1.},
          {HOP4X(14), PIP_Y(id_N818) - 1., HOP4X(12), PIP_Y(id_N818) - left_wire_dist - 1.},
          {HOP4X(12), PIP_Y(id_N818) - left_wire_dist - 1., HOP4X(12), PIP_Y(id_N818) - 2.},
          {HOP4X(12), PIP_Y(id_N818) - 2., HOP4X(10), PIP_Y(id_N818) - left_wire_dist - 2.},
          {HOP4X(10), PIP_Y(id_N818) - left_wire_dist - 2., HOP4X(10), -wrap_len - 2.},
          {HOP4X(10), -wrap_len - 2., HOP4X(11), -wrap_len - 2.},
          {HOP4X(11), -wrap_len - 2., HOP4X(11), PIP_Y(id_N810) - 2.},
          {HOP4X(11), PIP_Y(id_N810) - 2., HOP4X(9), PIP_Y(id_N810) + left_wire_dist - 2.},
          {HOP4X(9), PIP_Y(id_N810) + left_wire_dist - 2., HOP4X(9), PIP_Y(id_N810) - 1.},
          {HOP4X(9), PIP_Y(id_N810) - 1., HOP4X(7), PIP_Y(id_N810) + left_wire_dist - 1.},
          {HOP4X(9), PIP_Y(id_N814) - 1., WIRE_X(0), PIP_Y(id_N814) - 1.},
          {HOP4X(7), PIP_Y(id_N810) + left_wire_dist - 1., HOP4X(7), PIP_Y(id_N810) - 0.},
          {HOP4X(7), PIP_Y(id_N810) - 0., HOP4X(5), PIP_Y(id_N810) + left_wire_dist - 0.},
          {HOP4X(5), PIP_Y(id_N810) + left_wire_dist - 0., HOP4X(5), PIP_Y(id_N810) + 1.},
          {HOP4X(5), PIP_Y(id_N810) + 1., HOP4X(3), PIP_Y(id_N810) + left_wire_dist + 1.},
          {HOP4X(3), PIP_Y(id_N810) + left_wire_dist + 1., HOP4X(3), PIP_Y(id_N810) + 2.},
          {HOP4X(3), PIP_Y(id_N810) + 2., HOP4X(1), PIP_Y(id_N810) + left_wire_dist + 2.},
          {HOP4X(1), PIP_Y(id_N810) + left_wire_dist + 2., HOP4X(1), PIP_Y(id_N818) + 3.},
          {HOP4X(1), PIP_Y(id_N818) + 3., WIRE_X(0), PIP_Y(id_N818) + 3.}}},
        {id_S81_loop3,
         {{WIRE_X(0), PIP_Y(id_S810), HOP4X(16), PIP_Y(id_S810)},
          {HOP4X(16), PIP_Y(id_S810), HOP4X(16), PIP_Y(id_N818)},
          {HOP4X(16), PIP_Y(id_N818) - 0., HOP4X(14), PIP_Y(id_N818) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N818) - left_wire_dist - 0., HOP4X(14), PIP_Y(id_N818) - 1.},
          {HOP4X(14), PIP_Y(id_N818) - 1., HOP4X(12), PIP_Y(id_N818) - left_wire_dist - 1.},
          {HOP4X(12), PIP_Y(id_N818) - left_wire_dist - 1., HOP4X(12), PIP_Y(id_N818) - 2.},
          {HOP4X(12), PIP_Y(id_N818) - 2., HOP4X(10), PIP_Y(id_N818) - left_wire_dist - 2.},
          {HOP4X(10), PIP_Y(id_N818) - left_wire_dist - 2., HOP4X(10), PIP_Y(id_N818) - 3.},
          {HOP4X(10), PIP_Y(id_N818) - 3., HOP4X(8), PIP_Y(id_N818) - left_wire_dist - 3.},
          {HOP4X(8), PIP_Y(id_N818) - left_wire_dist - 3., HOP4X(8), -wrap_len - 3.},
          {HOP4X(8), -wrap_len - 3., HOP4X(9), -wrap_len - 3.},
          {HOP4X(9), -wrap_len - 3., HOP4X(9), PIP_Y(id_N810) - 3.},
          {HOP4X(9), PIP_Y(id_N814) - 3., WIRE_X(0), PIP_Y(id_N814) - 3.},
          {HOP4X(9), PIP_Y(id_N810) - 3., HOP4X(7), PIP_Y(id_N810) + left_wire_dist - 3.},
          {HOP4X(7), PIP_Y(id_N810) + left_wire_dist - 3., HOP4X(7), PIP_Y(id_N810) - 2.},
          {HOP4X(7), PIP_Y(id_N810) - 2., HOP4X(5), PIP_Y(id_N810) + left_wire_dist - 2.},
          {HOP4X(5), PIP_Y(id_N810) + left_wire_dist - 2., HOP4X(5), PIP_Y(id_N810) - 1.},
          {HOP4X(5), PIP_Y(id_N810) - 1., HOP4X(3), PIP_Y(id_N810) + left_wire_dist - 1.},
          {HOP4X(3), PIP_Y(id_N810) + left_wire_dist - 1., HOP4X(3), PIP_Y(id_N810) - 0.},
          {HOP4X(3), PIP_Y(id_N810) - 0., HOP4X(1), PIP_Y(id_N810) + left_wire_dist - 0.},
          {HOP4X(1), PIP_Y(id_N810) + left_wire_dist - 0., HOP4X(1), PIP_Y(id_N818) + 1.},
          {HOP4X(1), PIP_Y(id_N818) + 1., WIRE_X(0), PIP_Y(id_N818) + 1.}}},
        {id_S81_loop4,
         {{WIRE_X(0), PIP_Y(id_S810), HOP4X(16), PIP_Y(id_S810)},
          {HOP4X(16), PIP_Y(id_S810), HOP4X(16), PIP_Y(id_N818)},
          {HOP4X(16), PIP_Y(id_N818) - 0., HOP4X(14), PIP_Y(id_N818) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N818) - left_wire_dist - 0., HOP4X(14), PIP_Y(id_N818) - 1.},
          {HOP4X(14), PIP_Y(id_N818) - 1., HOP4X(12), PIP_Y(id_N818) - left_wire_dist - 1.},
          {HOP4X(12), PIP_Y(id_N818) - left_wire_dist - 1., HOP4X(12), PIP_Y(id_N818) - 2.},
          {HOP4X(12), PIP_Y(id_N818) - 2., HOP4X(10), PIP_Y(id_N818) - left_wire_dist - 2.},
          {HOP4X(10), PIP_Y(id_N818) - left_wire_dist - 2., HOP4X(10), PIP_Y(id_N818) - 3.},
          {HOP4X(10), PIP_Y(id_N818) - 3., HOP4X(8), PIP_Y(id_N818) - left_wire_dist - 3.},
          {HOP4X(8), PIP_Y(id_N818) - left_wire_dist - 3., HOP4X(8), PIP_Y(id_N818) - 4.},
          {HOP4X(8), PIP_Y(id_S814) - 4., WIRE_X(0), PIP_Y(id_S814) - 4.},
          {HOP4X(8), PIP_Y(id_N818) - 4., HOP4X(6), PIP_Y(id_N818) - left_wire_dist - 4.},
          {HOP4X(6), PIP_Y(id_N818) - left_wire_dist - 4., HOP4X(6), -wrap_len - 4.},
          {HOP4X(6), -wrap_len - 4., HOP4X(7), -wrap_len - 4.},
          {HOP4X(7), -wrap_len - 4., HOP4X(7), PIP_Y(id_N810) - 4.},
          {HOP4X(7), PIP_Y(id_N810) - 4., HOP4X(5), PIP_Y(id_N810) + left_wire_dist - 4.},
          {HOP4X(5), PIP_Y(id_N810) + left_wire_dist - 4., HOP4X(5), PIP_Y(id_N810) - 3.},
          {HOP4X(5), PIP_Y(id_N810) - 3., HOP4X(3), PIP_Y(id_N810) + left_wire_dist - 3.},
          {HOP4X(3), PIP_Y(id_N810) + left_wire_dist - 3., HOP4X(3), PIP_Y(id_N810) - 2.},
          {HOP4X(3), PIP_Y(id_N810) - 2., HOP4X(1), PIP_Y(id_N810) + left_wire_dist - 2.},
          {HOP4X(1), PIP_Y(id_N810) + left_wire_dist - 2., HOP4X(1), PIP_Y(id_N818) - 1.},
          {HOP4X(1), PIP_Y(id_N818) - 1., WIRE_X(0), PIP_Y(id_N818) - 1.}}},
        {id_S81_loop5,
         {{WIRE_X(0), PIP_Y(id_S810), HOP4X(16), PIP_Y(id_S810)},
          {HOP4X(16), PIP_Y(id_S810), HOP4X(16), PIP_Y(id_N818)},
          {HOP4X(16), PIP_Y(id_N818) - 0., HOP4X(14), PIP_Y(id_N818) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N818) - left_wire_dist - 0., HOP4X(14), PIP_Y(id_N818) - 1.},
          {HOP4X(14), PIP_Y(id_N818) - 1., HOP4X(12), PIP_Y(id_N818) - left_wire_dist - 1.},
          {HOP4X(12), PIP_Y(id_N818) - left_wire_dist - 1., HOP4X(12), PIP_Y(id_N818) - 2.},
          {HOP4X(12), PIP_Y(id_N818) - 2., HOP4X(10), PIP_Y(id_N818) - left_wire_dist - 2.},
          {HOP4X(10), PIP_Y(id_N818) - left_wire_dist - 2., HOP4X(10), PIP_Y(id_N818) - 3.},
          {HOP4X(10), PIP_Y(id_N818) - 3., HOP4X(8), PIP_Y(id_N818) - left_wire_dist - 3.},
          {HOP4X(8), PIP_Y(id_N818) - left_wire_dist - 3., HOP4X(8), PIP_Y(id_N818) - 4.},
          {HOP4X(8), PIP_Y(id_S814) - 4., WIRE_X(0), PIP_Y(id_S814) - 4.},
          {HOP4X(8), PIP_Y(id_N818) - 4., HOP4X(6), PIP_Y(id_N818) - left_wire_dist - 4.},
          {HOP4X(6), PIP_Y(id_N818) - left_wire_dist - 4., HOP4X(6), PIP_Y(id_N818) - 5.},
          {HOP4X(6), PIP_Y(id_N818) - 5., HOP4X(4), PIP_Y(id_N818) - left_wire_dist - 5.},
          {HOP4X(4), PIP_Y(id_N818) - left_wire_dist - 5., HOP4X(4), -wrap_len - 5.},
          {HOP4X(4), -wrap_len - 5., HOP4X(5), -wrap_len - 5.},
          {HOP4X(5), -wrap_len - 5., HOP4X(5), PIP_Y(id_N810) - 5.},
          {HOP4X(5), PIP_Y(id_N810) - 5., HOP4X(3), PIP_Y(id_N810) + left_wire_dist - 5.},
          {HOP4X(3), PIP_Y(id_N810) + left_wire_dist - 5., HOP4X(3), PIP_Y(id_N810) - 4.},
          {HOP4X(3), PIP_Y(id_N810) - 4., HOP4X(1), PIP_Y(id_N810) + left_wire_dist - 4.},
          {HOP4X(1), PIP_Y(id_N810) + left_wire_dist - 4., HOP4X(1), PIP_Y(id_N818) - 3.},
          {HOP4X(1), PIP_Y(id_N818) - 3., WIRE_X(0), PIP_Y(id_N818) - 3.}}},
        {id_S81_loop6,
         {{WIRE_X(0), PIP_Y(id_S810), HOP4X(16), PIP_Y(id_S810)},
          {HOP4X(16), PIP_Y(id_S810), HOP4X(16), PIP_Y(id_N818)},
          {HOP4X(16), PIP_Y(id_N818) - 0., HOP4X(14), PIP_Y(id_N818) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N818) - left_wire_dist - 0., HOP4X(14), PIP_Y(id_N818) - 1.},
          {HOP4X(14), PIP_Y(id_N818) - 1., HOP4X(12), PIP_Y(id_N818) - left_wire_dist - 1.},
          {HOP4X(12), PIP_Y(id_N818) - left_wire_dist - 1., HOP4X(12), PIP_Y(id_N818) - 2.},
          {HOP4X(12), PIP_Y(id_N818) - 2., HOP4X(10), PIP_Y(id_N818) - left_wire_dist - 2.},
          {HOP4X(10), PIP_Y(id_N818) - left_wire_dist - 2., HOP4X(10), PIP_Y(id_N818) - 3.},
          {HOP4X(10), PIP_Y(id_N818) - 3., HOP4X(8), PIP_Y(id_N818) - left_wire_dist - 3.},
          {HOP4X(8), PIP_Y(id_N818) - left_wire_dist - 3., HOP4X(8), PIP_Y(id_N818) - 4.},
          {HOP4X(8), PIP_Y(id_S814) - 4., WIRE_X(0), PIP_Y(id_S814) - 4.},
          {HOP4X(8), PIP_Y(id_N818) - 4., HOP4X(6), PIP_Y(id_N818) - left_wire_dist - 4.},
          {HOP4X(6), PIP_Y(id_N818) - left_wire_dist - 4., HOP4X(6), PIP_Y(id_N818) - 5.},
          {HOP4X(6), PIP_Y(id_N818) - 5., HOP4X(4), PIP_Y(id_N818) - left_wire_dist - 5.},
          {HOP4X(4), PIP_Y(id_N818) - left_wire_dist - 5., HOP4X(4), PIP_Y(id_N818) - 6.},
          {HOP4X(4), PIP_Y(id_N818) - 6., HOP4X(2), PIP_Y(id_N818) - left_wire_dist - 6.},
          {HOP4X(2), PIP_Y(id_N818) - left_wire_dist - 6., HOP4X(2), -wrap_len - 6.},
          {HOP4X(2), -wrap_len - 6., HOP4X(3), -wrap_len - 6.},
          {HOP4X(3), -wrap_len - 6., HOP4X(3), PIP_Y(id_N810) - 6.},
          {HOP4X(3), PIP_Y(id_N810) - 6., HOP4X(1), PIP_Y(id_N810) + left_wire_dist - 6.},
          {HOP4X(1), PIP_Y(id_N810) + left_wire_dist - 6., HOP4X(1), PIP_Y(id_N818) - 5.},
          {HOP4X(1), PIP_Y(id_N818) - 5., WIRE_X(0), PIP_Y(id_N818) - 5.}}},
        {id_S81_loop7,
         {{WIRE_X(0), PIP_Y(id_S810), HOP4X(16), PIP_Y(id_S810)},
          {HOP4X(16), PIP_Y(id_S810), HOP4X(16), PIP_Y(id_N818)},
          {HOP4X(16), PIP_Y(id_N818) - 0., HOP4X(14), PIP_Y(id_N818) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N818) - left_wire_dist - 0., HOP4X(14), PIP_Y(id_N818) - 1.},
          {HOP4X(14), PIP_Y(id_N818) - 1., HOP4X(12), PIP_Y(id_N818) - left_wire_dist - 1.},
          {HOP4X(12), PIP_Y(id_N818) - left_wire_dist - 1., HOP4X(12), PIP_Y(id_N818) - 2.},
          {HOP4X(12), PIP_Y(id_N818) - 2., HOP4X(10), PIP_Y(id_N818) - left_wire_dist - 2.},
          {HOP4X(10), PIP_Y(id_N818) - left_wire_dist - 2., HOP4X(10), PIP_Y(id_N818) - 3.},
          {HOP4X(10), PIP_Y(id_N818) - 3., HOP4X(8), PIP_Y(id_N818) - left_wire_dist - 3.},
          {HOP4X(8), PIP_Y(id_N818) - left_wire_dist - 3., HOP4X(8), PIP_Y(id_N818) - 4.},
          {HOP4X(8), PIP_Y(id_S814) - 4., WIRE_X(0), PIP_Y(id_S814) - 4.},
          {HOP4X(8), PIP_Y(id_N818) - 4., HOP4X(6), PIP_Y(id_N818) - left_wire_dist - 4.},
          {HOP4X(6), PIP_Y(id_N818) - left_wire_dist - 4., HOP4X(6), PIP_Y(id_N818) - 5.},
          {HOP4X(6), PIP_Y(id_N818) - 5., HOP4X(4), PIP_Y(id_N818) - left_wire_dist - 5.},
          {HOP4X(4), PIP_Y(id_N818) - left_wire_dist - 5., HOP4X(4), PIP_Y(id_N818) - 6.},
          {HOP4X(4), PIP_Y(id_N818) - 6., HOP4X(2), PIP_Y(id_N818) - left_wire_dist - 6.},
          {HOP4X(2), PIP_Y(id_N818) - left_wire_dist - 6., HOP4X(2), PIP_Y(id_N818) - 7.},
          {HOP4X(2), PIP_Y(id_N818) - 7., HOP4X(0), PIP_Y(id_N818) - left_wire_dist - 7.},
          {HOP4X(0), PIP_Y(id_N818) - left_wire_dist - 7., HOP4X(0), -wrap_len - 7.},
          {HOP4X(0), -wrap_len - 7., HOP4X(1), -wrap_len - 7.},
          {HOP4X(1), -wrap_len - 7., HOP4X(1), PIP_Y(id_N818) - 7.},
          {HOP4X(1), PIP_Y(id_N818) - 7., WIRE_X(0), PIP_Y(id_N818) - 7.}}},
        {id_N81_loop0,
         {{WIRE_X(0), PIP_Y(id_N810), HOP4X(17), PIP_Y(id_N810)},
          {HOP4X(17), PIP_Y(id_N810) + 0., HOP4X(15), PIP_Y(id_N810) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N810) + left_wire_dist + 0., HOP4X(15), wrap_len + 1.},
          {HOP4X(15), wrap_len + 1., HOP4X(14), wrap_len + 1.},
          {HOP4X(14), wrap_len + 1., HOP4X(14), PIP_Y(id_N818) + 0.},
          {HOP4X(14), PIP_Y(id_N818) + 0., HOP4X(12), PIP_Y(id_N818) - left_wire_dist + 0.},
          {HOP4X(12), PIP_Y(id_N818) - left_wire_dist + 0., HOP4X(12), PIP_Y(id_N818) - 1.},
          {HOP4X(12), PIP_Y(id_N818) - 1., HOP4X(10), PIP_Y(id_N818) - left_wire_dist - 1.},
          {HOP4X(10), PIP_Y(id_N818) - left_wire_dist - 1., HOP4X(10), PIP_Y(id_N818) - 2.},
          {HOP4X(10), PIP_Y(id_N818) - 2., HOP4X(8), PIP_Y(id_N818) - left_wire_dist - 2.},
          {HOP4X(8), PIP_Y(id_N818) - left_wire_dist - 2., HOP4X(8), PIP_Y(id_N818) - 3.},
          {HOP4X(8), PIP_Y(id_S814) - 3., WIRE_X(0), PIP_Y(id_S814) - 3.},
          {HOP4X(8), PIP_Y(id_N818) - 3., HOP4X(6), PIP_Y(id_N818) - left_wire_dist - 3.},
          {HOP4X(6), PIP_Y(id_N818) - left_wire_dist - 3., HOP4X(6), PIP_Y(id_N818) - 4.},
          {HOP4X(6), PIP_Y(id_N818) - 4., HOP4X(4), PIP_Y(id_N818) - left_wire_dist - 4.},
          {HOP4X(4), PIP_Y(id_N818) - left_wire_dist - 4., HOP4X(4), PIP_Y(id_N818) - 5.},
          {HOP4X(4), PIP_Y(id_N818) - 5., HOP4X(2), PIP_Y(id_N818) - left_wire_dist - 5.},
          {HOP4X(2), PIP_Y(id_N818) - left_wire_dist - 5., HOP4X(2), PIP_Y(id_N818) - 6.},
          {HOP4X(2), PIP_Y(id_N818) - 6., HOP4X(0), PIP_Y(id_N818) - left_wire_dist - 6.},
          {HOP4X(0), PIP_Y(id_N818) - left_wire_dist - 6., HOP4X(0), PIP_Y(id_S818) - 7.},
          {HOP4X(0), PIP_Y(id_S818) - 7., WIRE_X(0), PIP_Y(id_S818) - 7.}}},
        {id_N81_loop1,
         {{WIRE_X(0), PIP_Y(id_N810), HOP4X(17), PIP_Y(id_N810)},
          {HOP4X(17), PIP_Y(id_N810) + 0., HOP4X(15), PIP_Y(id_N810) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N810) + left_wire_dist + 0., HOP4X(15), PIP_Y(id_N810) + 1.},
          {HOP4X(15), PIP_Y(id_N810) + 1., HOP4X(13), PIP_Y(id_N810) + left_wire_dist + 1.},
          {HOP4X(13), PIP_Y(id_N810) + left_wire_dist + 1., HOP4X(13), wrap_len + 2.},
          {HOP4X(13), wrap_len + 2., HOP4X(12), wrap_len + 2.},
          {HOP4X(12), wrap_len + 2., HOP4X(12), PIP_Y(id_N818) + 1.},
          {HOP4X(12), PIP_Y(id_N818) + 1., HOP4X(10), PIP_Y(id_N818) - left_wire_dist + 1.},
          {HOP4X(10), PIP_Y(id_N818) - left_wire_dist + 1., HOP4X(10), PIP_Y(id_N818) + 0.},
          {HOP4X(10), PIP_Y(id_N818) + 0., HOP4X(8), PIP_Y(id_N818) - left_wire_dist + 0.},
          {HOP4X(8), PIP_Y(id_N818) - left_wire_dist + 0., HOP4X(8), PIP_Y(id_N818) - 1.},
          {HOP4X(8), PIP_Y(id_S814) - 1., WIRE_X(0), PIP_Y(id_S814) - 1.},
          {HOP4X(8), PIP_Y(id_N818) - 1., HOP4X(6), PIP_Y(id_N818) - left_wire_dist - 1.},
          {HOP4X(6), PIP_Y(id_N818) - left_wire_dist - 1., HOP4X(6), PIP_Y(id_N818) - 2.},
          {HOP4X(6), PIP_Y(id_N818) - 2., HOP4X(4), PIP_Y(id_N818) - left_wire_dist - 2.},
          {HOP4X(4), PIP_Y(id_N818) - left_wire_dist - 2., HOP4X(4), PIP_Y(id_N818) - 3.},
          {HOP4X(4), PIP_Y(id_N818) - 3., HOP4X(2), PIP_Y(id_N818) - left_wire_dist - 3.},
          {HOP4X(2), PIP_Y(id_N818) - left_wire_dist - 3., HOP4X(2), PIP_Y(id_N818) - 4.},
          {HOP4X(2), PIP_Y(id_N818) - 4., HOP4X(0), PIP_Y(id_N818) - left_wire_dist - 4.},
          {HOP4X(0), PIP_Y(id_N818) - left_wire_dist - 4., HOP4X(0), PIP_Y(id_S818) - 5.},
          {HOP4X(0), PIP_Y(id_S818) - 5., WIRE_X(0), PIP_Y(id_S818) - 5.}}},
        {id_N81_loop2,
         {{WIRE_X(0), PIP_Y(id_N810), HOP4X(17), PIP_Y(id_N810)},
          {HOP4X(17), PIP_Y(id_N810) + 0., HOP4X(15), PIP_Y(id_N810) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N810) + left_wire_dist + 0., HOP4X(15), PIP_Y(id_N810) + 1.},
          {HOP4X(15), PIP_Y(id_N810) + 1., HOP4X(13), PIP_Y(id_N810) + left_wire_dist + 1.},
          {HOP4X(13), PIP_Y(id_N810) + left_wire_dist + 1., HOP4X(13), PIP_Y(id_N810) + 2.},
          {HOP4X(13), PIP_Y(id_N810) + 2., HOP4X(11), PIP_Y(id_N810) + left_wire_dist + 2.},
          {HOP4X(11), PIP_Y(id_N810) + left_wire_dist + 2., HOP4X(11), wrap_len + 3.},
          {HOP4X(11), wrap_len + 3., HOP4X(10), wrap_len + 3.},
          {HOP4X(10), wrap_len + 3., HOP4X(10), PIP_Y(id_N818) + 2.},
          {HOP4X(10), PIP_Y(id_N818) + 2., HOP4X(8), PIP_Y(id_N818) - left_wire_dist + 2.},
          {HOP4X(8), PIP_Y(id_N818) - left_wire_dist + 2., HOP4X(8), PIP_Y(id_N818) + 1.},
          {HOP4X(8), PIP_Y(id_S814) + 1., WIRE_X(0), PIP_Y(id_S814) + 1.},
          {HOP4X(8), PIP_Y(id_N818) + 1., HOP4X(6), PIP_Y(id_N818) - left_wire_dist + 1.},
          {HOP4X(6), PIP_Y(id_N818) - left_wire_dist + 1., HOP4X(6), PIP_Y(id_N818) + 0.},
          {HOP4X(6), PIP_Y(id_N818) + 0., HOP4X(4), PIP_Y(id_N818) - left_wire_dist + 0.},
          {HOP4X(4), PIP_Y(id_N818) - left_wire_dist + 0., HOP4X(4), PIP_Y(id_N818) - 1.},
          {HOP4X(4), PIP_Y(id_N818) - 1., HOP4X(2), PIP_Y(id_N818) - left_wire_dist - 1.},
          {HOP4X(2), PIP_Y(id_N818) - left_wire_dist - 1., HOP4X(2), PIP_Y(id_N818) - 2.},
          {HOP4X(2), PIP_Y(id_N818) - 2., HOP4X(0), PIP_Y(id_N818) - left_wire_dist - 2.},
          {HOP4X(0), PIP_Y(id_N818) - left_wire_dist - 2., HOP4X(0), PIP_Y(id_S818) - 3.},
          {HOP4X(0), PIP_Y(id_S818) - 3., WIRE_X(0), PIP_Y(id_S818) - 3.}}},
        {id_N81_loop3,
         {{WIRE_X(0), PIP_Y(id_N810), HOP4X(17), PIP_Y(id_N810)},
          {HOP4X(17), PIP_Y(id_N810) + 0., HOP4X(15), PIP_Y(id_N810) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N810) + left_wire_dist + 0., HOP4X(15), PIP_Y(id_N810) + 1.},
          {HOP4X(15), PIP_Y(id_N810) + 1., HOP4X(13), PIP_Y(id_N810) + left_wire_dist + 1.},
          {HOP4X(13), PIP_Y(id_N810) + left_wire_dist + 1., HOP4X(13), PIP_Y(id_N810) + 2.},
          {HOP4X(13), PIP_Y(id_N810) + 2., HOP4X(11), PIP_Y(id_N810) + left_wire_dist + 2.},
          {HOP4X(11), PIP_Y(id_N810) + left_wire_dist + 2., HOP4X(11), PIP_Y(id_N810) + 3.},
          {HOP4X(11), PIP_Y(id_N810) + 3., HOP4X(9), PIP_Y(id_N810) + left_wire_dist + 3.},
          {HOP4X(9), PIP_Y(id_N810) + left_wire_dist + 3., HOP4X(9), wrap_len + 4.},
          {HOP4X(9), wrap_len + 4., HOP4X(8), wrap_len + 4.},
          {HOP4X(8), wrap_len + 4., HOP4X(8), PIP_Y(id_N818) + 3.},
          {HOP4X(8), PIP_Y(id_S814) + 3., WIRE_X(0), PIP_Y(id_S814) + 3.},
          {HOP4X(8), PIP_Y(id_N818) + 3., HOP4X(6), PIP_Y(id_N818) - left_wire_dist + 3.},
          {HOP4X(6), PIP_Y(id_N818) - left_wire_dist + 3., HOP4X(6), PIP_Y(id_N818) + 2.},
          {HOP4X(6), PIP_Y(id_N818) + 2., HOP4X(4), PIP_Y(id_N818) - left_wire_dist + 2.},
          {HOP4X(4), PIP_Y(id_N818) - left_wire_dist + 2., HOP4X(4), PIP_Y(id_N818) + 1.},
          {HOP4X(4), PIP_Y(id_N818) + 1., HOP4X(2), PIP_Y(id_N818) - left_wire_dist + 1.},
          {HOP4X(2), PIP_Y(id_N818) - left_wire_dist + 1., HOP4X(2), PIP_Y(id_N818) + 0.},
          {HOP4X(2), PIP_Y(id_N818) + 0., HOP4X(0), PIP_Y(id_N818) - left_wire_dist + 0.},
          {HOP4X(0), PIP_Y(id_N818) - left_wire_dist + 0., HOP4X(0), PIP_Y(id_S818) - 1.},
          {HOP4X(0), PIP_Y(id_S818) - 1., WIRE_X(0), PIP_Y(id_S818) - 1.}}},
        {id_N81_loop4,
         {{WIRE_X(0), PIP_Y(id_N810), HOP4X(17), PIP_Y(id_N810)},
          {HOP4X(17), PIP_Y(id_N810) + 0., HOP4X(15), PIP_Y(id_N810) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N810) + left_wire_dist + 0., HOP4X(15), PIP_Y(id_N810) + 1.},
          {HOP4X(15), PIP_Y(id_N810) + 1., HOP4X(13), PIP_Y(id_N810) + left_wire_dist + 1.},
          {HOP4X(13), PIP_Y(id_N810) + left_wire_dist + 1., HOP4X(13), PIP_Y(id_N810) + 2.},
          {HOP4X(13), PIP_Y(id_N810) + 2., HOP4X(11), PIP_Y(id_N810) + left_wire_dist + 2.},
          {HOP4X(11), PIP_Y(id_N810) + left_wire_dist + 2., HOP4X(11), PIP_Y(id_N810) + 3.},
          {HOP4X(11), PIP_Y(id_N810) + 3., HOP4X(9), PIP_Y(id_N810) + left_wire_dist + 3.},
          {HOP4X(9), PIP_Y(id_N810) + left_wire_dist + 3., HOP4X(9), PIP_Y(id_N810) + 4.},
          {HOP4X(9), PIP_Y(id_N814) + 4., WIRE_X(0), PIP_Y(id_N814) + 4.},
          {HOP4X(9), PIP_Y(id_N810) + 4., HOP4X(7), PIP_Y(id_N810) + left_wire_dist + 4.},
          {HOP4X(7), PIP_Y(id_N810) + left_wire_dist + 4., HOP4X(7), wrap_len + 5.},
          {HOP4X(7), wrap_len + 5., HOP4X(6), wrap_len + 5.},
          {HOP4X(6), wrap_len + 5., HOP4X(6), PIP_Y(id_N818) + 4.},
          {HOP4X(6), PIP_Y(id_N818) + 4., HOP4X(4), PIP_Y(id_N818) - left_wire_dist + 4.},
          {HOP4X(4), PIP_Y(id_N818) - left_wire_dist + 4., HOP4X(4), PIP_Y(id_N818) + 3.},
          {HOP4X(4), PIP_Y(id_N818) + 3., HOP4X(2), PIP_Y(id_N818) - left_wire_dist + 3.},
          {HOP4X(2), PIP_Y(id_N818) - left_wire_dist + 3., HOP4X(2), PIP_Y(id_N818) + 2.},
          {HOP4X(2), PIP_Y(id_N818) + 2., HOP4X(0), PIP_Y(id_N818) - left_wire_dist + 2.},
          {HOP4X(0), PIP_Y(id_N818) - left_wire_dist + 2., HOP4X(0), PIP_Y(id_S818) + 1.},
          {HOP4X(0), PIP_Y(id_S818) + 1., WIRE_X(0), PIP_Y(id_S818) + 1.}}},
        {id_N81_loop5,
         {{WIRE_X(0), PIP_Y(id_N810), HOP4X(17), PIP_Y(id_N810)},
          {HOP4X(17), PIP_Y(id_N810) + 0., HOP4X(15), PIP_Y(id_N810) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N810) + left_wire_dist + 0., HOP4X(15), PIP_Y(id_N810) + 1.},
          {HOP4X(15), PIP_Y(id_N810) + 1., HOP4X(13), PIP_Y(id_N810) + left_wire_dist + 1.},
          {HOP4X(13), PIP_Y(id_N810) + left_wire_dist + 1., HOP4X(13), PIP_Y(id_N810) + 2.},
          {HOP4X(13), PIP_Y(id_N810) + 2., HOP4X(11), PIP_Y(id_N810) + left_wire_dist + 2.},
          {HOP4X(11), PIP_Y(id_N810) + left_wire_dist + 2., HOP4X(11), PIP_Y(id_N810) + 3.},
          {HOP4X(11), PIP_Y(id_N810) + 3., HOP4X(9), PIP_Y(id_N810) + left_wire_dist + 3.},
          {HOP4X(9), PIP_Y(id_N810) + left_wire_dist + 3., HOP4X(9), PIP_Y(id_N810) + 4.},
          {HOP4X(9), PIP_Y(id_N814) + 4., WIRE_X(0), PIP_Y(id_N814) + 4.},
          {HOP4X(9), PIP_Y(id_N810) + 4., HOP4X(7), PIP_Y(id_N810) + left_wire_dist + 4.},
          {HOP4X(7), PIP_Y(id_N810) + left_wire_dist + 4., HOP4X(7), PIP_Y(id_N810) + 5.},
          {HOP4X(7), PIP_Y(id_N810) + 5., HOP4X(5), PIP_Y(id_N810) + left_wire_dist + 5.},
          {HOP4X(5), PIP_Y(id_N810) + left_wire_dist + 5., HOP4X(5), wrap_len + 6.},
          {HOP4X(5), wrap_len + 6., HOP4X(4), wrap_len + 6.},
          {HOP4X(4), wrap_len + 6., HOP4X(4), PIP_Y(id_N818) + 5.},
          {HOP4X(4), PIP_Y(id_N818) + 5., HOP4X(2), PIP_Y(id_N818) - left_wire_dist + 5.},
          {HOP4X(2), PIP_Y(id_N818) - left_wire_dist + 5., HOP4X(2), PIP_Y(id_N818) + 4.},
          {HOP4X(2), PIP_Y(id_N818) + 4., HOP4X(0), PIP_Y(id_N818) - left_wire_dist + 4.},
          {HOP4X(0), PIP_Y(id_N818) - left_wire_dist + 4., HOP4X(0), PIP_Y(id_S818) + 3.},
          {HOP4X(0), PIP_Y(id_S818) + 3., WIRE_X(0), PIP_Y(id_S818) + 3.}}},
        {id_N81_loop6,
         {{WIRE_X(0), PIP_Y(id_N810), HOP4X(17), PIP_Y(id_N810)},
          {HOP4X(17), PIP_Y(id_N810) + 0., HOP4X(15), PIP_Y(id_N810) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N810) + left_wire_dist + 0., HOP4X(15), PIP_Y(id_N810) + 1.},
          {HOP4X(15), PIP_Y(id_N810) + 1., HOP4X(13), PIP_Y(id_N810) + left_wire_dist + 1.},
          {HOP4X(13), PIP_Y(id_N810) + left_wire_dist + 1., HOP4X(13), PIP_Y(id_N810) + 2.},
          {HOP4X(13), PIP_Y(id_N810) + 2., HOP4X(11), PIP_Y(id_N810) + left_wire_dist + 2.},
          {HOP4X(11), PIP_Y(id_N810) + left_wire_dist + 2., HOP4X(11), PIP_Y(id_N810) + 3.},
          {HOP4X(11), PIP_Y(id_N810) + 3., HOP4X(9), PIP_Y(id_N810) + left_wire_dist + 3.},
          {HOP4X(9), PIP_Y(id_N810) + left_wire_dist + 3., HOP4X(9), PIP_Y(id_N810) + 4.},
          {HOP4X(9), PIP_Y(id_N814) + 4., WIRE_X(0), PIP_Y(id_N814) + 4.},
          {HOP4X(9), PIP_Y(id_N810) + 4., HOP4X(7), PIP_Y(id_N810) + left_wire_dist + 4.},
          {HOP4X(7), PIP_Y(id_N810) + left_wire_dist + 4., HOP4X(7), PIP_Y(id_N810) + 5.},
          {HOP4X(7), PIP_Y(id_N810) + 5., HOP4X(5), PIP_Y(id_N810) + left_wire_dist + 5.},
          {HOP4X(5), PIP_Y(id_N810) + left_wire_dist + 5., HOP4X(5), PIP_Y(id_N810) + 6.},
          {HOP4X(5), PIP_Y(id_N810) + 6., HOP4X(3), PIP_Y(id_N810) + left_wire_dist + 6.},
          {HOP4X(3), PIP_Y(id_N810) + left_wire_dist + 6., HOP4X(3), wrap_len + 7.},
          {HOP4X(3), wrap_len + 7., HOP4X(2), wrap_len + 7.},
          {HOP4X(2), wrap_len + 7., HOP4X(2), PIP_Y(id_N818) + 6.},
          {HOP4X(2), PIP_Y(id_N818) + 6., HOP4X(0), PIP_Y(id_N818) - left_wire_dist + 6.},
          {HOP4X(0), PIP_Y(id_N818) - left_wire_dist + 6., HOP4X(0), PIP_Y(id_S818) + 5.},
          {HOP4X(0), PIP_Y(id_S818) + 5., WIRE_X(0), PIP_Y(id_S818) + 5.}}},
        {id_N81_loop7,
         {{WIRE_X(0), PIP_Y(id_N810), HOP4X(17), PIP_Y(id_N810)},
          {HOP4X(17), PIP_Y(id_N810) + 0., HOP4X(15), PIP_Y(id_N810) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N810) + left_wire_dist + 0., HOP4X(15), PIP_Y(id_N810) + 1.},
          {HOP4X(15), PIP_Y(id_N810) + 1., HOP4X(13), PIP_Y(id_N810) + left_wire_dist + 1.},
          {HOP4X(13), PIP_Y(id_N810) + left_wire_dist + 1., HOP4X(13), PIP_Y(id_N810) + 2.},
          {HOP4X(13), PIP_Y(id_N810) + 2., HOP4X(11), PIP_Y(id_N810) + left_wire_dist + 2.},
          {HOP4X(11), PIP_Y(id_N810) + left_wire_dist + 2., HOP4X(11), PIP_Y(id_N810) + 3.},
          {HOP4X(11), PIP_Y(id_N810) + 3., HOP4X(9), PIP_Y(id_N810) + left_wire_dist + 3.},
          {HOP4X(9), PIP_Y(id_N810) + left_wire_dist + 3., HOP4X(9), PIP_Y(id_N810) + 4.},
          {HOP4X(9), PIP_Y(id_N814) + 4., WIRE_X(0), PIP_Y(id_N814) + 4.},
          {HOP4X(9), PIP_Y(id_N810) + 4., HOP4X(7), PIP_Y(id_N810) + left_wire_dist + 4.},
          {HOP4X(7), PIP_Y(id_N810) + left_wire_dist + 4., HOP4X(7), PIP_Y(id_N810) + 5.},
          {HOP4X(7), PIP_Y(id_N810) + 5., HOP4X(5), PIP_Y(id_N810) + left_wire_dist + 5.},
          {HOP4X(5), PIP_Y(id_N810) + left_wire_dist + 5., HOP4X(5), PIP_Y(id_N810) + 6.},
          {HOP4X(5), PIP_Y(id_N810) + 6., HOP4X(3), PIP_Y(id_N810) + left_wire_dist + 6.},
          {HOP4X(3), PIP_Y(id_N810) + left_wire_dist + 6., HOP4X(3), PIP_Y(id_N810) + 7.},
          {HOP4X(3), PIP_Y(id_N810) + 7., HOP4X(1), PIP_Y(id_N810) + left_wire_dist + 7.},
          {HOP4X(1), PIP_Y(id_N810) + left_wire_dist + 7., HOP4X(1), wrap_len + 8.},
          {HOP4X(1), wrap_len + 8., HOP4X(0), wrap_len + 8.},
          {HOP4X(0), wrap_len + 8., HOP4X(0), PIP_Y(id_S818) + 7.},
          {HOP4X(0), PIP_Y(id_S818) + 7., WIRE_X(0), PIP_Y(id_S818) + 7.}}},

#undef HOP4X
#define HOP4X(offset) WIRE_X((float)offset + HOP4X_START + 18.f + 18.f)
        {id_S82,
         {{WIRE_X(0), PIP_Y(id_S820), HOP4X(16), PIP_Y(id_S820)},
          {HOP4X(16), PIP_Y(id_S820), HOP4X(16), PIP_Y(id_N828)},
          {HOP4X(16), PIP_Y(id_N828) - 0., HOP4X(14), PIP_Y(id_N828) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N828) - left_wire_dist - 0., HOP4X(14), PIP_Y(id_N828) - 1.},
          {HOP4X(14), PIP_Y(id_N828) - 1., HOP4X(12), PIP_Y(id_N828) - left_wire_dist - 1.},
          {HOP4X(12), PIP_Y(id_N828) - left_wire_dist - 1., HOP4X(12), PIP_Y(id_N828) - 2.},
          {HOP4X(12), PIP_Y(id_N828) - 2., HOP4X(10), PIP_Y(id_N828) - left_wire_dist - 2.},
          {HOP4X(10), PIP_Y(id_N828) - left_wire_dist - 2., HOP4X(10), PIP_Y(id_N828) - 3.},
          {HOP4X(10), PIP_Y(id_N828) - 3., HOP4X(8), PIP_Y(id_N828) - left_wire_dist - 3.},
          {HOP4X(8), PIP_Y(id_N828) - left_wire_dist - 3., HOP4X(8), PIP_Y(id_N828) - 4.},
          {HOP4X(8), PIP_Y(id_S824) - 4., WIRE_X(0), PIP_Y(id_S824) - 4.},
          {HOP4X(8), PIP_Y(id_N828) - 4., HOP4X(6), PIP_Y(id_N828) - left_wire_dist - 4.},
          {HOP4X(6), PIP_Y(id_N828) - left_wire_dist - 4., HOP4X(6), PIP_Y(id_N828) - 5.},
          {HOP4X(6), PIP_Y(id_N828) - 5., HOP4X(4), PIP_Y(id_N828) - left_wire_dist - 5.},
          {HOP4X(4), PIP_Y(id_N828) - left_wire_dist - 5., HOP4X(4), PIP_Y(id_N828) - 6.},
          {HOP4X(4), PIP_Y(id_N828) - 6., HOP4X(2), PIP_Y(id_N828) - left_wire_dist - 6.},
          {HOP4X(2), PIP_Y(id_N828) - left_wire_dist - 6., HOP4X(2), PIP_Y(id_N828) - 7.},
          {HOP4X(2), PIP_Y(id_N828) - 7., HOP4X(0), PIP_Y(id_N828) - left_wire_dist - 7.},
          {HOP4X(0), PIP_Y(id_N828) - left_wire_dist - 7., HOP4X(0), PIP_Y(id_S828) - 8.},
          {HOP4X(0), PIP_Y(id_S828) - 8., WIRE_X(0), PIP_Y(id_S828) - 8.}}},
        {id_N82,
         {{WIRE_X(0), PIP_Y(id_N820), HOP4X(17), PIP_Y(id_N820)},
          {HOP4X(17), PIP_Y(id_N820) + 0., HOP4X(15), PIP_Y(id_N820) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N820) + left_wire_dist + 0., HOP4X(15), PIP_Y(id_N820) + 1.},
          {HOP4X(15), PIP_Y(id_N820) + 1., HOP4X(13), PIP_Y(id_N820) + left_wire_dist + 1.},
          {HOP4X(13), PIP_Y(id_N820) + left_wire_dist + 1., HOP4X(13), PIP_Y(id_N820) + 2.},
          {HOP4X(13), PIP_Y(id_N820) + 2., HOP4X(11), PIP_Y(id_N820) + left_wire_dist + 2.},
          {HOP4X(11), PIP_Y(id_N820) + left_wire_dist + 2., HOP4X(11), PIP_Y(id_N820) + 3.},
          {HOP4X(11), PIP_Y(id_N820) + 3., HOP4X(9), PIP_Y(id_N820) + left_wire_dist + 3.},
          {HOP4X(9), PIP_Y(id_N820) + left_wire_dist + 3., HOP4X(9), PIP_Y(id_N820) + 4.},
          {HOP4X(9), PIP_Y(id_N824) + 4., WIRE_X(0), PIP_Y(id_N824) + 4.},
          {HOP4X(9), PIP_Y(id_N820) + 4., HOP4X(7), PIP_Y(id_N820) + left_wire_dist + 4.},
          {HOP4X(7), PIP_Y(id_N820) + left_wire_dist + 4., HOP4X(7), PIP_Y(id_N820) + 5.},
          {HOP4X(7), PIP_Y(id_N820) + 5., HOP4X(5), PIP_Y(id_N820) + left_wire_dist + 5.},
          {HOP4X(5), PIP_Y(id_N820) + left_wire_dist + 5., HOP4X(5), PIP_Y(id_N820) + 6.},
          {HOP4X(5), PIP_Y(id_N820) + 6., HOP4X(3), PIP_Y(id_N820) + left_wire_dist + 6.},
          {HOP4X(3), PIP_Y(id_N820) + left_wire_dist + 6., HOP4X(3), PIP_Y(id_N820) + 7.},
          {HOP4X(3), PIP_Y(id_N820) + 7., HOP4X(1), PIP_Y(id_N820) + left_wire_dist + 7.},
          {HOP4X(1), PIP_Y(id_N820) + left_wire_dist + 7., HOP4X(1), PIP_Y(id_N828) + 8.},
          {HOP4X(1), PIP_Y(id_N828) + 8., WIRE_X(0), PIP_Y(id_N828) + 8.}}},
        {id_S82_loop0,
         {{WIRE_X(0), PIP_Y(id_S820), HOP4X(16), PIP_Y(id_S820)},
          {HOP4X(16), PIP_Y(id_S820), HOP4X(16), PIP_Y(id_N828)},
          {HOP4X(16), PIP_Y(id_N828) - 0., HOP4X(14), PIP_Y(id_N828) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N828) - left_wire_dist - 0., HOP4X(14), -wrap_len - 0.},
          {HOP4X(14), -wrap_len - 0., HOP4X(15), -wrap_len - 0.},
          {HOP4X(15), -wrap_len - 0., HOP4X(15), PIP_Y(id_N820) - 0.},
          {HOP4X(15), PIP_Y(id_N820) - 0., HOP4X(13), PIP_Y(id_N820) + left_wire_dist - 0.},
          {HOP4X(13), PIP_Y(id_N820) + left_wire_dist - 0., HOP4X(13), PIP_Y(id_N820) + 1.},
          {HOP4X(13), PIP_Y(id_N820) + 1., HOP4X(11), PIP_Y(id_N820) + left_wire_dist + 1.},
          {HOP4X(11), PIP_Y(id_N820) + left_wire_dist + 1., HOP4X(11), PIP_Y(id_N820) + 2.},
          {HOP4X(11), PIP_Y(id_N820) + 2., HOP4X(9), PIP_Y(id_N820) + left_wire_dist + 2.},
          {HOP4X(9), PIP_Y(id_N820) + left_wire_dist + 2., HOP4X(9), PIP_Y(id_N820) + 3.},
          {HOP4X(9), PIP_Y(id_N820) + 3., HOP4X(7), PIP_Y(id_N820) + left_wire_dist + 3.},
          {HOP4X(9), PIP_Y(id_N824) + 3., WIRE_X(0), PIP_Y(id_N824) + 3.},
          {HOP4X(7), PIP_Y(id_N820) + left_wire_dist + 3., HOP4X(7), PIP_Y(id_N820) + 4.},
          {HOP4X(7), PIP_Y(id_N820) + 4., HOP4X(5), PIP_Y(id_N820) + left_wire_dist + 4.},
          {HOP4X(5), PIP_Y(id_N820) + left_wire_dist + 4., HOP4X(5), PIP_Y(id_N820) + 5.},
          {HOP4X(5), PIP_Y(id_N820) + 5., HOP4X(3), PIP_Y(id_N820) + left_wire_dist + 5.},
          {HOP4X(3), PIP_Y(id_N820) + left_wire_dist + 5., HOP4X(3), PIP_Y(id_N820) + 6.},
          {HOP4X(3), PIP_Y(id_N820) + 6., HOP4X(1), PIP_Y(id_N820) + left_wire_dist + 6.},
          {HOP4X(1), PIP_Y(id_N820) + left_wire_dist + 6., HOP4X(1), PIP_Y(id_N828) + 7.},
          {HOP4X(1), PIP_Y(id_N828) + 7., WIRE_X(0), PIP_Y(id_N828) + 7.}}},
        {id_S82_loop1,
         {{WIRE_X(0), PIP_Y(id_S820), HOP4X(16), PIP_Y(id_S820)},
          {HOP4X(16), PIP_Y(id_S820), HOP4X(16), PIP_Y(id_N828)},
          {HOP4X(16), PIP_Y(id_N828) - 0., HOP4X(14), PIP_Y(id_N828) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N828) - left_wire_dist - 0., HOP4X(14), PIP_Y(id_N828) - 1.},
          {HOP4X(14), PIP_Y(id_N828) - 1., HOP4X(12), PIP_Y(id_N828) - left_wire_dist - 1.},
          {HOP4X(12), PIP_Y(id_N828) - left_wire_dist - 1., HOP4X(12), -wrap_len - 1.},
          {HOP4X(12), -wrap_len - 1., HOP4X(13), -wrap_len - 1.},
          {HOP4X(13), -wrap_len - 1., HOP4X(13), PIP_Y(id_N820) - 1.},
          {HOP4X(13), PIP_Y(id_N820) - 1., HOP4X(11), PIP_Y(id_N820) + left_wire_dist - 1.},
          {HOP4X(11), PIP_Y(id_N820) + left_wire_dist - 1., HOP4X(11), PIP_Y(id_N820) - 0.},
          {HOP4X(11), PIP_Y(id_N820) - 0., HOP4X(9), PIP_Y(id_N820) + left_wire_dist - 0.},
          {HOP4X(9), PIP_Y(id_N820) + left_wire_dist - 0., HOP4X(9), PIP_Y(id_N820) + 1.},
          {HOP4X(9), PIP_Y(id_N820) + 1., HOP4X(7), PIP_Y(id_N820) + left_wire_dist + 1.},
          {HOP4X(9), PIP_Y(id_N824) + 1., WIRE_X(0), PIP_Y(id_N824) + 1.},
          {HOP4X(7), PIP_Y(id_N820) + left_wire_dist + 1., HOP4X(7), PIP_Y(id_N820) + 2.},
          {HOP4X(7), PIP_Y(id_N820) + 2., HOP4X(5), PIP_Y(id_N820) + left_wire_dist + 2.},
          {HOP4X(5), PIP_Y(id_N820) + left_wire_dist + 2., HOP4X(5), PIP_Y(id_N820) + 3.},
          {HOP4X(5), PIP_Y(id_N820) + 3., HOP4X(3), PIP_Y(id_N820) + left_wire_dist + 3.},
          {HOP4X(3), PIP_Y(id_N820) + left_wire_dist + 3., HOP4X(3), PIP_Y(id_N820) + 4.},
          {HOP4X(3), PIP_Y(id_N820) + 4., HOP4X(1), PIP_Y(id_N820) + left_wire_dist + 4.},
          {HOP4X(1), PIP_Y(id_N820) + left_wire_dist + 4., HOP4X(1), PIP_Y(id_N828) + 5.},
          {HOP4X(1), PIP_Y(id_N828) + 5., WIRE_X(0), PIP_Y(id_N828) + 5.}}},
        {id_S82_loop2,
         {{WIRE_X(0), PIP_Y(id_S820), HOP4X(16), PIP_Y(id_S820)},
          {HOP4X(16), PIP_Y(id_S820), HOP4X(16), PIP_Y(id_N828)},
          {HOP4X(16), PIP_Y(id_N828) - 0., HOP4X(14), PIP_Y(id_N828) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N828) - left_wire_dist - 0., HOP4X(14), PIP_Y(id_N828) - 1.},
          {HOP4X(14), PIP_Y(id_N828) - 1., HOP4X(12), PIP_Y(id_N828) - left_wire_dist - 1.},
          {HOP4X(12), PIP_Y(id_N828) - left_wire_dist - 1., HOP4X(12), PIP_Y(id_N828) - 2.},
          {HOP4X(12), PIP_Y(id_N828) - 2., HOP4X(10), PIP_Y(id_N828) - left_wire_dist - 2.},
          {HOP4X(10), PIP_Y(id_N828) - left_wire_dist - 2., HOP4X(10), -wrap_len - 2.},
          {HOP4X(10), -wrap_len - 2., HOP4X(11), -wrap_len - 2.},
          {HOP4X(11), -wrap_len - 2., HOP4X(11), PIP_Y(id_N820) - 2.},
          {HOP4X(11), PIP_Y(id_N820) - 2., HOP4X(9), PIP_Y(id_N820) + left_wire_dist - 2.},
          {HOP4X(9), PIP_Y(id_N820) + left_wire_dist - 2., HOP4X(9), PIP_Y(id_N820) - 1.},
          {HOP4X(9), PIP_Y(id_N820) - 1., HOP4X(7), PIP_Y(id_N820) + left_wire_dist - 1.},
          {HOP4X(9), PIP_Y(id_N824) - 1., WIRE_X(0), PIP_Y(id_N824) - 1.},
          {HOP4X(7), PIP_Y(id_N820) + left_wire_dist - 1., HOP4X(7), PIP_Y(id_N820) - 0.},
          {HOP4X(7), PIP_Y(id_N820) - 0., HOP4X(5), PIP_Y(id_N820) + left_wire_dist - 0.},
          {HOP4X(5), PIP_Y(id_N820) + left_wire_dist - 0., HOP4X(5), PIP_Y(id_N820) + 1.},
          {HOP4X(5), PIP_Y(id_N820) + 1., HOP4X(3), PIP_Y(id_N820) + left_wire_dist + 1.},
          {HOP4X(3), PIP_Y(id_N820) + left_wire_dist + 1., HOP4X(3), PIP_Y(id_N820) + 2.},
          {HOP4X(3), PIP_Y(id_N820) + 2., HOP4X(1), PIP_Y(id_N820) + left_wire_dist + 2.},
          {HOP4X(1), PIP_Y(id_N820) + left_wire_dist + 2., HOP4X(1), PIP_Y(id_N828) + 3.},
          {HOP4X(1), PIP_Y(id_N828) + 3., WIRE_X(0), PIP_Y(id_N828) + 3.}}},
        {id_S82_loop3,
         {{WIRE_X(0), PIP_Y(id_S820), HOP4X(16), PIP_Y(id_S820)},
          {HOP4X(16), PIP_Y(id_S820), HOP4X(16), PIP_Y(id_N828)},
          {HOP4X(16), PIP_Y(id_N828) - 0., HOP4X(14), PIP_Y(id_N828) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N828) - left_wire_dist - 0., HOP4X(14), PIP_Y(id_N828) - 1.},
          {HOP4X(14), PIP_Y(id_N828) - 1., HOP4X(12), PIP_Y(id_N828) - left_wire_dist - 1.},
          {HOP4X(12), PIP_Y(id_N828) - left_wire_dist - 1., HOP4X(12), PIP_Y(id_N828) - 2.},
          {HOP4X(12), PIP_Y(id_N828) - 2., HOP4X(10), PIP_Y(id_N828) - left_wire_dist - 2.},
          {HOP4X(10), PIP_Y(id_N828) - left_wire_dist - 2., HOP4X(10), PIP_Y(id_N828) - 3.},
          {HOP4X(10), PIP_Y(id_N828) - 3., HOP4X(8), PIP_Y(id_N828) - left_wire_dist - 3.},
          {HOP4X(8), PIP_Y(id_N828) - left_wire_dist - 3., HOP4X(8), -wrap_len - 3.},
          {HOP4X(8), -wrap_len - 3., HOP4X(9), -wrap_len - 3.},
          {HOP4X(9), -wrap_len - 3., HOP4X(9), PIP_Y(id_N820) - 3.},
          {HOP4X(9), PIP_Y(id_N824) - 3., WIRE_X(0), PIP_Y(id_N824) - 3.},
          {HOP4X(9), PIP_Y(id_N820) - 3., HOP4X(7), PIP_Y(id_N820) + left_wire_dist - 3.},
          {HOP4X(7), PIP_Y(id_N820) + left_wire_dist - 3., HOP4X(7), PIP_Y(id_N820) - 2.},
          {HOP4X(7), PIP_Y(id_N820) - 2., HOP4X(5), PIP_Y(id_N820) + left_wire_dist - 2.},
          {HOP4X(5), PIP_Y(id_N820) + left_wire_dist - 2., HOP4X(5), PIP_Y(id_N820) - 1.},
          {HOP4X(5), PIP_Y(id_N820) - 1., HOP4X(3), PIP_Y(id_N820) + left_wire_dist - 1.},
          {HOP4X(3), PIP_Y(id_N820) + left_wire_dist - 1., HOP4X(3), PIP_Y(id_N820) - 0.},
          {HOP4X(3), PIP_Y(id_N820) - 0., HOP4X(1), PIP_Y(id_N820) + left_wire_dist - 0.},
          {HOP4X(1), PIP_Y(id_N820) + left_wire_dist - 0., HOP4X(1), PIP_Y(id_N828) + 1.},
          {HOP4X(1), PIP_Y(id_N828) + 1., WIRE_X(0), PIP_Y(id_N828) + 1.}}},
        {id_S82_loop4,
         {{WIRE_X(0), PIP_Y(id_S820), HOP4X(16), PIP_Y(id_S820)},
          {HOP4X(16), PIP_Y(id_S820), HOP4X(16), PIP_Y(id_N828)},
          {HOP4X(16), PIP_Y(id_N828) - 0., HOP4X(14), PIP_Y(id_N828) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N828) - left_wire_dist - 0., HOP4X(14), PIP_Y(id_N828) - 1.},
          {HOP4X(14), PIP_Y(id_N828) - 1., HOP4X(12), PIP_Y(id_N828) - left_wire_dist - 1.},
          {HOP4X(12), PIP_Y(id_N828) - left_wire_dist - 1., HOP4X(12), PIP_Y(id_N828) - 2.},
          {HOP4X(12), PIP_Y(id_N828) - 2., HOP4X(10), PIP_Y(id_N828) - left_wire_dist - 2.},
          {HOP4X(10), PIP_Y(id_N828) - left_wire_dist - 2., HOP4X(10), PIP_Y(id_N828) - 3.},
          {HOP4X(10), PIP_Y(id_N828) - 3., HOP4X(8), PIP_Y(id_N828) - left_wire_dist - 3.},
          {HOP4X(8), PIP_Y(id_N828) - left_wire_dist - 3., HOP4X(8), PIP_Y(id_N828) - 4.},
          {HOP4X(8), PIP_Y(id_S824) - 4., WIRE_X(0), PIP_Y(id_S824) - 4.},
          {HOP4X(8), PIP_Y(id_N828) - 4., HOP4X(6), PIP_Y(id_N828) - left_wire_dist - 4.},
          {HOP4X(6), PIP_Y(id_N828) - left_wire_dist - 4., HOP4X(6), -wrap_len - 4.},
          {HOP4X(6), -wrap_len - 4., HOP4X(7), -wrap_len - 4.},
          {HOP4X(7), -wrap_len - 4., HOP4X(7), PIP_Y(id_N820) - 4.},
          {HOP4X(7), PIP_Y(id_N820) - 4., HOP4X(5), PIP_Y(id_N820) + left_wire_dist - 4.},
          {HOP4X(5), PIP_Y(id_N820) + left_wire_dist - 4., HOP4X(5), PIP_Y(id_N820) - 3.},
          {HOP4X(5), PIP_Y(id_N820) - 3., HOP4X(3), PIP_Y(id_N820) + left_wire_dist - 3.},
          {HOP4X(3), PIP_Y(id_N820) + left_wire_dist - 3., HOP4X(3), PIP_Y(id_N820) - 2.},
          {HOP4X(3), PIP_Y(id_N820) - 2., HOP4X(1), PIP_Y(id_N820) + left_wire_dist - 2.},
          {HOP4X(1), PIP_Y(id_N820) + left_wire_dist - 2., HOP4X(1), PIP_Y(id_N828) - 1.},
          {HOP4X(1), PIP_Y(id_N828) - 1., WIRE_X(0), PIP_Y(id_N828) - 1.}}},
        {id_S82_loop5,
         {{WIRE_X(0), PIP_Y(id_S820), HOP4X(16), PIP_Y(id_S820)},
          {HOP4X(16), PIP_Y(id_S820), HOP4X(16), PIP_Y(id_N828)},
          {HOP4X(16), PIP_Y(id_N828) - 0., HOP4X(14), PIP_Y(id_N828) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N828) - left_wire_dist - 0., HOP4X(14), PIP_Y(id_N828) - 1.},
          {HOP4X(14), PIP_Y(id_N828) - 1., HOP4X(12), PIP_Y(id_N828) - left_wire_dist - 1.},
          {HOP4X(12), PIP_Y(id_N828) - left_wire_dist - 1., HOP4X(12), PIP_Y(id_N828) - 2.},
          {HOP4X(12), PIP_Y(id_N828) - 2., HOP4X(10), PIP_Y(id_N828) - left_wire_dist - 2.},
          {HOP4X(10), PIP_Y(id_N828) - left_wire_dist - 2., HOP4X(10), PIP_Y(id_N828) - 3.},
          {HOP4X(10), PIP_Y(id_N828) - 3., HOP4X(8), PIP_Y(id_N828) - left_wire_dist - 3.},
          {HOP4X(8), PIP_Y(id_N828) - left_wire_dist - 3., HOP4X(8), PIP_Y(id_N828) - 4.},
          {HOP4X(8), PIP_Y(id_S824) - 4., WIRE_X(0), PIP_Y(id_S824) - 4.},
          {HOP4X(8), PIP_Y(id_N828) - 4., HOP4X(6), PIP_Y(id_N828) - left_wire_dist - 4.},
          {HOP4X(6), PIP_Y(id_N828) - left_wire_dist - 4., HOP4X(6), PIP_Y(id_N828) - 5.},
          {HOP4X(6), PIP_Y(id_N828) - 5., HOP4X(4), PIP_Y(id_N828) - left_wire_dist - 5.},
          {HOP4X(4), PIP_Y(id_N828) - left_wire_dist - 5., HOP4X(4), -wrap_len - 5.},
          {HOP4X(4), -wrap_len - 5., HOP4X(5), -wrap_len - 5.},
          {HOP4X(5), -wrap_len - 5., HOP4X(5), PIP_Y(id_N820) - 5.},
          {HOP4X(5), PIP_Y(id_N820) - 5., HOP4X(3), PIP_Y(id_N820) + left_wire_dist - 5.},
          {HOP4X(3), PIP_Y(id_N820) + left_wire_dist - 5., HOP4X(3), PIP_Y(id_N820) - 4.},
          {HOP4X(3), PIP_Y(id_N820) - 4., HOP4X(1), PIP_Y(id_N820) + left_wire_dist - 4.},
          {HOP4X(1), PIP_Y(id_N820) + left_wire_dist - 4., HOP4X(1), PIP_Y(id_N828) - 3.},
          {HOP4X(1), PIP_Y(id_N828) - 3., WIRE_X(0), PIP_Y(id_N828) - 3.}}},
        {id_S82_loop6,
         {{WIRE_X(0), PIP_Y(id_S820), HOP4X(16), PIP_Y(id_S820)},
          {HOP4X(16), PIP_Y(id_S820), HOP4X(16), PIP_Y(id_N828)},
          {HOP4X(16), PIP_Y(id_N828) - 0., HOP4X(14), PIP_Y(id_N828) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N828) - left_wire_dist - 0., HOP4X(14), PIP_Y(id_N828) - 1.},
          {HOP4X(14), PIP_Y(id_N828) - 1., HOP4X(12), PIP_Y(id_N828) - left_wire_dist - 1.},
          {HOP4X(12), PIP_Y(id_N828) - left_wire_dist - 1., HOP4X(12), PIP_Y(id_N828) - 2.},
          {HOP4X(12), PIP_Y(id_N828) - 2., HOP4X(10), PIP_Y(id_N828) - left_wire_dist - 2.},
          {HOP4X(10), PIP_Y(id_N828) - left_wire_dist - 2., HOP4X(10), PIP_Y(id_N828) - 3.},
          {HOP4X(10), PIP_Y(id_N828) - 3., HOP4X(8), PIP_Y(id_N828) - left_wire_dist - 3.},
          {HOP4X(8), PIP_Y(id_N828) - left_wire_dist - 3., HOP4X(8), PIP_Y(id_N828) - 4.},
          {HOP4X(8), PIP_Y(id_S824) - 4., WIRE_X(0), PIP_Y(id_S824) - 4.},
          {HOP4X(8), PIP_Y(id_N828) - 4., HOP4X(6), PIP_Y(id_N828) - left_wire_dist - 4.},
          {HOP4X(6), PIP_Y(id_N828) - left_wire_dist - 4., HOP4X(6), PIP_Y(id_N828) - 5.},
          {HOP4X(6), PIP_Y(id_N828) - 5., HOP4X(4), PIP_Y(id_N828) - left_wire_dist - 5.},
          {HOP4X(4), PIP_Y(id_N828) - left_wire_dist - 5., HOP4X(4), PIP_Y(id_N828) - 6.},
          {HOP4X(4), PIP_Y(id_N828) - 6., HOP4X(2), PIP_Y(id_N828) - left_wire_dist - 6.},
          {HOP4X(2), PIP_Y(id_N828) - left_wire_dist - 6., HOP4X(2), -wrap_len - 6.},
          {HOP4X(2), -wrap_len - 6., HOP4X(3), -wrap_len - 6.},
          {HOP4X(3), -wrap_len - 6., HOP4X(3), PIP_Y(id_N820) - 6.},
          {HOP4X(3), PIP_Y(id_N820) - 6., HOP4X(1), PIP_Y(id_N820) + left_wire_dist - 6.},
          {HOP4X(1), PIP_Y(id_N820) + left_wire_dist - 6., HOP4X(1), PIP_Y(id_N828) - 5.},
          {HOP4X(1), PIP_Y(id_N828) - 5., WIRE_X(0), PIP_Y(id_N828) - 5.}}},
        {id_S82_loop7,
         {{WIRE_X(0), PIP_Y(id_S820), HOP4X(16), PIP_Y(id_S820)},
          {HOP4X(16), PIP_Y(id_S820), HOP4X(16), PIP_Y(id_N828)},
          {HOP4X(16), PIP_Y(id_N828) - 0., HOP4X(14), PIP_Y(id_N828) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N828) - left_wire_dist - 0., HOP4X(14), PIP_Y(id_N828) - 1.},
          {HOP4X(14), PIP_Y(id_N828) - 1., HOP4X(12), PIP_Y(id_N828) - left_wire_dist - 1.},
          {HOP4X(12), PIP_Y(id_N828) - left_wire_dist - 1., HOP4X(12), PIP_Y(id_N828) - 2.},
          {HOP4X(12), PIP_Y(id_N828) - 2., HOP4X(10), PIP_Y(id_N828) - left_wire_dist - 2.},
          {HOP4X(10), PIP_Y(id_N828) - left_wire_dist - 2., HOP4X(10), PIP_Y(id_N828) - 3.},
          {HOP4X(10), PIP_Y(id_N828) - 3., HOP4X(8), PIP_Y(id_N828) - left_wire_dist - 3.},
          {HOP4X(8), PIP_Y(id_N828) - left_wire_dist - 3., HOP4X(8), PIP_Y(id_N828) - 4.},
          {HOP4X(8), PIP_Y(id_S824) - 4., WIRE_X(0), PIP_Y(id_S824) - 4.},
          {HOP4X(8), PIP_Y(id_N828) - 4., HOP4X(6), PIP_Y(id_N828) - left_wire_dist - 4.},
          {HOP4X(6), PIP_Y(id_N828) - left_wire_dist - 4., HOP4X(6), PIP_Y(id_N828) - 5.},
          {HOP4X(6), PIP_Y(id_N828) - 5., HOP4X(4), PIP_Y(id_N828) - left_wire_dist - 5.},
          {HOP4X(4), PIP_Y(id_N828) - left_wire_dist - 5., HOP4X(4), PIP_Y(id_N828) - 6.},
          {HOP4X(4), PIP_Y(id_N828) - 6., HOP4X(2), PIP_Y(id_N828) - left_wire_dist - 6.},
          {HOP4X(2), PIP_Y(id_N828) - left_wire_dist - 6., HOP4X(2), PIP_Y(id_N828) - 7.},
          {HOP4X(2), PIP_Y(id_N828) - 7., HOP4X(0), PIP_Y(id_N828) - left_wire_dist - 7.},
          {HOP4X(0), PIP_Y(id_N828) - left_wire_dist - 7., HOP4X(0), -wrap_len - 7.},
          {HOP4X(0), -wrap_len - 7., HOP4X(1), -wrap_len - 7.},
          {HOP4X(1), -wrap_len - 7., HOP4X(1), PIP_Y(id_N828) - 7.},
          {HOP4X(1), PIP_Y(id_N828) - 7., WIRE_X(0), PIP_Y(id_N828) - 7.}}},
        {id_N82_loop0,
         {{WIRE_X(0), PIP_Y(id_N820), HOP4X(17), PIP_Y(id_N820)},
          {HOP4X(17), PIP_Y(id_N820) + 0., HOP4X(15), PIP_Y(id_N820) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N820) + left_wire_dist + 0., HOP4X(15), wrap_len + 1.},
          {HOP4X(15), wrap_len + 1., HOP4X(14), wrap_len + 1.},
          {HOP4X(14), wrap_len + 1., HOP4X(14), PIP_Y(id_N828) + 0.},
          {HOP4X(14), PIP_Y(id_N828) + 0., HOP4X(12), PIP_Y(id_N828) - left_wire_dist + 0.},
          {HOP4X(12), PIP_Y(id_N828) - left_wire_dist + 0., HOP4X(12), PIP_Y(id_N828) - 1.},
          {HOP4X(12), PIP_Y(id_N828) - 1., HOP4X(10), PIP_Y(id_N828) - left_wire_dist - 1.},
          {HOP4X(10), PIP_Y(id_N828) - left_wire_dist - 1., HOP4X(10), PIP_Y(id_N828) - 2.},
          {HOP4X(10), PIP_Y(id_N828) - 2., HOP4X(8), PIP_Y(id_N828) - left_wire_dist - 2.},
          {HOP4X(8), PIP_Y(id_N828) - left_wire_dist - 2., HOP4X(8), PIP_Y(id_N828) - 3.},
          {HOP4X(8), PIP_Y(id_S824) - 3., WIRE_X(0), PIP_Y(id_S824) - 3.},
          {HOP4X(8), PIP_Y(id_N828) - 3., HOP4X(6), PIP_Y(id_N828) - left_wire_dist - 3.},
          {HOP4X(6), PIP_Y(id_N828) - left_wire_dist - 3., HOP4X(6), PIP_Y(id_N828) - 4.},
          {HOP4X(6), PIP_Y(id_N828) - 4., HOP4X(4), PIP_Y(id_N828) - left_wire_dist - 4.},
          {HOP4X(4), PIP_Y(id_N828) - left_wire_dist - 4., HOP4X(4), PIP_Y(id_N828) - 5.},
          {HOP4X(4), PIP_Y(id_N828) - 5., HOP4X(2), PIP_Y(id_N828) - left_wire_dist - 5.},
          {HOP4X(2), PIP_Y(id_N828) - left_wire_dist - 5., HOP4X(2), PIP_Y(id_N828) - 6.},
          {HOP4X(2), PIP_Y(id_N828) - 6., HOP4X(0), PIP_Y(id_N828) - left_wire_dist - 6.},
          {HOP4X(0), PIP_Y(id_N828) - left_wire_dist - 6., HOP4X(0), PIP_Y(id_S828) - 7.},
          {HOP4X(0), PIP_Y(id_S828) - 7., WIRE_X(0), PIP_Y(id_S828) - 7.}}},
        {id_N82_loop1,
         {{WIRE_X(0), PIP_Y(id_N820), HOP4X(17), PIP_Y(id_N820)},
          {HOP4X(17), PIP_Y(id_N820) + 0., HOP4X(15), PIP_Y(id_N820) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N820) + left_wire_dist + 0., HOP4X(15), PIP_Y(id_N820) + 1.},
          {HOP4X(15), PIP_Y(id_N820) + 1., HOP4X(13), PIP_Y(id_N820) + left_wire_dist + 1.},
          {HOP4X(13), PIP_Y(id_N820) + left_wire_dist + 1., HOP4X(13), wrap_len + 2.},
          {HOP4X(13), wrap_len + 2., HOP4X(12), wrap_len + 2.},
          {HOP4X(12), wrap_len + 2., HOP4X(12), PIP_Y(id_N828) + 1.},
          {HOP4X(12), PIP_Y(id_N828) + 1., HOP4X(10), PIP_Y(id_N828) - left_wire_dist + 1.},
          {HOP4X(10), PIP_Y(id_N828) - left_wire_dist + 1., HOP4X(10), PIP_Y(id_N828) + 0.},
          {HOP4X(10), PIP_Y(id_N828) + 0., HOP4X(8), PIP_Y(id_N828) - left_wire_dist + 0.},
          {HOP4X(8), PIP_Y(id_N828) - left_wire_dist + 0., HOP4X(8), PIP_Y(id_N828) - 1.},
          {HOP4X(8), PIP_Y(id_S824) - 1., WIRE_X(0), PIP_Y(id_S824) - 1.},
          {HOP4X(8), PIP_Y(id_N828) - 1., HOP4X(6), PIP_Y(id_N828) - left_wire_dist - 1.},
          {HOP4X(6), PIP_Y(id_N828) - left_wire_dist - 1., HOP4X(6), PIP_Y(id_N828) - 2.},
          {HOP4X(6), PIP_Y(id_N828) - 2., HOP4X(4), PIP_Y(id_N828) - left_wire_dist - 2.},
          {HOP4X(4), PIP_Y(id_N828) - left_wire_dist - 2., HOP4X(4), PIP_Y(id_N828) - 3.},
          {HOP4X(4), PIP_Y(id_N828) - 3., HOP4X(2), PIP_Y(id_N828) - left_wire_dist - 3.},
          {HOP4X(2), PIP_Y(id_N828) - left_wire_dist - 3., HOP4X(2), PIP_Y(id_N828) - 4.},
          {HOP4X(2), PIP_Y(id_N828) - 4., HOP4X(0), PIP_Y(id_N828) - left_wire_dist - 4.},
          {HOP4X(0), PIP_Y(id_N828) - left_wire_dist - 4., HOP4X(0), PIP_Y(id_S828) - 5.},
          {HOP4X(0), PIP_Y(id_S828) - 5., WIRE_X(0), PIP_Y(id_S828) - 5.}}},
        {id_N82_loop2,
         {{WIRE_X(0), PIP_Y(id_N820), HOP4X(17), PIP_Y(id_N820)},
          {HOP4X(17), PIP_Y(id_N820) + 0., HOP4X(15), PIP_Y(id_N820) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N820) + left_wire_dist + 0., HOP4X(15), PIP_Y(id_N820) + 1.},
          {HOP4X(15), PIP_Y(id_N820) + 1., HOP4X(13), PIP_Y(id_N820) + left_wire_dist + 1.},
          {HOP4X(13), PIP_Y(id_N820) + left_wire_dist + 1., HOP4X(13), PIP_Y(id_N820) + 2.},
          {HOP4X(13), PIP_Y(id_N820) + 2., HOP4X(11), PIP_Y(id_N820) + left_wire_dist + 2.},
          {HOP4X(11), PIP_Y(id_N820) + left_wire_dist + 2., HOP4X(11), wrap_len + 3.},
          {HOP4X(11), wrap_len + 3., HOP4X(10), wrap_len + 3.},
          {HOP4X(10), wrap_len + 3., HOP4X(10), PIP_Y(id_N828) + 2.},
          {HOP4X(10), PIP_Y(id_N828) + 2., HOP4X(8), PIP_Y(id_N828) - left_wire_dist + 2.},
          {HOP4X(8), PIP_Y(id_N828) - left_wire_dist + 2., HOP4X(8), PIP_Y(id_N828) + 1.},
          {HOP4X(8), PIP_Y(id_S824) + 1., WIRE_X(0), PIP_Y(id_S824) + 1.},
          {HOP4X(8), PIP_Y(id_N828) + 1., HOP4X(6), PIP_Y(id_N828) - left_wire_dist + 1.},
          {HOP4X(6), PIP_Y(id_N828) - left_wire_dist + 1., HOP4X(6), PIP_Y(id_N828) + 0.},
          {HOP4X(6), PIP_Y(id_N828) + 0., HOP4X(4), PIP_Y(id_N828) - left_wire_dist + 0.},
          {HOP4X(4), PIP_Y(id_N828) - left_wire_dist + 0., HOP4X(4), PIP_Y(id_N828) - 1.},
          {HOP4X(4), PIP_Y(id_N828) - 1., HOP4X(2), PIP_Y(id_N828) - left_wire_dist - 1.},
          {HOP4X(2), PIP_Y(id_N828) - left_wire_dist - 1., HOP4X(2), PIP_Y(id_N828) - 2.},
          {HOP4X(2), PIP_Y(id_N828) - 2., HOP4X(0), PIP_Y(id_N828) - left_wire_dist - 2.},
          {HOP4X(0), PIP_Y(id_N828) - left_wire_dist - 2., HOP4X(0), PIP_Y(id_S828) - 3.},
          {HOP4X(0), PIP_Y(id_S828) - 3., WIRE_X(0), PIP_Y(id_S828) - 3.}}},
        {id_N82_loop3,
         {{WIRE_X(0), PIP_Y(id_N820), HOP4X(17), PIP_Y(id_N820)},
          {HOP4X(17), PIP_Y(id_N820) + 0., HOP4X(15), PIP_Y(id_N820) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N820) + left_wire_dist + 0., HOP4X(15), PIP_Y(id_N820) + 1.},
          {HOP4X(15), PIP_Y(id_N820) + 1., HOP4X(13), PIP_Y(id_N820) + left_wire_dist + 1.},
          {HOP4X(13), PIP_Y(id_N820) + left_wire_dist + 1., HOP4X(13), PIP_Y(id_N820) + 2.},
          {HOP4X(13), PIP_Y(id_N820) + 2., HOP4X(11), PIP_Y(id_N820) + left_wire_dist + 2.},
          {HOP4X(11), PIP_Y(id_N820) + left_wire_dist + 2., HOP4X(11), PIP_Y(id_N820) + 3.},
          {HOP4X(11), PIP_Y(id_N820) + 3., HOP4X(9), PIP_Y(id_N820) + left_wire_dist + 3.},
          {HOP4X(9), PIP_Y(id_N820) + left_wire_dist + 3., HOP4X(9), wrap_len + 4.},
          {HOP4X(9), wrap_len + 4., HOP4X(8), wrap_len + 4.},
          {HOP4X(8), wrap_len + 4., HOP4X(8), PIP_Y(id_N828) + 3.},
          {HOP4X(8), PIP_Y(id_S824) + 3., WIRE_X(0), PIP_Y(id_S824) + 3.},
          {HOP4X(8), PIP_Y(id_N828) + 3., HOP4X(6), PIP_Y(id_N828) - left_wire_dist + 3.},
          {HOP4X(6), PIP_Y(id_N828) - left_wire_dist + 3., HOP4X(6), PIP_Y(id_N828) + 2.},
          {HOP4X(6), PIP_Y(id_N828) + 2., HOP4X(4), PIP_Y(id_N828) - left_wire_dist + 2.},
          {HOP4X(4), PIP_Y(id_N828) - left_wire_dist + 2., HOP4X(4), PIP_Y(id_N828) + 1.},
          {HOP4X(4), PIP_Y(id_N828) + 1., HOP4X(2), PIP_Y(id_N828) - left_wire_dist + 1.},
          {HOP4X(2), PIP_Y(id_N828) - left_wire_dist + 1., HOP4X(2), PIP_Y(id_N828) + 0.},
          {HOP4X(2), PIP_Y(id_N828) + 0., HOP4X(0), PIP_Y(id_N828) - left_wire_dist + 0.},
          {HOP4X(0), PIP_Y(id_N828) - left_wire_dist + 0., HOP4X(0), PIP_Y(id_S828) - 1.},
          {HOP4X(0), PIP_Y(id_S828) - 1., WIRE_X(0), PIP_Y(id_S828) - 1.}}},
        {id_N82_loop4,
         {{WIRE_X(0), PIP_Y(id_N820), HOP4X(17), PIP_Y(id_N820)},
          {HOP4X(17), PIP_Y(id_N820) + 0., HOP4X(15), PIP_Y(id_N820) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N820) + left_wire_dist + 0., HOP4X(15), PIP_Y(id_N820) + 1.},
          {HOP4X(15), PIP_Y(id_N820) + 1., HOP4X(13), PIP_Y(id_N820) + left_wire_dist + 1.},
          {HOP4X(13), PIP_Y(id_N820) + left_wire_dist + 1., HOP4X(13), PIP_Y(id_N820) + 2.},
          {HOP4X(13), PIP_Y(id_N820) + 2., HOP4X(11), PIP_Y(id_N820) + left_wire_dist + 2.},
          {HOP4X(11), PIP_Y(id_N820) + left_wire_dist + 2., HOP4X(11), PIP_Y(id_N820) + 3.},
          {HOP4X(11), PIP_Y(id_N820) + 3., HOP4X(9), PIP_Y(id_N820) + left_wire_dist + 3.},
          {HOP4X(9), PIP_Y(id_N820) + left_wire_dist + 3., HOP4X(9), PIP_Y(id_N820) + 4.},
          {HOP4X(9), PIP_Y(id_N824) + 4., WIRE_X(0), PIP_Y(id_N824) + 4.},
          {HOP4X(9), PIP_Y(id_N820) + 4., HOP4X(7), PIP_Y(id_N820) + left_wire_dist + 4.},
          {HOP4X(7), PIP_Y(id_N820) + left_wire_dist + 4., HOP4X(7), wrap_len + 5.},
          {HOP4X(7), wrap_len + 5., HOP4X(6), wrap_len + 5.},
          {HOP4X(6), wrap_len + 5., HOP4X(6), PIP_Y(id_N828) + 4.},
          {HOP4X(6), PIP_Y(id_N828) + 4., HOP4X(4), PIP_Y(id_N828) - left_wire_dist + 4.},
          {HOP4X(4), PIP_Y(id_N828) - left_wire_dist + 4., HOP4X(4), PIP_Y(id_N828) + 3.},
          {HOP4X(4), PIP_Y(id_N828) + 3., HOP4X(2), PIP_Y(id_N828) - left_wire_dist + 3.},
          {HOP4X(2), PIP_Y(id_N828) - left_wire_dist + 3., HOP4X(2), PIP_Y(id_N828) + 2.},
          {HOP4X(2), PIP_Y(id_N828) + 2., HOP4X(0), PIP_Y(id_N828) - left_wire_dist + 2.},
          {HOP4X(0), PIP_Y(id_N828) - left_wire_dist + 2., HOP4X(0), PIP_Y(id_S828) + 1.},
          {HOP4X(0), PIP_Y(id_S828) + 1., WIRE_X(0), PIP_Y(id_S828) + 1.}}},
        {id_N82_loop5,
         {{WIRE_X(0), PIP_Y(id_N820), HOP4X(17), PIP_Y(id_N820)},
          {HOP4X(17), PIP_Y(id_N820) + 0., HOP4X(15), PIP_Y(id_N820) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N820) + left_wire_dist + 0., HOP4X(15), PIP_Y(id_N820) + 1.},
          {HOP4X(15), PIP_Y(id_N820) + 1., HOP4X(13), PIP_Y(id_N820) + left_wire_dist + 1.},
          {HOP4X(13), PIP_Y(id_N820) + left_wire_dist + 1., HOP4X(13), PIP_Y(id_N820) + 2.},
          {HOP4X(13), PIP_Y(id_N820) + 2., HOP4X(11), PIP_Y(id_N820) + left_wire_dist + 2.},
          {HOP4X(11), PIP_Y(id_N820) + left_wire_dist + 2., HOP4X(11), PIP_Y(id_N820) + 3.},
          {HOP4X(11), PIP_Y(id_N820) + 3., HOP4X(9), PIP_Y(id_N820) + left_wire_dist + 3.},
          {HOP4X(9), PIP_Y(id_N820) + left_wire_dist + 3., HOP4X(9), PIP_Y(id_N820) + 4.},
          {HOP4X(9), PIP_Y(id_N824) + 4., WIRE_X(0), PIP_Y(id_N824) + 4.},
          {HOP4X(9), PIP_Y(id_N820) + 4., HOP4X(7), PIP_Y(id_N820) + left_wire_dist + 4.},
          {HOP4X(7), PIP_Y(id_N820) + left_wire_dist + 4., HOP4X(7), PIP_Y(id_N820) + 5.},
          {HOP4X(7), PIP_Y(id_N820) + 5., HOP4X(5), PIP_Y(id_N820) + left_wire_dist + 5.},
          {HOP4X(5), PIP_Y(id_N820) + left_wire_dist + 5., HOP4X(5), wrap_len + 6.},
          {HOP4X(5), wrap_len + 6., HOP4X(4), wrap_len + 6.},
          {HOP4X(4), wrap_len + 6., HOP4X(4), PIP_Y(id_N828) + 5.},
          {HOP4X(4), PIP_Y(id_N828) + 5., HOP4X(2), PIP_Y(id_N828) - left_wire_dist + 5.},
          {HOP4X(2), PIP_Y(id_N828) - left_wire_dist + 5., HOP4X(2), PIP_Y(id_N828) + 4.},
          {HOP4X(2), PIP_Y(id_N828) + 4., HOP4X(0), PIP_Y(id_N828) - left_wire_dist + 4.},
          {HOP4X(0), PIP_Y(id_N828) - left_wire_dist + 4., HOP4X(0), PIP_Y(id_S828) + 3.},
          {HOP4X(0), PIP_Y(id_S828) + 3., WIRE_X(0), PIP_Y(id_S828) + 3.}}},
        {id_N82_loop6,
         {{WIRE_X(0), PIP_Y(id_N820), HOP4X(17), PIP_Y(id_N820)},
          {HOP4X(17), PIP_Y(id_N820) + 0., HOP4X(15), PIP_Y(id_N820) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N820) + left_wire_dist + 0., HOP4X(15), PIP_Y(id_N820) + 1.},
          {HOP4X(15), PIP_Y(id_N820) + 1., HOP4X(13), PIP_Y(id_N820) + left_wire_dist + 1.},
          {HOP4X(13), PIP_Y(id_N820) + left_wire_dist + 1., HOP4X(13), PIP_Y(id_N820) + 2.},
          {HOP4X(13), PIP_Y(id_N820) + 2., HOP4X(11), PIP_Y(id_N820) + left_wire_dist + 2.},
          {HOP4X(11), PIP_Y(id_N820) + left_wire_dist + 2., HOP4X(11), PIP_Y(id_N820) + 3.},
          {HOP4X(11), PIP_Y(id_N820) + 3., HOP4X(9), PIP_Y(id_N820) + left_wire_dist + 3.},
          {HOP4X(9), PIP_Y(id_N820) + left_wire_dist + 3., HOP4X(9), PIP_Y(id_N820) + 4.},
          {HOP4X(9), PIP_Y(id_N824) + 4., WIRE_X(0), PIP_Y(id_N824) + 4.},
          {HOP4X(9), PIP_Y(id_N820) + 4., HOP4X(7), PIP_Y(id_N820) + left_wire_dist + 4.},
          {HOP4X(7), PIP_Y(id_N820) + left_wire_dist + 4., HOP4X(7), PIP_Y(id_N820) + 5.},
          {HOP4X(7), PIP_Y(id_N820) + 5., HOP4X(5), PIP_Y(id_N820) + left_wire_dist + 5.},
          {HOP4X(5), PIP_Y(id_N820) + left_wire_dist + 5., HOP4X(5), PIP_Y(id_N820) + 6.},
          {HOP4X(5), PIP_Y(id_N820) + 6., HOP4X(3), PIP_Y(id_N820) + left_wire_dist + 6.},
          {HOP4X(3), PIP_Y(id_N820) + left_wire_dist + 6., HOP4X(3), wrap_len + 7.},
          {HOP4X(3), wrap_len + 7., HOP4X(2), wrap_len + 7.},
          {HOP4X(2), wrap_len + 7., HOP4X(2), PIP_Y(id_N828) + 6.},
          {HOP4X(2), PIP_Y(id_N828) + 6., HOP4X(0), PIP_Y(id_N828) - left_wire_dist + 6.},
          {HOP4X(0), PIP_Y(id_N828) - left_wire_dist + 6., HOP4X(0), PIP_Y(id_S828) + 5.},
          {HOP4X(0), PIP_Y(id_S828) + 5., WIRE_X(0), PIP_Y(id_S828) + 5.}}},
        {id_N82_loop7,
         {{WIRE_X(0), PIP_Y(id_N820), HOP4X(17), PIP_Y(id_N820)},
          {HOP4X(17), PIP_Y(id_N820) + 0., HOP4X(15), PIP_Y(id_N820) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N820) + left_wire_dist + 0., HOP4X(15), PIP_Y(id_N820) + 1.},
          {HOP4X(15), PIP_Y(id_N820) + 1., HOP4X(13), PIP_Y(id_N820) + left_wire_dist + 1.},
          {HOP4X(13), PIP_Y(id_N820) + left_wire_dist + 1., HOP4X(13), PIP_Y(id_N820) + 2.},
          {HOP4X(13), PIP_Y(id_N820) + 2., HOP4X(11), PIP_Y(id_N820) + left_wire_dist + 2.},
          {HOP4X(11), PIP_Y(id_N820) + left_wire_dist + 2., HOP4X(11), PIP_Y(id_N820) + 3.},
          {HOP4X(11), PIP_Y(id_N820) + 3., HOP4X(9), PIP_Y(id_N820) + left_wire_dist + 3.},
          {HOP4X(9), PIP_Y(id_N820) + left_wire_dist + 3., HOP4X(9), PIP_Y(id_N820) + 4.},
          {HOP4X(9), PIP_Y(id_N824) + 4., WIRE_X(0), PIP_Y(id_N824) + 4.},
          {HOP4X(9), PIP_Y(id_N820) + 4., HOP4X(7), PIP_Y(id_N820) + left_wire_dist + 4.},
          {HOP4X(7), PIP_Y(id_N820) + left_wire_dist + 4., HOP4X(7), PIP_Y(id_N820) + 5.},
          {HOP4X(7), PIP_Y(id_N820) + 5., HOP4X(5), PIP_Y(id_N820) + left_wire_dist + 5.},
          {HOP4X(5), PIP_Y(id_N820) + left_wire_dist + 5., HOP4X(5), PIP_Y(id_N820) + 6.},
          {HOP4X(5), PIP_Y(id_N820) + 6., HOP4X(3), PIP_Y(id_N820) + left_wire_dist + 6.},
          {HOP4X(3), PIP_Y(id_N820) + left_wire_dist + 6., HOP4X(3), PIP_Y(id_N820) + 7.},
          {HOP4X(3), PIP_Y(id_N820) + 7., HOP4X(1), PIP_Y(id_N820) + left_wire_dist + 7.},
          {HOP4X(1), PIP_Y(id_N820) + left_wire_dist + 7., HOP4X(1), wrap_len + 8.},
          {HOP4X(1), wrap_len + 8., HOP4X(0), wrap_len + 8.},
          {HOP4X(0), wrap_len + 8., HOP4X(0), PIP_Y(id_S828) + 7.},
          {HOP4X(0), PIP_Y(id_S828) + 7., WIRE_X(0), PIP_Y(id_S828) + 7.}}},

#undef HOP4X
#define HOP4X(offset) WIRE_X((float)offset + HOP4X_START + 18.f + 18.f + 18.f)
        {id_S83,
         {{WIRE_X(0), PIP_Y(id_S830), HOP4X(16), PIP_Y(id_S830)},
          {HOP4X(16), PIP_Y(id_S830), HOP4X(16), PIP_Y(id_N838)},
          {HOP4X(16), PIP_Y(id_N838) - 0., HOP4X(14), PIP_Y(id_N838) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N838) - left_wire_dist - 0., HOP4X(14), PIP_Y(id_N838) - 1.},
          {HOP4X(14), PIP_Y(id_N838) - 1., HOP4X(12), PIP_Y(id_N838) - left_wire_dist - 1.},
          {HOP4X(12), PIP_Y(id_N838) - left_wire_dist - 1., HOP4X(12), PIP_Y(id_N838) - 2.},
          {HOP4X(12), PIP_Y(id_N838) - 2., HOP4X(10), PIP_Y(id_N838) - left_wire_dist - 2.},
          {HOP4X(10), PIP_Y(id_N838) - left_wire_dist - 2., HOP4X(10), PIP_Y(id_N838) - 3.},
          {HOP4X(10), PIP_Y(id_N838) - 3., HOP4X(8), PIP_Y(id_N838) - left_wire_dist - 3.},
          {HOP4X(8), PIP_Y(id_N838) - left_wire_dist - 3., HOP4X(8), PIP_Y(id_N838) - 4.},
          {HOP4X(8), PIP_Y(id_S834) - 4., WIRE_X(0), PIP_Y(id_S834) - 4.},
          {HOP4X(8), PIP_Y(id_N838) - 4., HOP4X(6), PIP_Y(id_N838) - left_wire_dist - 4.},
          {HOP4X(6), PIP_Y(id_N838) - left_wire_dist - 4., HOP4X(6), PIP_Y(id_N838) - 5.},
          {HOP4X(6), PIP_Y(id_N838) - 5., HOP4X(4), PIP_Y(id_N838) - left_wire_dist - 5.},
          {HOP4X(4), PIP_Y(id_N838) - left_wire_dist - 5., HOP4X(4), PIP_Y(id_N838) - 6.},
          {HOP4X(4), PIP_Y(id_N838) - 6., HOP4X(2), PIP_Y(id_N838) - left_wire_dist - 6.},
          {HOP4X(2), PIP_Y(id_N838) - left_wire_dist - 6., HOP4X(2), PIP_Y(id_N838) - 7.},
          {HOP4X(2), PIP_Y(id_N838) - 7., HOP4X(0), PIP_Y(id_N838) - left_wire_dist - 7.},
          {HOP4X(0), PIP_Y(id_N838) - left_wire_dist - 7., HOP4X(0), PIP_Y(id_S838) - 8.},
          {HOP4X(0), PIP_Y(id_S838) - 8., WIRE_X(0), PIP_Y(id_S838) - 8.}}},
        {id_N83,
         {{WIRE_X(0), PIP_Y(id_N830), HOP4X(17), PIP_Y(id_N830)},
          {HOP4X(17), PIP_Y(id_N830) + 0., HOP4X(15), PIP_Y(id_N830) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N830) + left_wire_dist + 0., HOP4X(15), PIP_Y(id_N830) + 1.},
          {HOP4X(15), PIP_Y(id_N830) + 1., HOP4X(13), PIP_Y(id_N830) + left_wire_dist + 1.},
          {HOP4X(13), PIP_Y(id_N830) + left_wire_dist + 1., HOP4X(13), PIP_Y(id_N830) + 2.},
          {HOP4X(13), PIP_Y(id_N830) + 2., HOP4X(11), PIP_Y(id_N830) + left_wire_dist + 2.},
          {HOP4X(11), PIP_Y(id_N830) + left_wire_dist + 2., HOP4X(11), PIP_Y(id_N830) + 3.},
          {HOP4X(11), PIP_Y(id_N830) + 3., HOP4X(9), PIP_Y(id_N830) + left_wire_dist + 3.},
          {HOP4X(9), PIP_Y(id_N830) + left_wire_dist + 3., HOP4X(9), PIP_Y(id_N830) + 4.},
          {HOP4X(9), PIP_Y(id_N834) + 4., WIRE_X(0), PIP_Y(id_N834) + 4.},
          {HOP4X(9), PIP_Y(id_N830) + 4., HOP4X(7), PIP_Y(id_N830) + left_wire_dist + 4.},
          {HOP4X(7), PIP_Y(id_N830) + left_wire_dist + 4., HOP4X(7), PIP_Y(id_N830) + 5.},
          {HOP4X(7), PIP_Y(id_N830) + 5., HOP4X(5), PIP_Y(id_N830) + left_wire_dist + 5.},
          {HOP4X(5), PIP_Y(id_N830) + left_wire_dist + 5., HOP4X(5), PIP_Y(id_N830) + 6.},
          {HOP4X(5), PIP_Y(id_N830) + 6., HOP4X(3), PIP_Y(id_N830) + left_wire_dist + 6.},
          {HOP4X(3), PIP_Y(id_N830) + left_wire_dist + 6., HOP4X(3), PIP_Y(id_N830) + 7.},
          {HOP4X(3), PIP_Y(id_N830) + 7., HOP4X(1), PIP_Y(id_N830) + left_wire_dist + 7.},
          {HOP4X(1), PIP_Y(id_N830) + left_wire_dist + 7., HOP4X(1), PIP_Y(id_N838) + 8.},
          {HOP4X(1), PIP_Y(id_N838) + 8., WIRE_X(0), PIP_Y(id_N838) + 8.}}},
        {id_S83_loop0,
         {{WIRE_X(0), PIP_Y(id_S830), HOP4X(16), PIP_Y(id_S830)},
          {HOP4X(16), PIP_Y(id_S830), HOP4X(16), PIP_Y(id_N838)},
          {HOP4X(16), PIP_Y(id_N838) - 0., HOP4X(14), PIP_Y(id_N838) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N838) - left_wire_dist - 0., HOP4X(14), -wrap_len - 0.},
          {HOP4X(14), -wrap_len - 0., HOP4X(15), -wrap_len - 0.},
          {HOP4X(15), -wrap_len - 0., HOP4X(15), PIP_Y(id_N830) - 0.},
          {HOP4X(15), PIP_Y(id_N830) - 0., HOP4X(13), PIP_Y(id_N830) + left_wire_dist - 0.},
          {HOP4X(13), PIP_Y(id_N830) + left_wire_dist - 0., HOP4X(13), PIP_Y(id_N830) + 1.},
          {HOP4X(13), PIP_Y(id_N830) + 1., HOP4X(11), PIP_Y(id_N830) + left_wire_dist + 1.},
          {HOP4X(11), PIP_Y(id_N830) + left_wire_dist + 1., HOP4X(11), PIP_Y(id_N830) + 2.},
          {HOP4X(11), PIP_Y(id_N830) + 2., HOP4X(9), PIP_Y(id_N830) + left_wire_dist + 2.},
          {HOP4X(9), PIP_Y(id_N830) + left_wire_dist + 2., HOP4X(9), PIP_Y(id_N830) + 3.},
          {HOP4X(9), PIP_Y(id_N830) + 3., HOP4X(7), PIP_Y(id_N830) + left_wire_dist + 3.},
          {HOP4X(9), PIP_Y(id_N834) + 3., WIRE_X(0), PIP_Y(id_N834) + 3.},
          {HOP4X(7), PIP_Y(id_N830) + left_wire_dist + 3., HOP4X(7), PIP_Y(id_N830) + 4.},
          {HOP4X(7), PIP_Y(id_N830) + 4., HOP4X(5), PIP_Y(id_N830) + left_wire_dist + 4.},
          {HOP4X(5), PIP_Y(id_N830) + left_wire_dist + 4., HOP4X(5), PIP_Y(id_N830) + 5.},
          {HOP4X(5), PIP_Y(id_N830) + 5., HOP4X(3), PIP_Y(id_N830) + left_wire_dist + 5.},
          {HOP4X(3), PIP_Y(id_N830) + left_wire_dist + 5., HOP4X(3), PIP_Y(id_N830) + 6.},
          {HOP4X(3), PIP_Y(id_N830) + 6., HOP4X(1), PIP_Y(id_N830) + left_wire_dist + 6.},
          {HOP4X(1), PIP_Y(id_N830) + left_wire_dist + 6., HOP4X(1), PIP_Y(id_N838) + 7.},
          {HOP4X(1), PIP_Y(id_N838) + 7., WIRE_X(0), PIP_Y(id_N838) + 7.}}},
        {id_S83_loop1,
         {{WIRE_X(0), PIP_Y(id_S830), HOP4X(16), PIP_Y(id_S830)},
          {HOP4X(16), PIP_Y(id_S830), HOP4X(16), PIP_Y(id_N838)},
          {HOP4X(16), PIP_Y(id_N838) - 0., HOP4X(14), PIP_Y(id_N838) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N838) - left_wire_dist - 0., HOP4X(14), PIP_Y(id_N838) - 1.},
          {HOP4X(14), PIP_Y(id_N838) - 1., HOP4X(12), PIP_Y(id_N838) - left_wire_dist - 1.},
          {HOP4X(12), PIP_Y(id_N838) - left_wire_dist - 1., HOP4X(12), -wrap_len - 1.},
          {HOP4X(12), -wrap_len - 1., HOP4X(13), -wrap_len - 1.},
          {HOP4X(13), -wrap_len - 1., HOP4X(13), PIP_Y(id_N830) - 1.},
          {HOP4X(13), PIP_Y(id_N830) - 1., HOP4X(11), PIP_Y(id_N830) + left_wire_dist - 1.},
          {HOP4X(11), PIP_Y(id_N830) + left_wire_dist - 1., HOP4X(11), PIP_Y(id_N830) - 0.},
          {HOP4X(11), PIP_Y(id_N830) - 0., HOP4X(9), PIP_Y(id_N830) + left_wire_dist - 0.},
          {HOP4X(9), PIP_Y(id_N830) + left_wire_dist - 0., HOP4X(9), PIP_Y(id_N830) + 1.},
          {HOP4X(9), PIP_Y(id_N830) + 1., HOP4X(7), PIP_Y(id_N830) + left_wire_dist + 1.},
          {HOP4X(9), PIP_Y(id_N834) + 1., WIRE_X(0), PIP_Y(id_N834) + 1.},
          {HOP4X(7), PIP_Y(id_N830) + left_wire_dist + 1., HOP4X(7), PIP_Y(id_N830) + 2.},
          {HOP4X(7), PIP_Y(id_N830) + 2., HOP4X(5), PIP_Y(id_N830) + left_wire_dist + 2.},
          {HOP4X(5), PIP_Y(id_N830) + left_wire_dist + 2., HOP4X(5), PIP_Y(id_N830) + 3.},
          {HOP4X(5), PIP_Y(id_N830) + 3., HOP4X(3), PIP_Y(id_N830) + left_wire_dist + 3.},
          {HOP4X(3), PIP_Y(id_N830) + left_wire_dist + 3., HOP4X(3), PIP_Y(id_N830) + 4.},
          {HOP4X(3), PIP_Y(id_N830) + 4., HOP4X(1), PIP_Y(id_N830) + left_wire_dist + 4.},
          {HOP4X(1), PIP_Y(id_N830) + left_wire_dist + 4., HOP4X(1), PIP_Y(id_N838) + 5.},
          {HOP4X(1), PIP_Y(id_N838) + 5., WIRE_X(0), PIP_Y(id_N838) + 5.}}},
        {id_S83_loop2,
         {{WIRE_X(0), PIP_Y(id_S830), HOP4X(16), PIP_Y(id_S830)},
          {HOP4X(16), PIP_Y(id_S830), HOP4X(16), PIP_Y(id_N838)},
          {HOP4X(16), PIP_Y(id_N838) - 0., HOP4X(14), PIP_Y(id_N838) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N838) - left_wire_dist - 0., HOP4X(14), PIP_Y(id_N838) - 1.},
          {HOP4X(14), PIP_Y(id_N838) - 1., HOP4X(12), PIP_Y(id_N838) - left_wire_dist - 1.},
          {HOP4X(12), PIP_Y(id_N838) - left_wire_dist - 1., HOP4X(12), PIP_Y(id_N838) - 2.},
          {HOP4X(12), PIP_Y(id_N838) - 2., HOP4X(10), PIP_Y(id_N838) - left_wire_dist - 2.},
          {HOP4X(10), PIP_Y(id_N838) - left_wire_dist - 2., HOP4X(10), -wrap_len - 2.},
          {HOP4X(10), -wrap_len - 2., HOP4X(11), -wrap_len - 2.},
          {HOP4X(11), -wrap_len - 2., HOP4X(11), PIP_Y(id_N830) - 2.},
          {HOP4X(11), PIP_Y(id_N830) - 2., HOP4X(9), PIP_Y(id_N830) + left_wire_dist - 2.},
          {HOP4X(9), PIP_Y(id_N830) + left_wire_dist - 2., HOP4X(9), PIP_Y(id_N830) - 1.},
          {HOP4X(9), PIP_Y(id_N830) - 1., HOP4X(7), PIP_Y(id_N830) + left_wire_dist - 1.},
          {HOP4X(9), PIP_Y(id_N834) - 1., WIRE_X(0), PIP_Y(id_N834) - 1.},
          {HOP4X(7), PIP_Y(id_N830) + left_wire_dist - 1., HOP4X(7), PIP_Y(id_N830) - 0.},
          {HOP4X(7), PIP_Y(id_N830) - 0., HOP4X(5), PIP_Y(id_N830) + left_wire_dist - 0.},
          {HOP4X(5), PIP_Y(id_N830) + left_wire_dist - 0., HOP4X(5), PIP_Y(id_N830) + 1.},
          {HOP4X(5), PIP_Y(id_N830) + 1., HOP4X(3), PIP_Y(id_N830) + left_wire_dist + 1.},
          {HOP4X(3), PIP_Y(id_N830) + left_wire_dist + 1., HOP4X(3), PIP_Y(id_N830) + 2.},
          {HOP4X(3), PIP_Y(id_N830) + 2., HOP4X(1), PIP_Y(id_N830) + left_wire_dist + 2.},
          {HOP4X(1), PIP_Y(id_N830) + left_wire_dist + 2., HOP4X(1), PIP_Y(id_N838) + 3.},
          {HOP4X(1), PIP_Y(id_N838) + 3., WIRE_X(0), PIP_Y(id_N838) + 3.}}},
        {id_S83_loop3,
         {{WIRE_X(0), PIP_Y(id_S830), HOP4X(16), PIP_Y(id_S830)},
          {HOP4X(16), PIP_Y(id_S830), HOP4X(16), PIP_Y(id_N838)},
          {HOP4X(16), PIP_Y(id_N838) - 0., HOP4X(14), PIP_Y(id_N838) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N838) - left_wire_dist - 0., HOP4X(14), PIP_Y(id_N838) - 1.},
          {HOP4X(14), PIP_Y(id_N838) - 1., HOP4X(12), PIP_Y(id_N838) - left_wire_dist - 1.},
          {HOP4X(12), PIP_Y(id_N838) - left_wire_dist - 1., HOP4X(12), PIP_Y(id_N838) - 2.},
          {HOP4X(12), PIP_Y(id_N838) - 2., HOP4X(10), PIP_Y(id_N838) - left_wire_dist - 2.},
          {HOP4X(10), PIP_Y(id_N838) - left_wire_dist - 2., HOP4X(10), PIP_Y(id_N838) - 3.},
          {HOP4X(10), PIP_Y(id_N838) - 3., HOP4X(8), PIP_Y(id_N838) - left_wire_dist - 3.},
          {HOP4X(8), PIP_Y(id_N838) - left_wire_dist - 3., HOP4X(8), -wrap_len - 3.},
          {HOP4X(8), -wrap_len - 3., HOP4X(9), -wrap_len - 3.},
          {HOP4X(9), -wrap_len - 3., HOP4X(9), PIP_Y(id_N830) - 3.},
          {HOP4X(9), PIP_Y(id_N834) - 3., WIRE_X(0), PIP_Y(id_N834) - 3.},
          {HOP4X(9), PIP_Y(id_N830) - 3., HOP4X(7), PIP_Y(id_N830) + left_wire_dist - 3.},
          {HOP4X(7), PIP_Y(id_N830) + left_wire_dist - 3., HOP4X(7), PIP_Y(id_N830) - 2.},
          {HOP4X(7), PIP_Y(id_N830) - 2., HOP4X(5), PIP_Y(id_N830) + left_wire_dist - 2.},
          {HOP4X(5), PIP_Y(id_N830) + left_wire_dist - 2., HOP4X(5), PIP_Y(id_N830) - 1.},
          {HOP4X(5), PIP_Y(id_N830) - 1., HOP4X(3), PIP_Y(id_N830) + left_wire_dist - 1.},
          {HOP4X(3), PIP_Y(id_N830) + left_wire_dist - 1., HOP4X(3), PIP_Y(id_N830) - 0.},
          {HOP4X(3), PIP_Y(id_N830) - 0., HOP4X(1), PIP_Y(id_N830) + left_wire_dist - 0.},
          {HOP4X(1), PIP_Y(id_N830) + left_wire_dist - 0., HOP4X(1), PIP_Y(id_N838) + 1.},
          {HOP4X(1), PIP_Y(id_N838) + 1., WIRE_X(0), PIP_Y(id_N838) + 1.}}},
        {id_S83_loop4,
         {{WIRE_X(0), PIP_Y(id_S830), HOP4X(16), PIP_Y(id_S830)},
          {HOP4X(16), PIP_Y(id_S830), HOP4X(16), PIP_Y(id_N838)},
          {HOP4X(16), PIP_Y(id_N838) - 0., HOP4X(14), PIP_Y(id_N838) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N838) - left_wire_dist - 0., HOP4X(14), PIP_Y(id_N838) - 1.},
          {HOP4X(14), PIP_Y(id_N838) - 1., HOP4X(12), PIP_Y(id_N838) - left_wire_dist - 1.},
          {HOP4X(12), PIP_Y(id_N838) - left_wire_dist - 1., HOP4X(12), PIP_Y(id_N838) - 2.},
          {HOP4X(12), PIP_Y(id_N838) - 2., HOP4X(10), PIP_Y(id_N838) - left_wire_dist - 2.},
          {HOP4X(10), PIP_Y(id_N838) - left_wire_dist - 2., HOP4X(10), PIP_Y(id_N838) - 3.},
          {HOP4X(10), PIP_Y(id_N838) - 3., HOP4X(8), PIP_Y(id_N838) - left_wire_dist - 3.},
          {HOP4X(8), PIP_Y(id_N838) - left_wire_dist - 3., HOP4X(8), PIP_Y(id_N838) - 4.},
          {HOP4X(8), PIP_Y(id_S834) - 4., WIRE_X(0), PIP_Y(id_S834) - 4.},
          {HOP4X(8), PIP_Y(id_N838) - 4., HOP4X(6), PIP_Y(id_N838) - left_wire_dist - 4.},
          {HOP4X(6), PIP_Y(id_N838) - left_wire_dist - 4., HOP4X(6), -wrap_len - 4.},
          {HOP4X(6), -wrap_len - 4., HOP4X(7), -wrap_len - 4.},
          {HOP4X(7), -wrap_len - 4., HOP4X(7), PIP_Y(id_N830) - 4.},
          {HOP4X(7), PIP_Y(id_N830) - 4., HOP4X(5), PIP_Y(id_N830) + left_wire_dist - 4.},
          {HOP4X(5), PIP_Y(id_N830) + left_wire_dist - 4., HOP4X(5), PIP_Y(id_N830) - 3.},
          {HOP4X(5), PIP_Y(id_N830) - 3., HOP4X(3), PIP_Y(id_N830) + left_wire_dist - 3.},
          {HOP4X(3), PIP_Y(id_N830) + left_wire_dist - 3., HOP4X(3), PIP_Y(id_N830) - 2.},
          {HOP4X(3), PIP_Y(id_N830) - 2., HOP4X(1), PIP_Y(id_N830) + left_wire_dist - 2.},
          {HOP4X(1), PIP_Y(id_N830) + left_wire_dist - 2., HOP4X(1), PIP_Y(id_N838) - 1.},
          {HOP4X(1), PIP_Y(id_N838) - 1., WIRE_X(0), PIP_Y(id_N838) - 1.}}},
        {id_S83_loop5,
         {{WIRE_X(0), PIP_Y(id_S830), HOP4X(16), PIP_Y(id_S830)},
          {HOP4X(16), PIP_Y(id_S830), HOP4X(16), PIP_Y(id_N838)},
          {HOP4X(16), PIP_Y(id_N838) - 0., HOP4X(14), PIP_Y(id_N838) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N838) - left_wire_dist - 0., HOP4X(14), PIP_Y(id_N838) - 1.},
          {HOP4X(14), PIP_Y(id_N838) - 1., HOP4X(12), PIP_Y(id_N838) - left_wire_dist - 1.},
          {HOP4X(12), PIP_Y(id_N838) - left_wire_dist - 1., HOP4X(12), PIP_Y(id_N838) - 2.},
          {HOP4X(12), PIP_Y(id_N838) - 2., HOP4X(10), PIP_Y(id_N838) - left_wire_dist - 2.},
          {HOP4X(10), PIP_Y(id_N838) - left_wire_dist - 2., HOP4X(10), PIP_Y(id_N838) - 3.},
          {HOP4X(10), PIP_Y(id_N838) - 3., HOP4X(8), PIP_Y(id_N838) - left_wire_dist - 3.},
          {HOP4X(8), PIP_Y(id_N838) - left_wire_dist - 3., HOP4X(8), PIP_Y(id_N838) - 4.},
          {HOP4X(8), PIP_Y(id_S834) - 4., WIRE_X(0), PIP_Y(id_S834) - 4.},
          {HOP4X(8), PIP_Y(id_N838) - 4., HOP4X(6), PIP_Y(id_N838) - left_wire_dist - 4.},
          {HOP4X(6), PIP_Y(id_N838) - left_wire_dist - 4., HOP4X(6), PIP_Y(id_N838) - 5.},
          {HOP4X(6), PIP_Y(id_N838) - 5., HOP4X(4), PIP_Y(id_N838) - left_wire_dist - 5.},
          {HOP4X(4), PIP_Y(id_N838) - left_wire_dist - 5., HOP4X(4), -wrap_len - 5.},
          {HOP4X(4), -wrap_len - 5., HOP4X(5), -wrap_len - 5.},
          {HOP4X(5), -wrap_len - 5., HOP4X(5), PIP_Y(id_N830) - 5.},
          {HOP4X(5), PIP_Y(id_N830) - 5., HOP4X(3), PIP_Y(id_N830) + left_wire_dist - 5.},
          {HOP4X(3), PIP_Y(id_N830) + left_wire_dist - 5., HOP4X(3), PIP_Y(id_N830) - 4.},
          {HOP4X(3), PIP_Y(id_N830) - 4., HOP4X(1), PIP_Y(id_N830) + left_wire_dist - 4.},
          {HOP4X(1), PIP_Y(id_N830) + left_wire_dist - 4., HOP4X(1), PIP_Y(id_N838) - 3.},
          {HOP4X(1), PIP_Y(id_N838) - 3., WIRE_X(0), PIP_Y(id_N838) - 3.}}},
        {id_S83_loop6,
         {{WIRE_X(0), PIP_Y(id_S830), HOP4X(16), PIP_Y(id_S830)},
          {HOP4X(16), PIP_Y(id_S830), HOP4X(16), PIP_Y(id_N838)},
          {HOP4X(16), PIP_Y(id_N838) - 0., HOP4X(14), PIP_Y(id_N838) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N838) - left_wire_dist - 0., HOP4X(14), PIP_Y(id_N838) - 1.},
          {HOP4X(14), PIP_Y(id_N838) - 1., HOP4X(12), PIP_Y(id_N838) - left_wire_dist - 1.},
          {HOP4X(12), PIP_Y(id_N838) - left_wire_dist - 1., HOP4X(12), PIP_Y(id_N838) - 2.},
          {HOP4X(12), PIP_Y(id_N838) - 2., HOP4X(10), PIP_Y(id_N838) - left_wire_dist - 2.},
          {HOP4X(10), PIP_Y(id_N838) - left_wire_dist - 2., HOP4X(10), PIP_Y(id_N838) - 3.},
          {HOP4X(10), PIP_Y(id_N838) - 3., HOP4X(8), PIP_Y(id_N838) - left_wire_dist - 3.},
          {HOP4X(8), PIP_Y(id_N838) - left_wire_dist - 3., HOP4X(8), PIP_Y(id_N838) - 4.},
          {HOP4X(8), PIP_Y(id_S834) - 4., WIRE_X(0), PIP_Y(id_S834) - 4.},
          {HOP4X(8), PIP_Y(id_N838) - 4., HOP4X(6), PIP_Y(id_N838) - left_wire_dist - 4.},
          {HOP4X(6), PIP_Y(id_N838) - left_wire_dist - 4., HOP4X(6), PIP_Y(id_N838) - 5.},
          {HOP4X(6), PIP_Y(id_N838) - 5., HOP4X(4), PIP_Y(id_N838) - left_wire_dist - 5.},
          {HOP4X(4), PIP_Y(id_N838) - left_wire_dist - 5., HOP4X(4), PIP_Y(id_N838) - 6.},
          {HOP4X(4), PIP_Y(id_N838) - 6., HOP4X(2), PIP_Y(id_N838) - left_wire_dist - 6.},
          {HOP4X(2), PIP_Y(id_N838) - left_wire_dist - 6., HOP4X(2), -wrap_len - 6.},
          {HOP4X(2), -wrap_len - 6., HOP4X(3), -wrap_len - 6.},
          {HOP4X(3), -wrap_len - 6., HOP4X(3), PIP_Y(id_N830) - 6.},
          {HOP4X(3), PIP_Y(id_N830) - 6., HOP4X(1), PIP_Y(id_N830) + left_wire_dist - 6.},
          {HOP4X(1), PIP_Y(id_N830) + left_wire_dist - 6., HOP4X(1), PIP_Y(id_N838) - 5.},
          {HOP4X(1), PIP_Y(id_N838) - 5., WIRE_X(0), PIP_Y(id_N838) - 5.}}},
        {id_S83_loop7,
         {{WIRE_X(0), PIP_Y(id_S830), HOP4X(16), PIP_Y(id_S830)},
          {HOP4X(16), PIP_Y(id_S830), HOP4X(16), PIP_Y(id_N838)},
          {HOP4X(16), PIP_Y(id_N838) - 0., HOP4X(14), PIP_Y(id_N838) - left_wire_dist - 0.},
          {HOP4X(14), PIP_Y(id_N838) - left_wire_dist - 0., HOP4X(14), PIP_Y(id_N838) - 1.},
          {HOP4X(14), PIP_Y(id_N838) - 1., HOP4X(12), PIP_Y(id_N838) - left_wire_dist - 1.},
          {HOP4X(12), PIP_Y(id_N838) - left_wire_dist - 1., HOP4X(12), PIP_Y(id_N838) - 2.},
          {HOP4X(12), PIP_Y(id_N838) - 2., HOP4X(10), PIP_Y(id_N838) - left_wire_dist - 2.},
          {HOP4X(10), PIP_Y(id_N838) - left_wire_dist - 2., HOP4X(10), PIP_Y(id_N838) - 3.},
          {HOP4X(10), PIP_Y(id_N838) - 3., HOP4X(8), PIP_Y(id_N838) - left_wire_dist - 3.},
          {HOP4X(8), PIP_Y(id_N838) - left_wire_dist - 3., HOP4X(8), PIP_Y(id_N838) - 4.},
          {HOP4X(8), PIP_Y(id_S834) - 4., WIRE_X(0), PIP_Y(id_S834) - 4.},
          {HOP4X(8), PIP_Y(id_N838) - 4., HOP4X(6), PIP_Y(id_N838) - left_wire_dist - 4.},
          {HOP4X(6), PIP_Y(id_N838) - left_wire_dist - 4., HOP4X(6), PIP_Y(id_N838) - 5.},
          {HOP4X(6), PIP_Y(id_N838) - 5., HOP4X(4), PIP_Y(id_N838) - left_wire_dist - 5.},
          {HOP4X(4), PIP_Y(id_N838) - left_wire_dist - 5., HOP4X(4), PIP_Y(id_N838) - 6.},
          {HOP4X(4), PIP_Y(id_N838) - 6., HOP4X(2), PIP_Y(id_N838) - left_wire_dist - 6.},
          {HOP4X(2), PIP_Y(id_N838) - left_wire_dist - 6., HOP4X(2), PIP_Y(id_N838) - 7.},
          {HOP4X(2), PIP_Y(id_N838) - 7., HOP4X(0), PIP_Y(id_N838) - left_wire_dist - 7.},
          {HOP4X(0), PIP_Y(id_N838) - left_wire_dist - 7., HOP4X(0), -wrap_len - 7.},
          {HOP4X(0), -wrap_len - 7., HOP4X(1), -wrap_len - 7.},
          {HOP4X(1), -wrap_len - 7., HOP4X(1), PIP_Y(id_N838) - 7.},
          {HOP4X(1), PIP_Y(id_N838) - 7., WIRE_X(0), PIP_Y(id_N838) - 7.}}},
        {id_N83_loop0,
         {{WIRE_X(0), PIP_Y(id_N830), HOP4X(17), PIP_Y(id_N830)},
          {HOP4X(17), PIP_Y(id_N830) + 0., HOP4X(15), PIP_Y(id_N830) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N830) + left_wire_dist + 0., HOP4X(15), wrap_len + 1.},
          {HOP4X(15), wrap_len + 1., HOP4X(14), wrap_len + 1.},
          {HOP4X(14), wrap_len + 1., HOP4X(14), PIP_Y(id_N838) + 0.},
          {HOP4X(14), PIP_Y(id_N838) + 0., HOP4X(12), PIP_Y(id_N838) - left_wire_dist + 0.},
          {HOP4X(12), PIP_Y(id_N838) - left_wire_dist + 0., HOP4X(12), PIP_Y(id_N838) - 1.},
          {HOP4X(12), PIP_Y(id_N838) - 1., HOP4X(10), PIP_Y(id_N838) - left_wire_dist - 1.},
          {HOP4X(10), PIP_Y(id_N838) - left_wire_dist - 1., HOP4X(10), PIP_Y(id_N838) - 2.},
          {HOP4X(10), PIP_Y(id_N838) - 2., HOP4X(8), PIP_Y(id_N838) - left_wire_dist - 2.},
          {HOP4X(8), PIP_Y(id_N838) - left_wire_dist - 2., HOP4X(8), PIP_Y(id_N838) - 3.},
          {HOP4X(8), PIP_Y(id_S834) - 3., WIRE_X(0), PIP_Y(id_S834) - 3.},
          {HOP4X(8), PIP_Y(id_N838) - 3., HOP4X(6), PIP_Y(id_N838) - left_wire_dist - 3.},
          {HOP4X(6), PIP_Y(id_N838) - left_wire_dist - 3., HOP4X(6), PIP_Y(id_N838) - 4.},
          {HOP4X(6), PIP_Y(id_N838) - 4., HOP4X(4), PIP_Y(id_N838) - left_wire_dist - 4.},
          {HOP4X(4), PIP_Y(id_N838) - left_wire_dist - 4., HOP4X(4), PIP_Y(id_N838) - 5.},
          {HOP4X(4), PIP_Y(id_N838) - 5., HOP4X(2), PIP_Y(id_N838) - left_wire_dist - 5.},
          {HOP4X(2), PIP_Y(id_N838) - left_wire_dist - 5., HOP4X(2), PIP_Y(id_N838) - 6.},
          {HOP4X(2), PIP_Y(id_N838) - 6., HOP4X(0), PIP_Y(id_N838) - left_wire_dist - 6.},
          {HOP4X(0), PIP_Y(id_N838) - left_wire_dist - 6., HOP4X(0), PIP_Y(id_S838) - 7.},
          {HOP4X(0), PIP_Y(id_S838) - 7., WIRE_X(0), PIP_Y(id_S838) - 7.}}},
        {id_N83_loop1,
         {{WIRE_X(0), PIP_Y(id_N830), HOP4X(17), PIP_Y(id_N830)},
          {HOP4X(17), PIP_Y(id_N830) + 0., HOP4X(15), PIP_Y(id_N830) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N830) + left_wire_dist + 0., HOP4X(15), PIP_Y(id_N830) + 1.},
          {HOP4X(15), PIP_Y(id_N830) + 1., HOP4X(13), PIP_Y(id_N830) + left_wire_dist + 1.},
          {HOP4X(13), PIP_Y(id_N830) + left_wire_dist + 1., HOP4X(13), wrap_len + 2.},
          {HOP4X(13), wrap_len + 2., HOP4X(12), wrap_len + 2.},
          {HOP4X(12), wrap_len + 2., HOP4X(12), PIP_Y(id_N838) + 1.},
          {HOP4X(12), PIP_Y(id_N838) + 1., HOP4X(10), PIP_Y(id_N838) - left_wire_dist + 1.},
          {HOP4X(10), PIP_Y(id_N838) - left_wire_dist + 1., HOP4X(10), PIP_Y(id_N838) + 0.},
          {HOP4X(10), PIP_Y(id_N838) + 0., HOP4X(8), PIP_Y(id_N838) - left_wire_dist + 0.},
          {HOP4X(8), PIP_Y(id_N838) - left_wire_dist + 0., HOP4X(8), PIP_Y(id_N838) - 1.},
          {HOP4X(8), PIP_Y(id_S834) - 1., WIRE_X(0), PIP_Y(id_S834) - 1.},
          {HOP4X(8), PIP_Y(id_N838) - 1., HOP4X(6), PIP_Y(id_N838) - left_wire_dist - 1.},
          {HOP4X(6), PIP_Y(id_N838) - left_wire_dist - 1., HOP4X(6), PIP_Y(id_N838) - 2.},
          {HOP4X(6), PIP_Y(id_N838) - 2., HOP4X(4), PIP_Y(id_N838) - left_wire_dist - 2.},
          {HOP4X(4), PIP_Y(id_N838) - left_wire_dist - 2., HOP4X(4), PIP_Y(id_N838) - 3.},
          {HOP4X(4), PIP_Y(id_N838) - 3., HOP4X(2), PIP_Y(id_N838) - left_wire_dist - 3.},
          {HOP4X(2), PIP_Y(id_N838) - left_wire_dist - 3., HOP4X(2), PIP_Y(id_N838) - 4.},
          {HOP4X(2), PIP_Y(id_N838) - 4., HOP4X(0), PIP_Y(id_N838) - left_wire_dist - 4.},
          {HOP4X(0), PIP_Y(id_N838) - left_wire_dist - 4., HOP4X(0), PIP_Y(id_S838) - 5.},
          {HOP4X(0), PIP_Y(id_S838) - 5., WIRE_X(0), PIP_Y(id_S838) - 5.}}},
        {id_N83_loop2,
         {{WIRE_X(0), PIP_Y(id_N830), HOP4X(17), PIP_Y(id_N830)},
          {HOP4X(17), PIP_Y(id_N830) + 0., HOP4X(15), PIP_Y(id_N830) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N830) + left_wire_dist + 0., HOP4X(15), PIP_Y(id_N830) + 1.},
          {HOP4X(15), PIP_Y(id_N830) + 1., HOP4X(13), PIP_Y(id_N830) + left_wire_dist + 1.},
          {HOP4X(13), PIP_Y(id_N830) + left_wire_dist + 1., HOP4X(13), PIP_Y(id_N830) + 2.},
          {HOP4X(13), PIP_Y(id_N830) + 2., HOP4X(11), PIP_Y(id_N830) + left_wire_dist + 2.},
          {HOP4X(11), PIP_Y(id_N830) + left_wire_dist + 2., HOP4X(11), wrap_len + 3.},
          {HOP4X(11), wrap_len + 3., HOP4X(10), wrap_len + 3.},
          {HOP4X(10), wrap_len + 3., HOP4X(10), PIP_Y(id_N838) + 2.},
          {HOP4X(10), PIP_Y(id_N838) + 2., HOP4X(8), PIP_Y(id_N838) - left_wire_dist + 2.},
          {HOP4X(8), PIP_Y(id_N838) - left_wire_dist + 2., HOP4X(8), PIP_Y(id_N838) + 1.},
          {HOP4X(8), PIP_Y(id_S834) + 1., WIRE_X(0), PIP_Y(id_S834) + 1.},
          {HOP4X(8), PIP_Y(id_N838) + 1., HOP4X(6), PIP_Y(id_N838) - left_wire_dist + 1.},
          {HOP4X(6), PIP_Y(id_N838) - left_wire_dist + 1., HOP4X(6), PIP_Y(id_N838) + 0.},
          {HOP4X(6), PIP_Y(id_N838) + 0., HOP4X(4), PIP_Y(id_N838) - left_wire_dist + 0.},
          {HOP4X(4), PIP_Y(id_N838) - left_wire_dist + 0., HOP4X(4), PIP_Y(id_N838) - 1.},
          {HOP4X(4), PIP_Y(id_N838) - 1., HOP4X(2), PIP_Y(id_N838) - left_wire_dist - 1.},
          {HOP4X(2), PIP_Y(id_N838) - left_wire_dist - 1., HOP4X(2), PIP_Y(id_N838) - 2.},
          {HOP4X(2), PIP_Y(id_N838) - 2., HOP4X(0), PIP_Y(id_N838) - left_wire_dist - 2.},
          {HOP4X(0), PIP_Y(id_N838) - left_wire_dist - 2., HOP4X(0), PIP_Y(id_S838) - 3.},
          {HOP4X(0), PIP_Y(id_S838) - 3., WIRE_X(0), PIP_Y(id_S838) - 3.}}},
        {id_N83_loop3,
         {{WIRE_X(0), PIP_Y(id_N830), HOP4X(17), PIP_Y(id_N830)},
          {HOP4X(17), PIP_Y(id_N830) + 0., HOP4X(15), PIP_Y(id_N830) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N830) + left_wire_dist + 0., HOP4X(15), PIP_Y(id_N830) + 1.},
          {HOP4X(15), PIP_Y(id_N830) + 1., HOP4X(13), PIP_Y(id_N830) + left_wire_dist + 1.},
          {HOP4X(13), PIP_Y(id_N830) + left_wire_dist + 1., HOP4X(13), PIP_Y(id_N830) + 2.},
          {HOP4X(13), PIP_Y(id_N830) + 2., HOP4X(11), PIP_Y(id_N830) + left_wire_dist + 2.},
          {HOP4X(11), PIP_Y(id_N830) + left_wire_dist + 2., HOP4X(11), PIP_Y(id_N830) + 3.},
          {HOP4X(11), PIP_Y(id_N830) + 3., HOP4X(9), PIP_Y(id_N830) + left_wire_dist + 3.},
          {HOP4X(9), PIP_Y(id_N830) + left_wire_dist + 3., HOP4X(9), wrap_len + 4.},
          {HOP4X(9), wrap_len + 4., HOP4X(8), wrap_len + 4.},
          {HOP4X(8), wrap_len + 4., HOP4X(8), PIP_Y(id_N838) + 3.},
          {HOP4X(8), PIP_Y(id_S834) + 3., WIRE_X(0), PIP_Y(id_S834) + 3.},
          {HOP4X(8), PIP_Y(id_N838) + 3., HOP4X(6), PIP_Y(id_N838) - left_wire_dist + 3.},
          {HOP4X(6), PIP_Y(id_N838) - left_wire_dist + 3., HOP4X(6), PIP_Y(id_N838) + 2.},
          {HOP4X(6), PIP_Y(id_N838) + 2., HOP4X(4), PIP_Y(id_N838) - left_wire_dist + 2.},
          {HOP4X(4), PIP_Y(id_N838) - left_wire_dist + 2., HOP4X(4), PIP_Y(id_N838) + 1.},
          {HOP4X(4), PIP_Y(id_N838) + 1., HOP4X(2), PIP_Y(id_N838) - left_wire_dist + 1.},
          {HOP4X(2), PIP_Y(id_N838) - left_wire_dist + 1., HOP4X(2), PIP_Y(id_N838) + 0.},
          {HOP4X(2), PIP_Y(id_N838) + 0., HOP4X(0), PIP_Y(id_N838) - left_wire_dist + 0.},
          {HOP4X(0), PIP_Y(id_N838) - left_wire_dist + 0., HOP4X(0), PIP_Y(id_S838) - 1.},
          {HOP4X(0), PIP_Y(id_S838) - 1., WIRE_X(0), PIP_Y(id_S838) - 1.}}},
        {id_N83_loop4,
         {{WIRE_X(0), PIP_Y(id_N830), HOP4X(17), PIP_Y(id_N830)},
          {HOP4X(17), PIP_Y(id_N830) + 0., HOP4X(15), PIP_Y(id_N830) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N830) + left_wire_dist + 0., HOP4X(15), PIP_Y(id_N830) + 1.},
          {HOP4X(15), PIP_Y(id_N830) + 1., HOP4X(13), PIP_Y(id_N830) + left_wire_dist + 1.},
          {HOP4X(13), PIP_Y(id_N830) + left_wire_dist + 1., HOP4X(13), PIP_Y(id_N830) + 2.},
          {HOP4X(13), PIP_Y(id_N830) + 2., HOP4X(11), PIP_Y(id_N830) + left_wire_dist + 2.},
          {HOP4X(11), PIP_Y(id_N830) + left_wire_dist + 2., HOP4X(11), PIP_Y(id_N830) + 3.},
          {HOP4X(11), PIP_Y(id_N830) + 3., HOP4X(9), PIP_Y(id_N830) + left_wire_dist + 3.},
          {HOP4X(9), PIP_Y(id_N830) + left_wire_dist + 3., HOP4X(9), PIP_Y(id_N830) + 4.},
          {HOP4X(9), PIP_Y(id_N834) + 4., WIRE_X(0), PIP_Y(id_N834) + 4.},
          {HOP4X(9), PIP_Y(id_N830) + 4., HOP4X(7), PIP_Y(id_N830) + left_wire_dist + 4.},
          {HOP4X(7), PIP_Y(id_N830) + left_wire_dist + 4., HOP4X(7), wrap_len + 5.},
          {HOP4X(7), wrap_len + 5., HOP4X(6), wrap_len + 5.},
          {HOP4X(6), wrap_len + 5., HOP4X(6), PIP_Y(id_N838) + 4.},
          {HOP4X(6), PIP_Y(id_N838) + 4., HOP4X(4), PIP_Y(id_N838) - left_wire_dist + 4.},
          {HOP4X(4), PIP_Y(id_N838) - left_wire_dist + 4., HOP4X(4), PIP_Y(id_N838) + 3.},
          {HOP4X(4), PIP_Y(id_N838) + 3., HOP4X(2), PIP_Y(id_N838) - left_wire_dist + 3.},
          {HOP4X(2), PIP_Y(id_N838) - left_wire_dist + 3., HOP4X(2), PIP_Y(id_N838) + 2.},
          {HOP4X(2), PIP_Y(id_N838) + 2., HOP4X(0), PIP_Y(id_N838) - left_wire_dist + 2.},
          {HOP4X(0), PIP_Y(id_N838) - left_wire_dist + 2., HOP4X(0), PIP_Y(id_S838) + 1.},
          {HOP4X(0), PIP_Y(id_S838) + 1., WIRE_X(0), PIP_Y(id_S838) + 1.}}},
        {id_N83_loop5,
         {{WIRE_X(0), PIP_Y(id_N830), HOP4X(17), PIP_Y(id_N830)},
          {HOP4X(17), PIP_Y(id_N830) + 0., HOP4X(15), PIP_Y(id_N830) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N830) + left_wire_dist + 0., HOP4X(15), PIP_Y(id_N830) + 1.},
          {HOP4X(15), PIP_Y(id_N830) + 1., HOP4X(13), PIP_Y(id_N830) + left_wire_dist + 1.},
          {HOP4X(13), PIP_Y(id_N830) + left_wire_dist + 1., HOP4X(13), PIP_Y(id_N830) + 2.},
          {HOP4X(13), PIP_Y(id_N830) + 2., HOP4X(11), PIP_Y(id_N830) + left_wire_dist + 2.},
          {HOP4X(11), PIP_Y(id_N830) + left_wire_dist + 2., HOP4X(11), PIP_Y(id_N830) + 3.},
          {HOP4X(11), PIP_Y(id_N830) + 3., HOP4X(9), PIP_Y(id_N830) + left_wire_dist + 3.},
          {HOP4X(9), PIP_Y(id_N830) + left_wire_dist + 3., HOP4X(9), PIP_Y(id_N830) + 4.},
          {HOP4X(9), PIP_Y(id_N834) + 4., WIRE_X(0), PIP_Y(id_N834) + 4.},
          {HOP4X(9), PIP_Y(id_N830) + 4., HOP4X(7), PIP_Y(id_N830) + left_wire_dist + 4.},
          {HOP4X(7), PIP_Y(id_N830) + left_wire_dist + 4., HOP4X(7), PIP_Y(id_N830) + 5.},
          {HOP4X(7), PIP_Y(id_N830) + 5., HOP4X(5), PIP_Y(id_N830) + left_wire_dist + 5.},
          {HOP4X(5), PIP_Y(id_N830) + left_wire_dist + 5., HOP4X(5), wrap_len + 6.},
          {HOP4X(5), wrap_len + 6., HOP4X(4), wrap_len + 6.},
          {HOP4X(4), wrap_len + 6., HOP4X(4), PIP_Y(id_N838) + 5.},
          {HOP4X(4), PIP_Y(id_N838) + 5., HOP4X(2), PIP_Y(id_N838) - left_wire_dist + 5.},
          {HOP4X(2), PIP_Y(id_N838) - left_wire_dist + 5., HOP4X(2), PIP_Y(id_N838) + 4.},
          {HOP4X(2), PIP_Y(id_N838) + 4., HOP4X(0), PIP_Y(id_N838) - left_wire_dist + 4.},
          {HOP4X(0), PIP_Y(id_N838) - left_wire_dist + 4., HOP4X(0), PIP_Y(id_S838) + 3.},
          {HOP4X(0), PIP_Y(id_S838) + 3., WIRE_X(0), PIP_Y(id_S838) + 3.}}},
        {id_N83_loop6,
         {{WIRE_X(0), PIP_Y(id_N830), HOP4X(17), PIP_Y(id_N830)},
          {HOP4X(17), PIP_Y(id_N830) + 0., HOP4X(15), PIP_Y(id_N830) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N830) + left_wire_dist + 0., HOP4X(15), PIP_Y(id_N830) + 1.},
          {HOP4X(15), PIP_Y(id_N830) + 1., HOP4X(13), PIP_Y(id_N830) + left_wire_dist + 1.},
          {HOP4X(13), PIP_Y(id_N830) + left_wire_dist + 1., HOP4X(13), PIP_Y(id_N830) + 2.},
          {HOP4X(13), PIP_Y(id_N830) + 2., HOP4X(11), PIP_Y(id_N830) + left_wire_dist + 2.},
          {HOP4X(11), PIP_Y(id_N830) + left_wire_dist + 2., HOP4X(11), PIP_Y(id_N830) + 3.},
          {HOP4X(11), PIP_Y(id_N830) + 3., HOP4X(9), PIP_Y(id_N830) + left_wire_dist + 3.},
          {HOP4X(9), PIP_Y(id_N830) + left_wire_dist + 3., HOP4X(9), PIP_Y(id_N830) + 4.},
          {HOP4X(9), PIP_Y(id_N834) + 4., WIRE_X(0), PIP_Y(id_N834) + 4.},
          {HOP4X(9), PIP_Y(id_N830) + 4., HOP4X(7), PIP_Y(id_N830) + left_wire_dist + 4.},
          {HOP4X(7), PIP_Y(id_N830) + left_wire_dist + 4., HOP4X(7), PIP_Y(id_N830) + 5.},
          {HOP4X(7), PIP_Y(id_N830) + 5., HOP4X(5), PIP_Y(id_N830) + left_wire_dist + 5.},
          {HOP4X(5), PIP_Y(id_N830) + left_wire_dist + 5., HOP4X(5), PIP_Y(id_N830) + 6.},
          {HOP4X(5), PIP_Y(id_N830) + 6., HOP4X(3), PIP_Y(id_N830) + left_wire_dist + 6.},
          {HOP4X(3), PIP_Y(id_N830) + left_wire_dist + 6., HOP4X(3), wrap_len + 7.},
          {HOP4X(3), wrap_len + 7., HOP4X(2), wrap_len + 7.},
          {HOP4X(2), wrap_len + 7., HOP4X(2), PIP_Y(id_N838) + 6.},
          {HOP4X(2), PIP_Y(id_N838) + 6., HOP4X(0), PIP_Y(id_N838) - left_wire_dist + 6.},
          {HOP4X(0), PIP_Y(id_N838) - left_wire_dist + 6., HOP4X(0), PIP_Y(id_S838) + 5.},
          {HOP4X(0), PIP_Y(id_S838) + 5., WIRE_X(0), PIP_Y(id_S838) + 5.}}},
        {id_N83_loop7,
         {{WIRE_X(0), PIP_Y(id_N830), HOP4X(17), PIP_Y(id_N830)},
          {HOP4X(17), PIP_Y(id_N830) + 0., HOP4X(15), PIP_Y(id_N830) + left_wire_dist + 0.},
          {HOP4X(15), PIP_Y(id_N830) + left_wire_dist + 0., HOP4X(15), PIP_Y(id_N830) + 1.},
          {HOP4X(15), PIP_Y(id_N830) + 1., HOP4X(13), PIP_Y(id_N830) + left_wire_dist + 1.},
          {HOP4X(13), PIP_Y(id_N830) + left_wire_dist + 1., HOP4X(13), PIP_Y(id_N830) + 2.},
          {HOP4X(13), PIP_Y(id_N830) + 2., HOP4X(11), PIP_Y(id_N830) + left_wire_dist + 2.},
          {HOP4X(11), PIP_Y(id_N830) + left_wire_dist + 2., HOP4X(11), PIP_Y(id_N830) + 3.},
          {HOP4X(11), PIP_Y(id_N830) + 3., HOP4X(9), PIP_Y(id_N830) + left_wire_dist + 3.},
          {HOP4X(9), PIP_Y(id_N830) + left_wire_dist + 3., HOP4X(9), PIP_Y(id_N830) + 4.},
          {HOP4X(9), PIP_Y(id_N834) + 4., WIRE_X(0), PIP_Y(id_N834) + 4.},
          {HOP4X(9), PIP_Y(id_N830) + 4., HOP4X(7), PIP_Y(id_N830) + left_wire_dist + 4.},
          {HOP4X(7), PIP_Y(id_N830) + left_wire_dist + 4., HOP4X(7), PIP_Y(id_N830) + 5.},
          {HOP4X(7), PIP_Y(id_N830) + 5., HOP4X(5), PIP_Y(id_N830) + left_wire_dist + 5.},
          {HOP4X(5), PIP_Y(id_N830) + left_wire_dist + 5., HOP4X(5), PIP_Y(id_N830) + 6.},
          {HOP4X(5), PIP_Y(id_N830) + 6., HOP4X(3), PIP_Y(id_N830) + left_wire_dist + 6.},
          {HOP4X(3), PIP_Y(id_N830) + left_wire_dist + 6., HOP4X(3), PIP_Y(id_N830) + 7.},
          {HOP4X(3), PIP_Y(id_N830) + 7., HOP4X(1), PIP_Y(id_N830) + left_wire_dist + 7.},
          {HOP4X(1), PIP_Y(id_N830) + left_wire_dist + 7., HOP4X(1), wrap_len + 8.},
          {HOP4X(1), wrap_len + 8., HOP4X(0), wrap_len + 8.},
          {HOP4X(0), wrap_len + 8., HOP4X(0), PIP_Y(id_S838) + 7.},
          {HOP4X(0), PIP_Y(id_S838) + 7., WIRE_X(0), PIP_Y(id_S838) + 7.}}},

#define PIP_X(pip_id) (pipPoint.at(pip_id).second)
#define WIRE_Y(offset) (cru_y + cru_h + ((float)offset) * ew_dist)
        // 1 hop
        {id_E10,
         {{PIP_X(id_E100), WIRE_Y(0), PIP_X(id_E100), WIRE_Y(1)},
          {PIP_X(id_E100), WIRE_Y(1), PIP_X(id_E101) + 1., WIRE_Y(1)},
          {PIP_X(id_E101) + 1., WIRE_Y(1), PIP_X(id_E101) + 1., WIRE_Y(0)}}},
        {id_W10,
         {{PIP_X(id_W100), WIRE_Y(0), PIP_X(id_W100), WIRE_Y(2)},
          {PIP_X(id_W100), WIRE_Y(2), PIP_X(id_W101) - 1., WIRE_Y(2)},
          {PIP_X(id_W101) - 1., WIRE_Y(2), PIP_X(id_W101) - 1., WIRE_Y(0)}}},
        {id_E10_loop0,
         {{PIP_X(id_E100), WIRE_Y(0), PIP_X(id_E100), WIRE_Y(1)},
          {PIP_X(id_E100), WIRE_Y(1), 1. + wrap_len, WIRE_Y(1)},
          {1. + wrap_len, WIRE_Y(1), 1. + wrap_len, WIRE_Y(2)},
          {1. + wrap_len, WIRE_Y(2), PIP_X(id_W101), WIRE_Y(2)},
          {PIP_X(id_W101), WIRE_Y(2), PIP_X(id_W101), WIRE_Y(0)}}},
        {id_W10_loop0,
         {{PIP_X(id_W100), WIRE_Y(0), PIP_X(id_W100), WIRE_Y(2)},
          {PIP_X(id_W100), WIRE_Y(2), -1. * wrap_len, WIRE_Y(2)},
          {-1. * wrap_len, WIRE_Y(2), -1. * wrap_len, WIRE_Y(1)},
          {-1. * wrap_len, WIRE_Y(1), PIP_X(id_E101), WIRE_Y(1)},
          {PIP_X(id_E101), WIRE_Y(1), PIP_X(id_E101), WIRE_Y(0)}}},
        {id_E13,
         {{PIP_X(id_E130), WIRE_Y(0), PIP_X(id_E130), WIRE_Y(3)},
          {PIP_X(id_E130), WIRE_Y(3), PIP_X(id_E131) + 1., WIRE_Y(3)},
          {PIP_X(id_E131) + 1., WIRE_Y(3), PIP_X(id_E131) + 1., WIRE_Y(0)}}},
        {id_W13,
         {{PIP_X(id_W130), WIRE_Y(0), PIP_X(id_W130), WIRE_Y(4)},
          {PIP_X(id_W130), WIRE_Y(4), PIP_X(id_W131) - 1., WIRE_Y(4)},
          {PIP_X(id_W131) - 1., WIRE_Y(4), PIP_X(id_W131) - 1., WIRE_Y(0)}}},
        {id_E13_loop0,
         {{PIP_X(id_E130), WIRE_Y(0), PIP_X(id_E130), WIRE_Y(3)},
          {PIP_X(id_E130), WIRE_Y(3), 1. + wrap_len, WIRE_Y(3)},
          {1. + wrap_len, WIRE_Y(3), 1. + wrap_len, WIRE_Y(4)},
          {1. + wrap_len, WIRE_Y(4), PIP_X(id_W131), WIRE_Y(4)},
          {PIP_X(id_W131), WIRE_Y(4), PIP_X(id_W131), WIRE_Y(0)}}},
        {id_W13_loop0,
         {{PIP_X(id_W130), WIRE_Y(0), PIP_X(id_W130), WIRE_Y(4)},
          {PIP_X(id_W130), WIRE_Y(4), -1. * wrap_len, WIRE_Y(4)},
          {-1. * wrap_len, WIRE_Y(4), -1. * wrap_len, WIRE_Y(3)},
          {-1. * wrap_len, WIRE_Y(3), PIP_X(id_E131), WIRE_Y(3)},
          {PIP_X(id_E131), WIRE_Y(3), PIP_X(id_E131), WIRE_Y(0)}}},
        // 1 hop EW
        {id_EW10,
         {{PIP_X(id_EW10), WIRE_Y(0), PIP_X(id_EW10), WIRE_Y(6)},
          {PIP_X(id_EW10), WIRE_Y(6), PIP_X(id_E111) + 1., WIRE_Y(6)},
          {PIP_X(id_E111) + 1., WIRE_Y(6), PIP_X(id_E111) + 1., WIRE_Y(0)},
          {PIP_X(id_EW10), WIRE_Y(5), PIP_X(id_W111) - 1., WIRE_Y(5)},
          {PIP_X(id_W111) - 1., WIRE_Y(5), PIP_X(id_W111) - 1., WIRE_Y(0)}}},
        {id_EW10_loop_e,
         {{PIP_X(id_EW10), WIRE_Y(0), PIP_X(id_EW10), WIRE_Y(6)},
          {PIP_X(id_EW10), WIRE_Y(6), wrap_len + 1., WIRE_Y(6)},
          {wrap_len + 1., WIRE_Y(6), wrap_len + 1., WIRE_Y(5)},
          {wrap_len + 1., WIRE_Y(5), PIP_X(id_W111), WIRE_Y(5)},
          {PIP_X(id_W111), WIRE_Y(5), PIP_X(id_W111), WIRE_Y(0)},
          {PIP_X(id_EW10), WIRE_Y(5), PIP_X(id_W111) - 1., WIRE_Y(5)},
          {PIP_X(id_W111) - 1., WIRE_Y(5), PIP_X(id_W111) - 1., WIRE_Y(0)}}},
        {id_EW10_loop_w,
         {{PIP_X(id_EW10), WIRE_Y(0), PIP_X(id_EW10), WIRE_Y(6)},
          {PIP_X(id_EW10), WIRE_Y(6), PIP_X(id_E111) + 1., WIRE_Y(6)},
          {PIP_X(id_E111) + 1., WIRE_Y(6), PIP_X(id_E111) + 1., WIRE_Y(0)},
          {PIP_X(id_EW10), WIRE_Y(5), -wrap_len, WIRE_Y(5)},
          {-wrap_len, WIRE_Y(5), -wrap_len, WIRE_Y(6)},
          {-wrap_len, WIRE_Y(6), PIP_X(id_E111), WIRE_Y(6)},
          {PIP_X(id_E111), WIRE_Y(6), PIP_X(id_E111), WIRE_Y(0)}}},
        {id_EW20,
         {{PIP_X(id_EW20), WIRE_Y(0), PIP_X(id_EW20), WIRE_Y(8)},
          {PIP_X(id_EW20), WIRE_Y(8), PIP_X(id_E121) + 1., WIRE_Y(8)},
          {PIP_X(id_E121) + 1., WIRE_Y(8), PIP_X(id_E121) + 1., WIRE_Y(0)},
          {PIP_X(id_EW20), WIRE_Y(7), PIP_X(id_W121) - 1., WIRE_Y(7)},
          {PIP_X(id_W121) - 1., WIRE_Y(7), PIP_X(id_W121) - 1., WIRE_Y(0)}}},
        {id_EW20_loop_e,
         {{PIP_X(id_EW20), WIRE_Y(0), PIP_X(id_EW20), WIRE_Y(8)},
          {PIP_X(id_EW20), WIRE_Y(8), wrap_len + 1., WIRE_Y(8)},
          {wrap_len + 1., WIRE_Y(8), wrap_len + 1., WIRE_Y(7)},
          {wrap_len + 1., WIRE_Y(7), PIP_X(id_W121), WIRE_Y(7)},
          {PIP_X(id_W121), WIRE_Y(7), PIP_X(id_W121), WIRE_Y(0)},
          {PIP_X(id_EW20), WIRE_Y(7), PIP_X(id_W121) - 1., WIRE_Y(7)},
          {PIP_X(id_W121) - 1., WIRE_Y(7), PIP_X(id_W121) - 1., WIRE_Y(0)}}},
        {id_EW20_loop_w,
         {{PIP_X(id_EW20), WIRE_Y(0), PIP_X(id_EW20), WIRE_Y(8)},
          {PIP_X(id_EW20), WIRE_Y(8), PIP_X(id_E121) + 1., WIRE_Y(8)},
          {PIP_X(id_E121) + 1., WIRE_Y(8), PIP_X(id_E121) + 1., WIRE_Y(0)},
          {PIP_X(id_EW20), WIRE_Y(7), -wrap_len, WIRE_Y(7)},
          {-wrap_len, WIRE_Y(7), -wrap_len, WIRE_Y(8)},
          {-wrap_len, WIRE_Y(8), PIP_X(id_E121), WIRE_Y(8)},
          {PIP_X(id_E121), WIRE_Y(8), PIP_X(id_E121), WIRE_Y(0)}}},
// 2 hop
#define HOP2Y(offset) WIRE_Y(offset + 9)
        {id_E20,
         {{PIP_X(id_E200), WIRE_Y(0), PIP_X(id_E200), HOP2Y(2)},
          {PIP_X(id_E200), HOP2Y(2), PIP_X(id_E201) + 1., HOP2Y(2)},
          {PIP_X(id_E201) + 1., HOP2Y(2), PIP_X(id_E201) + 1., WIRE_Y(0)},
          {PIP_X(id_E201) + 1., HOP2Y(0), PIP_X(id_E202) + 2., HOP2Y(0)},
          {PIP_X(id_E202) + 2., HOP2Y(0), PIP_X(id_E202) + 2., WIRE_Y(0)}}},
        {id_W20,
         {{PIP_X(id_W200), WIRE_Y(0), PIP_X(id_W200), HOP2Y(3)},
          {PIP_X(id_W200), HOP2Y(3), PIP_X(id_W201) - 1., HOP2Y(3)},
          {PIP_X(id_W201) - 1., HOP2Y(3), PIP_X(id_W201) - 1., WIRE_Y(0)},
          {PIP_X(id_W201) - 1., HOP2Y(1), PIP_X(id_W202) - 2., HOP2Y(1)},
          {PIP_X(id_W202) - 2., HOP2Y(1), PIP_X(id_W202) - 2., WIRE_Y(0)}}},
        {id_E20_loop0,
         {{PIP_X(id_E200), WIRE_Y(0), PIP_X(id_E200), HOP2Y(2)},
          {PIP_X(id_E200), HOP2Y(2), wrap_len + 1., HOP2Y(2)},
          {wrap_len + 1., HOP2Y(2), wrap_len + 1., HOP2Y(3)},
          {wrap_len + 1., HOP2Y(3), PIP_X(id_W201), HOP2Y(3)},
          {PIP_X(id_W201), HOP2Y(3), PIP_X(id_W201), WIRE_Y(0)},
          {PIP_X(id_W201), HOP2Y(1), PIP_X(id_W202) - 1., HOP2Y(1)},
          {PIP_X(id_W202) - 1., HOP2Y(1), PIP_X(id_W202) - 1., WIRE_Y(0)}}},
        {id_W20_loop0,
         {{PIP_X(id_W200), WIRE_Y(0), PIP_X(id_W200), HOP2Y(3)},
          {PIP_X(id_W200), HOP2Y(3), -wrap_len, HOP2Y(3)},
          {-wrap_len, HOP2Y(3), -wrap_len, HOP2Y(2)},
          {-wrap_len, HOP2Y(2), PIP_X(id_E201), HOP2Y(2)},
          {PIP_X(id_E201), HOP2Y(2), PIP_X(id_E201), WIRE_Y(0)},
          {PIP_X(id_E201), HOP2Y(0), PIP_X(id_E202) + 1., HOP2Y(0)},
          {PIP_X(id_E202) + 1., HOP2Y(0), PIP_X(id_E202) + 1., WIRE_Y(0)}}},
        {id_E20_loop1,
         {{PIP_X(id_E200), WIRE_Y(0), PIP_X(id_E200), HOP2Y(2)},
          {PIP_X(id_E200), HOP2Y(2), PIP_X(id_E201) + 1., HOP2Y(2)},
          {PIP_X(id_E201) + 1., HOP2Y(2), PIP_X(id_E201) + 1., WIRE_Y(0)},
          {PIP_X(id_E201) + 1., HOP2Y(0), wrap_len + 2., HOP2Y(0)},
          {wrap_len + 2., HOP2Y(0), wrap_len + 2., HOP2Y(1)},
          {wrap_len + 2., HOP2Y(1), PIP_X(id_W202) + 1., HOP2Y(1)},
          {PIP_X(id_W202) + 1., HOP2Y(1), PIP_X(id_W202) + 1., WIRE_Y(0)}}},
        {id_W20_loop1,
         {{PIP_X(id_W200), WIRE_Y(0), PIP_X(id_W200), HOP2Y(3)},
          {PIP_X(id_W200), HOP2Y(3), PIP_X(id_W201) - 1., HOP2Y(3)},
          {PIP_X(id_W201) - 1., HOP2Y(3), PIP_X(id_W201) - 1., WIRE_Y(0)},
          {PIP_X(id_W201) - 1., HOP2Y(1), -wrap_len - 1., HOP2Y(1)},
          {-wrap_len - 1., HOP2Y(1), -wrap_len - 1., HOP2Y(0)},
          {-wrap_len - 1., HOP2Y(0), PIP_X(id_E202) - 1., HOP2Y(0)},
          {PIP_X(id_E202) - 1., HOP2Y(0), PIP_X(id_E202) - 1., WIRE_Y(0)}}},

#undef HOP2Y
#define HOP2Y(offset) WIRE_Y(offset + 9 + 4 * 1)
        {id_E21,
         {{PIP_X(id_E210), WIRE_Y(0), PIP_X(id_E210), HOP2Y(2)},
          {PIP_X(id_E210), HOP2Y(2), PIP_X(id_E211) + 1., HOP2Y(2)},
          {PIP_X(id_E211) + 1., HOP2Y(2), PIP_X(id_E211) + 1., WIRE_Y(0)},
          {PIP_X(id_E211) + 1., HOP2Y(0), PIP_X(id_E212) + 2., HOP2Y(0)},
          {PIP_X(id_E212) + 2., HOP2Y(0), PIP_X(id_E212) + 2., WIRE_Y(0)}}},
        {id_W21,
         {{PIP_X(id_W210), WIRE_Y(0), PIP_X(id_W210), HOP2Y(3)},
          {PIP_X(id_W210), HOP2Y(3), PIP_X(id_W211) - 1., HOP2Y(3)},
          {PIP_X(id_W211) - 1., HOP2Y(3), PIP_X(id_W211) - 1., WIRE_Y(0)},
          {PIP_X(id_W211) - 1., HOP2Y(1), PIP_X(id_W212) - 2., HOP2Y(1)},
          {PIP_X(id_W212) - 2., HOP2Y(1), PIP_X(id_W212) - 2., WIRE_Y(0)}}},
        {id_E21_loop0,
         {{PIP_X(id_E210), WIRE_Y(0), PIP_X(id_E210), HOP2Y(2)},
          {PIP_X(id_E210), HOP2Y(2), wrap_len + 1., HOP2Y(2)},
          {wrap_len + 1., HOP2Y(2), wrap_len + 1., HOP2Y(3)},
          {wrap_len + 1., HOP2Y(3), PIP_X(id_W211), HOP2Y(3)},
          {PIP_X(id_W211), HOP2Y(3), PIP_X(id_W211), WIRE_Y(0)},
          {PIP_X(id_W211), HOP2Y(1), PIP_X(id_W212) - 1., HOP2Y(1)},
          {PIP_X(id_W212) - 1., HOP2Y(1), PIP_X(id_W212) - 1., WIRE_Y(0)}}},
        {id_W21_loop0,
         {{PIP_X(id_W210), WIRE_Y(0), PIP_X(id_W210), HOP2Y(3)},
          {PIP_X(id_W210), HOP2Y(3), -wrap_len, HOP2Y(3)},
          {-wrap_len, HOP2Y(3), -wrap_len, HOP2Y(2)},
          {-wrap_len, HOP2Y(2), PIP_X(id_E211), HOP2Y(2)},
          {PIP_X(id_E211), HOP2Y(2), PIP_X(id_E211), WIRE_Y(0)},
          {PIP_X(id_E211), HOP2Y(0), PIP_X(id_E212) + 1., HOP2Y(0)},
          {PIP_X(id_E212) + 1., HOP2Y(0), PIP_X(id_E212) + 1., WIRE_Y(0)}}},
        {id_E21_loop1,
         {{PIP_X(id_E210), WIRE_Y(0), PIP_X(id_E210), HOP2Y(2)},
          {PIP_X(id_E210), HOP2Y(2), PIP_X(id_E211) + 1., HOP2Y(2)},
          {PIP_X(id_E211) + 1., HOP2Y(2), PIP_X(id_E211) + 1., WIRE_Y(0)},
          {PIP_X(id_E211) + 1., HOP2Y(0), wrap_len + 2., HOP2Y(0)},
          {wrap_len + 2., HOP2Y(0), wrap_len + 2., HOP2Y(1)},
          {wrap_len + 2., HOP2Y(1), PIP_X(id_W212) + 1., HOP2Y(1)},
          {PIP_X(id_W212) + 1., HOP2Y(1), PIP_X(id_W212) + 1., WIRE_Y(0)}}},
        {id_W21_loop1,
         {{PIP_X(id_W210), WIRE_Y(0), PIP_X(id_W210), HOP2Y(3)},
          {PIP_X(id_W210), HOP2Y(3), PIP_X(id_W211) - 1., HOP2Y(3)},
          {PIP_X(id_W211) - 1., HOP2Y(3), PIP_X(id_W211) - 1., WIRE_Y(0)},
          {PIP_X(id_W211) - 1., HOP2Y(1), -wrap_len - 1., HOP2Y(1)},
          {-wrap_len - 1., HOP2Y(1), -wrap_len - 1., HOP2Y(0)},
          {-wrap_len - 1., HOP2Y(0), PIP_X(id_E212) - 1., HOP2Y(0)},
          {PIP_X(id_E212) - 1., HOP2Y(0), PIP_X(id_E212) - 1., WIRE_Y(0)}}},

#undef HOP2Y
#define HOP2Y(offset) WIRE_Y(offset + 9 + 4 * 2)
        {id_E22,
         {{PIP_X(id_E220), WIRE_Y(0), PIP_X(id_E220), HOP2Y(2)},
          {PIP_X(id_E220), HOP2Y(2), PIP_X(id_E221) + 1., HOP2Y(2)},
          {PIP_X(id_E221) + 1., HOP2Y(2), PIP_X(id_E221) + 1., WIRE_Y(0)},
          {PIP_X(id_E221) + 1., HOP2Y(0), PIP_X(id_E222) + 2., HOP2Y(0)},
          {PIP_X(id_E222) + 2., HOP2Y(0), PIP_X(id_E222) + 2., WIRE_Y(0)}}},
        {id_W22,
         {{PIP_X(id_W220), WIRE_Y(0), PIP_X(id_W220), HOP2Y(3)},
          {PIP_X(id_W220), HOP2Y(3), PIP_X(id_W221) - 1., HOP2Y(3)},
          {PIP_X(id_W221) - 1., HOP2Y(3), PIP_X(id_W221) - 1., WIRE_Y(0)},
          {PIP_X(id_W221) - 1., HOP2Y(1), PIP_X(id_W222) - 2., HOP2Y(1)},
          {PIP_X(id_W222) - 2., HOP2Y(1), PIP_X(id_W222) - 2., WIRE_Y(0)}}},
        {id_E22_loop0,
         {{PIP_X(id_E220), WIRE_Y(0), PIP_X(id_E220), HOP2Y(2)},
          {PIP_X(id_E220), HOP2Y(2), wrap_len + 1., HOP2Y(2)},
          {wrap_len + 1., HOP2Y(2), wrap_len + 1., HOP2Y(3)},
          {wrap_len + 1., HOP2Y(3), PIP_X(id_W221), HOP2Y(3)},
          {PIP_X(id_W221), HOP2Y(3), PIP_X(id_W221), WIRE_Y(0)},
          {PIP_X(id_W221), HOP2Y(1), PIP_X(id_W222) - 1., HOP2Y(1)},
          {PIP_X(id_W222) - 1., HOP2Y(1), PIP_X(id_W222) - 1., WIRE_Y(0)}}},
        {id_W22_loop0,
         {{PIP_X(id_W220), WIRE_Y(0), PIP_X(id_W220), HOP2Y(3)},
          {PIP_X(id_W220), HOP2Y(3), -wrap_len, HOP2Y(3)},
          {-wrap_len, HOP2Y(3), -wrap_len, HOP2Y(2)},
          {-wrap_len, HOP2Y(2), PIP_X(id_E221), HOP2Y(2)},
          {PIP_X(id_E221), HOP2Y(2), PIP_X(id_E221), WIRE_Y(0)},
          {PIP_X(id_E221), HOP2Y(0), PIP_X(id_E222) + 1., HOP2Y(0)},
          {PIP_X(id_E222) + 1., HOP2Y(0), PIP_X(id_E222) + 1., WIRE_Y(0)}}},
        {id_E22_loop1,
         {{PIP_X(id_E220), WIRE_Y(0), PIP_X(id_E220), HOP2Y(2)},
          {PIP_X(id_E220), HOP2Y(2), PIP_X(id_E221) + 1., HOP2Y(2)},
          {PIP_X(id_E221) + 1., HOP2Y(2), PIP_X(id_E221) + 1., WIRE_Y(0)},
          {PIP_X(id_E221) + 1., HOP2Y(0), wrap_len + 2., HOP2Y(0)},
          {wrap_len + 2., HOP2Y(0), wrap_len + 2., HOP2Y(1)},
          {wrap_len + 2., HOP2Y(1), PIP_X(id_W222) + 1., HOP2Y(1)},
          {PIP_X(id_W222) + 1., HOP2Y(1), PIP_X(id_W222) + 1., WIRE_Y(0)}}},
        {id_W22_loop1,
         {{PIP_X(id_W220), WIRE_Y(0), PIP_X(id_W220), HOP2Y(3)},
          {PIP_X(id_W220), HOP2Y(3), PIP_X(id_W221) - 1., HOP2Y(3)},
          {PIP_X(id_W221) - 1., HOP2Y(3), PIP_X(id_W221) - 1., WIRE_Y(0)},
          {PIP_X(id_W221) - 1., HOP2Y(1), -wrap_len - 1., HOP2Y(1)},
          {-wrap_len - 1., HOP2Y(1), -wrap_len - 1., HOP2Y(0)},
          {-wrap_len - 1., HOP2Y(0), PIP_X(id_E222) - 1., HOP2Y(0)},
          {PIP_X(id_E222) - 1., HOP2Y(0), PIP_X(id_E222) - 1., WIRE_Y(0)}}},

#undef HOP2Y
#define HOP2Y(offset) WIRE_Y(offset + 9 + 4 * 3)
        {id_E23,
         {{PIP_X(id_E230), WIRE_Y(0), PIP_X(id_E230), HOP2Y(2)},
          {PIP_X(id_E230), HOP2Y(2), PIP_X(id_E231) + 1., HOP2Y(2)},
          {PIP_X(id_E231) + 1., HOP2Y(2), PIP_X(id_E231) + 1., WIRE_Y(0)},
          {PIP_X(id_E231) + 1., HOP2Y(0), PIP_X(id_E232) + 2., HOP2Y(0)},
          {PIP_X(id_E232) + 2., HOP2Y(0), PIP_X(id_E232) + 2., WIRE_Y(0)}}},
        {id_W23,
         {{PIP_X(id_W230), WIRE_Y(0), PIP_X(id_W230), HOP2Y(3)},
          {PIP_X(id_W230), HOP2Y(3), PIP_X(id_W231) - 1., HOP2Y(3)},
          {PIP_X(id_W231) - 1., HOP2Y(3), PIP_X(id_W231) - 1., WIRE_Y(0)},
          {PIP_X(id_W231) - 1., HOP2Y(1), PIP_X(id_W232) - 2., HOP2Y(1)},
          {PIP_X(id_W232) - 2., HOP2Y(1), PIP_X(id_W232) - 2., WIRE_Y(0)}}},
        {id_E23_loop0,
         {{PIP_X(id_E230), WIRE_Y(0), PIP_X(id_E230), HOP2Y(2)},
          {PIP_X(id_E230), HOP2Y(2), wrap_len + 1., HOP2Y(2)},
          {wrap_len + 1., HOP2Y(2), wrap_len + 1., HOP2Y(3)},
          {wrap_len + 1., HOP2Y(3), PIP_X(id_W231), HOP2Y(3)},
          {PIP_X(id_W231), HOP2Y(3), PIP_X(id_W231), WIRE_Y(0)},
          {PIP_X(id_W231), HOP2Y(1), PIP_X(id_W232) - 1., HOP2Y(1)},
          {PIP_X(id_W232) - 1., HOP2Y(1), PIP_X(id_W232) - 1., WIRE_Y(0)}}},
        {id_W23_loop0,
         {{PIP_X(id_W230), WIRE_Y(0), PIP_X(id_W230), HOP2Y(3)},
          {PIP_X(id_W230), HOP2Y(3), -wrap_len, HOP2Y(3)},
          {-wrap_len, HOP2Y(3), -wrap_len, HOP2Y(2)},
          {-wrap_len, HOP2Y(2), PIP_X(id_E231), HOP2Y(2)},
          {PIP_X(id_E231), HOP2Y(2), PIP_X(id_E231), WIRE_Y(0)},
          {PIP_X(id_E231), HOP2Y(0), PIP_X(id_E232) + 1., HOP2Y(0)},
          {PIP_X(id_E232) + 1., HOP2Y(0), PIP_X(id_E232) + 1., WIRE_Y(0)}}},
        {id_E23_loop1,
         {{PIP_X(id_E230), WIRE_Y(0), PIP_X(id_E230), HOP2Y(2)},
          {PIP_X(id_E230), HOP2Y(2), PIP_X(id_E231) + 1., HOP2Y(2)},
          {PIP_X(id_E231) + 1., HOP2Y(2), PIP_X(id_E231) + 1., WIRE_Y(0)},
          {PIP_X(id_E231) + 1., HOP2Y(0), wrap_len + 2., HOP2Y(0)},
          {wrap_len + 2., HOP2Y(0), wrap_len + 2., HOP2Y(1)},
          {wrap_len + 2., HOP2Y(1), PIP_X(id_W232) + 1., HOP2Y(1)},
          {PIP_X(id_W232) + 1., HOP2Y(1), PIP_X(id_W232) + 1., WIRE_Y(0)}}},
        {id_W23_loop1,
         {{PIP_X(id_W230), WIRE_Y(0), PIP_X(id_W230), HOP2Y(3)},
          {PIP_X(id_W230), HOP2Y(3), PIP_X(id_W231) - 1., HOP2Y(3)},
          {PIP_X(id_W231) - 1., HOP2Y(3), PIP_X(id_W231) - 1., WIRE_Y(0)},
          {PIP_X(id_W231) - 1., HOP2Y(1), -wrap_len - 1., HOP2Y(1)},
          {-wrap_len - 1., HOP2Y(1), -wrap_len - 1., HOP2Y(0)},
          {-wrap_len - 1., HOP2Y(0), PIP_X(id_E232) - 1., HOP2Y(0)},
          {PIP_X(id_E232) - 1., HOP2Y(0), PIP_X(id_E232) - 1., WIRE_Y(0)}}},

#undef HOP2Y
#define HOP2Y(offset) WIRE_Y(offset + 9 + 4 * 4)
        {id_E24,
         {{PIP_X(id_E240), WIRE_Y(0), PIP_X(id_E240), HOP2Y(2)},
          {PIP_X(id_E240), HOP2Y(2), PIP_X(id_E241) + 1., HOP2Y(2)},
          {PIP_X(id_E241) + 1., HOP2Y(2), PIP_X(id_E241) + 1., WIRE_Y(0)},
          {PIP_X(id_E241) + 1., HOP2Y(0), PIP_X(id_E242) + 2., HOP2Y(0)},
          {PIP_X(id_E242) + 2., HOP2Y(0), PIP_X(id_E242) + 2., WIRE_Y(0)}}},
        {id_W24,
         {{PIP_X(id_W240), WIRE_Y(0), PIP_X(id_W240), HOP2Y(3)},
          {PIP_X(id_W240), HOP2Y(3), PIP_X(id_W241) - 1., HOP2Y(3)},
          {PIP_X(id_W241) - 1., HOP2Y(3), PIP_X(id_W241) - 1., WIRE_Y(0)},
          {PIP_X(id_W241) - 1., HOP2Y(1), PIP_X(id_W242) - 2., HOP2Y(1)},
          {PIP_X(id_W242) - 2., HOP2Y(1), PIP_X(id_W242) - 2., WIRE_Y(0)}}},
        {id_E24_loop0,
         {{PIP_X(id_E240), WIRE_Y(0), PIP_X(id_E240), HOP2Y(2)},
          {PIP_X(id_E240), HOP2Y(2), wrap_len + 1., HOP2Y(2)},
          {wrap_len + 1., HOP2Y(2), wrap_len + 1., HOP2Y(3)},
          {wrap_len + 1., HOP2Y(3), PIP_X(id_W241), HOP2Y(3)},
          {PIP_X(id_W241), HOP2Y(3), PIP_X(id_W241), WIRE_Y(0)},
          {PIP_X(id_W241), HOP2Y(1), PIP_X(id_W242) - 1., HOP2Y(1)},
          {PIP_X(id_W242) - 1., HOP2Y(1), PIP_X(id_W242) - 1., WIRE_Y(0)}}},
        {id_W24_loop0,
         {{PIP_X(id_W240), WIRE_Y(0), PIP_X(id_W240), HOP2Y(3)},
          {PIP_X(id_W240), HOP2Y(3), -wrap_len, HOP2Y(3)},
          {-wrap_len, HOP2Y(3), -wrap_len, HOP2Y(2)},
          {-wrap_len, HOP2Y(2), PIP_X(id_E241), HOP2Y(2)},
          {PIP_X(id_E241), HOP2Y(2), PIP_X(id_E241), WIRE_Y(0)},
          {PIP_X(id_E241), HOP2Y(0), PIP_X(id_E242) + 1., HOP2Y(0)},
          {PIP_X(id_E242) + 1., HOP2Y(0), PIP_X(id_E242) + 1., WIRE_Y(0)}}},
        {id_E24_loop1,
         {{PIP_X(id_E240), WIRE_Y(0), PIP_X(id_E240), HOP2Y(2)},
          {PIP_X(id_E240), HOP2Y(2), PIP_X(id_E241) + 1., HOP2Y(2)},
          {PIP_X(id_E241) + 1., HOP2Y(2), PIP_X(id_E241) + 1., WIRE_Y(0)},
          {PIP_X(id_E241) + 1., HOP2Y(0), wrap_len + 2., HOP2Y(0)},
          {wrap_len + 2., HOP2Y(0), wrap_len + 2., HOP2Y(1)},
          {wrap_len + 2., HOP2Y(1), PIP_X(id_W242) + 1., HOP2Y(1)},
          {PIP_X(id_W242) + 1., HOP2Y(1), PIP_X(id_W242) + 1., WIRE_Y(0)}}},
        {id_W24_loop1,
         {{PIP_X(id_W240), WIRE_Y(0), PIP_X(id_W240), HOP2Y(3)},
          {PIP_X(id_W240), HOP2Y(3), PIP_X(id_W241) - 1., HOP2Y(3)},
          {PIP_X(id_W241) - 1., HOP2Y(3), PIP_X(id_W241) - 1., WIRE_Y(0)},
          {PIP_X(id_W241) - 1., HOP2Y(1), -wrap_len - 1., HOP2Y(1)},
          {-wrap_len - 1., HOP2Y(1), -wrap_len - 1., HOP2Y(0)},
          {-wrap_len - 1., HOP2Y(0), PIP_X(id_E242) - 1., HOP2Y(0)},
          {PIP_X(id_E242) - 1., HOP2Y(0), PIP_X(id_E242) - 1., WIRE_Y(0)}}},

#undef HOP2Y
#define HOP2Y(offset) WIRE_Y(offset + 9 + 4 * 5)
        {id_E25,
         {{PIP_X(id_E250), WIRE_Y(0), PIP_X(id_E250), HOP2Y(2)},
          {PIP_X(id_E250), HOP2Y(2), PIP_X(id_E251) + 1., HOP2Y(2)},
          {PIP_X(id_E251) + 1., HOP2Y(2), PIP_X(id_E251) + 1., WIRE_Y(0)},
          {PIP_X(id_E251) + 1., HOP2Y(0), PIP_X(id_E252) + 2., HOP2Y(0)},
          {PIP_X(id_E252) + 2., HOP2Y(0), PIP_X(id_E252) + 2., WIRE_Y(0)}}},
        {id_W25,
         {{PIP_X(id_W250), WIRE_Y(0), PIP_X(id_W250), HOP2Y(3)},
          {PIP_X(id_W250), HOP2Y(3), PIP_X(id_W251) - 1., HOP2Y(3)},
          {PIP_X(id_W251) - 1., HOP2Y(3), PIP_X(id_W251) - 1., WIRE_Y(0)},
          {PIP_X(id_W251) - 1., HOP2Y(1), PIP_X(id_W252) - 2., HOP2Y(1)},
          {PIP_X(id_W252) - 2., HOP2Y(1), PIP_X(id_W252) - 2., WIRE_Y(0)}}},
        {id_E25_loop0,
         {{PIP_X(id_E250), WIRE_Y(0), PIP_X(id_E250), HOP2Y(2)},
          {PIP_X(id_E250), HOP2Y(2), wrap_len + 1., HOP2Y(2)},
          {wrap_len + 1., HOP2Y(2), wrap_len + 1., HOP2Y(3)},
          {wrap_len + 1., HOP2Y(3), PIP_X(id_W251), HOP2Y(3)},
          {PIP_X(id_W251), HOP2Y(3), PIP_X(id_W251), WIRE_Y(0)},
          {PIP_X(id_W251), HOP2Y(1), PIP_X(id_W252) - 1., HOP2Y(1)},
          {PIP_X(id_W252) - 1., HOP2Y(1), PIP_X(id_W252) - 1., WIRE_Y(0)}}},
        {id_W25_loop0,
         {{PIP_X(id_W250), WIRE_Y(0), PIP_X(id_W250), HOP2Y(3)},
          {PIP_X(id_W250), HOP2Y(3), -wrap_len, HOP2Y(3)},
          {-wrap_len, HOP2Y(3), -wrap_len, HOP2Y(2)},
          {-wrap_len, HOP2Y(2), PIP_X(id_E251), HOP2Y(2)},
          {PIP_X(id_E251), HOP2Y(2), PIP_X(id_E251), WIRE_Y(0)},
          {PIP_X(id_E251), HOP2Y(0), PIP_X(id_E252) + 1., HOP2Y(0)},
          {PIP_X(id_E252) + 1., HOP2Y(0), PIP_X(id_E252) + 1., WIRE_Y(0)}}},
        {id_E25_loop1,
         {{PIP_X(id_E250), WIRE_Y(0), PIP_X(id_E250), HOP2Y(2)},
          {PIP_X(id_E250), HOP2Y(2), PIP_X(id_E251) + 1., HOP2Y(2)},
          {PIP_X(id_E251) + 1., HOP2Y(2), PIP_X(id_E251) + 1., WIRE_Y(0)},
          {PIP_X(id_E251) + 1., HOP2Y(0), wrap_len + 2., HOP2Y(0)},
          {wrap_len + 2., HOP2Y(0), wrap_len + 2., HOP2Y(1)},
          {wrap_len + 2., HOP2Y(1), PIP_X(id_W252) + 1., HOP2Y(1)},
          {PIP_X(id_W252) + 1., HOP2Y(1), PIP_X(id_W252) + 1., WIRE_Y(0)}}},
        {id_W25_loop1,
         {{PIP_X(id_W250), WIRE_Y(0), PIP_X(id_W250), HOP2Y(3)},
          {PIP_X(id_W250), HOP2Y(3), PIP_X(id_W251) - 1., HOP2Y(3)},
          {PIP_X(id_W251) - 1., HOP2Y(3), PIP_X(id_W251) - 1., WIRE_Y(0)},
          {PIP_X(id_W251) - 1., HOP2Y(1), -wrap_len - 1., HOP2Y(1)},
          {-wrap_len - 1., HOP2Y(1), -wrap_len - 1., HOP2Y(0)},
          {-wrap_len - 1., HOP2Y(0), PIP_X(id_E252) - 1., HOP2Y(0)},
          {PIP_X(id_E252) - 1., HOP2Y(0), PIP_X(id_E252) - 1., WIRE_Y(0)}}},

#undef HOP2Y
#define HOP2Y(offset) WIRE_Y(offset + 9 + 4 * 6)
        {id_E26,
         {{PIP_X(id_E260), WIRE_Y(0), PIP_X(id_E260), HOP2Y(2)},
          {PIP_X(id_E260), HOP2Y(2), PIP_X(id_E261) + 1., HOP2Y(2)},
          {PIP_X(id_E261) + 1., HOP2Y(2), PIP_X(id_E261) + 1., WIRE_Y(0)},
          {PIP_X(id_E261) + 1., HOP2Y(0), PIP_X(id_E262) + 2., HOP2Y(0)},
          {PIP_X(id_E262) + 2., HOP2Y(0), PIP_X(id_E262) + 2., WIRE_Y(0)}}},
        {id_W26,
         {{PIP_X(id_W260), WIRE_Y(0), PIP_X(id_W260), HOP2Y(3)},
          {PIP_X(id_W260), HOP2Y(3), PIP_X(id_W261) - 1., HOP2Y(3)},
          {PIP_X(id_W261) - 1., HOP2Y(3), PIP_X(id_W261) - 1., WIRE_Y(0)},
          {PIP_X(id_W261) - 1., HOP2Y(1), PIP_X(id_W262) - 2., HOP2Y(1)},
          {PIP_X(id_W262) - 2., HOP2Y(1), PIP_X(id_W262) - 2., WIRE_Y(0)}}},
        {id_E26_loop0,
         {{PIP_X(id_E260), WIRE_Y(0), PIP_X(id_E260), HOP2Y(2)},
          {PIP_X(id_E260), HOP2Y(2), wrap_len + 1., HOP2Y(2)},
          {wrap_len + 1., HOP2Y(2), wrap_len + 1., HOP2Y(3)},
          {wrap_len + 1., HOP2Y(3), PIP_X(id_W261), HOP2Y(3)},
          {PIP_X(id_W261), HOP2Y(3), PIP_X(id_W261), WIRE_Y(0)},
          {PIP_X(id_W261), HOP2Y(1), PIP_X(id_W262) - 1., HOP2Y(1)},
          {PIP_X(id_W262) - 1., HOP2Y(1), PIP_X(id_W262) - 1., WIRE_Y(0)}}},
        {id_W26_loop0,
         {{PIP_X(id_W260), WIRE_Y(0), PIP_X(id_W260), HOP2Y(3)},
          {PIP_X(id_W260), HOP2Y(3), -wrap_len, HOP2Y(3)},
          {-wrap_len, HOP2Y(3), -wrap_len, HOP2Y(2)},
          {-wrap_len, HOP2Y(2), PIP_X(id_E261), HOP2Y(2)},
          {PIP_X(id_E261), HOP2Y(2), PIP_X(id_E261), WIRE_Y(0)},
          {PIP_X(id_E261), HOP2Y(0), PIP_X(id_E262) + 1., HOP2Y(0)},
          {PIP_X(id_E262) + 1., HOP2Y(0), PIP_X(id_E262) + 1., WIRE_Y(0)}}},
        {id_E26_loop1,
         {{PIP_X(id_E260), WIRE_Y(0), PIP_X(id_E260), HOP2Y(2)},
          {PIP_X(id_E260), HOP2Y(2), PIP_X(id_E261) + 1., HOP2Y(2)},
          {PIP_X(id_E261) + 1., HOP2Y(2), PIP_X(id_E261) + 1., WIRE_Y(0)},
          {PIP_X(id_E261) + 1., HOP2Y(0), wrap_len + 2., HOP2Y(0)},
          {wrap_len + 2., HOP2Y(0), wrap_len + 2., HOP2Y(1)},
          {wrap_len + 2., HOP2Y(1), PIP_X(id_W262) + 1., HOP2Y(1)},
          {PIP_X(id_W262) + 1., HOP2Y(1), PIP_X(id_W262) + 1., WIRE_Y(0)}}},
        {id_W26_loop1,
         {{PIP_X(id_W260), WIRE_Y(0), PIP_X(id_W260), HOP2Y(3)},
          {PIP_X(id_W260), HOP2Y(3), PIP_X(id_W261) - 1., HOP2Y(3)},
          {PIP_X(id_W261) - 1., HOP2Y(3), PIP_X(id_W261) - 1., WIRE_Y(0)},
          {PIP_X(id_W261) - 1., HOP2Y(1), -wrap_len - 1., HOP2Y(1)},
          {-wrap_len - 1., HOP2Y(1), -wrap_len - 1., HOP2Y(0)},
          {-wrap_len - 1., HOP2Y(0), PIP_X(id_E262) - 1., HOP2Y(0)},
          {PIP_X(id_E262) - 1., HOP2Y(0), PIP_X(id_E262) - 1., WIRE_Y(0)}}},

#undef HOP2Y
#define HOP2Y(offset) WIRE_Y(offset + 9 + 4 * 7)
        {id_E27,
         {{PIP_X(id_E270), WIRE_Y(0), PIP_X(id_E270), HOP2Y(2)},
          {PIP_X(id_E270), HOP2Y(2), PIP_X(id_E271) + 1., HOP2Y(2)},
          {PIP_X(id_E271) + 1., HOP2Y(2), PIP_X(id_E271) + 1., WIRE_Y(0)},
          {PIP_X(id_E271) + 1., HOP2Y(0), PIP_X(id_E272) + 2., HOP2Y(0)},
          {PIP_X(id_E272) + 2., HOP2Y(0), PIP_X(id_E272) + 2., WIRE_Y(0)}}},
        {id_W27,
         {{PIP_X(id_W270), WIRE_Y(0), PIP_X(id_W270), HOP2Y(3)},
          {PIP_X(id_W270), HOP2Y(3), PIP_X(id_W271) - 1., HOP2Y(3)},
          {PIP_X(id_W271) - 1., HOP2Y(3), PIP_X(id_W271) - 1., WIRE_Y(0)},
          {PIP_X(id_W271) - 1., HOP2Y(1), PIP_X(id_W272) - 2., HOP2Y(1)},
          {PIP_X(id_W272) - 2., HOP2Y(1), PIP_X(id_W272) - 2., WIRE_Y(0)}}},
        {id_E27_loop0,
         {{PIP_X(id_E270), WIRE_Y(0), PIP_X(id_E270), HOP2Y(2)},
          {PIP_X(id_E270), HOP2Y(2), wrap_len + 1., HOP2Y(2)},
          {wrap_len + 1., HOP2Y(2), wrap_len + 1., HOP2Y(3)},
          {wrap_len + 1., HOP2Y(3), PIP_X(id_W271), HOP2Y(3)},
          {PIP_X(id_W271), HOP2Y(3), PIP_X(id_W271), WIRE_Y(0)},
          {PIP_X(id_W271), HOP2Y(1), PIP_X(id_W272) - 1., HOP2Y(1)},
          {PIP_X(id_W272) - 1., HOP2Y(1), PIP_X(id_W272) - 1., WIRE_Y(0)}}},
        {id_W27_loop0,
         {{PIP_X(id_W270), WIRE_Y(0), PIP_X(id_W270), HOP2Y(3)},
          {PIP_X(id_W270), HOP2Y(3), -wrap_len, HOP2Y(3)},
          {-wrap_len, HOP2Y(3), -wrap_len, HOP2Y(2)},
          {-wrap_len, HOP2Y(2), PIP_X(id_E271), HOP2Y(2)},
          {PIP_X(id_E271), HOP2Y(2), PIP_X(id_E271), WIRE_Y(0)},
          {PIP_X(id_E271), HOP2Y(0), PIP_X(id_E272) + 1., HOP2Y(0)},
          {PIP_X(id_E272) + 1., HOP2Y(0), PIP_X(id_E272) + 1., WIRE_Y(0)}}},
        {id_E27_loop1,
         {{PIP_X(id_E270), WIRE_Y(0), PIP_X(id_E270), HOP2Y(2)},
          {PIP_X(id_E270), HOP2Y(2), PIP_X(id_E271) + 1., HOP2Y(2)},
          {PIP_X(id_E271) + 1., HOP2Y(2), PIP_X(id_E271) + 1., WIRE_Y(0)},
          {PIP_X(id_E271) + 1., HOP2Y(0), wrap_len + 2., HOP2Y(0)},
          {wrap_len + 2., HOP2Y(0), wrap_len + 2., HOP2Y(1)},
          {wrap_len + 2., HOP2Y(1), PIP_X(id_W272) + 1., HOP2Y(1)},
          {PIP_X(id_W272) + 1., HOP2Y(1), PIP_X(id_W272) + 1., WIRE_Y(0)}}},
        {id_W27_loop1,
         {{PIP_X(id_W270), WIRE_Y(0), PIP_X(id_W270), HOP2Y(3)},
          {PIP_X(id_W270), HOP2Y(3), PIP_X(id_W271) - 1., HOP2Y(3)},
          {PIP_X(id_W271) - 1., HOP2Y(3), PIP_X(id_W271) - 1., WIRE_Y(0)},
          {PIP_X(id_W271) - 1., HOP2Y(1), -wrap_len - 1., HOP2Y(1)},
          {-wrap_len - 1., HOP2Y(1), -wrap_len - 1., HOP2Y(0)},
          {-wrap_len - 1., HOP2Y(0), PIP_X(id_E272) - 1., HOP2Y(0)},
          {PIP_X(id_E272) - 1., HOP2Y(0), PIP_X(id_E272) - 1., WIRE_Y(0)}}},

// clock branches
#define CLK_GBO0_Y 41.f
#define CLK_GBO1_Y 46.f
// 4 hop
#define HOP4Y_START (CLK_GBO0_Y + 10.f)
#define HOP4Y(offset) WIRE_Y((float)offset + HOP4Y_START)
        {id_E80,
         {{PIP_X(id_E800), WIRE_Y(0), PIP_X(id_E800), HOP4Y(16)},
          {PIP_X(id_E800), HOP4Y(16), PIP_X(id_W808), HOP4Y(16)},
          {PIP_X(id_W808) + 0., HOP4Y(16), PIP_X(id_W808) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W808) + top_wire_dist + 0., HOP4Y(14), PIP_X(id_W808) + 1., HOP4Y(14)},
          {PIP_X(id_W808) + 1., HOP4Y(14), PIP_X(id_W808) + top_wire_dist + 1., HOP4Y(12)},
          {PIP_X(id_W808) + top_wire_dist + 1., HOP4Y(12), PIP_X(id_W808) + 2., HOP4Y(12)},
          {PIP_X(id_W808) + 2., HOP4Y(12), PIP_X(id_W808) + top_wire_dist + 2., HOP4Y(10)},
          {PIP_X(id_W808) + top_wire_dist + 2., HOP4Y(10), PIP_X(id_W808) + 3., HOP4Y(10)},
          {PIP_X(id_W808) + 3., HOP4Y(10), PIP_X(id_W808) + top_wire_dist + 3., HOP4Y(8)},
          {PIP_X(id_W808) + top_wire_dist + 3., HOP4Y(8), PIP_X(id_W808) + 4., HOP4Y(8)},
          {PIP_X(id_W808) + 4., HOP4Y(8), PIP_X(id_W808) + top_wire_dist + 4., HOP4Y(6)},
          {PIP_X(id_E804) + 4., HOP4Y(8), PIP_X(id_E804) + 4., WIRE_Y(0)},
          {PIP_X(id_W808) + top_wire_dist + 4., HOP4Y(6), PIP_X(id_W808) + 5., HOP4Y(6)},
          {PIP_X(id_W808) + 5., HOP4Y(6), PIP_X(id_W808) + top_wire_dist + 5., HOP4Y(4)},
          {PIP_X(id_W808) + top_wire_dist + 5., HOP4Y(4), PIP_X(id_W808) + 6., HOP4Y(4)},
          {PIP_X(id_W808) + 6., HOP4Y(4), PIP_X(id_W808) + top_wire_dist + 6., HOP4Y(2)},
          {PIP_X(id_W808) + top_wire_dist + 6., HOP4Y(2), PIP_X(id_W808) + 7., HOP4Y(2)},
          {PIP_X(id_W808) + 7., HOP4Y(2), PIP_X(id_W808) + top_wire_dist + 7., HOP4Y(0)},
          {PIP_X(id_W808) + top_wire_dist + 7., HOP4Y(0), PIP_X(id_E808) + 8., HOP4Y(0)},
          {PIP_X(id_E808) + 8, HOP4Y(0), PIP_X(id_E808) + 8., WIRE_Y(0)}}},
        {id_W80,
         {{PIP_X(id_W800), WIRE_Y(0), PIP_X(id_W800), HOP4Y(17)},
          {PIP_X(id_W800) - 0., HOP4Y(17), PIP_X(id_W800) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W800) - top_wire_dist - 0., HOP4Y(15), PIP_X(id_W800) - 1., HOP4Y(15)},
          {PIP_X(id_W800) - 1., HOP4Y(15), PIP_X(id_W800) - top_wire_dist - 1., HOP4Y(13)},
          {PIP_X(id_W800) - top_wire_dist - 1., HOP4Y(13), PIP_X(id_W800) - 2., HOP4Y(13)},
          {PIP_X(id_W800) - 2., HOP4Y(13), PIP_X(id_W800) - top_wire_dist - 2., HOP4Y(11)},
          {PIP_X(id_W800) - top_wire_dist - 2., HOP4Y(11), PIP_X(id_W800) - 3., HOP4Y(11)},
          {PIP_X(id_W800) - 3., HOP4Y(11), PIP_X(id_W800) - top_wire_dist - 3., HOP4Y(9)},
          {PIP_X(id_W800) - top_wire_dist - 3., HOP4Y(9), PIP_X(id_W800) - 4., HOP4Y(9)},
          {PIP_X(id_W800) - 4., HOP4Y(9), PIP_X(id_W800) - top_wire_dist - 4., HOP4Y(7)},
          {PIP_X(id_W804) - 4., HOP4Y(9), PIP_X(id_W804) - 4., WIRE_Y(0)},
          {PIP_X(id_W800) - top_wire_dist - 4., HOP4Y(7), PIP_X(id_W800) - 5., HOP4Y(7)},
          {PIP_X(id_W800) - 5., HOP4Y(7), PIP_X(id_W800) - top_wire_dist - 5., HOP4Y(5)},
          {PIP_X(id_W800) - top_wire_dist - 5., HOP4Y(5), PIP_X(id_W800) - 6., HOP4Y(5)},
          {PIP_X(id_W800) - 6., HOP4Y(5), PIP_X(id_W800) - top_wire_dist - 6., HOP4Y(3)},
          {PIP_X(id_W800) - top_wire_dist - 6., HOP4Y(3), PIP_X(id_W800) - 7., HOP4Y(3)},
          {PIP_X(id_W800) - 7., HOP4Y(3), PIP_X(id_W800) - top_wire_dist - 7., HOP4Y(1)},
          {PIP_X(id_W800) - top_wire_dist - 7., HOP4Y(1), PIP_X(id_W808) - 8., HOP4Y(1)},
          {PIP_X(id_W808) - 8, HOP4Y(1), PIP_X(id_W808) - 8., WIRE_Y(0)}}},
        {id_E80_loop0,
         {{PIP_X(id_E800), WIRE_Y(0), PIP_X(id_E800), HOP4Y(16)},
          {PIP_X(id_E800), HOP4Y(16), PIP_X(id_W808), HOP4Y(16)},
          {PIP_X(id_W808) + 0., HOP4Y(16), PIP_X(id_W808) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W808) + top_wire_dist + 0., HOP4Y(14), wrap_len + 1., HOP4Y(14)},
          {wrap_len + 1., HOP4Y(14), wrap_len + 1., HOP4Y(15)},
          {wrap_len + 1., HOP4Y(15), PIP_X(id_W800) - 0., HOP4Y(15)},
          {PIP_X(id_W800) - 0., HOP4Y(15), PIP_X(id_W800) - top_wire_dist - 0., HOP4Y(13)},
          {PIP_X(id_W800) - top_wire_dist - 0., HOP4Y(13), PIP_X(id_W800) - 1., HOP4Y(13)},
          {PIP_X(id_W800) - 1., HOP4Y(13), PIP_X(id_W800) - top_wire_dist - 1., HOP4Y(11)},
          {PIP_X(id_W800) - top_wire_dist - 1., HOP4Y(11), PIP_X(id_W800) - 2., HOP4Y(11)},
          {PIP_X(id_W800) - 2., HOP4Y(11), PIP_X(id_W800) - top_wire_dist - 2., HOP4Y(9)},
          {PIP_X(id_W800) - top_wire_dist - 2., HOP4Y(9), PIP_X(id_W800) - 3., HOP4Y(9)},
          {PIP_X(id_W800) - 3., HOP4Y(9), PIP_X(id_W800) - top_wire_dist - 3., HOP4Y(7)},
          {PIP_X(id_W804) - 3., HOP4Y(9), PIP_X(id_W804) - 3., WIRE_Y(0)},
          {PIP_X(id_W800) - top_wire_dist - 3., HOP4Y(7), PIP_X(id_W800) - 4., HOP4Y(7)},
          {PIP_X(id_W800) - 4., HOP4Y(7), PIP_X(id_W800) - top_wire_dist - 4., HOP4Y(5)},
          {PIP_X(id_W800) - top_wire_dist - 4., HOP4Y(5), PIP_X(id_W800) - 5., HOP4Y(5)},
          {PIP_X(id_W800) - 5., HOP4Y(5), PIP_X(id_W800) - top_wire_dist - 5., HOP4Y(3)},
          {PIP_X(id_W800) - top_wire_dist - 5., HOP4Y(3), PIP_X(id_W800) - 6., HOP4Y(3)},
          {PIP_X(id_W800) - 6., HOP4Y(3), PIP_X(id_W800) - top_wire_dist - 6., HOP4Y(1)},
          {PIP_X(id_W800) - top_wire_dist - 6., HOP4Y(1), PIP_X(id_W808) - 7., HOP4Y(1)},
          {PIP_X(id_W808) - 7, HOP4Y(1), PIP_X(id_W808) - 7., WIRE_Y(0)}}},
        {id_E80_loop1,
         {{PIP_X(id_E800), WIRE_Y(0), PIP_X(id_E800), HOP4Y(16)},
          {PIP_X(id_E800), HOP4Y(16), PIP_X(id_W808), HOP4Y(16)},
          {PIP_X(id_W808) + 0., HOP4Y(16), PIP_X(id_W808) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W808) + top_wire_dist + 0., HOP4Y(14), PIP_X(id_W808) + 1., HOP4Y(14)},
          {PIP_X(id_W808) + 1., HOP4Y(14), PIP_X(id_W808) + top_wire_dist + 1., HOP4Y(12)},
          {PIP_X(id_W808) + top_wire_dist + 1., HOP4Y(12), wrap_len + 2., HOP4Y(12)},
          {wrap_len + 2., HOP4Y(12), wrap_len + 2., HOP4Y(13)},
          {wrap_len + 2., HOP4Y(13), PIP_X(id_W800) + 1., HOP4Y(13)},
          {PIP_X(id_W800) + 1., HOP4Y(13), PIP_X(id_W800) - top_wire_dist + 1., HOP4Y(11)},
          {PIP_X(id_W800) - top_wire_dist + 1., HOP4Y(11), PIP_X(id_W800) - 0., HOP4Y(11)},
          {PIP_X(id_W800) - 0., HOP4Y(11), PIP_X(id_W800) - top_wire_dist - 0., HOP4Y(9)},
          {PIP_X(id_W800) - top_wire_dist - 0., HOP4Y(9), PIP_X(id_W800) - 1., HOP4Y(9)},
          {PIP_X(id_W804) - 1., HOP4Y(9), PIP_X(id_W804) - 1., WIRE_Y(0)},
          {PIP_X(id_W800) - 1., HOP4Y(9), PIP_X(id_W800) - top_wire_dist - 1., HOP4Y(7)},
          {PIP_X(id_W800) - top_wire_dist - 1., HOP4Y(7), PIP_X(id_W800) - 2., HOP4Y(7)},
          {PIP_X(id_W800) - 2., HOP4Y(7), PIP_X(id_W800) - top_wire_dist - 2., HOP4Y(5)},
          {PIP_X(id_W800) - top_wire_dist - 2., HOP4Y(5), PIP_X(id_W800) - 3., HOP4Y(5)},
          {PIP_X(id_W800) - 3., HOP4Y(5), PIP_X(id_W800) - top_wire_dist - 3., HOP4Y(3)},
          {PIP_X(id_W800) - top_wire_dist - 3., HOP4Y(3), PIP_X(id_W800) - 4., HOP4Y(3)},
          {PIP_X(id_W800) - 4., HOP4Y(3), PIP_X(id_W800) - top_wire_dist - 4., HOP4Y(1)},
          {PIP_X(id_W800) - top_wire_dist - 4., HOP4Y(1), PIP_X(id_W808) - 5., HOP4Y(1)},
          {PIP_X(id_W808) - 5., HOP4Y(1), PIP_X(id_W808) - 5., WIRE_Y(0)}}},
        {id_E80_loop2,
         {{PIP_X(id_E800), WIRE_Y(0), PIP_X(id_E800), HOP4Y(16)},
          {PIP_X(id_E800), HOP4Y(16), PIP_X(id_W808), HOP4Y(16)},
          {PIP_X(id_W808) + 0., HOP4Y(16), PIP_X(id_W808) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W808) + top_wire_dist + 0., HOP4Y(14), PIP_X(id_W808) + 1., HOP4Y(14)},
          {PIP_X(id_W808) + 1., HOP4Y(14), PIP_X(id_W808) + top_wire_dist + 1., HOP4Y(12)},
          {PIP_X(id_W808) + top_wire_dist + 1., HOP4Y(12), PIP_X(id_W808) + 2., HOP4Y(12)},
          {PIP_X(id_W808) + 2., HOP4Y(12), PIP_X(id_W808) + top_wire_dist + 2., HOP4Y(10)},
          {PIP_X(id_W808) + top_wire_dist + 2., HOP4Y(10), wrap_len + 3., HOP4Y(10)},
          {wrap_len + 3., HOP4Y(10), wrap_len + 3., HOP4Y(11)},
          {wrap_len + 3., HOP4Y(11), PIP_X(id_W800) + 2., HOP4Y(11)},
          {PIP_X(id_W800) + 2., HOP4Y(11), PIP_X(id_W800) - top_wire_dist + 2., HOP4Y(9)},
          {PIP_X(id_W800) - top_wire_dist + 2., HOP4Y(9), PIP_X(id_W800) + 1., HOP4Y(9)},
          {PIP_X(id_W804) + 1., HOP4Y(9), PIP_X(id_W804) + 1., WIRE_Y(0)},
          {PIP_X(id_W800) + 1., HOP4Y(9), PIP_X(id_W800) - top_wire_dist + 1., HOP4Y(7)},
          {PIP_X(id_W800) - top_wire_dist + 1., HOP4Y(7), PIP_X(id_W800) + 0., HOP4Y(7)},
          {PIP_X(id_W800) + 0., HOP4Y(7), PIP_X(id_W800) - top_wire_dist - 0., HOP4Y(5)},
          {PIP_X(id_W800) - top_wire_dist - 0., HOP4Y(5), PIP_X(id_W800) - 1., HOP4Y(5)},
          {PIP_X(id_W800) - 1., HOP4Y(5), PIP_X(id_W800) - top_wire_dist - 1., HOP4Y(3)},
          {PIP_X(id_W800) - top_wire_dist - 1., HOP4Y(3), PIP_X(id_W800) - 2., HOP4Y(3)},
          {PIP_X(id_W800) - 2., HOP4Y(3), PIP_X(id_W800) - top_wire_dist - 2., HOP4Y(1)},
          {PIP_X(id_W800) - top_wire_dist - 2., HOP4Y(1), PIP_X(id_W808) - 3., HOP4Y(1)},
          {PIP_X(id_W808) - 3., HOP4Y(1), PIP_X(id_W808) - 3., WIRE_Y(0)}}},
        {id_E80_loop3,
         {{PIP_X(id_E800), WIRE_Y(0), PIP_X(id_E800), HOP4Y(16)},
          {PIP_X(id_E800), HOP4Y(16), PIP_X(id_W808), HOP4Y(16)},
          {PIP_X(id_W808) + 0., HOP4Y(16), PIP_X(id_W808) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W808) + top_wire_dist + 0., HOP4Y(14), PIP_X(id_W808) + 1., HOP4Y(14)},
          {PIP_X(id_W808) + 1., HOP4Y(14), PIP_X(id_W808) + top_wire_dist + 1., HOP4Y(12)},
          {PIP_X(id_W808) + top_wire_dist + 1., HOP4Y(12), PIP_X(id_W808) + 2., HOP4Y(12)},
          {PIP_X(id_W808) + 2., HOP4Y(12), PIP_X(id_W808) + top_wire_dist + 2., HOP4Y(10)},
          {PIP_X(id_W808) + top_wire_dist + 2., HOP4Y(10), PIP_X(id_W808) + 3., HOP4Y(10)},
          {PIP_X(id_W808) + 3., HOP4Y(10), PIP_X(id_W808) + top_wire_dist + 3., HOP4Y(8)},
          {PIP_X(id_W808) + top_wire_dist + 3., HOP4Y(8), wrap_len + 4., HOP4Y(8)},
          {wrap_len + 4., HOP4Y(8), wrap_len + 4., HOP4Y(9)},
          {wrap_len + 4., HOP4Y(9), PIP_X(id_W800) + 3., HOP4Y(9)},
          {PIP_X(id_W804) + 3., HOP4Y(9), PIP_X(id_W804) + 3., WIRE_Y(0)},
          {PIP_X(id_W800) + 3., HOP4Y(9), PIP_X(id_W800) - top_wire_dist + 3., HOP4Y(7)},
          {PIP_X(id_W800) - top_wire_dist + 3., HOP4Y(7), PIP_X(id_W800) + 2., HOP4Y(7)},
          {PIP_X(id_W800) + 2., HOP4Y(7), PIP_X(id_W800) - top_wire_dist + 2., HOP4Y(5)},
          {PIP_X(id_W800) - top_wire_dist + 2., HOP4Y(5), PIP_X(id_W800) + 1., HOP4Y(5)},
          {PIP_X(id_W800) + 1., HOP4Y(5), PIP_X(id_W800) - top_wire_dist + 1., HOP4Y(3)},
          {PIP_X(id_W800) - top_wire_dist + 1., HOP4Y(3), PIP_X(id_W800) - 0., HOP4Y(3)},
          {PIP_X(id_W800) - 0., HOP4Y(3), PIP_X(id_W800) - top_wire_dist - 0., HOP4Y(1)},
          {PIP_X(id_W800) - top_wire_dist - 0., HOP4Y(1), PIP_X(id_W808) - 1., HOP4Y(1)},
          {PIP_X(id_W808) - 1., HOP4Y(1), PIP_X(id_W808) - 1., WIRE_Y(0)}}},
        {id_E80_loop4,
         {{PIP_X(id_E800), WIRE_Y(0), PIP_X(id_E800), HOP4Y(16)},
          {PIP_X(id_E800), HOP4Y(16), PIP_X(id_W808), HOP4Y(16)},
          {PIP_X(id_W808) + 0., HOP4Y(16), PIP_X(id_W808) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W808) + top_wire_dist + 0., HOP4Y(14), PIP_X(id_W808) + 1., HOP4Y(14)},
          {PIP_X(id_W808) + 1., HOP4Y(14), PIP_X(id_W808) + top_wire_dist + 1., HOP4Y(12)},
          {PIP_X(id_W808) + top_wire_dist + 1., HOP4Y(12), PIP_X(id_W808) + 2., HOP4Y(12)},
          {PIP_X(id_W808) + 2., HOP4Y(12), PIP_X(id_W808) + top_wire_dist + 2., HOP4Y(10)},
          {PIP_X(id_W808) + top_wire_dist + 2., HOP4Y(10), PIP_X(id_W808) + 3., HOP4Y(10)},
          {PIP_X(id_W808) + 3., HOP4Y(10), PIP_X(id_W808) + top_wire_dist + 3., HOP4Y(8)},
          {PIP_X(id_W808) + top_wire_dist + 3., HOP4Y(8), PIP_X(id_W808) + 4., HOP4Y(8)},
          {PIP_X(id_E804) + 4., HOP4Y(8), PIP_X(id_E804) + 4., WIRE_Y(0)},
          {PIP_X(id_W808) + 4., HOP4Y(8), PIP_X(id_W808) + top_wire_dist + 4., HOP4Y(6)},
          {PIP_X(id_W808) + top_wire_dist + 4., HOP4Y(6), wrap_len + 5., HOP4Y(6)},
          {wrap_len + 5., HOP4Y(6), wrap_len + 5., HOP4Y(7)},
          {wrap_len + 5., HOP4Y(7), PIP_X(id_W800) + 4., HOP4Y(7)},
          {PIP_X(id_W800) + 4., HOP4Y(7), PIP_X(id_W800) - top_wire_dist + 4., HOP4Y(5)},
          {PIP_X(id_W800) - top_wire_dist + 4., HOP4Y(5), PIP_X(id_W800) + 3., HOP4Y(5)},
          {PIP_X(id_W800) + 3., HOP4Y(5), PIP_X(id_W800) - top_wire_dist + 3., HOP4Y(3)},
          {PIP_X(id_W800) - top_wire_dist + 3., HOP4Y(3), PIP_X(id_W800) + 2., HOP4Y(3)},
          {PIP_X(id_W800) + 2., HOP4Y(3), PIP_X(id_W800) - top_wire_dist + 2., HOP4Y(1)},
          {PIP_X(id_W800) - top_wire_dist + 2., HOP4Y(1), PIP_X(id_W808) + 1., HOP4Y(1)},
          {PIP_X(id_W808) + 1., HOP4Y(1), PIP_X(id_W808) + 1., WIRE_Y(0)}}},
        {id_E80_loop5,
         {{PIP_X(id_E800), WIRE_Y(0), PIP_X(id_E800), HOP4Y(16)},
          {PIP_X(id_E800), HOP4Y(16), PIP_X(id_W808), HOP4Y(16)},
          {PIP_X(id_W808) + 0., HOP4Y(16), PIP_X(id_W808) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W808) + top_wire_dist + 0., HOP4Y(14), PIP_X(id_W808) + 1., HOP4Y(14)},
          {PIP_X(id_W808) + 1., HOP4Y(14), PIP_X(id_W808) + top_wire_dist + 1., HOP4Y(12)},
          {PIP_X(id_W808) + top_wire_dist + 1., HOP4Y(12), PIP_X(id_W808) + 2., HOP4Y(12)},
          {PIP_X(id_W808) + 2., HOP4Y(12), PIP_X(id_W808) + top_wire_dist + 2., HOP4Y(10)},
          {PIP_X(id_W808) + top_wire_dist + 2., HOP4Y(10), PIP_X(id_W808) + 3., HOP4Y(10)},
          {PIP_X(id_W808) + 3., HOP4Y(10), PIP_X(id_W808) + top_wire_dist + 3., HOP4Y(8)},
          {PIP_X(id_W808) + top_wire_dist + 3., HOP4Y(8), PIP_X(id_W808) + 4., HOP4Y(8)},
          {PIP_X(id_E804) + 4., HOP4Y(8), PIP_X(id_E804) + 4., WIRE_Y(0)},
          {PIP_X(id_W808) + 4., HOP4Y(8), PIP_X(id_W808) + top_wire_dist + 4., HOP4Y(6)},
          {PIP_X(id_W808) + top_wire_dist + 4., HOP4Y(6), PIP_X(id_W808) + 5., HOP4Y(6)},
          {PIP_X(id_W808) + 5., HOP4Y(6), PIP_X(id_W808) + top_wire_dist + 5., HOP4Y(4)},
          {PIP_X(id_W808) + top_wire_dist + 5., HOP4Y(4), wrap_len + 6., HOP4Y(4)},
          {wrap_len + 6., HOP4Y(4), wrap_len + 6., HOP4Y(5)},
          {wrap_len + 6., HOP4Y(5), PIP_X(id_W800) + 5., HOP4Y(5)},
          {PIP_X(id_W800) + 5., HOP4Y(5), PIP_X(id_W800) - top_wire_dist + 5., HOP4Y(3)},
          {PIP_X(id_W800) - top_wire_dist + 5., HOP4Y(3), PIP_X(id_W800) + 4., HOP4Y(3)},
          {PIP_X(id_W800) + 4., HOP4Y(3), PIP_X(id_W800) - top_wire_dist + 4., HOP4Y(1)},
          {PIP_X(id_W800) - top_wire_dist + 4., HOP4Y(1), PIP_X(id_W808) + 3., HOP4Y(1)},
          {PIP_X(id_W808) + 3., HOP4Y(1), PIP_X(id_W808) + 3., WIRE_Y(0)}}},
        {id_E80_loop6,
         {{PIP_X(id_E800), WIRE_Y(0), PIP_X(id_E800), HOP4Y(16)},
          {PIP_X(id_E800), HOP4Y(16), PIP_X(id_W808), HOP4Y(16)},
          {PIP_X(id_W808) + 0., HOP4Y(16), PIP_X(id_W808) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W808) + top_wire_dist + 0., HOP4Y(14), PIP_X(id_W808) + 1., HOP4Y(14)},
          {PIP_X(id_W808) + 1., HOP4Y(14), PIP_X(id_W808) + top_wire_dist + 1., HOP4Y(12)},
          {PIP_X(id_W808) + top_wire_dist + 1., HOP4Y(12), PIP_X(id_W808) + 2., HOP4Y(12)},
          {PIP_X(id_W808) + 2., HOP4Y(12), PIP_X(id_W808) + top_wire_dist + 2., HOP4Y(10)},
          {PIP_X(id_W808) + top_wire_dist + 2., HOP4Y(10), PIP_X(id_W808) + 3., HOP4Y(10)},
          {PIP_X(id_W808) + 3., HOP4Y(10), PIP_X(id_W808) + top_wire_dist + 3., HOP4Y(8)},
          {PIP_X(id_W808) + top_wire_dist + 3., HOP4Y(8), PIP_X(id_W808) + 4., HOP4Y(8)},
          {PIP_X(id_W808) + 4., HOP4Y(8), PIP_X(id_W808) + top_wire_dist + 4., HOP4Y(6)},
          {PIP_X(id_E804) + 4., HOP4Y(8), PIP_X(id_E804) + 4., WIRE_Y(0)},
          {PIP_X(id_W808) + top_wire_dist + 4., HOP4Y(6), PIP_X(id_W808) + 5., HOP4Y(6)},
          {PIP_X(id_W808) + 5., HOP4Y(6), PIP_X(id_W808) + top_wire_dist + 5., HOP4Y(4)},
          {PIP_X(id_W808) + top_wire_dist + 5., HOP4Y(4), PIP_X(id_W808) + 6., HOP4Y(4)},
          {PIP_X(id_W808) + 6., HOP4Y(4), PIP_X(id_W808) + top_wire_dist + 6., HOP4Y(2)},
          {PIP_X(id_W808) + top_wire_dist + 6., HOP4Y(2), wrap_len + 7., HOP4Y(2)},
          {wrap_len + 7., HOP4Y(2), wrap_len + 7., HOP4Y(3)},
          {wrap_len + 7., HOP4Y(3), PIP_X(id_W800) + 6., HOP4Y(3)},
          {PIP_X(id_W800) + 6., HOP4Y(3), PIP_X(id_W800) - top_wire_dist + 6., HOP4Y(1)},
          {PIP_X(id_W800) - top_wire_dist + 6., HOP4Y(1), PIP_X(id_W808) + 5., HOP4Y(1)},
          {PIP_X(id_W808) + 5., HOP4Y(1), PIP_X(id_W808) + 5., WIRE_Y(0)}}},
        {id_E80_loop7,
         {{PIP_X(id_E800), WIRE_Y(0), PIP_X(id_E800), HOP4Y(16)},
          {PIP_X(id_E800), HOP4Y(16), PIP_X(id_W808), HOP4Y(16)},
          {PIP_X(id_W808) + 0., HOP4Y(16), PIP_X(id_W808) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W808) + top_wire_dist + 0., HOP4Y(14), PIP_X(id_W808) + 1., HOP4Y(14)},
          {PIP_X(id_W808) + 1., HOP4Y(14), PIP_X(id_W808) + top_wire_dist + 1., HOP4Y(12)},
          {PIP_X(id_W808) + top_wire_dist + 1., HOP4Y(12), PIP_X(id_W808) + 2., HOP4Y(12)},
          {PIP_X(id_W808) + 2., HOP4Y(12), PIP_X(id_W808) + top_wire_dist + 2., HOP4Y(10)},
          {PIP_X(id_W808) + top_wire_dist + 2., HOP4Y(10), PIP_X(id_W808) + 3., HOP4Y(10)},
          {PIP_X(id_W808) + 3., HOP4Y(10), PIP_X(id_W808) + top_wire_dist + 3., HOP4Y(8)},
          {PIP_X(id_W808) + top_wire_dist + 3., HOP4Y(8), PIP_X(id_W808) + 4., HOP4Y(8)},
          {PIP_X(id_W808) + 4., HOP4Y(8), PIP_X(id_W808) + top_wire_dist + 4., HOP4Y(6)},
          {PIP_X(id_E804) + 4., HOP4Y(8), PIP_X(id_E804) + 4., WIRE_Y(0)},
          {PIP_X(id_W808) + top_wire_dist + 4., HOP4Y(6), PIP_X(id_W808) + 5., HOP4Y(6)},
          {PIP_X(id_W808) + 5., HOP4Y(6), PIP_X(id_W808) + top_wire_dist + 5., HOP4Y(4)},
          {PIP_X(id_W808) + top_wire_dist + 5., HOP4Y(4), PIP_X(id_W808) + 6., HOP4Y(4)},
          {PIP_X(id_W808) + 6., HOP4Y(4), PIP_X(id_W808) + top_wire_dist + 6., HOP4Y(2)},
          {PIP_X(id_W808) + top_wire_dist + 6., HOP4Y(2), PIP_X(id_W808) + 7., HOP4Y(2)},
          {PIP_X(id_W808) + 7., HOP4Y(2), PIP_X(id_W808) + top_wire_dist + 7., HOP4Y(0)},
          {PIP_X(id_W808) + top_wire_dist + 7., HOP4Y(0), wrap_len + 8., HOP4Y(0)},
          {wrap_len + 8., HOP4Y(0), wrap_len + 8., HOP4Y(1)},
          {wrap_len + 8., HOP4Y(1), PIP_X(id_W808) + 7., HOP4Y(1)},
          {PIP_X(id_W808) + 7., HOP4Y(1), PIP_X(id_W808) + 7., WIRE_Y(0)}}},
        {id_W80_loop0,
         {{PIP_X(id_W800), WIRE_Y(0), PIP_X(id_W800), HOP4Y(17)},
          {PIP_X(id_W800) - 0., HOP4Y(17), PIP_X(id_W800) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W800) - top_wire_dist - 0., HOP4Y(15), -wrap_len - 0., HOP4Y(15)},
          {-wrap_len - 0., HOP4Y(15), -wrap_len - 0., HOP4Y(14)},
          {-wrap_len - 0., HOP4Y(14), PIP_X(id_W808) + 0., HOP4Y(14)},
          {PIP_X(id_W808) + 0., HOP4Y(14), PIP_X(id_W808) + top_wire_dist + 0., HOP4Y(12)},
          {PIP_X(id_W808) + top_wire_dist + 0., HOP4Y(12), PIP_X(id_W808) + 1., HOP4Y(12)},
          {PIP_X(id_W808) + 1., HOP4Y(12), PIP_X(id_W808) + top_wire_dist + 1., HOP4Y(10)},
          {PIP_X(id_W808) + top_wire_dist + 1., HOP4Y(10), PIP_X(id_W808) + 2., HOP4Y(10)},
          {PIP_X(id_W808) + 2., HOP4Y(10), PIP_X(id_W808) + top_wire_dist + 2., HOP4Y(8)},
          {PIP_X(id_W808) + top_wire_dist + 2., HOP4Y(8), PIP_X(id_W808) + 3., HOP4Y(8)},
          {PIP_X(id_W808) + 3., HOP4Y(8), PIP_X(id_W808) + top_wire_dist + 3., HOP4Y(6)},
          {PIP_X(id_E804) + 3., HOP4Y(8), PIP_X(id_E804) + 3., WIRE_Y(0)},
          {PIP_X(id_W808) + top_wire_dist + 3., HOP4Y(6), PIP_X(id_W808) + 4., HOP4Y(6)},
          {PIP_X(id_W808) + 4., HOP4Y(6), PIP_X(id_W808) + top_wire_dist + 4., HOP4Y(4)},
          {PIP_X(id_W808) + top_wire_dist + 4., HOP4Y(4), PIP_X(id_W808) + 5., HOP4Y(4)},
          {PIP_X(id_W808) + 5., HOP4Y(4), PIP_X(id_W808) + top_wire_dist + 5., HOP4Y(2)},
          {PIP_X(id_W808) + top_wire_dist + 5., HOP4Y(2), PIP_X(id_W808) + 6., HOP4Y(2)},
          {PIP_X(id_W808) + 6., HOP4Y(2), PIP_X(id_W808) + top_wire_dist + 6., HOP4Y(0)},
          {PIP_X(id_W808) + top_wire_dist + 6., HOP4Y(0), PIP_X(id_E808) + 7., HOP4Y(0)},
          {PIP_X(id_E808) + 7., HOP4Y(0), PIP_X(id_E808) + 7., WIRE_Y(0)}}},
        {id_W80_loop1,
         {{PIP_X(id_W800), WIRE_Y(0), PIP_X(id_W800), HOP4Y(17)},
          {PIP_X(id_W800) - 0., HOP4Y(17), PIP_X(id_W800) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W800) - top_wire_dist - 0., HOP4Y(15), PIP_X(id_W800) - 1., HOP4Y(15)},
          {PIP_X(id_W800) - 1., HOP4Y(15), PIP_X(id_W800) - top_wire_dist - 1., HOP4Y(13)},
          {PIP_X(id_W800) - top_wire_dist - 1., HOP4Y(13), -wrap_len - 1., HOP4Y(13)},
          {-wrap_len - 1., HOP4Y(13), -wrap_len - 1., HOP4Y(12)},
          {-wrap_len - 1., HOP4Y(12), PIP_X(id_W808) - 1., HOP4Y(12)},
          {PIP_X(id_W808) - 1., HOP4Y(12), PIP_X(id_W808) + top_wire_dist - 1., HOP4Y(10)},
          {PIP_X(id_W808) + top_wire_dist - 1., HOP4Y(10), PIP_X(id_W808) + 0., HOP4Y(10)},
          {PIP_X(id_W808) + 0., HOP4Y(10), PIP_X(id_W808) + top_wire_dist + 0., HOP4Y(8)},
          {PIP_X(id_W808) + top_wire_dist + 0., HOP4Y(8), PIP_X(id_W808) + 1., HOP4Y(8)},
          {PIP_X(id_W808) + 1., HOP4Y(8), PIP_X(id_W808) + top_wire_dist + 1., HOP4Y(6)},
          {PIP_X(id_E804) + 1., HOP4Y(8), PIP_X(id_E804) + 1., WIRE_Y(0)},
          {PIP_X(id_W808) + top_wire_dist + 1., HOP4Y(6), PIP_X(id_W808) + 2., HOP4Y(6)},
          {PIP_X(id_W808) + 2., HOP4Y(6), PIP_X(id_W808) + top_wire_dist + 2., HOP4Y(4)},
          {PIP_X(id_W808) + top_wire_dist + 2., HOP4Y(4), PIP_X(id_W808) + 3., HOP4Y(4)},
          {PIP_X(id_W808) + 3., HOP4Y(4), PIP_X(id_W808) + top_wire_dist + 3., HOP4Y(2)},
          {PIP_X(id_W808) + top_wire_dist + 3., HOP4Y(2), PIP_X(id_W808) + 4., HOP4Y(2)},
          {PIP_X(id_W808) + 4., HOP4Y(2), PIP_X(id_W808) + top_wire_dist + 4., HOP4Y(0)},
          {PIP_X(id_W808) + top_wire_dist + 4., HOP4Y(0), PIP_X(id_E808) + 5., HOP4Y(0)},
          {PIP_X(id_E808) + 5., HOP4Y(0), PIP_X(id_E808) + 5., WIRE_Y(0)}}},
        {id_W80_loop2,
         {{PIP_X(id_W800), WIRE_Y(0), PIP_X(id_W800), HOP4Y(17)},
          {PIP_X(id_W800) - 0., HOP4Y(17), PIP_X(id_W800) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W800) - top_wire_dist - 0., HOP4Y(15), PIP_X(id_W800) - 1., HOP4Y(15)},
          {PIP_X(id_W800) - 1., HOP4Y(15), PIP_X(id_W800) - top_wire_dist - 1., HOP4Y(13)},
          {PIP_X(id_W800) - top_wire_dist - 1., HOP4Y(13), PIP_X(id_W800) - 2., HOP4Y(13)},
          {PIP_X(id_W800) - 2., HOP4Y(13), PIP_X(id_W800) - top_wire_dist - 2., HOP4Y(11)},
          {PIP_X(id_W800) - top_wire_dist - 2., HOP4Y(11), -wrap_len - 2., HOP4Y(11)},
          {-wrap_len - 2., HOP4Y(11), -wrap_len - 2., HOP4Y(10)},
          {-wrap_len - 2., HOP4Y(10), PIP_X(id_W808) - 2., HOP4Y(10)},
          {PIP_X(id_W808) - 2., HOP4Y(10), PIP_X(id_W808) + top_wire_dist - 2., HOP4Y(8)},
          {PIP_X(id_W808) + top_wire_dist - 2., HOP4Y(8), PIP_X(id_W808) - 1., HOP4Y(8)},
          {PIP_X(id_W808) - 1., HOP4Y(8), PIP_X(id_W808) + top_wire_dist - 1., HOP4Y(6)},
          {PIP_X(id_E804) - 1., HOP4Y(8), PIP_X(id_E804) - 1., WIRE_Y(0)},
          {PIP_X(id_W808) + top_wire_dist - 1., HOP4Y(6), PIP_X(id_W808) + 0., HOP4Y(6)},
          {PIP_X(id_W808) + 0., HOP4Y(6), PIP_X(id_W808) + top_wire_dist + 0., HOP4Y(4)},
          {PIP_X(id_W808) + top_wire_dist + 0., HOP4Y(4), PIP_X(id_W808) + 1., HOP4Y(4)},
          {PIP_X(id_W808) + 1., HOP4Y(4), PIP_X(id_W808) + top_wire_dist + 1., HOP4Y(2)},
          {PIP_X(id_W808) + top_wire_dist + 1., HOP4Y(2), PIP_X(id_W808) + 2., HOP4Y(2)},
          {PIP_X(id_W808) + 2., HOP4Y(2), PIP_X(id_W808) + top_wire_dist + 2., HOP4Y(0)},
          {PIP_X(id_W808) + top_wire_dist + 2., HOP4Y(0), PIP_X(id_E808) + 3., HOP4Y(0)},
          {PIP_X(id_E808) + 3., HOP4Y(0), PIP_X(id_E808) + 3., WIRE_Y(0)}}},
        {id_W80_loop3,
         {{PIP_X(id_W800), WIRE_Y(0), PIP_X(id_W800), HOP4Y(17)},
          {PIP_X(id_W800) - 0., HOP4Y(17), PIP_X(id_W800) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W800) - top_wire_dist - 0., HOP4Y(15), PIP_X(id_W800) - 1., HOP4Y(15)},
          {PIP_X(id_W800) - 1., HOP4Y(15), PIP_X(id_W800) - top_wire_dist - 1., HOP4Y(13)},
          {PIP_X(id_W800) - top_wire_dist - 1., HOP4Y(13), PIP_X(id_W800) - 2., HOP4Y(13)},
          {PIP_X(id_W800) - 2., HOP4Y(13), PIP_X(id_W800) - top_wire_dist - 2., HOP4Y(11)},
          {PIP_X(id_W800) - top_wire_dist - 2., HOP4Y(11), PIP_X(id_W800) - 3., HOP4Y(11)},
          {PIP_X(id_W800) - 3., HOP4Y(11), PIP_X(id_W800) - top_wire_dist - 3., HOP4Y(9)},
          {PIP_X(id_W800) - top_wire_dist - 3., HOP4Y(9), -wrap_len - 3., HOP4Y(9)},
          {-wrap_len - 3., HOP4Y(9), -wrap_len - 3., HOP4Y(8)},
          {-wrap_len - 3., HOP4Y(8), PIP_X(id_W808) - 3., HOP4Y(8)},
          {PIP_X(id_W808) - 3., HOP4Y(8), PIP_X(id_W808) + top_wire_dist - 3., HOP4Y(6)},
          {PIP_X(id_E804) - 3., HOP4Y(8), PIP_X(id_E804) - 3., WIRE_Y(0)},
          {PIP_X(id_W808) + top_wire_dist - 3., HOP4Y(6), PIP_X(id_W808) - 2., HOP4Y(6)},
          {PIP_X(id_W808) - 2., HOP4Y(6), PIP_X(id_W808) + top_wire_dist - 2., HOP4Y(4)},
          {PIP_X(id_W808) + top_wire_dist - 2., HOP4Y(4), PIP_X(id_W808) - 1., HOP4Y(4)},
          {PIP_X(id_W808) - 1., HOP4Y(4), PIP_X(id_W808) + top_wire_dist - 1., HOP4Y(2)},
          {PIP_X(id_W808) + top_wire_dist - 1., HOP4Y(2), PIP_X(id_W808) + 0., HOP4Y(2)},
          {PIP_X(id_W808) + 0., HOP4Y(2), PIP_X(id_W808) + top_wire_dist + 0., HOP4Y(0)},
          {PIP_X(id_W808) + top_wire_dist + 0., HOP4Y(0), PIP_X(id_E808) + 1., HOP4Y(0)},
          {PIP_X(id_E808) + 1., HOP4Y(0), PIP_X(id_E808) + 1., WIRE_Y(0)}}},
        {id_W80_loop4,
         {{PIP_X(id_W800), WIRE_Y(0), PIP_X(id_W800), HOP4Y(17)},
          {PIP_X(id_W800) - 0., HOP4Y(17), PIP_X(id_W800) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W800) - top_wire_dist - 0., HOP4Y(15), PIP_X(id_W800) - 1., HOP4Y(15)},
          {PIP_X(id_W800) - 1., HOP4Y(15), PIP_X(id_W800) - top_wire_dist - 1., HOP4Y(13)},
          {PIP_X(id_W800) - top_wire_dist - 1., HOP4Y(13), PIP_X(id_W800) - 2., HOP4Y(13)},
          {PIP_X(id_W800) - 2., HOP4Y(13), PIP_X(id_W800) - top_wire_dist - 2., HOP4Y(11)},
          {PIP_X(id_W800) - top_wire_dist - 2., HOP4Y(11), PIP_X(id_W800) - 3., HOP4Y(11)},
          {PIP_X(id_W800) - 3., HOP4Y(11), PIP_X(id_W800) - top_wire_dist - 3., HOP4Y(9)},
          {PIP_X(id_W800) - top_wire_dist - 3., HOP4Y(9), PIP_X(id_W800) - 4., HOP4Y(9)},
          {PIP_X(id_W800) - 4., HOP4Y(9), PIP_X(id_W800) - top_wire_dist - 4., HOP4Y(7)},
          {PIP_X(id_W800) - top_wire_dist - 4., HOP4Y(7), -wrap_len - 4., HOP4Y(7)},
          {-wrap_len - 4., HOP4Y(7), -wrap_len - 4., HOP4Y(6)},
          {PIP_X(id_W804) - 4., HOP4Y(6), PIP_X(id_W804) - 4., WIRE_Y(0)},
          {-wrap_len - 4., HOP4Y(6), PIP_X(id_W808) - 4., HOP4Y(6)},
          {PIP_X(id_W808) - 4., HOP4Y(6), PIP_X(id_W808) + top_wire_dist - 4., HOP4Y(4)},
          {PIP_X(id_W808) + top_wire_dist - 4., HOP4Y(4), PIP_X(id_W808) - 3., HOP4Y(4)},
          {PIP_X(id_W808) - 3., HOP4Y(4), PIP_X(id_W808) + top_wire_dist - 3., HOP4Y(2)},
          {PIP_X(id_W808) + top_wire_dist - 3., HOP4Y(2), PIP_X(id_W808) - 2., HOP4Y(2)},
          {PIP_X(id_W808) - 2., HOP4Y(2), PIP_X(id_W808) + top_wire_dist - 2., HOP4Y(0)},
          {PIP_X(id_W808) + top_wire_dist - 2., HOP4Y(0), PIP_X(id_E808) - 1., HOP4Y(0)},
          {PIP_X(id_E808) - 1., HOP4Y(0), PIP_X(id_E808) - 1., WIRE_Y(0)}}},
        {id_W80_loop5,
         {{PIP_X(id_W800), WIRE_Y(0), PIP_X(id_W800), HOP4Y(17)},
          {PIP_X(id_W800) - 0., HOP4Y(17), PIP_X(id_W800) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W800) - top_wire_dist - 0., HOP4Y(15), PIP_X(id_W800) - 1., HOP4Y(15)},
          {PIP_X(id_W800) - 1., HOP4Y(15), PIP_X(id_W800) - top_wire_dist - 1., HOP4Y(13)},
          {PIP_X(id_W800) - top_wire_dist - 1., HOP4Y(13), PIP_X(id_W800) - 2., HOP4Y(13)},
          {PIP_X(id_W800) - 2., HOP4Y(13), PIP_X(id_W800) - top_wire_dist - 2., HOP4Y(11)},
          {PIP_X(id_W800) - top_wire_dist - 2., HOP4Y(11), PIP_X(id_W800) - 3., HOP4Y(11)},
          {PIP_X(id_W800) - 3., HOP4Y(11), PIP_X(id_W800) - top_wire_dist - 3., HOP4Y(9)},
          {PIP_X(id_W800) - top_wire_dist - 3., HOP4Y(9), PIP_X(id_W800) - 4., HOP4Y(9)},
          {PIP_X(id_W800) - 4., HOP4Y(9), PIP_X(id_W800) - top_wire_dist - 4., HOP4Y(7)},
          {PIP_X(id_W804) - 4., HOP4Y(9), PIP_X(id_W804) - 4., WIRE_Y(0)},
          {PIP_X(id_W800) - top_wire_dist - 4., HOP4Y(7), PIP_X(id_W800) - 5., HOP4Y(7)},
          {PIP_X(id_W800) - 5., HOP4Y(7), PIP_X(id_W800) - top_wire_dist - 5., HOP4Y(5)},
          {PIP_X(id_W800) - top_wire_dist - 5., HOP4Y(5), -wrap_len - 5., HOP4Y(5)},
          {-wrap_len - 5., HOP4Y(5), -wrap_len - 5., HOP4Y(4)},
          {-wrap_len - 5., HOP4Y(4), PIP_X(id_W808) - 5., HOP4Y(4)},
          {PIP_X(id_W808) - 5., HOP4Y(4), PIP_X(id_W808) + top_wire_dist - 5., HOP4Y(2)},
          {PIP_X(id_W808) + top_wire_dist - 5., HOP4Y(2), PIP_X(id_W808) - 4., HOP4Y(2)},
          {PIP_X(id_W808) - 4., HOP4Y(2), PIP_X(id_W808) + top_wire_dist - 4., HOP4Y(0)},
          {PIP_X(id_W808) + top_wire_dist - 4., HOP4Y(0), PIP_X(id_E808) - 3., HOP4Y(0)},
          {PIP_X(id_E808) - 3., HOP4Y(0), PIP_X(id_E808) - 3., WIRE_Y(0)}}},
        {id_W80_loop6,
         {{PIP_X(id_W800), WIRE_Y(0), PIP_X(id_W800), HOP4Y(17)},
          {PIP_X(id_W800) - 0., HOP4Y(17), PIP_X(id_W800) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W800) - top_wire_dist - 0., HOP4Y(15), PIP_X(id_W800) - 1., HOP4Y(15)},
          {PIP_X(id_W800) - 1., HOP4Y(15), PIP_X(id_W800) - top_wire_dist - 1., HOP4Y(13)},
          {PIP_X(id_W800) - top_wire_dist - 1., HOP4Y(13), PIP_X(id_W800) - 2., HOP4Y(13)},
          {PIP_X(id_W800) - 2., HOP4Y(13), PIP_X(id_W800) - top_wire_dist - 2., HOP4Y(11)},
          {PIP_X(id_W800) - top_wire_dist - 2., HOP4Y(11), PIP_X(id_W800) - 3., HOP4Y(11)},
          {PIP_X(id_W800) - 3., HOP4Y(11), PIP_X(id_W800) - top_wire_dist - 3., HOP4Y(9)},
          {PIP_X(id_W800) - top_wire_dist - 3., HOP4Y(9), PIP_X(id_W800) - 4., HOP4Y(9)},
          {PIP_X(id_W800) - 4., HOP4Y(9), PIP_X(id_W800) - top_wire_dist - 4., HOP4Y(7)},
          {PIP_X(id_W804) - 4., HOP4Y(9), PIP_X(id_W804) - 4., WIRE_Y(0)},
          {PIP_X(id_W800) - top_wire_dist - 4., HOP4Y(7), PIP_X(id_W800) - 5., HOP4Y(7)},
          {PIP_X(id_W800) - 5., HOP4Y(7), PIP_X(id_W800) - top_wire_dist - 5., HOP4Y(5)},
          {PIP_X(id_W800) - top_wire_dist - 5., HOP4Y(5), PIP_X(id_W800) - 6., HOP4Y(5)},
          {PIP_X(id_W800) - 6., HOP4Y(5), PIP_X(id_W800) - top_wire_dist - 6., HOP4Y(3)},
          {PIP_X(id_W800) - top_wire_dist - 6., HOP4Y(3), -wrap_len - 6., HOP4Y(3)},
          {-wrap_len - 6., HOP4Y(3), -wrap_len - 6., HOP4Y(2)},
          {-wrap_len - 6., HOP4Y(2), PIP_X(id_W808) - 6., HOP4Y(2)},
          {PIP_X(id_W808) - 6., HOP4Y(2), PIP_X(id_W808) + top_wire_dist - 6., HOP4Y(0)},
          {PIP_X(id_W808) + top_wire_dist - 6., HOP4Y(0), PIP_X(id_E808) - 5., HOP4Y(0)},
          {PIP_X(id_E808) - 5., HOP4Y(0), PIP_X(id_E808) - 5., WIRE_Y(0)}}},
        {id_W80_loop7,
         {{PIP_X(id_W800), WIRE_Y(0), PIP_X(id_W800), HOP4Y(17)},
          {PIP_X(id_W800) - 0., HOP4Y(17), PIP_X(id_W800) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W800) - top_wire_dist - 0., HOP4Y(15), PIP_X(id_W800) - 1., HOP4Y(15)},
          {PIP_X(id_W800) - 1., HOP4Y(15), PIP_X(id_W800) - top_wire_dist - 1., HOP4Y(13)},
          {PIP_X(id_W800) - top_wire_dist - 1., HOP4Y(13), PIP_X(id_W800) - 2., HOP4Y(13)},
          {PIP_X(id_W800) - 2., HOP4Y(13), PIP_X(id_W800) - top_wire_dist - 2., HOP4Y(11)},
          {PIP_X(id_W800) - top_wire_dist - 2., HOP4Y(11), PIP_X(id_W800) - 3., HOP4Y(11)},
          {PIP_X(id_W800) - 3., HOP4Y(11), PIP_X(id_W800) - top_wire_dist - 3., HOP4Y(9)},
          {PIP_X(id_W800) - top_wire_dist - 3., HOP4Y(9), PIP_X(id_W800) - 4., HOP4Y(9)},
          {PIP_X(id_W800) - 4., HOP4Y(9), PIP_X(id_W800) - top_wire_dist - 4., HOP4Y(7)},
          {PIP_X(id_W804) - 4., HOP4Y(9), PIP_X(id_W804) - 4., WIRE_Y(0)},
          {PIP_X(id_W800) - top_wire_dist - 4., HOP4Y(7), PIP_X(id_W800) - 5., HOP4Y(7)},
          {PIP_X(id_W800) - 5., HOP4Y(7), PIP_X(id_W800) - top_wire_dist - 5., HOP4Y(5)},
          {PIP_X(id_W800) - top_wire_dist - 5., HOP4Y(5), PIP_X(id_W800) - 6., HOP4Y(5)},
          {PIP_X(id_W800) - 6., HOP4Y(5), PIP_X(id_W800) - top_wire_dist - 6., HOP4Y(3)},
          {PIP_X(id_W800) - top_wire_dist - 6., HOP4Y(3), PIP_X(id_W800) - 7., HOP4Y(3)},
          {PIP_X(id_W800) - 7., HOP4Y(3), PIP_X(id_W800) - top_wire_dist - 7., HOP4Y(1)},
          {PIP_X(id_W800) - top_wire_dist - 7., HOP4Y(1), -wrap_len - 7., HOP4Y(1)},
          {-wrap_len - 7., HOP4Y(1), -wrap_len - 7., HOP4Y(0)},
          {-wrap_len - 7., HOP4Y(0), PIP_X(id_E808) - 7., HOP4Y(0)},
          {PIP_X(id_E808) - 7., HOP4Y(0), PIP_X(id_E808) - 7., WIRE_Y(0)}}},

#undef HOP4Y
#define HOP4Y(offset) WIRE_Y((float)offset + HOP4Y_START + 18.f)
        {id_E81,
         {{PIP_X(id_E810), WIRE_Y(0), PIP_X(id_E810), HOP4Y(16)},
          {PIP_X(id_E810), HOP4Y(16), PIP_X(id_W818), HOP4Y(16)},
          {PIP_X(id_W818) + 0., HOP4Y(16), PIP_X(id_W818) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W818) + top_wire_dist + 0., HOP4Y(14), PIP_X(id_W818) + 1., HOP4Y(14)},
          {PIP_X(id_W818) + 1., HOP4Y(14), PIP_X(id_W818) + top_wire_dist + 1., HOP4Y(12)},
          {PIP_X(id_W818) + top_wire_dist + 1., HOP4Y(12), PIP_X(id_W818) + 2., HOP4Y(12)},
          {PIP_X(id_W818) + 2., HOP4Y(12), PIP_X(id_W818) + top_wire_dist + 2., HOP4Y(10)},
          {PIP_X(id_W818) + top_wire_dist + 2., HOP4Y(10), PIP_X(id_W818) + 3., HOP4Y(10)},
          {PIP_X(id_W818) + 3., HOP4Y(10), PIP_X(id_W818) + top_wire_dist + 3., HOP4Y(8)},
          {PIP_X(id_W818) + top_wire_dist + 3., HOP4Y(8), PIP_X(id_W818) + 4., HOP4Y(8)},
          {PIP_X(id_W818) + 4., HOP4Y(8), PIP_X(id_W818) + top_wire_dist + 4., HOP4Y(6)},
          {PIP_X(id_E814) + 4., HOP4Y(8), PIP_X(id_E814) + 4., WIRE_Y(0)},
          {PIP_X(id_W818) + top_wire_dist + 4., HOP4Y(6), PIP_X(id_W818) + 5., HOP4Y(6)},
          {PIP_X(id_W818) + 5., HOP4Y(6), PIP_X(id_W818) + top_wire_dist + 5., HOP4Y(4)},
          {PIP_X(id_W818) + top_wire_dist + 5., HOP4Y(4), PIP_X(id_W818) + 6., HOP4Y(4)},
          {PIP_X(id_W818) + 6., HOP4Y(4), PIP_X(id_W818) + top_wire_dist + 6., HOP4Y(2)},
          {PIP_X(id_W818) + top_wire_dist + 6., HOP4Y(2), PIP_X(id_W818) + 7., HOP4Y(2)},
          {PIP_X(id_W818) + 7., HOP4Y(2), PIP_X(id_W818) + top_wire_dist + 7., HOP4Y(0)},
          {PIP_X(id_W818) + top_wire_dist + 7., HOP4Y(0), PIP_X(id_E818) + 8., HOP4Y(0)},
          {PIP_X(id_E818) + 8, HOP4Y(0), PIP_X(id_E818) + 8., WIRE_Y(0)}}},
        {id_W81,
         {{PIP_X(id_W810), WIRE_Y(0), PIP_X(id_W810), HOP4Y(17)},
          {PIP_X(id_W810) - 0., HOP4Y(17), PIP_X(id_W810) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W810) - top_wire_dist - 0., HOP4Y(15), PIP_X(id_W810) - 1., HOP4Y(15)},
          {PIP_X(id_W810) - 1., HOP4Y(15), PIP_X(id_W810) - top_wire_dist - 1., HOP4Y(13)},
          {PIP_X(id_W810) - top_wire_dist - 1., HOP4Y(13), PIP_X(id_W810) - 2., HOP4Y(13)},
          {PIP_X(id_W810) - 2., HOP4Y(13), PIP_X(id_W810) - top_wire_dist - 2., HOP4Y(11)},
          {PIP_X(id_W810) - top_wire_dist - 2., HOP4Y(11), PIP_X(id_W810) - 3., HOP4Y(11)},
          {PIP_X(id_W810) - 3., HOP4Y(11), PIP_X(id_W810) - top_wire_dist - 3., HOP4Y(9)},
          {PIP_X(id_W810) - top_wire_dist - 3., HOP4Y(9), PIP_X(id_W810) - 4., HOP4Y(9)},
          {PIP_X(id_W810) - 4., HOP4Y(9), PIP_X(id_W810) - top_wire_dist - 4., HOP4Y(7)},
          {PIP_X(id_W814) - 4., HOP4Y(9), PIP_X(id_W814) - 4., WIRE_Y(0)},
          {PIP_X(id_W810) - top_wire_dist - 4., HOP4Y(7), PIP_X(id_W810) - 5., HOP4Y(7)},
          {PIP_X(id_W810) - 5., HOP4Y(7), PIP_X(id_W810) - top_wire_dist - 5., HOP4Y(5)},
          {PIP_X(id_W810) - top_wire_dist - 5., HOP4Y(5), PIP_X(id_W810) - 6., HOP4Y(5)},
          {PIP_X(id_W810) - 6., HOP4Y(5), PIP_X(id_W810) - top_wire_dist - 6., HOP4Y(3)},
          {PIP_X(id_W810) - top_wire_dist - 6., HOP4Y(3), PIP_X(id_W810) - 7., HOP4Y(3)},
          {PIP_X(id_W810) - 7., HOP4Y(3), PIP_X(id_W810) - top_wire_dist - 7., HOP4Y(1)},
          {PIP_X(id_W810) - top_wire_dist - 7., HOP4Y(1), PIP_X(id_W818) - 8., HOP4Y(1)},
          {PIP_X(id_W818) - 8, HOP4Y(1), PIP_X(id_W818) - 8., WIRE_Y(0)}}},
        {id_E81_loop0,
         {{PIP_X(id_E810), WIRE_Y(0), PIP_X(id_E810), HOP4Y(16)},
          {PIP_X(id_E810), HOP4Y(16), PIP_X(id_W818), HOP4Y(16)},
          {PIP_X(id_W818) + 0., HOP4Y(16), PIP_X(id_W818) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W818) + top_wire_dist + 0., HOP4Y(14), wrap_len + 1., HOP4Y(14)},
          {wrap_len + 1., HOP4Y(14), wrap_len + 1., HOP4Y(15)},
          {wrap_len + 1., HOP4Y(15), PIP_X(id_W810) - 0., HOP4Y(15)},
          {PIP_X(id_W810) - 0., HOP4Y(15), PIP_X(id_W810) - top_wire_dist - 0., HOP4Y(13)},
          {PIP_X(id_W810) - top_wire_dist - 0., HOP4Y(13), PIP_X(id_W810) - 1., HOP4Y(13)},
          {PIP_X(id_W810) - 1., HOP4Y(13), PIP_X(id_W810) - top_wire_dist - 1., HOP4Y(11)},
          {PIP_X(id_W810) - top_wire_dist - 1., HOP4Y(11), PIP_X(id_W810) - 2., HOP4Y(11)},
          {PIP_X(id_W810) - 2., HOP4Y(11), PIP_X(id_W810) - top_wire_dist - 2., HOP4Y(9)},
          {PIP_X(id_W810) - top_wire_dist - 2., HOP4Y(9), PIP_X(id_W810) - 3., HOP4Y(9)},
          {PIP_X(id_W810) - 3., HOP4Y(9), PIP_X(id_W810) - top_wire_dist - 3., HOP4Y(7)},
          {PIP_X(id_W814) - 3., HOP4Y(9), PIP_X(id_W814) - 3., WIRE_Y(0)},
          {PIP_X(id_W810) - top_wire_dist - 3., HOP4Y(7), PIP_X(id_W810) - 4., HOP4Y(7)},
          {PIP_X(id_W810) - 4., HOP4Y(7), PIP_X(id_W810) - top_wire_dist - 4., HOP4Y(5)},
          {PIP_X(id_W810) - top_wire_dist - 4., HOP4Y(5), PIP_X(id_W810) - 5., HOP4Y(5)},
          {PIP_X(id_W810) - 5., HOP4Y(5), PIP_X(id_W810) - top_wire_dist - 5., HOP4Y(3)},
          {PIP_X(id_W810) - top_wire_dist - 5., HOP4Y(3), PIP_X(id_W810) - 6., HOP4Y(3)},
          {PIP_X(id_W810) - 6., HOP4Y(3), PIP_X(id_W810) - top_wire_dist - 6., HOP4Y(1)},
          {PIP_X(id_W810) - top_wire_dist - 6., HOP4Y(1), PIP_X(id_W818) - 7., HOP4Y(1)},
          {PIP_X(id_W818) - 7, HOP4Y(1), PIP_X(id_W818) - 7., WIRE_Y(0)}}},
        {id_E81_loop1,
         {{PIP_X(id_E810), WIRE_Y(0), PIP_X(id_E810), HOP4Y(16)},
          {PIP_X(id_E810), HOP4Y(16), PIP_X(id_W818), HOP4Y(16)},
          {PIP_X(id_W818) + 0., HOP4Y(16), PIP_X(id_W818) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W818) + top_wire_dist + 0., HOP4Y(14), PIP_X(id_W818) + 1., HOP4Y(14)},
          {PIP_X(id_W818) + 1., HOP4Y(14), PIP_X(id_W818) + top_wire_dist + 1., HOP4Y(12)},
          {PIP_X(id_W818) + top_wire_dist + 1., HOP4Y(12), wrap_len + 2., HOP4Y(12)},
          {wrap_len + 2., HOP4Y(12), wrap_len + 2., HOP4Y(13)},
          {wrap_len + 2., HOP4Y(13), PIP_X(id_W810) + 1., HOP4Y(13)},
          {PIP_X(id_W810) + 1., HOP4Y(13), PIP_X(id_W810) - top_wire_dist + 1., HOP4Y(11)},
          {PIP_X(id_W810) - top_wire_dist + 1., HOP4Y(11), PIP_X(id_W810) - 0., HOP4Y(11)},
          {PIP_X(id_W810) - 0., HOP4Y(11), PIP_X(id_W810) - top_wire_dist - 0., HOP4Y(9)},
          {PIP_X(id_W810) - top_wire_dist - 0., HOP4Y(9), PIP_X(id_W810) - 1., HOP4Y(9)},
          {PIP_X(id_W814) - 1., HOP4Y(9), PIP_X(id_W814) - 1., WIRE_Y(0)},
          {PIP_X(id_W810) - 1., HOP4Y(9), PIP_X(id_W810) - top_wire_dist - 1., HOP4Y(7)},
          {PIP_X(id_W810) - top_wire_dist - 1., HOP4Y(7), PIP_X(id_W810) - 2., HOP4Y(7)},
          {PIP_X(id_W810) - 2., HOP4Y(7), PIP_X(id_W810) - top_wire_dist - 2., HOP4Y(5)},
          {PIP_X(id_W810) - top_wire_dist - 2., HOP4Y(5), PIP_X(id_W810) - 3., HOP4Y(5)},
          {PIP_X(id_W810) - 3., HOP4Y(5), PIP_X(id_W810) - top_wire_dist - 3., HOP4Y(3)},
          {PIP_X(id_W810) - top_wire_dist - 3., HOP4Y(3), PIP_X(id_W810) - 4., HOP4Y(3)},
          {PIP_X(id_W810) - 4., HOP4Y(3), PIP_X(id_W810) - top_wire_dist - 4., HOP4Y(1)},
          {PIP_X(id_W810) - top_wire_dist - 4., HOP4Y(1), PIP_X(id_W818) - 5., HOP4Y(1)},
          {PIP_X(id_W818) - 5., HOP4Y(1), PIP_X(id_W818) - 5., WIRE_Y(0)}}},
        {id_E81_loop2,
         {{PIP_X(id_E810), WIRE_Y(0), PIP_X(id_E810), HOP4Y(16)},
          {PIP_X(id_E810), HOP4Y(16), PIP_X(id_W818), HOP4Y(16)},
          {PIP_X(id_W818) + 0., HOP4Y(16), PIP_X(id_W818) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W818) + top_wire_dist + 0., HOP4Y(14), PIP_X(id_W818) + 1., HOP4Y(14)},
          {PIP_X(id_W818) + 1., HOP4Y(14), PIP_X(id_W818) + top_wire_dist + 1., HOP4Y(12)},
          {PIP_X(id_W818) + top_wire_dist + 1., HOP4Y(12), PIP_X(id_W818) + 2., HOP4Y(12)},
          {PIP_X(id_W818) + 2., HOP4Y(12), PIP_X(id_W818) + top_wire_dist + 2., HOP4Y(10)},
          {PIP_X(id_W818) + top_wire_dist + 2., HOP4Y(10), wrap_len + 3., HOP4Y(10)},
          {wrap_len + 3., HOP4Y(10), wrap_len + 3., HOP4Y(11)},
          {wrap_len + 3., HOP4Y(11), PIP_X(id_W810) + 2., HOP4Y(11)},
          {PIP_X(id_W810) + 2., HOP4Y(11), PIP_X(id_W810) - top_wire_dist + 2., HOP4Y(9)},
          {PIP_X(id_W810) - top_wire_dist + 2., HOP4Y(9), PIP_X(id_W810) + 1., HOP4Y(9)},
          {PIP_X(id_W814) + 1., HOP4Y(9), PIP_X(id_W814) + 1., WIRE_Y(0)},
          {PIP_X(id_W810) + 1., HOP4Y(9), PIP_X(id_W810) - top_wire_dist + 1., HOP4Y(7)},
          {PIP_X(id_W810) - top_wire_dist + 1., HOP4Y(7), PIP_X(id_W810) + 0., HOP4Y(7)},
          {PIP_X(id_W810) + 0., HOP4Y(7), PIP_X(id_W810) - top_wire_dist - 0., HOP4Y(5)},
          {PIP_X(id_W810) - top_wire_dist - 0., HOP4Y(5), PIP_X(id_W810) - 1., HOP4Y(5)},
          {PIP_X(id_W810) - 1., HOP4Y(5), PIP_X(id_W810) - top_wire_dist - 1., HOP4Y(3)},
          {PIP_X(id_W810) - top_wire_dist - 1., HOP4Y(3), PIP_X(id_W810) - 2., HOP4Y(3)},
          {PIP_X(id_W810) - 2., HOP4Y(3), PIP_X(id_W810) - top_wire_dist - 2., HOP4Y(1)},
          {PIP_X(id_W810) - top_wire_dist - 2., HOP4Y(1), PIP_X(id_W818) - 3., HOP4Y(1)},
          {PIP_X(id_W818) - 3., HOP4Y(1), PIP_X(id_W818) - 3., WIRE_Y(0)}}},
        {id_E81_loop3,
         {{PIP_X(id_E810), WIRE_Y(0), PIP_X(id_E810), HOP4Y(16)},
          {PIP_X(id_E810), HOP4Y(16), PIP_X(id_W818), HOP4Y(16)},
          {PIP_X(id_W818) + 0., HOP4Y(16), PIP_X(id_W818) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W818) + top_wire_dist + 0., HOP4Y(14), PIP_X(id_W818) + 1., HOP4Y(14)},
          {PIP_X(id_W818) + 1., HOP4Y(14), PIP_X(id_W818) + top_wire_dist + 1., HOP4Y(12)},
          {PIP_X(id_W818) + top_wire_dist + 1., HOP4Y(12), PIP_X(id_W818) + 2., HOP4Y(12)},
          {PIP_X(id_W818) + 2., HOP4Y(12), PIP_X(id_W818) + top_wire_dist + 2., HOP4Y(10)},
          {PIP_X(id_W818) + top_wire_dist + 2., HOP4Y(10), PIP_X(id_W818) + 3., HOP4Y(10)},
          {PIP_X(id_W818) + 3., HOP4Y(10), PIP_X(id_W818) + top_wire_dist + 3., HOP4Y(8)},
          {PIP_X(id_W818) + top_wire_dist + 3., HOP4Y(8), wrap_len + 4., HOP4Y(8)},
          {wrap_len + 4., HOP4Y(8), wrap_len + 4., HOP4Y(9)},
          {wrap_len + 4., HOP4Y(9), PIP_X(id_W810) + 3., HOP4Y(9)},
          {PIP_X(id_W814) + 3., HOP4Y(9), PIP_X(id_W814) + 3., WIRE_Y(0)},
          {PIP_X(id_W810) + 3., HOP4Y(9), PIP_X(id_W810) - top_wire_dist + 3., HOP4Y(7)},
          {PIP_X(id_W810) - top_wire_dist + 3., HOP4Y(7), PIP_X(id_W810) + 2., HOP4Y(7)},
          {PIP_X(id_W810) + 2., HOP4Y(7), PIP_X(id_W810) - top_wire_dist + 2., HOP4Y(5)},
          {PIP_X(id_W810) - top_wire_dist + 2., HOP4Y(5), PIP_X(id_W810) + 1., HOP4Y(5)},
          {PIP_X(id_W810) + 1., HOP4Y(5), PIP_X(id_W810) - top_wire_dist + 1., HOP4Y(3)},
          {PIP_X(id_W810) - top_wire_dist + 1., HOP4Y(3), PIP_X(id_W810) - 0., HOP4Y(3)},
          {PIP_X(id_W810) - 0., HOP4Y(3), PIP_X(id_W810) - top_wire_dist - 0., HOP4Y(1)},
          {PIP_X(id_W810) - top_wire_dist - 0., HOP4Y(1), PIP_X(id_W818) - 1., HOP4Y(1)},
          {PIP_X(id_W818) - 1., HOP4Y(1), PIP_X(id_W818) - 1., WIRE_Y(0)}}},
        {id_E81_loop4,
         {{PIP_X(id_E810), WIRE_Y(0), PIP_X(id_E810), HOP4Y(16)},
          {PIP_X(id_E810), HOP4Y(16), PIP_X(id_W818), HOP4Y(16)},
          {PIP_X(id_W818) + 0., HOP4Y(16), PIP_X(id_W818) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W818) + top_wire_dist + 0., HOP4Y(14), PIP_X(id_W818) + 1., HOP4Y(14)},
          {PIP_X(id_W818) + 1., HOP4Y(14), PIP_X(id_W818) + top_wire_dist + 1., HOP4Y(12)},
          {PIP_X(id_W818) + top_wire_dist + 1., HOP4Y(12), PIP_X(id_W818) + 2., HOP4Y(12)},
          {PIP_X(id_W818) + 2., HOP4Y(12), PIP_X(id_W818) + top_wire_dist + 2., HOP4Y(10)},
          {PIP_X(id_W818) + top_wire_dist + 2., HOP4Y(10), PIP_X(id_W818) + 3., HOP4Y(10)},
          {PIP_X(id_W818) + 3., HOP4Y(10), PIP_X(id_W818) + top_wire_dist + 3., HOP4Y(8)},
          {PIP_X(id_W818) + top_wire_dist + 3., HOP4Y(8), PIP_X(id_W818) + 4., HOP4Y(8)},
          {PIP_X(id_E814) + 4., HOP4Y(8), PIP_X(id_E814) + 4., WIRE_Y(0)},
          {PIP_X(id_W818) + 4., HOP4Y(8), PIP_X(id_W818) + top_wire_dist + 4., HOP4Y(6)},
          {PIP_X(id_W818) + top_wire_dist + 4., HOP4Y(6), wrap_len + 5., HOP4Y(6)},
          {wrap_len + 5., HOP4Y(6), wrap_len + 5., HOP4Y(7)},
          {wrap_len + 5., HOP4Y(7), PIP_X(id_W810) + 4., HOP4Y(7)},
          {PIP_X(id_W810) + 4., HOP4Y(7), PIP_X(id_W810) - top_wire_dist + 4., HOP4Y(5)},
          {PIP_X(id_W810) - top_wire_dist + 4., HOP4Y(5), PIP_X(id_W810) + 3., HOP4Y(5)},
          {PIP_X(id_W810) + 3., HOP4Y(5), PIP_X(id_W810) - top_wire_dist + 3., HOP4Y(3)},
          {PIP_X(id_W810) - top_wire_dist + 3., HOP4Y(3), PIP_X(id_W810) + 2., HOP4Y(3)},
          {PIP_X(id_W810) + 2., HOP4Y(3), PIP_X(id_W810) - top_wire_dist + 2., HOP4Y(1)},
          {PIP_X(id_W810) - top_wire_dist + 2., HOP4Y(1), PIP_X(id_W818) + 1., HOP4Y(1)},
          {PIP_X(id_W818) + 1., HOP4Y(1), PIP_X(id_W818) + 1., WIRE_Y(0)}}},
        {id_E81_loop5,
         {{PIP_X(id_E810), WIRE_Y(0), PIP_X(id_E810), HOP4Y(16)},
          {PIP_X(id_E810), HOP4Y(16), PIP_X(id_W818), HOP4Y(16)},
          {PIP_X(id_W818) + 0., HOP4Y(16), PIP_X(id_W818) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W818) + top_wire_dist + 0., HOP4Y(14), PIP_X(id_W818) + 1., HOP4Y(14)},
          {PIP_X(id_W818) + 1., HOP4Y(14), PIP_X(id_W818) + top_wire_dist + 1., HOP4Y(12)},
          {PIP_X(id_W818) + top_wire_dist + 1., HOP4Y(12), PIP_X(id_W818) + 2., HOP4Y(12)},
          {PIP_X(id_W818) + 2., HOP4Y(12), PIP_X(id_W818) + top_wire_dist + 2., HOP4Y(10)},
          {PIP_X(id_W818) + top_wire_dist + 2., HOP4Y(10), PIP_X(id_W818) + 3., HOP4Y(10)},
          {PIP_X(id_W818) + 3., HOP4Y(10), PIP_X(id_W818) + top_wire_dist + 3., HOP4Y(8)},
          {PIP_X(id_W818) + top_wire_dist + 3., HOP4Y(8), PIP_X(id_W818) + 4., HOP4Y(8)},
          {PIP_X(id_E814) + 4., HOP4Y(8), PIP_X(id_E814) + 4., WIRE_Y(0)},
          {PIP_X(id_W818) + 4., HOP4Y(8), PIP_X(id_W818) + top_wire_dist + 4., HOP4Y(6)},
          {PIP_X(id_W818) + top_wire_dist + 4., HOP4Y(6), PIP_X(id_W818) + 5., HOP4Y(6)},
          {PIP_X(id_W818) + 5., HOP4Y(6), PIP_X(id_W818) + top_wire_dist + 5., HOP4Y(4)},
          {PIP_X(id_W818) + top_wire_dist + 5., HOP4Y(4), wrap_len + 6., HOP4Y(4)},
          {wrap_len + 6., HOP4Y(4), wrap_len + 6., HOP4Y(5)},
          {wrap_len + 6., HOP4Y(5), PIP_X(id_W810) + 5., HOP4Y(5)},
          {PIP_X(id_W810) + 5., HOP4Y(5), PIP_X(id_W810) - top_wire_dist + 5., HOP4Y(3)},
          {PIP_X(id_W810) - top_wire_dist + 5., HOP4Y(3), PIP_X(id_W810) + 4., HOP4Y(3)},
          {PIP_X(id_W810) + 4., HOP4Y(3), PIP_X(id_W810) - top_wire_dist + 4., HOP4Y(1)},
          {PIP_X(id_W810) - top_wire_dist + 4., HOP4Y(1), PIP_X(id_W818) + 3., HOP4Y(1)},
          {PIP_X(id_W818) + 3., HOP4Y(1), PIP_X(id_W818) + 3., WIRE_Y(0)}}},
        {id_E81_loop6,
         {{PIP_X(id_E810), WIRE_Y(0), PIP_X(id_E810), HOP4Y(16)},
          {PIP_X(id_E810), HOP4Y(16), PIP_X(id_W818), HOP4Y(16)},
          {PIP_X(id_W818) + 0., HOP4Y(16), PIP_X(id_W818) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W818) + top_wire_dist + 0., HOP4Y(14), PIP_X(id_W818) + 1., HOP4Y(14)},
          {PIP_X(id_W818) + 1., HOP4Y(14), PIP_X(id_W818) + top_wire_dist + 1., HOP4Y(12)},
          {PIP_X(id_W818) + top_wire_dist + 1., HOP4Y(12), PIP_X(id_W818) + 2., HOP4Y(12)},
          {PIP_X(id_W818) + 2., HOP4Y(12), PIP_X(id_W818) + top_wire_dist + 2., HOP4Y(10)},
          {PIP_X(id_W818) + top_wire_dist + 2., HOP4Y(10), PIP_X(id_W818) + 3., HOP4Y(10)},
          {PIP_X(id_W818) + 3., HOP4Y(10), PIP_X(id_W818) + top_wire_dist + 3., HOP4Y(8)},
          {PIP_X(id_W818) + top_wire_dist + 3., HOP4Y(8), PIP_X(id_W818) + 4., HOP4Y(8)},
          {PIP_X(id_W818) + 4., HOP4Y(8), PIP_X(id_W818) + top_wire_dist + 4., HOP4Y(6)},
          {PIP_X(id_E814) + 4., HOP4Y(8), PIP_X(id_E814) + 4., WIRE_Y(0)},
          {PIP_X(id_W818) + top_wire_dist + 4., HOP4Y(6), PIP_X(id_W818) + 5., HOP4Y(6)},
          {PIP_X(id_W818) + 5., HOP4Y(6), PIP_X(id_W818) + top_wire_dist + 5., HOP4Y(4)},
          {PIP_X(id_W818) + top_wire_dist + 5., HOP4Y(4), PIP_X(id_W818) + 6., HOP4Y(4)},
          {PIP_X(id_W818) + 6., HOP4Y(4), PIP_X(id_W818) + top_wire_dist + 6., HOP4Y(2)},
          {PIP_X(id_W818) + top_wire_dist + 6., HOP4Y(2), wrap_len + 7., HOP4Y(2)},
          {wrap_len + 7., HOP4Y(2), wrap_len + 7., HOP4Y(3)},
          {wrap_len + 7., HOP4Y(3), PIP_X(id_W810) + 6., HOP4Y(3)},
          {PIP_X(id_W810) + 6., HOP4Y(3), PIP_X(id_W810) - top_wire_dist + 6., HOP4Y(1)},
          {PIP_X(id_W810) - top_wire_dist + 6., HOP4Y(1), PIP_X(id_W818) + 5., HOP4Y(1)},
          {PIP_X(id_W818) + 5., HOP4Y(1), PIP_X(id_W818) + 5., WIRE_Y(0)}}},
        {id_E81_loop7,
         {{PIP_X(id_E810), WIRE_Y(0), PIP_X(id_E810), HOP4Y(16)},
          {PIP_X(id_E810), HOP4Y(16), PIP_X(id_W818), HOP4Y(16)},
          {PIP_X(id_W818) + 0., HOP4Y(16), PIP_X(id_W818) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W818) + top_wire_dist + 0., HOP4Y(14), PIP_X(id_W818) + 1., HOP4Y(14)},
          {PIP_X(id_W818) + 1., HOP4Y(14), PIP_X(id_W818) + top_wire_dist + 1., HOP4Y(12)},
          {PIP_X(id_W818) + top_wire_dist + 1., HOP4Y(12), PIP_X(id_W818) + 2., HOP4Y(12)},
          {PIP_X(id_W818) + 2., HOP4Y(12), PIP_X(id_W818) + top_wire_dist + 2., HOP4Y(10)},
          {PIP_X(id_W818) + top_wire_dist + 2., HOP4Y(10), PIP_X(id_W818) + 3., HOP4Y(10)},
          {PIP_X(id_W818) + 3., HOP4Y(10), PIP_X(id_W818) + top_wire_dist + 3., HOP4Y(8)},
          {PIP_X(id_W818) + top_wire_dist + 3., HOP4Y(8), PIP_X(id_W818) + 4., HOP4Y(8)},
          {PIP_X(id_W818) + 4., HOP4Y(8), PIP_X(id_W818) + top_wire_dist + 4., HOP4Y(6)},
          {PIP_X(id_E814) + 4., HOP4Y(8), PIP_X(id_E814) + 4., WIRE_Y(0)},
          {PIP_X(id_W818) + top_wire_dist + 4., HOP4Y(6), PIP_X(id_W818) + 5., HOP4Y(6)},
          {PIP_X(id_W818) + 5., HOP4Y(6), PIP_X(id_W818) + top_wire_dist + 5., HOP4Y(4)},
          {PIP_X(id_W818) + top_wire_dist + 5., HOP4Y(4), PIP_X(id_W818) + 6., HOP4Y(4)},
          {PIP_X(id_W818) + 6., HOP4Y(4), PIP_X(id_W818) + top_wire_dist + 6., HOP4Y(2)},
          {PIP_X(id_W818) + top_wire_dist + 6., HOP4Y(2), PIP_X(id_W818) + 7., HOP4Y(2)},
          {PIP_X(id_W818) + 7., HOP4Y(2), PIP_X(id_W818) + top_wire_dist + 7., HOP4Y(0)},
          {PIP_X(id_W818) + top_wire_dist + 7., HOP4Y(0), wrap_len + 8., HOP4Y(0)},
          {wrap_len + 8., HOP4Y(0), wrap_len + 8., HOP4Y(1)},
          {wrap_len + 8., HOP4Y(1), PIP_X(id_W818) + 7., HOP4Y(1)},
          {PIP_X(id_W818) + 7., HOP4Y(1), PIP_X(id_W818) + 7., WIRE_Y(0)}}},
        {id_W81_loop0,
         {{PIP_X(id_W810), WIRE_Y(0), PIP_X(id_W810), HOP4Y(17)},
          {PIP_X(id_W810) - 0., HOP4Y(17), PIP_X(id_W810) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W810) - top_wire_dist - 0., HOP4Y(15), -wrap_len - 0., HOP4Y(15)},
          {-wrap_len - 0., HOP4Y(15), -wrap_len - 0., HOP4Y(14)},
          {-wrap_len - 0., HOP4Y(14), PIP_X(id_W818) + 0., HOP4Y(14)},
          {PIP_X(id_W818) + 0., HOP4Y(14), PIP_X(id_W818) + top_wire_dist + 0., HOP4Y(12)},
          {PIP_X(id_W818) + top_wire_dist + 0., HOP4Y(12), PIP_X(id_W818) + 1., HOP4Y(12)},
          {PIP_X(id_W818) + 1., HOP4Y(12), PIP_X(id_W818) + top_wire_dist + 1., HOP4Y(10)},
          {PIP_X(id_W818) + top_wire_dist + 1., HOP4Y(10), PIP_X(id_W818) + 2., HOP4Y(10)},
          {PIP_X(id_W818) + 2., HOP4Y(10), PIP_X(id_W818) + top_wire_dist + 2., HOP4Y(8)},
          {PIP_X(id_W818) + top_wire_dist + 2., HOP4Y(8), PIP_X(id_W818) + 3., HOP4Y(8)},
          {PIP_X(id_W818) + 3., HOP4Y(8), PIP_X(id_W818) + top_wire_dist + 3., HOP4Y(6)},
          {PIP_X(id_E814) + 3., HOP4Y(8), PIP_X(id_E814) + 3., WIRE_Y(0)},
          {PIP_X(id_W818) + top_wire_dist + 3., HOP4Y(6), PIP_X(id_W818) + 4., HOP4Y(6)},
          {PIP_X(id_W818) + 4., HOP4Y(6), PIP_X(id_W818) + top_wire_dist + 4., HOP4Y(4)},
          {PIP_X(id_W818) + top_wire_dist + 4., HOP4Y(4), PIP_X(id_W818) + 5., HOP4Y(4)},
          {PIP_X(id_W818) + 5., HOP4Y(4), PIP_X(id_W818) + top_wire_dist + 5., HOP4Y(2)},
          {PIP_X(id_W818) + top_wire_dist + 5., HOP4Y(2), PIP_X(id_W818) + 6., HOP4Y(2)},
          {PIP_X(id_W818) + 6., HOP4Y(2), PIP_X(id_W818) + top_wire_dist + 6., HOP4Y(0)},
          {PIP_X(id_W818) + top_wire_dist + 6., HOP4Y(0), PIP_X(id_E818) + 7., HOP4Y(0)},
          {PIP_X(id_E818) + 7., HOP4Y(0), PIP_X(id_E818) + 7., WIRE_Y(0)}}},
        {id_W81_loop1,
         {{PIP_X(id_W810), WIRE_Y(0), PIP_X(id_W810), HOP4Y(17)},
          {PIP_X(id_W810) - 0., HOP4Y(17), PIP_X(id_W810) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W810) - top_wire_dist - 0., HOP4Y(15), PIP_X(id_W810) - 1., HOP4Y(15)},
          {PIP_X(id_W810) - 1., HOP4Y(15), PIP_X(id_W810) - top_wire_dist - 1., HOP4Y(13)},
          {PIP_X(id_W810) - top_wire_dist - 1., HOP4Y(13), -wrap_len - 1., HOP4Y(13)},
          {-wrap_len - 1., HOP4Y(13), -wrap_len - 1., HOP4Y(12)},
          {-wrap_len - 1., HOP4Y(12), PIP_X(id_W818) - 1., HOP4Y(12)},
          {PIP_X(id_W818) - 1., HOP4Y(12), PIP_X(id_W818) + top_wire_dist - 1., HOP4Y(10)},
          {PIP_X(id_W818) + top_wire_dist - 1., HOP4Y(10), PIP_X(id_W818) + 0., HOP4Y(10)},
          {PIP_X(id_W818) + 0., HOP4Y(10), PIP_X(id_W818) + top_wire_dist + 0., HOP4Y(8)},
          {PIP_X(id_W818) + top_wire_dist + 0., HOP4Y(8), PIP_X(id_W818) + 1., HOP4Y(8)},
          {PIP_X(id_W818) + 1., HOP4Y(8), PIP_X(id_W818) + top_wire_dist + 1., HOP4Y(6)},
          {PIP_X(id_E814) + 1., HOP4Y(8), PIP_X(id_E814) + 1., WIRE_Y(0)},
          {PIP_X(id_W818) + top_wire_dist + 1., HOP4Y(6), PIP_X(id_W818) + 2., HOP4Y(6)},
          {PIP_X(id_W818) + 2., HOP4Y(6), PIP_X(id_W818) + top_wire_dist + 2., HOP4Y(4)},
          {PIP_X(id_W818) + top_wire_dist + 2., HOP4Y(4), PIP_X(id_W818) + 3., HOP4Y(4)},
          {PIP_X(id_W818) + 3., HOP4Y(4), PIP_X(id_W818) + top_wire_dist + 3., HOP4Y(2)},
          {PIP_X(id_W818) + top_wire_dist + 3., HOP4Y(2), PIP_X(id_W818) + 4., HOP4Y(2)},
          {PIP_X(id_W818) + 4., HOP4Y(2), PIP_X(id_W818) + top_wire_dist + 4., HOP4Y(0)},
          {PIP_X(id_W818) + top_wire_dist + 4., HOP4Y(0), PIP_X(id_E818) + 5., HOP4Y(0)},
          {PIP_X(id_E818) + 5., HOP4Y(0), PIP_X(id_E818) + 5., WIRE_Y(0)}}},
        {id_W81_loop2,
         {{PIP_X(id_W810), WIRE_Y(0), PIP_X(id_W810), HOP4Y(17)},
          {PIP_X(id_W810) - 0., HOP4Y(17), PIP_X(id_W810) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W810) - top_wire_dist - 0., HOP4Y(15), PIP_X(id_W810) - 1., HOP4Y(15)},
          {PIP_X(id_W810) - 1., HOP4Y(15), PIP_X(id_W810) - top_wire_dist - 1., HOP4Y(13)},
          {PIP_X(id_W810) - top_wire_dist - 1., HOP4Y(13), PIP_X(id_W810) - 2., HOP4Y(13)},
          {PIP_X(id_W810) - 2., HOP4Y(13), PIP_X(id_W810) - top_wire_dist - 2., HOP4Y(11)},
          {PIP_X(id_W810) - top_wire_dist - 2., HOP4Y(11), -wrap_len - 2., HOP4Y(11)},
          {-wrap_len - 2., HOP4Y(11), -wrap_len - 2., HOP4Y(10)},
          {-wrap_len - 2., HOP4Y(10), PIP_X(id_W818) - 2., HOP4Y(10)},
          {PIP_X(id_W818) - 2., HOP4Y(10), PIP_X(id_W818) + top_wire_dist - 2., HOP4Y(8)},
          {PIP_X(id_W818) + top_wire_dist - 2., HOP4Y(8), PIP_X(id_W818) - 1., HOP4Y(8)},
          {PIP_X(id_W818) - 1., HOP4Y(8), PIP_X(id_W818) + top_wire_dist - 1., HOP4Y(6)},
          {PIP_X(id_E814) - 1., HOP4Y(8), PIP_X(id_E814) - 1., WIRE_Y(0)},
          {PIP_X(id_W818) + top_wire_dist - 1., HOP4Y(6), PIP_X(id_W818) + 0., HOP4Y(6)},
          {PIP_X(id_W818) + 0., HOP4Y(6), PIP_X(id_W818) + top_wire_dist + 0., HOP4Y(4)},
          {PIP_X(id_W818) + top_wire_dist + 0., HOP4Y(4), PIP_X(id_W818) + 1., HOP4Y(4)},
          {PIP_X(id_W818) + 1., HOP4Y(4), PIP_X(id_W818) + top_wire_dist + 1., HOP4Y(2)},
          {PIP_X(id_W818) + top_wire_dist + 1., HOP4Y(2), PIP_X(id_W818) + 2., HOP4Y(2)},
          {PIP_X(id_W818) + 2., HOP4Y(2), PIP_X(id_W818) + top_wire_dist + 2., HOP4Y(0)},
          {PIP_X(id_W818) + top_wire_dist + 2., HOP4Y(0), PIP_X(id_E818) + 3., HOP4Y(0)},
          {PIP_X(id_E818) + 3., HOP4Y(0), PIP_X(id_E818) + 3., WIRE_Y(0)}}},
        {id_W81_loop3,
         {{PIP_X(id_W810), WIRE_Y(0), PIP_X(id_W810), HOP4Y(17)},
          {PIP_X(id_W810) - 0., HOP4Y(17), PIP_X(id_W810) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W810) - top_wire_dist - 0., HOP4Y(15), PIP_X(id_W810) - 1., HOP4Y(15)},
          {PIP_X(id_W810) - 1., HOP4Y(15), PIP_X(id_W810) - top_wire_dist - 1., HOP4Y(13)},
          {PIP_X(id_W810) - top_wire_dist - 1., HOP4Y(13), PIP_X(id_W810) - 2., HOP4Y(13)},
          {PIP_X(id_W810) - 2., HOP4Y(13), PIP_X(id_W810) - top_wire_dist - 2., HOP4Y(11)},
          {PIP_X(id_W810) - top_wire_dist - 2., HOP4Y(11), PIP_X(id_W810) - 3., HOP4Y(11)},
          {PIP_X(id_W810) - 3., HOP4Y(11), PIP_X(id_W810) - top_wire_dist - 3., HOP4Y(9)},
          {PIP_X(id_W810) - top_wire_dist - 3., HOP4Y(9), -wrap_len - 3., HOP4Y(9)},
          {-wrap_len - 3., HOP4Y(9), -wrap_len - 3., HOP4Y(8)},
          {-wrap_len - 3., HOP4Y(8), PIP_X(id_W818) - 3., HOP4Y(8)},
          {PIP_X(id_W818) - 3., HOP4Y(8), PIP_X(id_W818) + top_wire_dist - 3., HOP4Y(6)},
          {PIP_X(id_E814) - 3., HOP4Y(8), PIP_X(id_E814) - 3., WIRE_Y(0)},
          {PIP_X(id_W818) + top_wire_dist - 3., HOP4Y(6), PIP_X(id_W818) - 2., HOP4Y(6)},
          {PIP_X(id_W818) - 2., HOP4Y(6), PIP_X(id_W818) + top_wire_dist - 2., HOP4Y(4)},
          {PIP_X(id_W818) + top_wire_dist - 2., HOP4Y(4), PIP_X(id_W818) - 1., HOP4Y(4)},
          {PIP_X(id_W818) - 1., HOP4Y(4), PIP_X(id_W818) + top_wire_dist - 1., HOP4Y(2)},
          {PIP_X(id_W818) + top_wire_dist - 1., HOP4Y(2), PIP_X(id_W818) + 0., HOP4Y(2)},
          {PIP_X(id_W818) + 0., HOP4Y(2), PIP_X(id_W818) + top_wire_dist + 0., HOP4Y(0)},
          {PIP_X(id_W818) + top_wire_dist + 0., HOP4Y(0), PIP_X(id_E818) + 1., HOP4Y(0)},
          {PIP_X(id_E818) + 1., HOP4Y(0), PIP_X(id_E818) + 1., WIRE_Y(0)}}},
        {id_W81_loop4,
         {{PIP_X(id_W810), WIRE_Y(0), PIP_X(id_W810), HOP4Y(17)},
          {PIP_X(id_W810) - 0., HOP4Y(17), PIP_X(id_W810) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W810) - top_wire_dist - 0., HOP4Y(15), PIP_X(id_W810) - 1., HOP4Y(15)},
          {PIP_X(id_W810) - 1., HOP4Y(15), PIP_X(id_W810) - top_wire_dist - 1., HOP4Y(13)},
          {PIP_X(id_W810) - top_wire_dist - 1., HOP4Y(13), PIP_X(id_W810) - 2., HOP4Y(13)},
          {PIP_X(id_W810) - 2., HOP4Y(13), PIP_X(id_W810) - top_wire_dist - 2., HOP4Y(11)},
          {PIP_X(id_W810) - top_wire_dist - 2., HOP4Y(11), PIP_X(id_W810) - 3., HOP4Y(11)},
          {PIP_X(id_W810) - 3., HOP4Y(11), PIP_X(id_W810) - top_wire_dist - 3., HOP4Y(9)},
          {PIP_X(id_W810) - top_wire_dist - 3., HOP4Y(9), PIP_X(id_W810) - 4., HOP4Y(9)},
          {PIP_X(id_W810) - 4., HOP4Y(9), PIP_X(id_W810) - top_wire_dist - 4., HOP4Y(7)},
          {PIP_X(id_W810) - top_wire_dist - 4., HOP4Y(7), -wrap_len - 4., HOP4Y(7)},
          {-wrap_len - 4., HOP4Y(7), -wrap_len - 4., HOP4Y(6)},
          {PIP_X(id_W814) - 4., HOP4Y(6), PIP_X(id_W814) - 4., WIRE_Y(0)},
          {-wrap_len - 4., HOP4Y(6), PIP_X(id_W818) - 4., HOP4Y(6)},
          {PIP_X(id_W818) - 4., HOP4Y(6), PIP_X(id_W818) + top_wire_dist - 4., HOP4Y(4)},
          {PIP_X(id_W818) + top_wire_dist - 4., HOP4Y(4), PIP_X(id_W818) - 3., HOP4Y(4)},
          {PIP_X(id_W818) - 3., HOP4Y(4), PIP_X(id_W818) + top_wire_dist - 3., HOP4Y(2)},
          {PIP_X(id_W818) + top_wire_dist - 3., HOP4Y(2), PIP_X(id_W818) - 2., HOP4Y(2)},
          {PIP_X(id_W818) - 2., HOP4Y(2), PIP_X(id_W818) + top_wire_dist - 2., HOP4Y(0)},
          {PIP_X(id_W818) + top_wire_dist - 2., HOP4Y(0), PIP_X(id_E818) - 1., HOP4Y(0)},
          {PIP_X(id_E818) - 1., HOP4Y(0), PIP_X(id_E818) - 1., WIRE_Y(0)}}},
        {id_W81_loop5,
         {{PIP_X(id_W810), WIRE_Y(0), PIP_X(id_W810), HOP4Y(17)},
          {PIP_X(id_W810) - 0., HOP4Y(17), PIP_X(id_W810) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W810) - top_wire_dist - 0., HOP4Y(15), PIP_X(id_W810) - 1., HOP4Y(15)},
          {PIP_X(id_W810) - 1., HOP4Y(15), PIP_X(id_W810) - top_wire_dist - 1., HOP4Y(13)},
          {PIP_X(id_W810) - top_wire_dist - 1., HOP4Y(13), PIP_X(id_W810) - 2., HOP4Y(13)},
          {PIP_X(id_W810) - 2., HOP4Y(13), PIP_X(id_W810) - top_wire_dist - 2., HOP4Y(11)},
          {PIP_X(id_W810) - top_wire_dist - 2., HOP4Y(11), PIP_X(id_W810) - 3., HOP4Y(11)},
          {PIP_X(id_W810) - 3., HOP4Y(11), PIP_X(id_W810) - top_wire_dist - 3., HOP4Y(9)},
          {PIP_X(id_W810) - top_wire_dist - 3., HOP4Y(9), PIP_X(id_W810) - 4., HOP4Y(9)},
          {PIP_X(id_W810) - 4., HOP4Y(9), PIP_X(id_W810) - top_wire_dist - 4., HOP4Y(7)},
          {PIP_X(id_W814) - 4., HOP4Y(9), PIP_X(id_W814) - 4., WIRE_Y(0)},
          {PIP_X(id_W810) - top_wire_dist - 4., HOP4Y(7), PIP_X(id_W810) - 5., HOP4Y(7)},
          {PIP_X(id_W810) - 5., HOP4Y(7), PIP_X(id_W810) - top_wire_dist - 5., HOP4Y(5)},
          {PIP_X(id_W810) - top_wire_dist - 5., HOP4Y(5), -wrap_len - 5., HOP4Y(5)},
          {-wrap_len - 5., HOP4Y(5), -wrap_len - 5., HOP4Y(4)},
          {-wrap_len - 5., HOP4Y(4), PIP_X(id_W818) - 5., HOP4Y(4)},
          {PIP_X(id_W818) - 5., HOP4Y(4), PIP_X(id_W818) + top_wire_dist - 5., HOP4Y(2)},
          {PIP_X(id_W818) + top_wire_dist - 5., HOP4Y(2), PIP_X(id_W818) - 4., HOP4Y(2)},
          {PIP_X(id_W818) - 4., HOP4Y(2), PIP_X(id_W818) + top_wire_dist - 4., HOP4Y(0)},
          {PIP_X(id_W818) + top_wire_dist - 4., HOP4Y(0), PIP_X(id_E818) - 3., HOP4Y(0)},
          {PIP_X(id_E818) - 3., HOP4Y(0), PIP_X(id_E818) - 3., WIRE_Y(0)}}},
        {id_W81_loop6,
         {{PIP_X(id_W810), WIRE_Y(0), PIP_X(id_W810), HOP4Y(17)},
          {PIP_X(id_W810) - 0., HOP4Y(17), PIP_X(id_W810) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W810) - top_wire_dist - 0., HOP4Y(15), PIP_X(id_W810) - 1., HOP4Y(15)},
          {PIP_X(id_W810) - 1., HOP4Y(15), PIP_X(id_W810) - top_wire_dist - 1., HOP4Y(13)},
          {PIP_X(id_W810) - top_wire_dist - 1., HOP4Y(13), PIP_X(id_W810) - 2., HOP4Y(13)},
          {PIP_X(id_W810) - 2., HOP4Y(13), PIP_X(id_W810) - top_wire_dist - 2., HOP4Y(11)},
          {PIP_X(id_W810) - top_wire_dist - 2., HOP4Y(11), PIP_X(id_W810) - 3., HOP4Y(11)},
          {PIP_X(id_W810) - 3., HOP4Y(11), PIP_X(id_W810) - top_wire_dist - 3., HOP4Y(9)},
          {PIP_X(id_W810) - top_wire_dist - 3., HOP4Y(9), PIP_X(id_W810) - 4., HOP4Y(9)},
          {PIP_X(id_W810) - 4., HOP4Y(9), PIP_X(id_W810) - top_wire_dist - 4., HOP4Y(7)},
          {PIP_X(id_W814) - 4., HOP4Y(9), PIP_X(id_W814) - 4., WIRE_Y(0)},
          {PIP_X(id_W810) - top_wire_dist - 4., HOP4Y(7), PIP_X(id_W810) - 5., HOP4Y(7)},
          {PIP_X(id_W810) - 5., HOP4Y(7), PIP_X(id_W810) - top_wire_dist - 5., HOP4Y(5)},
          {PIP_X(id_W810) - top_wire_dist - 5., HOP4Y(5), PIP_X(id_W810) - 6., HOP4Y(5)},
          {PIP_X(id_W810) - 6., HOP4Y(5), PIP_X(id_W810) - top_wire_dist - 6., HOP4Y(3)},
          {PIP_X(id_W810) - top_wire_dist - 6., HOP4Y(3), -wrap_len - 6., HOP4Y(3)},
          {-wrap_len - 6., HOP4Y(3), -wrap_len - 6., HOP4Y(2)},
          {-wrap_len - 6., HOP4Y(2), PIP_X(id_W818) - 6., HOP4Y(2)},
          {PIP_X(id_W818) - 6., HOP4Y(2), PIP_X(id_W818) + top_wire_dist - 6., HOP4Y(0)},
          {PIP_X(id_W818) + top_wire_dist - 6., HOP4Y(0), PIP_X(id_E818) - 5., HOP4Y(0)},
          {PIP_X(id_E818) - 5., HOP4Y(0), PIP_X(id_E818) - 5., WIRE_Y(0)}}},
        {id_W81_loop7,
         {{PIP_X(id_W810), WIRE_Y(0), PIP_X(id_W810), HOP4Y(17)},
          {PIP_X(id_W810) - 0., HOP4Y(17), PIP_X(id_W810) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W810) - top_wire_dist - 0., HOP4Y(15), PIP_X(id_W810) - 1., HOP4Y(15)},
          {PIP_X(id_W810) - 1., HOP4Y(15), PIP_X(id_W810) - top_wire_dist - 1., HOP4Y(13)},
          {PIP_X(id_W810) - top_wire_dist - 1., HOP4Y(13), PIP_X(id_W810) - 2., HOP4Y(13)},
          {PIP_X(id_W810) - 2., HOP4Y(13), PIP_X(id_W810) - top_wire_dist - 2., HOP4Y(11)},
          {PIP_X(id_W810) - top_wire_dist - 2., HOP4Y(11), PIP_X(id_W810) - 3., HOP4Y(11)},
          {PIP_X(id_W810) - 3., HOP4Y(11), PIP_X(id_W810) - top_wire_dist - 3., HOP4Y(9)},
          {PIP_X(id_W810) - top_wire_dist - 3., HOP4Y(9), PIP_X(id_W810) - 4., HOP4Y(9)},
          {PIP_X(id_W810) - 4., HOP4Y(9), PIP_X(id_W810) - top_wire_dist - 4., HOP4Y(7)},
          {PIP_X(id_W814) - 4., HOP4Y(9), PIP_X(id_W814) - 4., WIRE_Y(0)},
          {PIP_X(id_W810) - top_wire_dist - 4., HOP4Y(7), PIP_X(id_W810) - 5., HOP4Y(7)},
          {PIP_X(id_W810) - 5., HOP4Y(7), PIP_X(id_W810) - top_wire_dist - 5., HOP4Y(5)},
          {PIP_X(id_W810) - top_wire_dist - 5., HOP4Y(5), PIP_X(id_W810) - 6., HOP4Y(5)},
          {PIP_X(id_W810) - 6., HOP4Y(5), PIP_X(id_W810) - top_wire_dist - 6., HOP4Y(3)},
          {PIP_X(id_W810) - top_wire_dist - 6., HOP4Y(3), PIP_X(id_W810) - 7., HOP4Y(3)},
          {PIP_X(id_W810) - 7., HOP4Y(3), PIP_X(id_W810) - top_wire_dist - 7., HOP4Y(1)},
          {PIP_X(id_W810) - top_wire_dist - 7., HOP4Y(1), -wrap_len - 7., HOP4Y(1)},
          {-wrap_len - 7., HOP4Y(1), -wrap_len - 7., HOP4Y(0)},
          {-wrap_len - 7., HOP4Y(0), PIP_X(id_E818) - 7., HOP4Y(0)},
          {PIP_X(id_E818) - 7., HOP4Y(0), PIP_X(id_E818) - 7., WIRE_Y(0)}}},

#undef HOP4Y
#define HOP4Y(offset) WIRE_Y((float)offset + HOP4Y_START + 18.f + 18.f)
        {id_E82,
         {{PIP_X(id_E820), WIRE_Y(0), PIP_X(id_E820), HOP4Y(16)},
          {PIP_X(id_E820), HOP4Y(16), PIP_X(id_W828), HOP4Y(16)},
          {PIP_X(id_W828) + 0., HOP4Y(16), PIP_X(id_W828) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W828) + top_wire_dist + 0., HOP4Y(14), PIP_X(id_W828) + 1., HOP4Y(14)},
          {PIP_X(id_W828) + 1., HOP4Y(14), PIP_X(id_W828) + top_wire_dist + 1., HOP4Y(12)},
          {PIP_X(id_W828) + top_wire_dist + 1., HOP4Y(12), PIP_X(id_W828) + 2., HOP4Y(12)},
          {PIP_X(id_W828) + 2., HOP4Y(12), PIP_X(id_W828) + top_wire_dist + 2., HOP4Y(10)},
          {PIP_X(id_W828) + top_wire_dist + 2., HOP4Y(10), PIP_X(id_W828) + 3., HOP4Y(10)},
          {PIP_X(id_W828) + 3., HOP4Y(10), PIP_X(id_W828) + top_wire_dist + 3., HOP4Y(8)},
          {PIP_X(id_W828) + top_wire_dist + 3., HOP4Y(8), PIP_X(id_W828) + 4., HOP4Y(8)},
          {PIP_X(id_W828) + 4., HOP4Y(8), PIP_X(id_W828) + top_wire_dist + 4., HOP4Y(6)},
          {PIP_X(id_E824) + 4., HOP4Y(8), PIP_X(id_E824) + 4., WIRE_Y(0)},
          {PIP_X(id_W828) + top_wire_dist + 4., HOP4Y(6), PIP_X(id_W828) + 5., HOP4Y(6)},
          {PIP_X(id_W828) + 5., HOP4Y(6), PIP_X(id_W828) + top_wire_dist + 5., HOP4Y(4)},
          {PIP_X(id_W828) + top_wire_dist + 5., HOP4Y(4), PIP_X(id_W828) + 6., HOP4Y(4)},
          {PIP_X(id_W828) + 6., HOP4Y(4), PIP_X(id_W828) + top_wire_dist + 6., HOP4Y(2)},
          {PIP_X(id_W828) + top_wire_dist + 6., HOP4Y(2), PIP_X(id_W828) + 7., HOP4Y(2)},
          {PIP_X(id_W828) + 7., HOP4Y(2), PIP_X(id_W828) + top_wire_dist + 7., HOP4Y(0)},
          {PIP_X(id_W828) + top_wire_dist + 7., HOP4Y(0), PIP_X(id_E828) + 8., HOP4Y(0)},
          {PIP_X(id_E828) + 8, HOP4Y(0), PIP_X(id_E828) + 8., WIRE_Y(0)}}},
        {id_W82,
         {{PIP_X(id_W820), WIRE_Y(0), PIP_X(id_W820), HOP4Y(17)},
          {PIP_X(id_W820) - 0., HOP4Y(17), PIP_X(id_W820) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W820) - top_wire_dist - 0., HOP4Y(15), PIP_X(id_W820) - 1., HOP4Y(15)},
          {PIP_X(id_W820) - 1., HOP4Y(15), PIP_X(id_W820) - top_wire_dist - 1., HOP4Y(13)},
          {PIP_X(id_W820) - top_wire_dist - 1., HOP4Y(13), PIP_X(id_W820) - 2., HOP4Y(13)},
          {PIP_X(id_W820) - 2., HOP4Y(13), PIP_X(id_W820) - top_wire_dist - 2., HOP4Y(11)},
          {PIP_X(id_W820) - top_wire_dist - 2., HOP4Y(11), PIP_X(id_W820) - 3., HOP4Y(11)},
          {PIP_X(id_W820) - 3., HOP4Y(11), PIP_X(id_W820) - top_wire_dist - 3., HOP4Y(9)},
          {PIP_X(id_W820) - top_wire_dist - 3., HOP4Y(9), PIP_X(id_W820) - 4., HOP4Y(9)},
          {PIP_X(id_W820) - 4., HOP4Y(9), PIP_X(id_W820) - top_wire_dist - 4., HOP4Y(7)},
          {PIP_X(id_W824) - 4., HOP4Y(9), PIP_X(id_W824) - 4., WIRE_Y(0)},
          {PIP_X(id_W820) - top_wire_dist - 4., HOP4Y(7), PIP_X(id_W820) - 5., HOP4Y(7)},
          {PIP_X(id_W820) - 5., HOP4Y(7), PIP_X(id_W820) - top_wire_dist - 5., HOP4Y(5)},
          {PIP_X(id_W820) - top_wire_dist - 5., HOP4Y(5), PIP_X(id_W820) - 6., HOP4Y(5)},
          {PIP_X(id_W820) - 6., HOP4Y(5), PIP_X(id_W820) - top_wire_dist - 6., HOP4Y(3)},
          {PIP_X(id_W820) - top_wire_dist - 6., HOP4Y(3), PIP_X(id_W820) - 7., HOP4Y(3)},
          {PIP_X(id_W820) - 7., HOP4Y(3), PIP_X(id_W820) - top_wire_dist - 7., HOP4Y(1)},
          {PIP_X(id_W820) - top_wire_dist - 7., HOP4Y(1), PIP_X(id_W828) - 8., HOP4Y(1)},
          {PIP_X(id_W828) - 8, HOP4Y(1), PIP_X(id_W828) - 8., WIRE_Y(0)}}},
        {id_E82_loop0,
         {{PIP_X(id_E820), WIRE_Y(0), PIP_X(id_E820), HOP4Y(16)},
          {PIP_X(id_E820), HOP4Y(16), PIP_X(id_W828), HOP4Y(16)},
          {PIP_X(id_W828) + 0., HOP4Y(16), PIP_X(id_W828) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W828) + top_wire_dist + 0., HOP4Y(14), wrap_len + 1., HOP4Y(14)},
          {wrap_len + 1., HOP4Y(14), wrap_len + 1., HOP4Y(15)},
          {wrap_len + 1., HOP4Y(15), PIP_X(id_W820) - 0., HOP4Y(15)},
          {PIP_X(id_W820) - 0., HOP4Y(15), PIP_X(id_W820) - top_wire_dist - 0., HOP4Y(13)},
          {PIP_X(id_W820) - top_wire_dist - 0., HOP4Y(13), PIP_X(id_W820) - 1., HOP4Y(13)},
          {PIP_X(id_W820) - 1., HOP4Y(13), PIP_X(id_W820) - top_wire_dist - 1., HOP4Y(11)},
          {PIP_X(id_W820) - top_wire_dist - 1., HOP4Y(11), PIP_X(id_W820) - 2., HOP4Y(11)},
          {PIP_X(id_W820) - 2., HOP4Y(11), PIP_X(id_W820) - top_wire_dist - 2., HOP4Y(9)},
          {PIP_X(id_W820) - top_wire_dist - 2., HOP4Y(9), PIP_X(id_W820) - 3., HOP4Y(9)},
          {PIP_X(id_W820) - 3., HOP4Y(9), PIP_X(id_W820) - top_wire_dist - 3., HOP4Y(7)},
          {PIP_X(id_W824) - 3., HOP4Y(9), PIP_X(id_W824) - 3., WIRE_Y(0)},
          {PIP_X(id_W820) - top_wire_dist - 3., HOP4Y(7), PIP_X(id_W820) - 4., HOP4Y(7)},
          {PIP_X(id_W820) - 4., HOP4Y(7), PIP_X(id_W820) - top_wire_dist - 4., HOP4Y(5)},
          {PIP_X(id_W820) - top_wire_dist - 4., HOP4Y(5), PIP_X(id_W820) - 5., HOP4Y(5)},
          {PIP_X(id_W820) - 5., HOP4Y(5), PIP_X(id_W820) - top_wire_dist - 5., HOP4Y(3)},
          {PIP_X(id_W820) - top_wire_dist - 5., HOP4Y(3), PIP_X(id_W820) - 6., HOP4Y(3)},
          {PIP_X(id_W820) - 6., HOP4Y(3), PIP_X(id_W820) - top_wire_dist - 6., HOP4Y(1)},
          {PIP_X(id_W820) - top_wire_dist - 6., HOP4Y(1), PIP_X(id_W828) - 7., HOP4Y(1)},
          {PIP_X(id_W828) - 7, HOP4Y(1), PIP_X(id_W828) - 7., WIRE_Y(0)}}},
        {id_E82_loop1,
         {{PIP_X(id_E820), WIRE_Y(0), PIP_X(id_E820), HOP4Y(16)},
          {PIP_X(id_E820), HOP4Y(16), PIP_X(id_W828), HOP4Y(16)},
          {PIP_X(id_W828) + 0., HOP4Y(16), PIP_X(id_W828) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W828) + top_wire_dist + 0., HOP4Y(14), PIP_X(id_W828) + 1., HOP4Y(14)},
          {PIP_X(id_W828) + 1., HOP4Y(14), PIP_X(id_W828) + top_wire_dist + 1., HOP4Y(12)},
          {PIP_X(id_W828) + top_wire_dist + 1., HOP4Y(12), wrap_len + 2., HOP4Y(12)},
          {wrap_len + 2., HOP4Y(12), wrap_len + 2., HOP4Y(13)},
          {wrap_len + 2., HOP4Y(13), PIP_X(id_W820) + 1., HOP4Y(13)},
          {PIP_X(id_W820) + 1., HOP4Y(13), PIP_X(id_W820) - top_wire_dist + 1., HOP4Y(11)},
          {PIP_X(id_W820) - top_wire_dist + 1., HOP4Y(11), PIP_X(id_W820) - 0., HOP4Y(11)},
          {PIP_X(id_W820) - 0., HOP4Y(11), PIP_X(id_W820) - top_wire_dist - 0., HOP4Y(9)},
          {PIP_X(id_W820) - top_wire_dist - 0., HOP4Y(9), PIP_X(id_W820) - 1., HOP4Y(9)},
          {PIP_X(id_W824) - 1., HOP4Y(9), PIP_X(id_W824) - 1., WIRE_Y(0)},
          {PIP_X(id_W820) - 1., HOP4Y(9), PIP_X(id_W820) - top_wire_dist - 1., HOP4Y(7)},
          {PIP_X(id_W820) - top_wire_dist - 1., HOP4Y(7), PIP_X(id_W820) - 2., HOP4Y(7)},
          {PIP_X(id_W820) - 2., HOP4Y(7), PIP_X(id_W820) - top_wire_dist - 2., HOP4Y(5)},
          {PIP_X(id_W820) - top_wire_dist - 2., HOP4Y(5), PIP_X(id_W820) - 3., HOP4Y(5)},
          {PIP_X(id_W820) - 3., HOP4Y(5), PIP_X(id_W820) - top_wire_dist - 3., HOP4Y(3)},
          {PIP_X(id_W820) - top_wire_dist - 3., HOP4Y(3), PIP_X(id_W820) - 4., HOP4Y(3)},
          {PIP_X(id_W820) - 4., HOP4Y(3), PIP_X(id_W820) - top_wire_dist - 4., HOP4Y(1)},
          {PIP_X(id_W820) - top_wire_dist - 4., HOP4Y(1), PIP_X(id_W828) - 5., HOP4Y(1)},
          {PIP_X(id_W828) - 5., HOP4Y(1), PIP_X(id_W828) - 5., WIRE_Y(0)}}},
        {id_E82_loop2,
         {{PIP_X(id_E820), WIRE_Y(0), PIP_X(id_E820), HOP4Y(16)},
          {PIP_X(id_E820), HOP4Y(16), PIP_X(id_W828), HOP4Y(16)},
          {PIP_X(id_W828) + 0., HOP4Y(16), PIP_X(id_W828) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W828) + top_wire_dist + 0., HOP4Y(14), PIP_X(id_W828) + 1., HOP4Y(14)},
          {PIP_X(id_W828) + 1., HOP4Y(14), PIP_X(id_W828) + top_wire_dist + 1., HOP4Y(12)},
          {PIP_X(id_W828) + top_wire_dist + 1., HOP4Y(12), PIP_X(id_W828) + 2., HOP4Y(12)},
          {PIP_X(id_W828) + 2., HOP4Y(12), PIP_X(id_W828) + top_wire_dist + 2., HOP4Y(10)},
          {PIP_X(id_W828) + top_wire_dist + 2., HOP4Y(10), wrap_len + 3., HOP4Y(10)},
          {wrap_len + 3., HOP4Y(10), wrap_len + 3., HOP4Y(11)},
          {wrap_len + 3., HOP4Y(11), PIP_X(id_W820) + 2., HOP4Y(11)},
          {PIP_X(id_W820) + 2., HOP4Y(11), PIP_X(id_W820) - top_wire_dist + 2., HOP4Y(9)},
          {PIP_X(id_W820) - top_wire_dist + 2., HOP4Y(9), PIP_X(id_W820) + 1., HOP4Y(9)},
          {PIP_X(id_W824) + 1., HOP4Y(9), PIP_X(id_W824) + 1., WIRE_Y(0)},
          {PIP_X(id_W820) + 1., HOP4Y(9), PIP_X(id_W820) - top_wire_dist + 1., HOP4Y(7)},
          {PIP_X(id_W820) - top_wire_dist + 1., HOP4Y(7), PIP_X(id_W820) + 0., HOP4Y(7)},
          {PIP_X(id_W820) + 0., HOP4Y(7), PIP_X(id_W820) - top_wire_dist - 0., HOP4Y(5)},
          {PIP_X(id_W820) - top_wire_dist - 0., HOP4Y(5), PIP_X(id_W820) - 1., HOP4Y(5)},
          {PIP_X(id_W820) - 1., HOP4Y(5), PIP_X(id_W820) - top_wire_dist - 1., HOP4Y(3)},
          {PIP_X(id_W820) - top_wire_dist - 1., HOP4Y(3), PIP_X(id_W820) - 2., HOP4Y(3)},
          {PIP_X(id_W820) - 2., HOP4Y(3), PIP_X(id_W820) - top_wire_dist - 2., HOP4Y(1)},
          {PIP_X(id_W820) - top_wire_dist - 2., HOP4Y(1), PIP_X(id_W828) - 3., HOP4Y(1)},
          {PIP_X(id_W828) - 3., HOP4Y(1), PIP_X(id_W828) - 3., WIRE_Y(0)}}},
        {id_E82_loop3,
         {{PIP_X(id_E820), WIRE_Y(0), PIP_X(id_E820), HOP4Y(16)},
          {PIP_X(id_E820), HOP4Y(16), PIP_X(id_W828), HOP4Y(16)},
          {PIP_X(id_W828) + 0., HOP4Y(16), PIP_X(id_W828) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W828) + top_wire_dist + 0., HOP4Y(14), PIP_X(id_W828) + 1., HOP4Y(14)},
          {PIP_X(id_W828) + 1., HOP4Y(14), PIP_X(id_W828) + top_wire_dist + 1., HOP4Y(12)},
          {PIP_X(id_W828) + top_wire_dist + 1., HOP4Y(12), PIP_X(id_W828) + 2., HOP4Y(12)},
          {PIP_X(id_W828) + 2., HOP4Y(12), PIP_X(id_W828) + top_wire_dist + 2., HOP4Y(10)},
          {PIP_X(id_W828) + top_wire_dist + 2., HOP4Y(10), PIP_X(id_W828) + 3., HOP4Y(10)},
          {PIP_X(id_W828) + 3., HOP4Y(10), PIP_X(id_W828) + top_wire_dist + 3., HOP4Y(8)},
          {PIP_X(id_W828) + top_wire_dist + 3., HOP4Y(8), wrap_len + 4., HOP4Y(8)},
          {wrap_len + 4., HOP4Y(8), wrap_len + 4., HOP4Y(9)},
          {wrap_len + 4., HOP4Y(9), PIP_X(id_W820) + 3., HOP4Y(9)},
          {PIP_X(id_W824) + 3., HOP4Y(9), PIP_X(id_W824) + 3., WIRE_Y(0)},
          {PIP_X(id_W820) + 3., HOP4Y(9), PIP_X(id_W820) - top_wire_dist + 3., HOP4Y(7)},
          {PIP_X(id_W820) - top_wire_dist + 3., HOP4Y(7), PIP_X(id_W820) + 2., HOP4Y(7)},
          {PIP_X(id_W820) + 2., HOP4Y(7), PIP_X(id_W820) - top_wire_dist + 2., HOP4Y(5)},
          {PIP_X(id_W820) - top_wire_dist + 2., HOP4Y(5), PIP_X(id_W820) + 1., HOP4Y(5)},
          {PIP_X(id_W820) + 1., HOP4Y(5), PIP_X(id_W820) - top_wire_dist + 1., HOP4Y(3)},
          {PIP_X(id_W820) - top_wire_dist + 1., HOP4Y(3), PIP_X(id_W820) - 0., HOP4Y(3)},
          {PIP_X(id_W820) - 0., HOP4Y(3), PIP_X(id_W820) - top_wire_dist - 0., HOP4Y(1)},
          {PIP_X(id_W820) - top_wire_dist - 0., HOP4Y(1), PIP_X(id_W828) - 1., HOP4Y(1)},
          {PIP_X(id_W828) - 1., HOP4Y(1), PIP_X(id_W828) - 1., WIRE_Y(0)}}},
        {id_E82_loop4,
         {{PIP_X(id_E820), WIRE_Y(0), PIP_X(id_E820), HOP4Y(16)},
          {PIP_X(id_E820), HOP4Y(16), PIP_X(id_W828), HOP4Y(16)},
          {PIP_X(id_W828) + 0., HOP4Y(16), PIP_X(id_W828) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W828) + top_wire_dist + 0., HOP4Y(14), PIP_X(id_W828) + 1., HOP4Y(14)},
          {PIP_X(id_W828) + 1., HOP4Y(14), PIP_X(id_W828) + top_wire_dist + 1., HOP4Y(12)},
          {PIP_X(id_W828) + top_wire_dist + 1., HOP4Y(12), PIP_X(id_W828) + 2., HOP4Y(12)},
          {PIP_X(id_W828) + 2., HOP4Y(12), PIP_X(id_W828) + top_wire_dist + 2., HOP4Y(10)},
          {PIP_X(id_W828) + top_wire_dist + 2., HOP4Y(10), PIP_X(id_W828) + 3., HOP4Y(10)},
          {PIP_X(id_W828) + 3., HOP4Y(10), PIP_X(id_W828) + top_wire_dist + 3., HOP4Y(8)},
          {PIP_X(id_W828) + top_wire_dist + 3., HOP4Y(8), PIP_X(id_W828) + 4., HOP4Y(8)},
          {PIP_X(id_E824) + 4., HOP4Y(8), PIP_X(id_E824) + 4., WIRE_Y(0)},
          {PIP_X(id_W828) + 4., HOP4Y(8), PIP_X(id_W828) + top_wire_dist + 4., HOP4Y(6)},
          {PIP_X(id_W828) + top_wire_dist + 4., HOP4Y(6), wrap_len + 5., HOP4Y(6)},
          {wrap_len + 5., HOP4Y(6), wrap_len + 5., HOP4Y(7)},
          {wrap_len + 5., HOP4Y(7), PIP_X(id_W820) + 4., HOP4Y(7)},
          {PIP_X(id_W820) + 4., HOP4Y(7), PIP_X(id_W820) - top_wire_dist + 4., HOP4Y(5)},
          {PIP_X(id_W820) - top_wire_dist + 4., HOP4Y(5), PIP_X(id_W820) + 3., HOP4Y(5)},
          {PIP_X(id_W820) + 3., HOP4Y(5), PIP_X(id_W820) - top_wire_dist + 3., HOP4Y(3)},
          {PIP_X(id_W820) - top_wire_dist + 3., HOP4Y(3), PIP_X(id_W820) + 2., HOP4Y(3)},
          {PIP_X(id_W820) + 2., HOP4Y(3), PIP_X(id_W820) - top_wire_dist + 2., HOP4Y(1)},
          {PIP_X(id_W820) - top_wire_dist + 2., HOP4Y(1), PIP_X(id_W828) + 1., HOP4Y(1)},
          {PIP_X(id_W828) + 1., HOP4Y(1), PIP_X(id_W828) + 1., WIRE_Y(0)}}},
        {id_E82_loop5,
         {{PIP_X(id_E820), WIRE_Y(0), PIP_X(id_E820), HOP4Y(16)},
          {PIP_X(id_E820), HOP4Y(16), PIP_X(id_W828), HOP4Y(16)},
          {PIP_X(id_W828) + 0., HOP4Y(16), PIP_X(id_W828) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W828) + top_wire_dist + 0., HOP4Y(14), PIP_X(id_W828) + 1., HOP4Y(14)},
          {PIP_X(id_W828) + 1., HOP4Y(14), PIP_X(id_W828) + top_wire_dist + 1., HOP4Y(12)},
          {PIP_X(id_W828) + top_wire_dist + 1., HOP4Y(12), PIP_X(id_W828) + 2., HOP4Y(12)},
          {PIP_X(id_W828) + 2., HOP4Y(12), PIP_X(id_W828) + top_wire_dist + 2., HOP4Y(10)},
          {PIP_X(id_W828) + top_wire_dist + 2., HOP4Y(10), PIP_X(id_W828) + 3., HOP4Y(10)},
          {PIP_X(id_W828) + 3., HOP4Y(10), PIP_X(id_W828) + top_wire_dist + 3., HOP4Y(8)},
          {PIP_X(id_W828) + top_wire_dist + 3., HOP4Y(8), PIP_X(id_W828) + 4., HOP4Y(8)},
          {PIP_X(id_E824) + 4., HOP4Y(8), PIP_X(id_E824) + 4., WIRE_Y(0)},
          {PIP_X(id_W828) + 4., HOP4Y(8), PIP_X(id_W828) + top_wire_dist + 4., HOP4Y(6)},
          {PIP_X(id_W828) + top_wire_dist + 4., HOP4Y(6), PIP_X(id_W828) + 5., HOP4Y(6)},
          {PIP_X(id_W828) + 5., HOP4Y(6), PIP_X(id_W828) + top_wire_dist + 5., HOP4Y(4)},
          {PIP_X(id_W828) + top_wire_dist + 5., HOP4Y(4), wrap_len + 6., HOP4Y(4)},
          {wrap_len + 6., HOP4Y(4), wrap_len + 6., HOP4Y(5)},
          {wrap_len + 6., HOP4Y(5), PIP_X(id_W820) + 5., HOP4Y(5)},
          {PIP_X(id_W820) + 5., HOP4Y(5), PIP_X(id_W820) - top_wire_dist + 5., HOP4Y(3)},
          {PIP_X(id_W820) - top_wire_dist + 5., HOP4Y(3), PIP_X(id_W820) + 4., HOP4Y(3)},
          {PIP_X(id_W820) + 4., HOP4Y(3), PIP_X(id_W820) - top_wire_dist + 4., HOP4Y(1)},
          {PIP_X(id_W820) - top_wire_dist + 4., HOP4Y(1), PIP_X(id_W828) + 3., HOP4Y(1)},
          {PIP_X(id_W828) + 3., HOP4Y(1), PIP_X(id_W828) + 3., WIRE_Y(0)}}},
        {id_E82_loop6,
         {{PIP_X(id_E820), WIRE_Y(0), PIP_X(id_E820), HOP4Y(16)},
          {PIP_X(id_E820), HOP4Y(16), PIP_X(id_W828), HOP4Y(16)},
          {PIP_X(id_W828) + 0., HOP4Y(16), PIP_X(id_W828) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W828) + top_wire_dist + 0., HOP4Y(14), PIP_X(id_W828) + 1., HOP4Y(14)},
          {PIP_X(id_W828) + 1., HOP4Y(14), PIP_X(id_W828) + top_wire_dist + 1., HOP4Y(12)},
          {PIP_X(id_W828) + top_wire_dist + 1., HOP4Y(12), PIP_X(id_W828) + 2., HOP4Y(12)},
          {PIP_X(id_W828) + 2., HOP4Y(12), PIP_X(id_W828) + top_wire_dist + 2., HOP4Y(10)},
          {PIP_X(id_W828) + top_wire_dist + 2., HOP4Y(10), PIP_X(id_W828) + 3., HOP4Y(10)},
          {PIP_X(id_W828) + 3., HOP4Y(10), PIP_X(id_W828) + top_wire_dist + 3., HOP4Y(8)},
          {PIP_X(id_W828) + top_wire_dist + 3., HOP4Y(8), PIP_X(id_W828) + 4., HOP4Y(8)},
          {PIP_X(id_W828) + 4., HOP4Y(8), PIP_X(id_W828) + top_wire_dist + 4., HOP4Y(6)},
          {PIP_X(id_E824) + 4., HOP4Y(8), PIP_X(id_E824) + 4., WIRE_Y(0)},
          {PIP_X(id_W828) + top_wire_dist + 4., HOP4Y(6), PIP_X(id_W828) + 5., HOP4Y(6)},
          {PIP_X(id_W828) + 5., HOP4Y(6), PIP_X(id_W828) + top_wire_dist + 5., HOP4Y(4)},
          {PIP_X(id_W828) + top_wire_dist + 5., HOP4Y(4), PIP_X(id_W828) + 6., HOP4Y(4)},
          {PIP_X(id_W828) + 6., HOP4Y(4), PIP_X(id_W828) + top_wire_dist + 6., HOP4Y(2)},
          {PIP_X(id_W828) + top_wire_dist + 6., HOP4Y(2), wrap_len + 7., HOP4Y(2)},
          {wrap_len + 7., HOP4Y(2), wrap_len + 7., HOP4Y(3)},
          {wrap_len + 7., HOP4Y(3), PIP_X(id_W820) + 6., HOP4Y(3)},
          {PIP_X(id_W820) + 6., HOP4Y(3), PIP_X(id_W820) - top_wire_dist + 6., HOP4Y(1)},
          {PIP_X(id_W820) - top_wire_dist + 6., HOP4Y(1), PIP_X(id_W828) + 5., HOP4Y(1)},
          {PIP_X(id_W828) + 5., HOP4Y(1), PIP_X(id_W828) + 5., WIRE_Y(0)}}},
        {id_E82_loop7,
         {{PIP_X(id_E820), WIRE_Y(0), PIP_X(id_E820), HOP4Y(16)},
          {PIP_X(id_E820), HOP4Y(16), PIP_X(id_W828), HOP4Y(16)},
          {PIP_X(id_W828) + 0., HOP4Y(16), PIP_X(id_W828) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W828) + top_wire_dist + 0., HOP4Y(14), PIP_X(id_W828) + 1., HOP4Y(14)},
          {PIP_X(id_W828) + 1., HOP4Y(14), PIP_X(id_W828) + top_wire_dist + 1., HOP4Y(12)},
          {PIP_X(id_W828) + top_wire_dist + 1., HOP4Y(12), PIP_X(id_W828) + 2., HOP4Y(12)},
          {PIP_X(id_W828) + 2., HOP4Y(12), PIP_X(id_W828) + top_wire_dist + 2., HOP4Y(10)},
          {PIP_X(id_W828) + top_wire_dist + 2., HOP4Y(10), PIP_X(id_W828) + 3., HOP4Y(10)},
          {PIP_X(id_W828) + 3., HOP4Y(10), PIP_X(id_W828) + top_wire_dist + 3., HOP4Y(8)},
          {PIP_X(id_W828) + top_wire_dist + 3., HOP4Y(8), PIP_X(id_W828) + 4., HOP4Y(8)},
          {PIP_X(id_W828) + 4., HOP4Y(8), PIP_X(id_W828) + top_wire_dist + 4., HOP4Y(6)},
          {PIP_X(id_E824) + 4., HOP4Y(8), PIP_X(id_E824) + 4., WIRE_Y(0)},
          {PIP_X(id_W828) + top_wire_dist + 4., HOP4Y(6), PIP_X(id_W828) + 5., HOP4Y(6)},
          {PIP_X(id_W828) + 5., HOP4Y(6), PIP_X(id_W828) + top_wire_dist + 5., HOP4Y(4)},
          {PIP_X(id_W828) + top_wire_dist + 5., HOP4Y(4), PIP_X(id_W828) + 6., HOP4Y(4)},
          {PIP_X(id_W828) + 6., HOP4Y(4), PIP_X(id_W828) + top_wire_dist + 6., HOP4Y(2)},
          {PIP_X(id_W828) + top_wire_dist + 6., HOP4Y(2), PIP_X(id_W828) + 7., HOP4Y(2)},
          {PIP_X(id_W828) + 7., HOP4Y(2), PIP_X(id_W828) + top_wire_dist + 7., HOP4Y(0)},
          {PIP_X(id_W828) + top_wire_dist + 7., HOP4Y(0), wrap_len + 8., HOP4Y(0)},
          {wrap_len + 8., HOP4Y(0), wrap_len + 8., HOP4Y(1)},
          {wrap_len + 8., HOP4Y(1), PIP_X(id_W828) + 7., HOP4Y(1)},
          {PIP_X(id_W828) + 7., HOP4Y(1), PIP_X(id_W828) + 7., WIRE_Y(0)}}},
        {id_W82_loop0,
         {{PIP_X(id_W820), WIRE_Y(0), PIP_X(id_W820), HOP4Y(17)},
          {PIP_X(id_W820) - 0., HOP4Y(17), PIP_X(id_W820) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W820) - top_wire_dist - 0., HOP4Y(15), -wrap_len - 0., HOP4Y(15)},
          {-wrap_len - 0., HOP4Y(15), -wrap_len - 0., HOP4Y(14)},
          {-wrap_len - 0., HOP4Y(14), PIP_X(id_W828) + 0., HOP4Y(14)},
          {PIP_X(id_W828) + 0., HOP4Y(14), PIP_X(id_W828) + top_wire_dist + 0., HOP4Y(12)},
          {PIP_X(id_W828) + top_wire_dist + 0., HOP4Y(12), PIP_X(id_W828) + 1., HOP4Y(12)},
          {PIP_X(id_W828) + 1., HOP4Y(12), PIP_X(id_W828) + top_wire_dist + 1., HOP4Y(10)},
          {PIP_X(id_W828) + top_wire_dist + 1., HOP4Y(10), PIP_X(id_W828) + 2., HOP4Y(10)},
          {PIP_X(id_W828) + 2., HOP4Y(10), PIP_X(id_W828) + top_wire_dist + 2., HOP4Y(8)},
          {PIP_X(id_W828) + top_wire_dist + 2., HOP4Y(8), PIP_X(id_W828) + 3., HOP4Y(8)},
          {PIP_X(id_W828) + 3., HOP4Y(8), PIP_X(id_W828) + top_wire_dist + 3., HOP4Y(6)},
          {PIP_X(id_E824) + 3., HOP4Y(8), PIP_X(id_E824) + 3., WIRE_Y(0)},
          {PIP_X(id_W828) + top_wire_dist + 3., HOP4Y(6), PIP_X(id_W828) + 4., HOP4Y(6)},
          {PIP_X(id_W828) + 4., HOP4Y(6), PIP_X(id_W828) + top_wire_dist + 4., HOP4Y(4)},
          {PIP_X(id_W828) + top_wire_dist + 4., HOP4Y(4), PIP_X(id_W828) + 5., HOP4Y(4)},
          {PIP_X(id_W828) + 5., HOP4Y(4), PIP_X(id_W828) + top_wire_dist + 5., HOP4Y(2)},
          {PIP_X(id_W828) + top_wire_dist + 5., HOP4Y(2), PIP_X(id_W828) + 6., HOP4Y(2)},
          {PIP_X(id_W828) + 6., HOP4Y(2), PIP_X(id_W828) + top_wire_dist + 6., HOP4Y(0)},
          {PIP_X(id_W828) + top_wire_dist + 6., HOP4Y(0), PIP_X(id_E828) + 7., HOP4Y(0)},
          {PIP_X(id_E828) + 7., HOP4Y(0), PIP_X(id_E828) + 7., WIRE_Y(0)}}},
        {id_W82_loop1,
         {{PIP_X(id_W820), WIRE_Y(0), PIP_X(id_W820), HOP4Y(17)},
          {PIP_X(id_W820) - 0., HOP4Y(17), PIP_X(id_W820) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W820) - top_wire_dist - 0., HOP4Y(15), PIP_X(id_W820) - 1., HOP4Y(15)},
          {PIP_X(id_W820) - 1., HOP4Y(15), PIP_X(id_W820) - top_wire_dist - 1., HOP4Y(13)},
          {PIP_X(id_W820) - top_wire_dist - 1., HOP4Y(13), -wrap_len - 1., HOP4Y(13)},
          {-wrap_len - 1., HOP4Y(13), -wrap_len - 1., HOP4Y(12)},
          {-wrap_len - 1., HOP4Y(12), PIP_X(id_W828) - 1., HOP4Y(12)},
          {PIP_X(id_W828) - 1., HOP4Y(12), PIP_X(id_W828) + top_wire_dist - 1., HOP4Y(10)},
          {PIP_X(id_W828) + top_wire_dist - 1., HOP4Y(10), PIP_X(id_W828) + 0., HOP4Y(10)},
          {PIP_X(id_W828) + 0., HOP4Y(10), PIP_X(id_W828) + top_wire_dist + 0., HOP4Y(8)},
          {PIP_X(id_W828) + top_wire_dist + 0., HOP4Y(8), PIP_X(id_W828) + 1., HOP4Y(8)},
          {PIP_X(id_W828) + 1., HOP4Y(8), PIP_X(id_W828) + top_wire_dist + 1., HOP4Y(6)},
          {PIP_X(id_E824) + 1., HOP4Y(8), PIP_X(id_E824) + 1., WIRE_Y(0)},
          {PIP_X(id_W828) + top_wire_dist + 1., HOP4Y(6), PIP_X(id_W828) + 2., HOP4Y(6)},
          {PIP_X(id_W828) + 2., HOP4Y(6), PIP_X(id_W828) + top_wire_dist + 2., HOP4Y(4)},
          {PIP_X(id_W828) + top_wire_dist + 2., HOP4Y(4), PIP_X(id_W828) + 3., HOP4Y(4)},
          {PIP_X(id_W828) + 3., HOP4Y(4), PIP_X(id_W828) + top_wire_dist + 3., HOP4Y(2)},
          {PIP_X(id_W828) + top_wire_dist + 3., HOP4Y(2), PIP_X(id_W828) + 4., HOP4Y(2)},
          {PIP_X(id_W828) + 4., HOP4Y(2), PIP_X(id_W828) + top_wire_dist + 4., HOP4Y(0)},
          {PIP_X(id_W828) + top_wire_dist + 4., HOP4Y(0), PIP_X(id_E828) + 5., HOP4Y(0)},
          {PIP_X(id_E828) + 5., HOP4Y(0), PIP_X(id_E828) + 5., WIRE_Y(0)}}},
        {id_W82_loop2,
         {{PIP_X(id_W820), WIRE_Y(0), PIP_X(id_W820), HOP4Y(17)},
          {PIP_X(id_W820) - 0., HOP4Y(17), PIP_X(id_W820) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W820) - top_wire_dist - 0., HOP4Y(15), PIP_X(id_W820) - 1., HOP4Y(15)},
          {PIP_X(id_W820) - 1., HOP4Y(15), PIP_X(id_W820) - top_wire_dist - 1., HOP4Y(13)},
          {PIP_X(id_W820) - top_wire_dist - 1., HOP4Y(13), PIP_X(id_W820) - 2., HOP4Y(13)},
          {PIP_X(id_W820) - 2., HOP4Y(13), PIP_X(id_W820) - top_wire_dist - 2., HOP4Y(11)},
          {PIP_X(id_W820) - top_wire_dist - 2., HOP4Y(11), -wrap_len - 2., HOP4Y(11)},
          {-wrap_len - 2., HOP4Y(11), -wrap_len - 2., HOP4Y(10)},
          {-wrap_len - 2., HOP4Y(10), PIP_X(id_W828) - 2., HOP4Y(10)},
          {PIP_X(id_W828) - 2., HOP4Y(10), PIP_X(id_W828) + top_wire_dist - 2., HOP4Y(8)},
          {PIP_X(id_W828) + top_wire_dist - 2., HOP4Y(8), PIP_X(id_W828) - 1., HOP4Y(8)},
          {PIP_X(id_W828) - 1., HOP4Y(8), PIP_X(id_W828) + top_wire_dist - 1., HOP4Y(6)},
          {PIP_X(id_E824) - 1., HOP4Y(8), PIP_X(id_E824) - 1., WIRE_Y(0)},
          {PIP_X(id_W828) + top_wire_dist - 1., HOP4Y(6), PIP_X(id_W828) + 0., HOP4Y(6)},
          {PIP_X(id_W828) + 0., HOP4Y(6), PIP_X(id_W828) + top_wire_dist + 0., HOP4Y(4)},
          {PIP_X(id_W828) + top_wire_dist + 0., HOP4Y(4), PIP_X(id_W828) + 1., HOP4Y(4)},
          {PIP_X(id_W828) + 1., HOP4Y(4), PIP_X(id_W828) + top_wire_dist + 1., HOP4Y(2)},
          {PIP_X(id_W828) + top_wire_dist + 1., HOP4Y(2), PIP_X(id_W828) + 2., HOP4Y(2)},
          {PIP_X(id_W828) + 2., HOP4Y(2), PIP_X(id_W828) + top_wire_dist + 2., HOP4Y(0)},
          {PIP_X(id_W828) + top_wire_dist + 2., HOP4Y(0), PIP_X(id_E828) + 3., HOP4Y(0)},
          {PIP_X(id_E828) + 3., HOP4Y(0), PIP_X(id_E828) + 3., WIRE_Y(0)}}},
        {id_W82_loop3,
         {{PIP_X(id_W820), WIRE_Y(0), PIP_X(id_W820), HOP4Y(17)},
          {PIP_X(id_W820) - 0., HOP4Y(17), PIP_X(id_W820) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W820) - top_wire_dist - 0., HOP4Y(15), PIP_X(id_W820) - 1., HOP4Y(15)},
          {PIP_X(id_W820) - 1., HOP4Y(15), PIP_X(id_W820) - top_wire_dist - 1., HOP4Y(13)},
          {PIP_X(id_W820) - top_wire_dist - 1., HOP4Y(13), PIP_X(id_W820) - 2., HOP4Y(13)},
          {PIP_X(id_W820) - 2., HOP4Y(13), PIP_X(id_W820) - top_wire_dist - 2., HOP4Y(11)},
          {PIP_X(id_W820) - top_wire_dist - 2., HOP4Y(11), PIP_X(id_W820) - 3., HOP4Y(11)},
          {PIP_X(id_W820) - 3., HOP4Y(11), PIP_X(id_W820) - top_wire_dist - 3., HOP4Y(9)},
          {PIP_X(id_W820) - top_wire_dist - 3., HOP4Y(9), -wrap_len - 3., HOP4Y(9)},
          {-wrap_len - 3., HOP4Y(9), -wrap_len - 3., HOP4Y(8)},
          {-wrap_len - 3., HOP4Y(8), PIP_X(id_W828) - 3., HOP4Y(8)},
          {PIP_X(id_W828) - 3., HOP4Y(8), PIP_X(id_W828) + top_wire_dist - 3., HOP4Y(6)},
          {PIP_X(id_E824) - 3., HOP4Y(8), PIP_X(id_E824) - 3., WIRE_Y(0)},
          {PIP_X(id_W828) + top_wire_dist - 3., HOP4Y(6), PIP_X(id_W828) - 2., HOP4Y(6)},
          {PIP_X(id_W828) - 2., HOP4Y(6), PIP_X(id_W828) + top_wire_dist - 2., HOP4Y(4)},
          {PIP_X(id_W828) + top_wire_dist - 2., HOP4Y(4), PIP_X(id_W828) - 1., HOP4Y(4)},
          {PIP_X(id_W828) - 1., HOP4Y(4), PIP_X(id_W828) + top_wire_dist - 1., HOP4Y(2)},
          {PIP_X(id_W828) + top_wire_dist - 1., HOP4Y(2), PIP_X(id_W828) + 0., HOP4Y(2)},
          {PIP_X(id_W828) + 0., HOP4Y(2), PIP_X(id_W828) + top_wire_dist + 0., HOP4Y(0)},
          {PIP_X(id_W828) + top_wire_dist + 0., HOP4Y(0), PIP_X(id_E828) + 1., HOP4Y(0)},
          {PIP_X(id_E828) + 1., HOP4Y(0), PIP_X(id_E828) + 1., WIRE_Y(0)}}},
        {id_W82_loop4,
         {{PIP_X(id_W820), WIRE_Y(0), PIP_X(id_W820), HOP4Y(17)},
          {PIP_X(id_W820) - 0., HOP4Y(17), PIP_X(id_W820) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W820) - top_wire_dist - 0., HOP4Y(15), PIP_X(id_W820) - 1., HOP4Y(15)},
          {PIP_X(id_W820) - 1., HOP4Y(15), PIP_X(id_W820) - top_wire_dist - 1., HOP4Y(13)},
          {PIP_X(id_W820) - top_wire_dist - 1., HOP4Y(13), PIP_X(id_W820) - 2., HOP4Y(13)},
          {PIP_X(id_W820) - 2., HOP4Y(13), PIP_X(id_W820) - top_wire_dist - 2., HOP4Y(11)},
          {PIP_X(id_W820) - top_wire_dist - 2., HOP4Y(11), PIP_X(id_W820) - 3., HOP4Y(11)},
          {PIP_X(id_W820) - 3., HOP4Y(11), PIP_X(id_W820) - top_wire_dist - 3., HOP4Y(9)},
          {PIP_X(id_W820) - top_wire_dist - 3., HOP4Y(9), PIP_X(id_W820) - 4., HOP4Y(9)},
          {PIP_X(id_W820) - 4., HOP4Y(9), PIP_X(id_W820) - top_wire_dist - 4., HOP4Y(7)},
          {PIP_X(id_W820) - top_wire_dist - 4., HOP4Y(7), -wrap_len - 4., HOP4Y(7)},
          {-wrap_len - 4., HOP4Y(7), -wrap_len - 4., HOP4Y(6)},
          {PIP_X(id_W824) - 4., HOP4Y(6), PIP_X(id_W824) - 4., WIRE_Y(0)},
          {-wrap_len - 4., HOP4Y(6), PIP_X(id_W828) - 4., HOP4Y(6)},
          {PIP_X(id_W828) - 4., HOP4Y(6), PIP_X(id_W828) + top_wire_dist - 4., HOP4Y(4)},
          {PIP_X(id_W828) + top_wire_dist - 4., HOP4Y(4), PIP_X(id_W828) - 3., HOP4Y(4)},
          {PIP_X(id_W828) - 3., HOP4Y(4), PIP_X(id_W828) + top_wire_dist - 3., HOP4Y(2)},
          {PIP_X(id_W828) + top_wire_dist - 3., HOP4Y(2), PIP_X(id_W828) - 2., HOP4Y(2)},
          {PIP_X(id_W828) - 2., HOP4Y(2), PIP_X(id_W828) + top_wire_dist - 2., HOP4Y(0)},
          {PIP_X(id_W828) + top_wire_dist - 2., HOP4Y(0), PIP_X(id_E828) - 1., HOP4Y(0)},
          {PIP_X(id_E828) - 1., HOP4Y(0), PIP_X(id_E828) - 1., WIRE_Y(0)}}},
        {id_W82_loop5,
         {{PIP_X(id_W820), WIRE_Y(0), PIP_X(id_W820), HOP4Y(17)},
          {PIP_X(id_W820) - 0., HOP4Y(17), PIP_X(id_W820) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W820) - top_wire_dist - 0., HOP4Y(15), PIP_X(id_W820) - 1., HOP4Y(15)},
          {PIP_X(id_W820) - 1., HOP4Y(15), PIP_X(id_W820) - top_wire_dist - 1., HOP4Y(13)},
          {PIP_X(id_W820) - top_wire_dist - 1., HOP4Y(13), PIP_X(id_W820) - 2., HOP4Y(13)},
          {PIP_X(id_W820) - 2., HOP4Y(13), PIP_X(id_W820) - top_wire_dist - 2., HOP4Y(11)},
          {PIP_X(id_W820) - top_wire_dist - 2., HOP4Y(11), PIP_X(id_W820) - 3., HOP4Y(11)},
          {PIP_X(id_W820) - 3., HOP4Y(11), PIP_X(id_W820) - top_wire_dist - 3., HOP4Y(9)},
          {PIP_X(id_W820) - top_wire_dist - 3., HOP4Y(9), PIP_X(id_W820) - 4., HOP4Y(9)},
          {PIP_X(id_W820) - 4., HOP4Y(9), PIP_X(id_W820) - top_wire_dist - 4., HOP4Y(7)},
          {PIP_X(id_W824) - 4., HOP4Y(9), PIP_X(id_W824) - 4., WIRE_Y(0)},
          {PIP_X(id_W820) - top_wire_dist - 4., HOP4Y(7), PIP_X(id_W820) - 5., HOP4Y(7)},
          {PIP_X(id_W820) - 5., HOP4Y(7), PIP_X(id_W820) - top_wire_dist - 5., HOP4Y(5)},
          {PIP_X(id_W820) - top_wire_dist - 5., HOP4Y(5), -wrap_len - 5., HOP4Y(5)},
          {-wrap_len - 5., HOP4Y(5), -wrap_len - 5., HOP4Y(4)},
          {-wrap_len - 5., HOP4Y(4), PIP_X(id_W828) - 5., HOP4Y(4)},
          {PIP_X(id_W828) - 5., HOP4Y(4), PIP_X(id_W828) + top_wire_dist - 5., HOP4Y(2)},
          {PIP_X(id_W828) + top_wire_dist - 5., HOP4Y(2), PIP_X(id_W828) - 4., HOP4Y(2)},
          {PIP_X(id_W828) - 4., HOP4Y(2), PIP_X(id_W828) + top_wire_dist - 4., HOP4Y(0)},
          {PIP_X(id_W828) + top_wire_dist - 4., HOP4Y(0), PIP_X(id_E828) - 3., HOP4Y(0)},
          {PIP_X(id_E828) - 3., HOP4Y(0), PIP_X(id_E828) - 3., WIRE_Y(0)}}},
        {id_W82_loop6,
         {{PIP_X(id_W820), WIRE_Y(0), PIP_X(id_W820), HOP4Y(17)},
          {PIP_X(id_W820) - 0., HOP4Y(17), PIP_X(id_W820) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W820) - top_wire_dist - 0., HOP4Y(15), PIP_X(id_W820) - 1., HOP4Y(15)},
          {PIP_X(id_W820) - 1., HOP4Y(15), PIP_X(id_W820) - top_wire_dist - 1., HOP4Y(13)},
          {PIP_X(id_W820) - top_wire_dist - 1., HOP4Y(13), PIP_X(id_W820) - 2., HOP4Y(13)},
          {PIP_X(id_W820) - 2., HOP4Y(13), PIP_X(id_W820) - top_wire_dist - 2., HOP4Y(11)},
          {PIP_X(id_W820) - top_wire_dist - 2., HOP4Y(11), PIP_X(id_W820) - 3., HOP4Y(11)},
          {PIP_X(id_W820) - 3., HOP4Y(11), PIP_X(id_W820) - top_wire_dist - 3., HOP4Y(9)},
          {PIP_X(id_W820) - top_wire_dist - 3., HOP4Y(9), PIP_X(id_W820) - 4., HOP4Y(9)},
          {PIP_X(id_W820) - 4., HOP4Y(9), PIP_X(id_W820) - top_wire_dist - 4., HOP4Y(7)},
          {PIP_X(id_W824) - 4., HOP4Y(9), PIP_X(id_W824) - 4., WIRE_Y(0)},
          {PIP_X(id_W820) - top_wire_dist - 4., HOP4Y(7), PIP_X(id_W820) - 5., HOP4Y(7)},
          {PIP_X(id_W820) - 5., HOP4Y(7), PIP_X(id_W820) - top_wire_dist - 5., HOP4Y(5)},
          {PIP_X(id_W820) - top_wire_dist - 5., HOP4Y(5), PIP_X(id_W820) - 6., HOP4Y(5)},
          {PIP_X(id_W820) - 6., HOP4Y(5), PIP_X(id_W820) - top_wire_dist - 6., HOP4Y(3)},
          {PIP_X(id_W820) - top_wire_dist - 6., HOP4Y(3), -wrap_len - 6., HOP4Y(3)},
          {-wrap_len - 6., HOP4Y(3), -wrap_len - 6., HOP4Y(2)},
          {-wrap_len - 6., HOP4Y(2), PIP_X(id_W828) - 6., HOP4Y(2)},
          {PIP_X(id_W828) - 6., HOP4Y(2), PIP_X(id_W828) + top_wire_dist - 6., HOP4Y(0)},
          {PIP_X(id_W828) + top_wire_dist - 6., HOP4Y(0), PIP_X(id_E828) - 5., HOP4Y(0)},
          {PIP_X(id_E828) - 5., HOP4Y(0), PIP_X(id_E828) - 5., WIRE_Y(0)}}},
        {id_W82_loop7,
         {{PIP_X(id_W820), WIRE_Y(0), PIP_X(id_W820), HOP4Y(17)},
          {PIP_X(id_W820) - 0., HOP4Y(17), PIP_X(id_W820) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W820) - top_wire_dist - 0., HOP4Y(15), PIP_X(id_W820) - 1., HOP4Y(15)},
          {PIP_X(id_W820) - 1., HOP4Y(15), PIP_X(id_W820) - top_wire_dist - 1., HOP4Y(13)},
          {PIP_X(id_W820) - top_wire_dist - 1., HOP4Y(13), PIP_X(id_W820) - 2., HOP4Y(13)},
          {PIP_X(id_W820) - 2., HOP4Y(13), PIP_X(id_W820) - top_wire_dist - 2., HOP4Y(11)},
          {PIP_X(id_W820) - top_wire_dist - 2., HOP4Y(11), PIP_X(id_W820) - 3., HOP4Y(11)},
          {PIP_X(id_W820) - 3., HOP4Y(11), PIP_X(id_W820) - top_wire_dist - 3., HOP4Y(9)},
          {PIP_X(id_W820) - top_wire_dist - 3., HOP4Y(9), PIP_X(id_W820) - 4., HOP4Y(9)},
          {PIP_X(id_W820) - 4., HOP4Y(9), PIP_X(id_W820) - top_wire_dist - 4., HOP4Y(7)},
          {PIP_X(id_W824) - 4., HOP4Y(9), PIP_X(id_W824) - 4., WIRE_Y(0)},
          {PIP_X(id_W820) - top_wire_dist - 4., HOP4Y(7), PIP_X(id_W820) - 5., HOP4Y(7)},
          {PIP_X(id_W820) - 5., HOP4Y(7), PIP_X(id_W820) - top_wire_dist - 5., HOP4Y(5)},
          {PIP_X(id_W820) - top_wire_dist - 5., HOP4Y(5), PIP_X(id_W820) - 6., HOP4Y(5)},
          {PIP_X(id_W820) - 6., HOP4Y(5), PIP_X(id_W820) - top_wire_dist - 6., HOP4Y(3)},
          {PIP_X(id_W820) - top_wire_dist - 6., HOP4Y(3), PIP_X(id_W820) - 7., HOP4Y(3)},
          {PIP_X(id_W820) - 7., HOP4Y(3), PIP_X(id_W820) - top_wire_dist - 7., HOP4Y(1)},
          {PIP_X(id_W820) - top_wire_dist - 7., HOP4Y(1), -wrap_len - 7., HOP4Y(1)},
          {-wrap_len - 7., HOP4Y(1), -wrap_len - 7., HOP4Y(0)},
          {-wrap_len - 7., HOP4Y(0), PIP_X(id_E828) - 7., HOP4Y(0)},
          {PIP_X(id_E828) - 7., HOP4Y(0), PIP_X(id_E828) - 7., WIRE_Y(0)}}},

#undef HOP4Y
#define HOP4Y(offset) WIRE_Y((float)offset + HOP4Y_START + 18.f + 18.f + 18.f)
        {id_E83,
         {{PIP_X(id_E830), WIRE_Y(0), PIP_X(id_E830), HOP4Y(16)},
          {PIP_X(id_E830), HOP4Y(16), PIP_X(id_W838), HOP4Y(16)},
          {PIP_X(id_W838) + 0., HOP4Y(16), PIP_X(id_W838) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W838) + top_wire_dist + 0., HOP4Y(14), PIP_X(id_W838) + 1., HOP4Y(14)},
          {PIP_X(id_W838) + 1., HOP4Y(14), PIP_X(id_W838) + top_wire_dist + 1., HOP4Y(12)},
          {PIP_X(id_W838) + top_wire_dist + 1., HOP4Y(12), PIP_X(id_W838) + 2., HOP4Y(12)},
          {PIP_X(id_W838) + 2., HOP4Y(12), PIP_X(id_W838) + top_wire_dist + 2., HOP4Y(10)},
          {PIP_X(id_W838) + top_wire_dist + 2., HOP4Y(10), PIP_X(id_W838) + 3., HOP4Y(10)},
          {PIP_X(id_W838) + 3., HOP4Y(10), PIP_X(id_W838) + top_wire_dist + 3., HOP4Y(8)},
          {PIP_X(id_W838) + top_wire_dist + 3., HOP4Y(8), PIP_X(id_W838) + 4., HOP4Y(8)},
          {PIP_X(id_W838) + 4., HOP4Y(8), PIP_X(id_W838) + top_wire_dist + 4., HOP4Y(6)},
          {PIP_X(id_E834) + 4., HOP4Y(8), PIP_X(id_E834) + 4., WIRE_Y(0)},
          {PIP_X(id_W838) + top_wire_dist + 4., HOP4Y(6), PIP_X(id_W838) + 5., HOP4Y(6)},
          {PIP_X(id_W838) + 5., HOP4Y(6), PIP_X(id_W838) + top_wire_dist + 5., HOP4Y(4)},
          {PIP_X(id_W838) + top_wire_dist + 5., HOP4Y(4), PIP_X(id_W838) + 6., HOP4Y(4)},
          {PIP_X(id_W838) + 6., HOP4Y(4), PIP_X(id_W838) + top_wire_dist + 6., HOP4Y(2)},
          {PIP_X(id_W838) + top_wire_dist + 6., HOP4Y(2), PIP_X(id_W838) + 7., HOP4Y(2)},
          {PIP_X(id_W838) + 7., HOP4Y(2), PIP_X(id_W838) + top_wire_dist + 7., HOP4Y(0)},
          {PIP_X(id_W838) + top_wire_dist + 7., HOP4Y(0), PIP_X(id_E838) + 8., HOP4Y(0)},
          {PIP_X(id_E838) + 8, HOP4Y(0), PIP_X(id_E838) + 8., WIRE_Y(0)}}},
        {id_W83,
         {{PIP_X(id_W830), WIRE_Y(0), PIP_X(id_W830), HOP4Y(17)},
          {PIP_X(id_W830) - 0., HOP4Y(17), PIP_X(id_W830) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W830) - top_wire_dist - 0., HOP4Y(15), PIP_X(id_W830) - 1., HOP4Y(15)},
          {PIP_X(id_W830) - 1., HOP4Y(15), PIP_X(id_W830) - top_wire_dist - 1., HOP4Y(13)},
          {PIP_X(id_W830) - top_wire_dist - 1., HOP4Y(13), PIP_X(id_W830) - 2., HOP4Y(13)},
          {PIP_X(id_W830) - 2., HOP4Y(13), PIP_X(id_W830) - top_wire_dist - 2., HOP4Y(11)},
          {PIP_X(id_W830) - top_wire_dist - 2., HOP4Y(11), PIP_X(id_W830) - 3., HOP4Y(11)},
          {PIP_X(id_W830) - 3., HOP4Y(11), PIP_X(id_W830) - top_wire_dist - 3., HOP4Y(9)},
          {PIP_X(id_W830) - top_wire_dist - 3., HOP4Y(9), PIP_X(id_W830) - 4., HOP4Y(9)},
          {PIP_X(id_W830) - 4., HOP4Y(9), PIP_X(id_W830) - top_wire_dist - 4., HOP4Y(7)},
          {PIP_X(id_W834) - 4., HOP4Y(9), PIP_X(id_W834) - 4., WIRE_Y(0)},
          {PIP_X(id_W830) - top_wire_dist - 4., HOP4Y(7), PIP_X(id_W830) - 5., HOP4Y(7)},
          {PIP_X(id_W830) - 5., HOP4Y(7), PIP_X(id_W830) - top_wire_dist - 5., HOP4Y(5)},
          {PIP_X(id_W830) - top_wire_dist - 5., HOP4Y(5), PIP_X(id_W830) - 6., HOP4Y(5)},
          {PIP_X(id_W830) - 6., HOP4Y(5), PIP_X(id_W830) - top_wire_dist - 6., HOP4Y(3)},
          {PIP_X(id_W830) - top_wire_dist - 6., HOP4Y(3), PIP_X(id_W830) - 7., HOP4Y(3)},
          {PIP_X(id_W830) - 7., HOP4Y(3), PIP_X(id_W830) - top_wire_dist - 7., HOP4Y(1)},
          {PIP_X(id_W830) - top_wire_dist - 7., HOP4Y(1), PIP_X(id_W838) - 8., HOP4Y(1)},
          {PIP_X(id_W838) - 8, HOP4Y(1), PIP_X(id_W838) - 8., WIRE_Y(0)}}},
        {id_E83_loop0,
         {{PIP_X(id_E830), WIRE_Y(0), PIP_X(id_E830), HOP4Y(16)},
          {PIP_X(id_E830), HOP4Y(16), PIP_X(id_W838), HOP4Y(16)},
          {PIP_X(id_W838) + 0., HOP4Y(16), PIP_X(id_W838) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W838) + top_wire_dist + 0., HOP4Y(14), wrap_len + 1., HOP4Y(14)},
          {wrap_len + 1., HOP4Y(14), wrap_len + 1., HOP4Y(15)},
          {wrap_len + 1., HOP4Y(15), PIP_X(id_W830) - 0., HOP4Y(15)},
          {PIP_X(id_W830) - 0., HOP4Y(15), PIP_X(id_W830) - top_wire_dist - 0., HOP4Y(13)},
          {PIP_X(id_W830) - top_wire_dist - 0., HOP4Y(13), PIP_X(id_W830) - 1., HOP4Y(13)},
          {PIP_X(id_W830) - 1., HOP4Y(13), PIP_X(id_W830) - top_wire_dist - 1., HOP4Y(11)},
          {PIP_X(id_W830) - top_wire_dist - 1., HOP4Y(11), PIP_X(id_W830) - 2., HOP4Y(11)},
          {PIP_X(id_W830) - 2., HOP4Y(11), PIP_X(id_W830) - top_wire_dist - 2., HOP4Y(9)},
          {PIP_X(id_W830) - top_wire_dist - 2., HOP4Y(9), PIP_X(id_W830) - 3., HOP4Y(9)},
          {PIP_X(id_W830) - 3., HOP4Y(9), PIP_X(id_W830) - top_wire_dist - 3., HOP4Y(7)},
          {PIP_X(id_W834) - 3., HOP4Y(9), PIP_X(id_W834) - 3., WIRE_Y(0)},
          {PIP_X(id_W830) - top_wire_dist - 3., HOP4Y(7), PIP_X(id_W830) - 4., HOP4Y(7)},
          {PIP_X(id_W830) - 4., HOP4Y(7), PIP_X(id_W830) - top_wire_dist - 4., HOP4Y(5)},
          {PIP_X(id_W830) - top_wire_dist - 4., HOP4Y(5), PIP_X(id_W830) - 5., HOP4Y(5)},
          {PIP_X(id_W830) - 5., HOP4Y(5), PIP_X(id_W830) - top_wire_dist - 5., HOP4Y(3)},
          {PIP_X(id_W830) - top_wire_dist - 5., HOP4Y(3), PIP_X(id_W830) - 6., HOP4Y(3)},
          {PIP_X(id_W830) - 6., HOP4Y(3), PIP_X(id_W830) - top_wire_dist - 6., HOP4Y(1)},
          {PIP_X(id_W830) - top_wire_dist - 6., HOP4Y(1), PIP_X(id_W838) - 7., HOP4Y(1)},
          {PIP_X(id_W838) - 7, HOP4Y(1), PIP_X(id_W838) - 7., WIRE_Y(0)}}},
        {id_E83_loop1,
         {{PIP_X(id_E830), WIRE_Y(0), PIP_X(id_E830), HOP4Y(16)},
          {PIP_X(id_E830), HOP4Y(16), PIP_X(id_W838), HOP4Y(16)},
          {PIP_X(id_W838) + 0., HOP4Y(16), PIP_X(id_W838) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W838) + top_wire_dist + 0., HOP4Y(14), PIP_X(id_W838) + 1., HOP4Y(14)},
          {PIP_X(id_W838) + 1., HOP4Y(14), PIP_X(id_W838) + top_wire_dist + 1., HOP4Y(12)},
          {PIP_X(id_W838) + top_wire_dist + 1., HOP4Y(12), wrap_len + 2., HOP4Y(12)},
          {wrap_len + 2., HOP4Y(12), wrap_len + 2., HOP4Y(13)},
          {wrap_len + 2., HOP4Y(13), PIP_X(id_W830) + 1., HOP4Y(13)},
          {PIP_X(id_W830) + 1., HOP4Y(13), PIP_X(id_W830) - top_wire_dist + 1., HOP4Y(11)},
          {PIP_X(id_W830) - top_wire_dist + 1., HOP4Y(11), PIP_X(id_W830) - 0., HOP4Y(11)},
          {PIP_X(id_W830) - 0., HOP4Y(11), PIP_X(id_W830) - top_wire_dist - 0., HOP4Y(9)},
          {PIP_X(id_W830) - top_wire_dist - 0., HOP4Y(9), PIP_X(id_W830) - 1., HOP4Y(9)},
          {PIP_X(id_W834) - 1., HOP4Y(9), PIP_X(id_W834) - 1., WIRE_Y(0)},
          {PIP_X(id_W830) - 1., HOP4Y(9), PIP_X(id_W830) - top_wire_dist - 1., HOP4Y(7)},
          {PIP_X(id_W830) - top_wire_dist - 1., HOP4Y(7), PIP_X(id_W830) - 2., HOP4Y(7)},
          {PIP_X(id_W830) - 2., HOP4Y(7), PIP_X(id_W830) - top_wire_dist - 2., HOP4Y(5)},
          {PIP_X(id_W830) - top_wire_dist - 2., HOP4Y(5), PIP_X(id_W830) - 3., HOP4Y(5)},
          {PIP_X(id_W830) - 3., HOP4Y(5), PIP_X(id_W830) - top_wire_dist - 3., HOP4Y(3)},
          {PIP_X(id_W830) - top_wire_dist - 3., HOP4Y(3), PIP_X(id_W830) - 4., HOP4Y(3)},
          {PIP_X(id_W830) - 4., HOP4Y(3), PIP_X(id_W830) - top_wire_dist - 4., HOP4Y(1)},
          {PIP_X(id_W830) - top_wire_dist - 4., HOP4Y(1), PIP_X(id_W838) - 5., HOP4Y(1)},
          {PIP_X(id_W838) - 5., HOP4Y(1), PIP_X(id_W838) - 5., WIRE_Y(0)}}},
        {id_E83_loop2,
         {{PIP_X(id_E830), WIRE_Y(0), PIP_X(id_E830), HOP4Y(16)},
          {PIP_X(id_E830), HOP4Y(16), PIP_X(id_W838), HOP4Y(16)},
          {PIP_X(id_W838) + 0., HOP4Y(16), PIP_X(id_W838) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W838) + top_wire_dist + 0., HOP4Y(14), PIP_X(id_W838) + 1., HOP4Y(14)},
          {PIP_X(id_W838) + 1., HOP4Y(14), PIP_X(id_W838) + top_wire_dist + 1., HOP4Y(12)},
          {PIP_X(id_W838) + top_wire_dist + 1., HOP4Y(12), PIP_X(id_W838) + 2., HOP4Y(12)},
          {PIP_X(id_W838) + 2., HOP4Y(12), PIP_X(id_W838) + top_wire_dist + 2., HOP4Y(10)},
          {PIP_X(id_W838) + top_wire_dist + 2., HOP4Y(10), wrap_len + 3., HOP4Y(10)},
          {wrap_len + 3., HOP4Y(10), wrap_len + 3., HOP4Y(11)},
          {wrap_len + 3., HOP4Y(11), PIP_X(id_W830) + 2., HOP4Y(11)},
          {PIP_X(id_W830) + 2., HOP4Y(11), PIP_X(id_W830) - top_wire_dist + 2., HOP4Y(9)},
          {PIP_X(id_W830) - top_wire_dist + 2., HOP4Y(9), PIP_X(id_W830) + 1., HOP4Y(9)},
          {PIP_X(id_W834) + 1., HOP4Y(9), PIP_X(id_W834) + 1., WIRE_Y(0)},
          {PIP_X(id_W830) + 1., HOP4Y(9), PIP_X(id_W830) - top_wire_dist + 1., HOP4Y(7)},
          {PIP_X(id_W830) - top_wire_dist + 1., HOP4Y(7), PIP_X(id_W830) + 0., HOP4Y(7)},
          {PIP_X(id_W830) + 0., HOP4Y(7), PIP_X(id_W830) - top_wire_dist - 0., HOP4Y(5)},
          {PIP_X(id_W830) - top_wire_dist - 0., HOP4Y(5), PIP_X(id_W830) - 1., HOP4Y(5)},
          {PIP_X(id_W830) - 1., HOP4Y(5), PIP_X(id_W830) - top_wire_dist - 1., HOP4Y(3)},
          {PIP_X(id_W830) - top_wire_dist - 1., HOP4Y(3), PIP_X(id_W830) - 2., HOP4Y(3)},
          {PIP_X(id_W830) - 2., HOP4Y(3), PIP_X(id_W830) - top_wire_dist - 2., HOP4Y(1)},
          {PIP_X(id_W830) - top_wire_dist - 2., HOP4Y(1), PIP_X(id_W838) - 3., HOP4Y(1)},
          {PIP_X(id_W838) - 3., HOP4Y(1), PIP_X(id_W838) - 3., WIRE_Y(0)}}},
        {id_E83_loop3,
         {{PIP_X(id_E830), WIRE_Y(0), PIP_X(id_E830), HOP4Y(16)},
          {PIP_X(id_E830), HOP4Y(16), PIP_X(id_W838), HOP4Y(16)},
          {PIP_X(id_W838) + 0., HOP4Y(16), PIP_X(id_W838) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W838) + top_wire_dist + 0., HOP4Y(14), PIP_X(id_W838) + 1., HOP4Y(14)},
          {PIP_X(id_W838) + 1., HOP4Y(14), PIP_X(id_W838) + top_wire_dist + 1., HOP4Y(12)},
          {PIP_X(id_W838) + top_wire_dist + 1., HOP4Y(12), PIP_X(id_W838) + 2., HOP4Y(12)},
          {PIP_X(id_W838) + 2., HOP4Y(12), PIP_X(id_W838) + top_wire_dist + 2., HOP4Y(10)},
          {PIP_X(id_W838) + top_wire_dist + 2., HOP4Y(10), PIP_X(id_W838) + 3., HOP4Y(10)},
          {PIP_X(id_W838) + 3., HOP4Y(10), PIP_X(id_W838) + top_wire_dist + 3., HOP4Y(8)},
          {PIP_X(id_W838) + top_wire_dist + 3., HOP4Y(8), wrap_len + 4., HOP4Y(8)},
          {wrap_len + 4., HOP4Y(8), wrap_len + 4., HOP4Y(9)},
          {wrap_len + 4., HOP4Y(9), PIP_X(id_W830) + 3., HOP4Y(9)},
          {PIP_X(id_W834) + 3., HOP4Y(9), PIP_X(id_W834) + 3., WIRE_Y(0)},
          {PIP_X(id_W830) + 3., HOP4Y(9), PIP_X(id_W830) - top_wire_dist + 3., HOP4Y(7)},
          {PIP_X(id_W830) - top_wire_dist + 3., HOP4Y(7), PIP_X(id_W830) + 2., HOP4Y(7)},
          {PIP_X(id_W830) + 2., HOP4Y(7), PIP_X(id_W830) - top_wire_dist + 2., HOP4Y(5)},
          {PIP_X(id_W830) - top_wire_dist + 2., HOP4Y(5), PIP_X(id_W830) + 1., HOP4Y(5)},
          {PIP_X(id_W830) + 1., HOP4Y(5), PIP_X(id_W830) - top_wire_dist + 1., HOP4Y(3)},
          {PIP_X(id_W830) - top_wire_dist + 1., HOP4Y(3), PIP_X(id_W830) - 0., HOP4Y(3)},
          {PIP_X(id_W830) - 0., HOP4Y(3), PIP_X(id_W830) - top_wire_dist - 0., HOP4Y(1)},
          {PIP_X(id_W830) - top_wire_dist - 0., HOP4Y(1), PIP_X(id_W838) - 1., HOP4Y(1)},
          {PIP_X(id_W838) - 1., HOP4Y(1), PIP_X(id_W838) - 1., WIRE_Y(0)}}},
        {id_E83_loop4,
         {{PIP_X(id_E830), WIRE_Y(0), PIP_X(id_E830), HOP4Y(16)},
          {PIP_X(id_E830), HOP4Y(16), PIP_X(id_W838), HOP4Y(16)},
          {PIP_X(id_W838) + 0., HOP4Y(16), PIP_X(id_W838) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W838) + top_wire_dist + 0., HOP4Y(14), PIP_X(id_W838) + 1., HOP4Y(14)},
          {PIP_X(id_W838) + 1., HOP4Y(14), PIP_X(id_W838) + top_wire_dist + 1., HOP4Y(12)},
          {PIP_X(id_W838) + top_wire_dist + 1., HOP4Y(12), PIP_X(id_W838) + 2., HOP4Y(12)},
          {PIP_X(id_W838) + 2., HOP4Y(12), PIP_X(id_W838) + top_wire_dist + 2., HOP4Y(10)},
          {PIP_X(id_W838) + top_wire_dist + 2., HOP4Y(10), PIP_X(id_W838) + 3., HOP4Y(10)},
          {PIP_X(id_W838) + 3., HOP4Y(10), PIP_X(id_W838) + top_wire_dist + 3., HOP4Y(8)},
          {PIP_X(id_W838) + top_wire_dist + 3., HOP4Y(8), PIP_X(id_W838) + 4., HOP4Y(8)},
          {PIP_X(id_E834) + 4., HOP4Y(8), PIP_X(id_E834) + 4., WIRE_Y(0)},
          {PIP_X(id_W838) + 4., HOP4Y(8), PIP_X(id_W838) + top_wire_dist + 4., HOP4Y(6)},
          {PIP_X(id_W838) + top_wire_dist + 4., HOP4Y(6), wrap_len + 5., HOP4Y(6)},
          {wrap_len + 5., HOP4Y(6), wrap_len + 5., HOP4Y(7)},
          {wrap_len + 5., HOP4Y(7), PIP_X(id_W830) + 4., HOP4Y(7)},
          {PIP_X(id_W830) + 4., HOP4Y(7), PIP_X(id_W830) - top_wire_dist + 4., HOP4Y(5)},
          {PIP_X(id_W830) - top_wire_dist + 4., HOP4Y(5), PIP_X(id_W830) + 3., HOP4Y(5)},
          {PIP_X(id_W830) + 3., HOP4Y(5), PIP_X(id_W830) - top_wire_dist + 3., HOP4Y(3)},
          {PIP_X(id_W830) - top_wire_dist + 3., HOP4Y(3), PIP_X(id_W830) + 2., HOP4Y(3)},
          {PIP_X(id_W830) + 2., HOP4Y(3), PIP_X(id_W830) - top_wire_dist + 2., HOP4Y(1)},
          {PIP_X(id_W830) - top_wire_dist + 2., HOP4Y(1), PIP_X(id_W838) + 1., HOP4Y(1)},
          {PIP_X(id_W838) + 1., HOP4Y(1), PIP_X(id_W838) + 1., WIRE_Y(0)}}},
        {id_E83_loop5,
         {{PIP_X(id_E830), WIRE_Y(0), PIP_X(id_E830), HOP4Y(16)},
          {PIP_X(id_E830), HOP4Y(16), PIP_X(id_W838), HOP4Y(16)},
          {PIP_X(id_W838) + 0., HOP4Y(16), PIP_X(id_W838) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W838) + top_wire_dist + 0., HOP4Y(14), PIP_X(id_W838) + 1., HOP4Y(14)},
          {PIP_X(id_W838) + 1., HOP4Y(14), PIP_X(id_W838) + top_wire_dist + 1., HOP4Y(12)},
          {PIP_X(id_W838) + top_wire_dist + 1., HOP4Y(12), PIP_X(id_W838) + 2., HOP4Y(12)},
          {PIP_X(id_W838) + 2., HOP4Y(12), PIP_X(id_W838) + top_wire_dist + 2., HOP4Y(10)},
          {PIP_X(id_W838) + top_wire_dist + 2., HOP4Y(10), PIP_X(id_W838) + 3., HOP4Y(10)},
          {PIP_X(id_W838) + 3., HOP4Y(10), PIP_X(id_W838) + top_wire_dist + 3., HOP4Y(8)},
          {PIP_X(id_W838) + top_wire_dist + 3., HOP4Y(8), PIP_X(id_W838) + 4., HOP4Y(8)},
          {PIP_X(id_E834) + 4., HOP4Y(8), PIP_X(id_E834) + 4., WIRE_Y(0)},
          {PIP_X(id_W838) + 4., HOP4Y(8), PIP_X(id_W838) + top_wire_dist + 4., HOP4Y(6)},
          {PIP_X(id_W838) + top_wire_dist + 4., HOP4Y(6), PIP_X(id_W838) + 5., HOP4Y(6)},
          {PIP_X(id_W838) + 5., HOP4Y(6), PIP_X(id_W838) + top_wire_dist + 5., HOP4Y(4)},
          {PIP_X(id_W838) + top_wire_dist + 5., HOP4Y(4), wrap_len + 6., HOP4Y(4)},
          {wrap_len + 6., HOP4Y(4), wrap_len + 6., HOP4Y(5)},
          {wrap_len + 6., HOP4Y(5), PIP_X(id_W830) + 5., HOP4Y(5)},
          {PIP_X(id_W830) + 5., HOP4Y(5), PIP_X(id_W830) - top_wire_dist + 5., HOP4Y(3)},
          {PIP_X(id_W830) - top_wire_dist + 5., HOP4Y(3), PIP_X(id_W830) + 4., HOP4Y(3)},
          {PIP_X(id_W830) + 4., HOP4Y(3), PIP_X(id_W830) - top_wire_dist + 4., HOP4Y(1)},
          {PIP_X(id_W830) - top_wire_dist + 4., HOP4Y(1), PIP_X(id_W838) + 3., HOP4Y(1)},
          {PIP_X(id_W838) + 3., HOP4Y(1), PIP_X(id_W838) + 3., WIRE_Y(0)}}},
        {id_E83_loop6,
         {{PIP_X(id_E830), WIRE_Y(0), PIP_X(id_E830), HOP4Y(16)},
          {PIP_X(id_E830), HOP4Y(16), PIP_X(id_W838), HOP4Y(16)},
          {PIP_X(id_W838) + 0., HOP4Y(16), PIP_X(id_W838) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W838) + top_wire_dist + 0., HOP4Y(14), PIP_X(id_W838) + 1., HOP4Y(14)},
          {PIP_X(id_W838) + 1., HOP4Y(14), PIP_X(id_W838) + top_wire_dist + 1., HOP4Y(12)},
          {PIP_X(id_W838) + top_wire_dist + 1., HOP4Y(12), PIP_X(id_W838) + 2., HOP4Y(12)},
          {PIP_X(id_W838) + 2., HOP4Y(12), PIP_X(id_W838) + top_wire_dist + 2., HOP4Y(10)},
          {PIP_X(id_W838) + top_wire_dist + 2., HOP4Y(10), PIP_X(id_W838) + 3., HOP4Y(10)},
          {PIP_X(id_W838) + 3., HOP4Y(10), PIP_X(id_W838) + top_wire_dist + 3., HOP4Y(8)},
          {PIP_X(id_W838) + top_wire_dist + 3., HOP4Y(8), PIP_X(id_W838) + 4., HOP4Y(8)},
          {PIP_X(id_W838) + 4., HOP4Y(8), PIP_X(id_W838) + top_wire_dist + 4., HOP4Y(6)},
          {PIP_X(id_E834) + 4., HOP4Y(8), PIP_X(id_E834) + 4., WIRE_Y(0)},
          {PIP_X(id_W838) + top_wire_dist + 4., HOP4Y(6), PIP_X(id_W838) + 5., HOP4Y(6)},
          {PIP_X(id_W838) + 5., HOP4Y(6), PIP_X(id_W838) + top_wire_dist + 5., HOP4Y(4)},
          {PIP_X(id_W838) + top_wire_dist + 5., HOP4Y(4), PIP_X(id_W838) + 6., HOP4Y(4)},
          {PIP_X(id_W838) + 6., HOP4Y(4), PIP_X(id_W838) + top_wire_dist + 6., HOP4Y(2)},
          {PIP_X(id_W838) + top_wire_dist + 6., HOP4Y(2), wrap_len + 7., HOP4Y(2)},
          {wrap_len + 7., HOP4Y(2), wrap_len + 7., HOP4Y(3)},
          {wrap_len + 7., HOP4Y(3), PIP_X(id_W830) + 6., HOP4Y(3)},
          {PIP_X(id_W830) + 6., HOP4Y(3), PIP_X(id_W830) - top_wire_dist + 6., HOP4Y(1)},
          {PIP_X(id_W830) - top_wire_dist + 6., HOP4Y(1), PIP_X(id_W838) + 5., HOP4Y(1)},
          {PIP_X(id_W838) + 5., HOP4Y(1), PIP_X(id_W838) + 5., WIRE_Y(0)}}},
        {id_E83_loop7,
         {{PIP_X(id_E830), WIRE_Y(0), PIP_X(id_E830), HOP4Y(16)},
          {PIP_X(id_E830), HOP4Y(16), PIP_X(id_W838), HOP4Y(16)},
          {PIP_X(id_W838) + 0., HOP4Y(16), PIP_X(id_W838) + top_wire_dist + 0., HOP4Y(14)},
          {PIP_X(id_W838) + top_wire_dist + 0., HOP4Y(14), PIP_X(id_W838) + 1., HOP4Y(14)},
          {PIP_X(id_W838) + 1., HOP4Y(14), PIP_X(id_W838) + top_wire_dist + 1., HOP4Y(12)},
          {PIP_X(id_W838) + top_wire_dist + 1., HOP4Y(12), PIP_X(id_W838) + 2., HOP4Y(12)},
          {PIP_X(id_W838) + 2., HOP4Y(12), PIP_X(id_W838) + top_wire_dist + 2., HOP4Y(10)},
          {PIP_X(id_W838) + top_wire_dist + 2., HOP4Y(10), PIP_X(id_W838) + 3., HOP4Y(10)},
          {PIP_X(id_W838) + 3., HOP4Y(10), PIP_X(id_W838) + top_wire_dist + 3., HOP4Y(8)},
          {PIP_X(id_W838) + top_wire_dist + 3., HOP4Y(8), PIP_X(id_W838) + 4., HOP4Y(8)},
          {PIP_X(id_W838) + 4., HOP4Y(8), PIP_X(id_W838) + top_wire_dist + 4., HOP4Y(6)},
          {PIP_X(id_E834) + 4., HOP4Y(8), PIP_X(id_E834) + 4., WIRE_Y(0)},
          {PIP_X(id_W838) + top_wire_dist + 4., HOP4Y(6), PIP_X(id_W838) + 5., HOP4Y(6)},
          {PIP_X(id_W838) + 5., HOP4Y(6), PIP_X(id_W838) + top_wire_dist + 5., HOP4Y(4)},
          {PIP_X(id_W838) + top_wire_dist + 5., HOP4Y(4), PIP_X(id_W838) + 6., HOP4Y(4)},
          {PIP_X(id_W838) + 6., HOP4Y(4), PIP_X(id_W838) + top_wire_dist + 6., HOP4Y(2)},
          {PIP_X(id_W838) + top_wire_dist + 6., HOP4Y(2), PIP_X(id_W838) + 7., HOP4Y(2)},
          {PIP_X(id_W838) + 7., HOP4Y(2), PIP_X(id_W838) + top_wire_dist + 7., HOP4Y(0)},
          {PIP_X(id_W838) + top_wire_dist + 7., HOP4Y(0), wrap_len + 8., HOP4Y(0)},
          {wrap_len + 8., HOP4Y(0), wrap_len + 8., HOP4Y(1)},
          {wrap_len + 8., HOP4Y(1), PIP_X(id_W838) + 7., HOP4Y(1)},
          {PIP_X(id_W838) + 7., HOP4Y(1), PIP_X(id_W838) + 7., WIRE_Y(0)}}},
        {id_W83_loop0,
         {{PIP_X(id_W830), WIRE_Y(0), PIP_X(id_W830), HOP4Y(17)},
          {PIP_X(id_W830) - 0., HOP4Y(17), PIP_X(id_W830) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W830) - top_wire_dist - 0., HOP4Y(15), -wrap_len - 0., HOP4Y(15)},
          {-wrap_len - 0., HOP4Y(15), -wrap_len - 0., HOP4Y(14)},
          {-wrap_len - 0., HOP4Y(14), PIP_X(id_W838) + 0., HOP4Y(14)},
          {PIP_X(id_W838) + 0., HOP4Y(14), PIP_X(id_W838) + top_wire_dist + 0., HOP4Y(12)},
          {PIP_X(id_W838) + top_wire_dist + 0., HOP4Y(12), PIP_X(id_W838) + 1., HOP4Y(12)},
          {PIP_X(id_W838) + 1., HOP4Y(12), PIP_X(id_W838) + top_wire_dist + 1., HOP4Y(10)},
          {PIP_X(id_W838) + top_wire_dist + 1., HOP4Y(10), PIP_X(id_W838) + 2., HOP4Y(10)},
          {PIP_X(id_W838) + 2., HOP4Y(10), PIP_X(id_W838) + top_wire_dist + 2., HOP4Y(8)},
          {PIP_X(id_W838) + top_wire_dist + 2., HOP4Y(8), PIP_X(id_W838) + 3., HOP4Y(8)},
          {PIP_X(id_W838) + 3., HOP4Y(8), PIP_X(id_W838) + top_wire_dist + 3., HOP4Y(6)},
          {PIP_X(id_E834) + 3., HOP4Y(8), PIP_X(id_E834) + 3., WIRE_Y(0)},
          {PIP_X(id_W838) + top_wire_dist + 3., HOP4Y(6), PIP_X(id_W838) + 4., HOP4Y(6)},
          {PIP_X(id_W838) + 4., HOP4Y(6), PIP_X(id_W838) + top_wire_dist + 4., HOP4Y(4)},
          {PIP_X(id_W838) + top_wire_dist + 4., HOP4Y(4), PIP_X(id_W838) + 5., HOP4Y(4)},
          {PIP_X(id_W838) + 5., HOP4Y(4), PIP_X(id_W838) + top_wire_dist + 5., HOP4Y(2)},
          {PIP_X(id_W838) + top_wire_dist + 5., HOP4Y(2), PIP_X(id_W838) + 6., HOP4Y(2)},
          {PIP_X(id_W838) + 6., HOP4Y(2), PIP_X(id_W838) + top_wire_dist + 6., HOP4Y(0)},
          {PIP_X(id_W838) + top_wire_dist + 6., HOP4Y(0), PIP_X(id_E838) + 7., HOP4Y(0)},
          {PIP_X(id_E838) + 7., HOP4Y(0), PIP_X(id_E838) + 7., WIRE_Y(0)}}},
        {id_W83_loop1,
         {{PIP_X(id_W830), WIRE_Y(0), PIP_X(id_W830), HOP4Y(17)},
          {PIP_X(id_W830) - 0., HOP4Y(17), PIP_X(id_W830) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W830) - top_wire_dist - 0., HOP4Y(15), PIP_X(id_W830) - 1., HOP4Y(15)},
          {PIP_X(id_W830) - 1., HOP4Y(15), PIP_X(id_W830) - top_wire_dist - 1., HOP4Y(13)},
          {PIP_X(id_W830) - top_wire_dist - 1., HOP4Y(13), -wrap_len - 1., HOP4Y(13)},
          {-wrap_len - 1., HOP4Y(13), -wrap_len - 1., HOP4Y(12)},
          {-wrap_len - 1., HOP4Y(12), PIP_X(id_W838) - 1., HOP4Y(12)},
          {PIP_X(id_W838) - 1., HOP4Y(12), PIP_X(id_W838) + top_wire_dist - 1., HOP4Y(10)},
          {PIP_X(id_W838) + top_wire_dist - 1., HOP4Y(10), PIP_X(id_W838) + 0., HOP4Y(10)},
          {PIP_X(id_W838) + 0., HOP4Y(10), PIP_X(id_W838) + top_wire_dist + 0., HOP4Y(8)},
          {PIP_X(id_W838) + top_wire_dist + 0., HOP4Y(8), PIP_X(id_W838) + 1., HOP4Y(8)},
          {PIP_X(id_W838) + 1., HOP4Y(8), PIP_X(id_W838) + top_wire_dist + 1., HOP4Y(6)},
          {PIP_X(id_E834) + 1., HOP4Y(8), PIP_X(id_E834) + 1., WIRE_Y(0)},
          {PIP_X(id_W838) + top_wire_dist + 1., HOP4Y(6), PIP_X(id_W838) + 2., HOP4Y(6)},
          {PIP_X(id_W838) + 2., HOP4Y(6), PIP_X(id_W838) + top_wire_dist + 2., HOP4Y(4)},
          {PIP_X(id_W838) + top_wire_dist + 2., HOP4Y(4), PIP_X(id_W838) + 3., HOP4Y(4)},
          {PIP_X(id_W838) + 3., HOP4Y(4), PIP_X(id_W838) + top_wire_dist + 3., HOP4Y(2)},
          {PIP_X(id_W838) + top_wire_dist + 3., HOP4Y(2), PIP_X(id_W838) + 4., HOP4Y(2)},
          {PIP_X(id_W838) + 4., HOP4Y(2), PIP_X(id_W838) + top_wire_dist + 4., HOP4Y(0)},
          {PIP_X(id_W838) + top_wire_dist + 4., HOP4Y(0), PIP_X(id_E838) + 5., HOP4Y(0)},
          {PIP_X(id_E838) + 5., HOP4Y(0), PIP_X(id_E838) + 5., WIRE_Y(0)}}},
        {id_W83_loop2,
         {{PIP_X(id_W830), WIRE_Y(0), PIP_X(id_W830), HOP4Y(17)},
          {PIP_X(id_W830) - 0., HOP4Y(17), PIP_X(id_W830) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W830) - top_wire_dist - 0., HOP4Y(15), PIP_X(id_W830) - 1., HOP4Y(15)},
          {PIP_X(id_W830) - 1., HOP4Y(15), PIP_X(id_W830) - top_wire_dist - 1., HOP4Y(13)},
          {PIP_X(id_W830) - top_wire_dist - 1., HOP4Y(13), PIP_X(id_W830) - 2., HOP4Y(13)},
          {PIP_X(id_W830) - 2., HOP4Y(13), PIP_X(id_W830) - top_wire_dist - 2., HOP4Y(11)},
          {PIP_X(id_W830) - top_wire_dist - 2., HOP4Y(11), -wrap_len - 2., HOP4Y(11)},
          {-wrap_len - 2., HOP4Y(11), -wrap_len - 2., HOP4Y(10)},
          {-wrap_len - 2., HOP4Y(10), PIP_X(id_W838) - 2., HOP4Y(10)},
          {PIP_X(id_W838) - 2., HOP4Y(10), PIP_X(id_W838) + top_wire_dist - 2., HOP4Y(8)},
          {PIP_X(id_W838) + top_wire_dist - 2., HOP4Y(8), PIP_X(id_W838) - 1., HOP4Y(8)},
          {PIP_X(id_W838) - 1., HOP4Y(8), PIP_X(id_W838) + top_wire_dist - 1., HOP4Y(6)},
          {PIP_X(id_E834) - 1., HOP4Y(8), PIP_X(id_E834) - 1., WIRE_Y(0)},
          {PIP_X(id_W838) + top_wire_dist - 1., HOP4Y(6), PIP_X(id_W838) + 0., HOP4Y(6)},
          {PIP_X(id_W838) + 0., HOP4Y(6), PIP_X(id_W838) + top_wire_dist + 0., HOP4Y(4)},
          {PIP_X(id_W838) + top_wire_dist + 0., HOP4Y(4), PIP_X(id_W838) + 1., HOP4Y(4)},
          {PIP_X(id_W838) + 1., HOP4Y(4), PIP_X(id_W838) + top_wire_dist + 1., HOP4Y(2)},
          {PIP_X(id_W838) + top_wire_dist + 1., HOP4Y(2), PIP_X(id_W838) + 2., HOP4Y(2)},
          {PIP_X(id_W838) + 2., HOP4Y(2), PIP_X(id_W838) + top_wire_dist + 2., HOP4Y(0)},
          {PIP_X(id_W838) + top_wire_dist + 2., HOP4Y(0), PIP_X(id_E838) + 3., HOP4Y(0)},
          {PIP_X(id_E838) + 3., HOP4Y(0), PIP_X(id_E838) + 3., WIRE_Y(0)}}},
        {id_W83_loop3,
         {{PIP_X(id_W830), WIRE_Y(0), PIP_X(id_W830), HOP4Y(17)},
          {PIP_X(id_W830) - 0., HOP4Y(17), PIP_X(id_W830) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W830) - top_wire_dist - 0., HOP4Y(15), PIP_X(id_W830) - 1., HOP4Y(15)},
          {PIP_X(id_W830) - 1., HOP4Y(15), PIP_X(id_W830) - top_wire_dist - 1., HOP4Y(13)},
          {PIP_X(id_W830) - top_wire_dist - 1., HOP4Y(13), PIP_X(id_W830) - 2., HOP4Y(13)},
          {PIP_X(id_W830) - 2., HOP4Y(13), PIP_X(id_W830) - top_wire_dist - 2., HOP4Y(11)},
          {PIP_X(id_W830) - top_wire_dist - 2., HOP4Y(11), PIP_X(id_W830) - 3., HOP4Y(11)},
          {PIP_X(id_W830) - 3., HOP4Y(11), PIP_X(id_W830) - top_wire_dist - 3., HOP4Y(9)},
          {PIP_X(id_W830) - top_wire_dist - 3., HOP4Y(9), -wrap_len - 3., HOP4Y(9)},
          {-wrap_len - 3., HOP4Y(9), -wrap_len - 3., HOP4Y(8)},
          {-wrap_len - 3., HOP4Y(8), PIP_X(id_W838) - 3., HOP4Y(8)},
          {PIP_X(id_W838) - 3., HOP4Y(8), PIP_X(id_W838) + top_wire_dist - 3., HOP4Y(6)},
          {PIP_X(id_E834) - 3., HOP4Y(8), PIP_X(id_E834) - 3., WIRE_Y(0)},
          {PIP_X(id_W838) + top_wire_dist - 3., HOP4Y(6), PIP_X(id_W838) - 2., HOP4Y(6)},
          {PIP_X(id_W838) - 2., HOP4Y(6), PIP_X(id_W838) + top_wire_dist - 2., HOP4Y(4)},
          {PIP_X(id_W838) + top_wire_dist - 2., HOP4Y(4), PIP_X(id_W838) - 1., HOP4Y(4)},
          {PIP_X(id_W838) - 1., HOP4Y(4), PIP_X(id_W838) + top_wire_dist - 1., HOP4Y(2)},
          {PIP_X(id_W838) + top_wire_dist - 1., HOP4Y(2), PIP_X(id_W838) + 0., HOP4Y(2)},
          {PIP_X(id_W838) + 0., HOP4Y(2), PIP_X(id_W838) + top_wire_dist + 0., HOP4Y(0)},
          {PIP_X(id_W838) + top_wire_dist + 0., HOP4Y(0), PIP_X(id_E838) + 1., HOP4Y(0)},
          {PIP_X(id_E838) + 1., HOP4Y(0), PIP_X(id_E838) + 1., WIRE_Y(0)}}},
        {id_W83_loop4,
         {{PIP_X(id_W830), WIRE_Y(0), PIP_X(id_W830), HOP4Y(17)},
          {PIP_X(id_W830) - 0., HOP4Y(17), PIP_X(id_W830) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W830) - top_wire_dist - 0., HOP4Y(15), PIP_X(id_W830) - 1., HOP4Y(15)},
          {PIP_X(id_W830) - 1., HOP4Y(15), PIP_X(id_W830) - top_wire_dist - 1., HOP4Y(13)},
          {PIP_X(id_W830) - top_wire_dist - 1., HOP4Y(13), PIP_X(id_W830) - 2., HOP4Y(13)},
          {PIP_X(id_W830) - 2., HOP4Y(13), PIP_X(id_W830) - top_wire_dist - 2., HOP4Y(11)},
          {PIP_X(id_W830) - top_wire_dist - 2., HOP4Y(11), PIP_X(id_W830) - 3., HOP4Y(11)},
          {PIP_X(id_W830) - 3., HOP4Y(11), PIP_X(id_W830) - top_wire_dist - 3., HOP4Y(9)},
          {PIP_X(id_W830) - top_wire_dist - 3., HOP4Y(9), PIP_X(id_W830) - 4., HOP4Y(9)},
          {PIP_X(id_W830) - 4., HOP4Y(9), PIP_X(id_W830) - top_wire_dist - 4., HOP4Y(7)},
          {PIP_X(id_W830) - top_wire_dist - 4., HOP4Y(7), -wrap_len - 4., HOP4Y(7)},
          {-wrap_len - 4., HOP4Y(7), -wrap_len - 4., HOP4Y(6)},
          {PIP_X(id_W834) - 4., HOP4Y(6), PIP_X(id_W834) - 4., WIRE_Y(0)},
          {-wrap_len - 4., HOP4Y(6), PIP_X(id_W838) - 4., HOP4Y(6)},
          {PIP_X(id_W838) - 4., HOP4Y(6), PIP_X(id_W838) + top_wire_dist - 4., HOP4Y(4)},
          {PIP_X(id_W838) + top_wire_dist - 4., HOP4Y(4), PIP_X(id_W838) - 3., HOP4Y(4)},
          {PIP_X(id_W838) - 3., HOP4Y(4), PIP_X(id_W838) + top_wire_dist - 3., HOP4Y(2)},
          {PIP_X(id_W838) + top_wire_dist - 3., HOP4Y(2), PIP_X(id_W838) - 2., HOP4Y(2)},
          {PIP_X(id_W838) - 2., HOP4Y(2), PIP_X(id_W838) + top_wire_dist - 2., HOP4Y(0)},
          {PIP_X(id_W838) + top_wire_dist - 2., HOP4Y(0), PIP_X(id_E838) - 1., HOP4Y(0)},
          {PIP_X(id_E838) - 1., HOP4Y(0), PIP_X(id_E838) - 1., WIRE_Y(0)}}},
        {id_W83_loop5,
         {{PIP_X(id_W830), WIRE_Y(0), PIP_X(id_W830), HOP4Y(17)},
          {PIP_X(id_W830) - 0., HOP4Y(17), PIP_X(id_W830) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W830) - top_wire_dist - 0., HOP4Y(15), PIP_X(id_W830) - 1., HOP4Y(15)},
          {PIP_X(id_W830) - 1., HOP4Y(15), PIP_X(id_W830) - top_wire_dist - 1., HOP4Y(13)},
          {PIP_X(id_W830) - top_wire_dist - 1., HOP4Y(13), PIP_X(id_W830) - 2., HOP4Y(13)},
          {PIP_X(id_W830) - 2., HOP4Y(13), PIP_X(id_W830) - top_wire_dist - 2., HOP4Y(11)},
          {PIP_X(id_W830) - top_wire_dist - 2., HOP4Y(11), PIP_X(id_W830) - 3., HOP4Y(11)},
          {PIP_X(id_W830) - 3., HOP4Y(11), PIP_X(id_W830) - top_wire_dist - 3., HOP4Y(9)},
          {PIP_X(id_W830) - top_wire_dist - 3., HOP4Y(9), PIP_X(id_W830) - 4., HOP4Y(9)},
          {PIP_X(id_W830) - 4., HOP4Y(9), PIP_X(id_W830) - top_wire_dist - 4., HOP4Y(7)},
          {PIP_X(id_W834) - 4., HOP4Y(9), PIP_X(id_W834) - 4., WIRE_Y(0)},
          {PIP_X(id_W830) - top_wire_dist - 4., HOP4Y(7), PIP_X(id_W830) - 5., HOP4Y(7)},
          {PIP_X(id_W830) - 5., HOP4Y(7), PIP_X(id_W830) - top_wire_dist - 5., HOP4Y(5)},
          {PIP_X(id_W830) - top_wire_dist - 5., HOP4Y(5), -wrap_len - 5., HOP4Y(5)},
          {-wrap_len - 5., HOP4Y(5), -wrap_len - 5., HOP4Y(4)},
          {-wrap_len - 5., HOP4Y(4), PIP_X(id_W838) - 5., HOP4Y(4)},
          {PIP_X(id_W838) - 5., HOP4Y(4), PIP_X(id_W838) + top_wire_dist - 5., HOP4Y(2)},
          {PIP_X(id_W838) + top_wire_dist - 5., HOP4Y(2), PIP_X(id_W838) - 4., HOP4Y(2)},
          {PIP_X(id_W838) - 4., HOP4Y(2), PIP_X(id_W838) + top_wire_dist - 4., HOP4Y(0)},
          {PIP_X(id_W838) + top_wire_dist - 4., HOP4Y(0), PIP_X(id_E838) - 3., HOP4Y(0)},
          {PIP_X(id_E838) - 3., HOP4Y(0), PIP_X(id_E838) - 3., WIRE_Y(0)}}},
        {id_W83_loop6,
         {{PIP_X(id_W830), WIRE_Y(0), PIP_X(id_W830), HOP4Y(17)},
          {PIP_X(id_W830) - 0., HOP4Y(17), PIP_X(id_W830) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W830) - top_wire_dist - 0., HOP4Y(15), PIP_X(id_W830) - 1., HOP4Y(15)},
          {PIP_X(id_W830) - 1., HOP4Y(15), PIP_X(id_W830) - top_wire_dist - 1., HOP4Y(13)},
          {PIP_X(id_W830) - top_wire_dist - 1., HOP4Y(13), PIP_X(id_W830) - 2., HOP4Y(13)},
          {PIP_X(id_W830) - 2., HOP4Y(13), PIP_X(id_W830) - top_wire_dist - 2., HOP4Y(11)},
          {PIP_X(id_W830) - top_wire_dist - 2., HOP4Y(11), PIP_X(id_W830) - 3., HOP4Y(11)},
          {PIP_X(id_W830) - 3., HOP4Y(11), PIP_X(id_W830) - top_wire_dist - 3., HOP4Y(9)},
          {PIP_X(id_W830) - top_wire_dist - 3., HOP4Y(9), PIP_X(id_W830) - 4., HOP4Y(9)},
          {PIP_X(id_W830) - 4., HOP4Y(9), PIP_X(id_W830) - top_wire_dist - 4., HOP4Y(7)},
          {PIP_X(id_W834) - 4., HOP4Y(9), PIP_X(id_W834) - 4., WIRE_Y(0)},
          {PIP_X(id_W830) - top_wire_dist - 4., HOP4Y(7), PIP_X(id_W830) - 5., HOP4Y(7)},
          {PIP_X(id_W830) - 5., HOP4Y(7), PIP_X(id_W830) - top_wire_dist - 5., HOP4Y(5)},
          {PIP_X(id_W830) - top_wire_dist - 5., HOP4Y(5), PIP_X(id_W830) - 6., HOP4Y(5)},
          {PIP_X(id_W830) - 6., HOP4Y(5), PIP_X(id_W830) - top_wire_dist - 6., HOP4Y(3)},
          {PIP_X(id_W830) - top_wire_dist - 6., HOP4Y(3), -wrap_len - 6., HOP4Y(3)},
          {-wrap_len - 6., HOP4Y(3), -wrap_len - 6., HOP4Y(2)},
          {-wrap_len - 6., HOP4Y(2), PIP_X(id_W838) - 6., HOP4Y(2)},
          {PIP_X(id_W838) - 6., HOP4Y(2), PIP_X(id_W838) + top_wire_dist - 6., HOP4Y(0)},
          {PIP_X(id_W838) + top_wire_dist - 6., HOP4Y(0), PIP_X(id_E838) - 5., HOP4Y(0)},
          {PIP_X(id_E838) - 5., HOP4Y(0), PIP_X(id_E838) - 5., WIRE_Y(0)}}},
        {id_W83_loop7,
         {{PIP_X(id_W830), WIRE_Y(0), PIP_X(id_W830), HOP4Y(17)},
          {PIP_X(id_W830) - 0., HOP4Y(17), PIP_X(id_W830) - top_wire_dist - 0., HOP4Y(15)},
          {PIP_X(id_W830) - top_wire_dist - 0., HOP4Y(15), PIP_X(id_W830) - 1., HOP4Y(15)},
          {PIP_X(id_W830) - 1., HOP4Y(15), PIP_X(id_W830) - top_wire_dist - 1., HOP4Y(13)},
          {PIP_X(id_W830) - top_wire_dist - 1., HOP4Y(13), PIP_X(id_W830) - 2., HOP4Y(13)},
          {PIP_X(id_W830) - 2., HOP4Y(13), PIP_X(id_W830) - top_wire_dist - 2., HOP4Y(11)},
          {PIP_X(id_W830) - top_wire_dist - 2., HOP4Y(11), PIP_X(id_W830) - 3., HOP4Y(11)},
          {PIP_X(id_W830) - 3., HOP4Y(11), PIP_X(id_W830) - top_wire_dist - 3., HOP4Y(9)},
          {PIP_X(id_W830) - top_wire_dist - 3., HOP4Y(9), PIP_X(id_W830) - 4., HOP4Y(9)},
          {PIP_X(id_W830) - 4., HOP4Y(9), PIP_X(id_W830) - top_wire_dist - 4., HOP4Y(7)},
          {PIP_X(id_W834) - 4., HOP4Y(9), PIP_X(id_W834) - 4., WIRE_Y(0)},
          {PIP_X(id_W830) - top_wire_dist - 4., HOP4Y(7), PIP_X(id_W830) - 5., HOP4Y(7)},
          {PIP_X(id_W830) - 5., HOP4Y(7), PIP_X(id_W830) - top_wire_dist - 5., HOP4Y(5)},
          {PIP_X(id_W830) - top_wire_dist - 5., HOP4Y(5), PIP_X(id_W830) - 6., HOP4Y(5)},
          {PIP_X(id_W830) - 6., HOP4Y(5), PIP_X(id_W830) - top_wire_dist - 6., HOP4Y(3)},
          {PIP_X(id_W830) - top_wire_dist - 6., HOP4Y(3), PIP_X(id_W830) - 7., HOP4Y(3)},
          {PIP_X(id_W830) - 7., HOP4Y(3), PIP_X(id_W830) - top_wire_dist - 7., HOP4Y(1)},
          {PIP_X(id_W830) - top_wire_dist - 7., HOP4Y(1), -wrap_len - 7., HOP4Y(1)},
          {-wrap_len - 7., HOP4Y(1), -wrap_len - 7., HOP4Y(0)},
          {-wrap_len - 7., HOP4Y(0), PIP_X(id_E838) - 7., HOP4Y(0)},
          {PIP_X(id_E838) - 7., HOP4Y(0), PIP_X(id_E838) - 7., WIRE_Y(0)}}},
};
const int PIP_SRC_DST_LEN = 20;

static void get_pip_xy(CruSide side, float &off, float &x, float &y)
{
    switch (side) {
    case Top:
        x = off;
        y = cru_y + cru_h;
        break;
    case Bottom:
        x = off;
        y = cru_y;
        break;
    case Left:
        x = cru_x;
        y = off;
        break;
    case Right:
        x = cru_x + cru_w;
        y = off;
        break;
    case Center:
        x = cru_x + cru_w / 2.f;
        y = off;
        break;
    }
}

void gfxSetPipDefaultDecal(Arch *arch, PipInfo &pip)
{
    DecalXY active, inactive;
    std::vector<std::string> split_res;
    IdString src_loc_id, dst_loc_id;
    char buf[PIP_SRC_DST_LEN];

    active.x = inactive.x = pip.loc.x;
    active.y = inactive.y = arch->gridDimY - 1. - pip.loc.y;
    boost::split(split_res, pip.name.str(arch), [](char c) { return c == '_'; });
    src_loc_id = arch->id(split_res.at(1));
    dst_loc_id = arch->id(split_res.at(2));
    snprintf(buf, PIP_SRC_DST_LEN, "%s_%s_active", src_loc_id.c_str(arch), dst_loc_id.c_str(arch));
    IdString active_id = arch->id(buf);
    active.decal = active_id;
    snprintf(buf, PIP_SRC_DST_LEN, "%s_%s_inactive", src_loc_id.c_str(arch), dst_loc_id.c_str(arch));
    IdString inactive_id = arch->id(buf);
    inactive.decal = inactive_id;
    // create if absent
    if (arch->decal_graphics.count(active_id) == 0) {
        // clock?
        if (dst_loc_id.in(id_GT00, id_GT10)) {
            WireInfo &wi = arch->wire_info(pip.srcWire);
            if (wi.type.str(arch).substr(0, 3) != "UNK") {
                // create pip
                GraphicElement el;
                el.type = GraphicElement::TYPE_LOCAL_LINE;
                el.style = GraphicElement::STYLE_ACTIVE;
                if (dst_loc_id == id_GT00) {
                    el.x1 = WIRE_X(CLK_GT00_X);
                } else {
                    el.x1 = WIRE_X(CLK_GT10_X);
                }
                el.x2 = el.x1 + spine_pip_off;
                el.y2 = spineY.at(arch->wire_info(pip.srcWire).type);
                el.y1 = el.y2 - spine_pip_off;
                arch->addDecalGraphic(active_id, el);
                el.style = GraphicElement::STYLE_INACTIVE;
                arch->addDecalGraphic(inactive_id, el);
            }
        } else {
            // XXX
            if (pipPoint.count(src_loc_id) == 0 || pipPoint.count(dst_loc_id) == 0) {
                // std::cout << "*R" << pip.loc.y + 1 << "C" << pip.loc.x + 1 << " no " << pip.name.str(arch) << " " <<
                // buf << std::endl;
            } else {
                GraphicElement el;
                el.type = GraphicElement::TYPE_LOCAL_ARROW;
                el.style = GraphicElement::STYLE_ACTIVE;
                CruSide srcSide = pipPoint.at(src_loc_id).first;
                float srcOff = pipPoint.at(src_loc_id).second;
                CruSide dstSide = pipPoint.at(dst_loc_id).first;
                float dstOff = pipPoint.at(dst_loc_id).second;
                if (srcSide != dstSide) {
                    get_pip_xy(srcSide, srcOff, el.x1, el.y1);
                    get_pip_xy(dstSide, dstOff, el.x2, el.y2);
                    arch->addDecalGraphic(active_id, el);
                    el.style = GraphicElement::STYLE_HIDDEN;
                    arch->addDecalGraphic(inactive_id, el);
                } else {
                    get_pip_xy(srcSide, srcOff, el.x1, el.y1);
                    float dst_x = 0, dst_y = 0, m_x = 0, m_y = 0;
                    get_pip_xy(dstSide, dstOff, dst_x, dst_y);
                    switch (dstSide) {
                    case Top:
                        m_x = el.x1 + (dst_x - el.x1) / 2.f;
                        m_y = dst_y - std::max(cru_h * 0.1f, std::min(cru_h * 0.4f, std::abs(el.x1 - dst_x)));
                        break;
                    case Bottom:
                        m_x = el.x1 + (dst_x - el.x1) / 2.f;
                        m_y = dst_y + std::max(cru_h * 0.1f, std::min(cru_h * 0.4f, std::abs(el.x1 - dst_x)));
                        break;
                    case Right:
                        m_x = dst_x - std::max(cru_w * 0.1f, std::min(cru_w * 0.4f, std::abs(el.y1 - dst_y)));
                        m_y = el.y1 + (dst_y - el.y1) / 2.f;
                        break;
                    case Left:
                        m_x = dst_x + std::max(cru_w * 0.1f, std::min(cru_w * 0.4f, std::abs(el.y1 - dst_y)));
                        m_y = el.y1 + (dst_y - el.y1) / 2.f;
                        break;
                    default: // unreachable
                        break;
                    }
                    el.x2 = m_x;
                    el.y2 = m_y;
                    arch->addDecalGraphic(active_id, el);
                    el.style = GraphicElement::STYLE_HIDDEN;
                    arch->addDecalGraphic(inactive_id, el);
                    el.style = GraphicElement::STYLE_ACTIVE;
                    el.x1 = m_x;
                    el.y1 = m_y;
                    el.x2 = dst_x;
                    el.y2 = dst_y;
                    arch->addDecalGraphic(active_id, el);
                    el.style = GraphicElement::STYLE_HIDDEN;
                    arch->addDecalGraphic(inactive_id, el);
                }
            }
        }
    }
    arch->setPipDecal(pip.name, active, inactive);
}

const int WIRE_ID_LEN = 30;

void gfxSetWireDefaultDecal(Arch *arch, WireInfo &wire)
{
    DecalXY active, inactive;
    IdString active_id, inactive_id;
    GraphicElement el;
    std::vector<std::string> split_res;
    char buf[WIRE_ID_LEN];

    if (std::find(decalless_wires.begin(), decalless_wires.end(), wire.name) != decalless_wires.end()) {
        arch->setWireDecal(wire.type, DecalXY(), DecalXY());
        return;
    }
    // local to cell
    if (arch->haveBelType(wire.x, wire.y, id_SLICE) && sliceLocalWires.count(wire.type) != 0) {
        snprintf(buf, sizeof(buf), "%s_active", wire.type.c_str(arch));
        active_id = arch->id(buf);
        active.decal = active_id;
        snprintf(buf, sizeof(buf), "%s_inactive", wire.type.c_str(arch));
        inactive_id = arch->id(buf);
        inactive.decal = inactive_id;
        active.x = inactive.x = wire.x;
        active.y = inactive.y = arch->gridDimY - 1. - wire.y;

        // create if absent
        if (arch->decal_graphics.count(active_id) == 0) {
            el.type = GraphicElement::TYPE_LOCAL_LINE;
            for (auto seg : sliceLocalWires.at(wire.type)) {
                el.style = GraphicElement::STYLE_ACTIVE;
                el.x1 = std::get<0>(seg);
                el.y1 = std::get<1>(seg);
                el.x2 = std::get<2>(seg);
                el.y2 = std::get<3>(seg);
                arch->addDecalGraphic(active_id, el);
                el.style = GraphicElement::STYLE_INACTIVE;
                arch->addDecalGraphic(inactive_id, el);
            }
        }
        arch->setWireDecal(wire.name, active, inactive);
        return;
    }
    // spines
    if (spineY.count(wire.type) != 0) {
        snprintf(buf, sizeof(buf), "%s_active", wire.type.c_str(arch));
        active_id = arch->id(buf);
        active.decal = active_id;
        snprintf(buf, sizeof(buf), "%s_inactive", wire.type.c_str(arch));
        inactive_id = arch->id(buf);
        inactive.decal = inactive_id;
        active.x = inactive.x = 0.;
        active.y = inactive.y = 0.;

        // update clock spines cache
        arch->updateClockSpinesCache(wire.type, wire.name);

        if (arch->decal_graphics.count(active_id) == 0) {
            el.type = GraphicElement::TYPE_LINE;
            el.style = GraphicElement::STYLE_ACTIVE;
            el.x1 = 0.2;                                        // cell's x will be added later in fixClockSpineDecals
            el.x2 = 0.7;                                        // cell's x will be added later in fixClockSpineDecals
            el.y1 = spineY.at(wire.type) + arch->gridDimY - 1.; // cell's y will be added later in fixClockSpineDecals
            el.y2 = el.y1;
            arch->addDecalGraphic(active_id, el);
            el.style = GraphicElement::STYLE_HIDDEN;
            arch->addDecalGraphic(inactive_id, el);
        }
        arch->setWireDecal(wire.name, active, inactive);
        return;
    }

    // global simple wires like IMUX
    if (globalSimpleWires.count(wire.type) != 0) {
        snprintf(buf, sizeof(buf), "%s_active", wire.name.c_str(arch));
        active_id = arch->id(buf);
        active.decal = active_id;
        snprintf(buf, sizeof(buf), "%s_inactive", wire.name.c_str(arch));
        inactive_id = arch->id(buf);
        inactive.decal = inactive_id;
        active.x = inactive.x = 0;
        active.y = inactive.y = 0;

        // create if absent
        if (arch->decal_graphics.count(active_id) == 0) {
            el.type = GraphicElement::TYPE_LINE;
            for (auto seg : globalSimpleWires.at(wire.type)) {
                el.style = GraphicElement::STYLE_ACTIVE;
                el.x1 = std::get<0>(seg) + wire.x;
                el.y1 = std::get<1>(seg) + arch->gridDimY - 1. - wire.y;
                el.x2 = std::get<2>(seg) + wire.x;
                el.y2 = std::get<3>(seg) + arch->gridDimY - 1. - wire.y;
                arch->addDecalGraphic(active_id, el);
                el.style = GraphicElement::STYLE_INACTIVE;
                arch->addDecalGraphic(inactive_id, el);
            }
        }
        arch->setWireDecal(wire.name, active, inactive);
        return;
    }

    // global
    boost::split(split_res, wire.name.str(arch), [](char c) { return c == '_'; });
    if (split_res.size() >= 2) {
        IdString wire_id = arch->id(split_res.at(1));
        // wrap
        if ((wire.y == (arch->gridDimY - 1) && split_res.at(1).at(0) == 'S') ||
            (wire.y == 0 && split_res.at(1).at(0) == 'N')) {
            wire_id = arch->id(split_res.at(1) + "_loop0");
        }
        if ((wire.x == (arch->gridDimX - 1) && split_res.at(1).at(0) == 'E') ||
            (wire.x == 0 && split_res.at(1).at(0) == 'W')) {
            wire_id = arch->id(split_res.at(1) + "_loop0");
        }
        // SN wires
        if (split_res.at(1).substr(0, 2) == "SN") {
            if (wire.y == 0) {
                wire_id = arch->id(split_res.at(1) + "_loop_n");
            } else {
                if (wire.y == (arch->gridDimY - 1)) {
                    wire_id = arch->id(split_res.at(1) + "_loop_s");
                }
            }
        } else {
            // wrap 2 hop
            if ((wire.y == (arch->gridDimY - 2) && split_res.at(1).substr(0, 2) == "S2") ||
                (wire.y == 1 && split_res.at(1).substr(0, 2) == "N2")) {
                wire_id = arch->id(split_res.at(1) + "_loop1");
            }
            // wrap 4 hop
            if (split_res.at(1).substr(0, 2) == "S8" || split_res.at(1).substr(0, 2) == "N8") {
                char loop_buf[5 + 2];
                if (split_res.at(1).substr(0, 2) == "N8") {
                    if (wire.y < 8) {
                        snprintf(loop_buf, sizeof(loop_buf), "_loop%1u", wire.y);
                        wire_id = arch->id(split_res.at(1) + loop_buf);
                    }
                } else {
                    if (arch->gridDimY - 1 - wire.y < 8) {
                        snprintf(loop_buf, sizeof(loop_buf), "_loop%1u", arch->gridDimY - 1 - wire.y);
                        wire_id = arch->id(split_res.at(1) + loop_buf);
                    }
                }
            }
        }
        // EW wires
        if (split_res.at(1).substr(0, 2) == "EW") {
            if (wire.x == 0) {
                wire_id = arch->id(split_res.at(1) + "_loop_w");
            } else {
                if (wire.x == (arch->gridDimX - 1)) {
                    wire_id = arch->id(split_res.at(1) + "_loop_e");
                }
            }
        } else {
            // wrap 2 hop
            if ((wire.x == (arch->gridDimX - 2) && split_res.at(1).substr(0, 2) == "E2") ||
                (wire.x == 1 && split_res.at(1).substr(0, 2) == "W2")) {
                wire_id = arch->id(split_res.at(1) + "_loop1");
            }
            // wrap 4 hop
            if (split_res.at(1).substr(0, 2) == "E8" || split_res.at(1).substr(0, 2) == "W8") {
                char loop_buf[5 + 2];
                if (split_res.at(1).substr(0, 2) == "W8") {
                    if (wire.x < 8) {
                        snprintf(loop_buf, sizeof(loop_buf), "_loop%1u", wire.x);
                        wire_id = arch->id(split_res.at(1) + loop_buf);
                    }
                } else {
                    if (arch->gridDimX - 1 - wire.x < 8) {
                        snprintf(loop_buf, sizeof(loop_buf), "_loop%1u", arch->gridDimX - 1 - wire.x);
                        wire_id = arch->id(split_res.at(1) + loop_buf);
                    }
                }
            }
        }
        // really create decal
        if (globalWires.count(wire_id) != 0) {
            snprintf(buf, sizeof(buf), "%s_active", wire.name.c_str(arch));
            active_id = arch->id(buf);
            active.decal = active_id;
            snprintf(buf, sizeof(buf), "%s_inactive", wire.name.c_str(arch));
            inactive_id = arch->id(buf);
            inactive.decal = inactive_id;
            active.x = inactive.x = 0;
            active.y = inactive.y = 0;

            // create if absent
            if (arch->decal_graphics.count(active_id) == 0) {
                el.type = GraphicElement::TYPE_LINE;
                for (auto seg : globalWires.at(wire_id)) {
                    el.style = GraphicElement::STYLE_ACTIVE;
                    el.x1 = std::get<0>(seg) + wire.x;
                    el.y1 = std::get<1>(seg) + arch->gridDimY - 1. - wire.y;
                    el.x2 = std::get<2>(seg) + wire.x;
                    el.y2 = std::get<3>(seg) + arch->gridDimY - 1. - wire.y;
                    arch->addDecalGraphic(active_id, el);
                    el.style = GraphicElement::STYLE_INACTIVE;
                    arch->addDecalGraphic(inactive_id, el);
                }
            }
            arch->setWireDecal(wire.name, active, inactive);
            return;
        }
        // clock branches
        // # of rows is unknown so generate wire ids at runtime
        if (split_res.at(1).substr(0, 3) == "GBO") {
            snprintf(buf, sizeof(buf), "%s_active", wire.name.c_str(arch));
            active_id = arch->id(buf);
            active.decal = active_id;
            snprintf(buf, sizeof(buf), "%s_inactive", wire.name.c_str(arch));
            inactive_id = arch->id(buf);
            inactive.decal = inactive_id;
            active.x = inactive.x = 0;
            active.y = inactive.y = 0;

            float pip_x = PIP_X(id_GBO0);
            float line_y = WIRE_Y(CLK_GBO0_Y) + arch->gridDimY - 1. - wire.y;
            float line_0 = WIRE_Y(0) + arch->gridDimY - 1. - wire.y;
            if (split_res.at(1).at(3) == '1') {
                pip_x = PIP_X(id_GBO1);
                line_y = WIRE_Y(CLK_GBO1_Y) + arch->gridDimY - 1. - wire.y;
            }

            // create if absent
            if (arch->decal_graphics.count(active_id) == 0) {
                el.type = GraphicElement::TYPE_LINE;
                el.style = GraphicElement::STYLE_ACTIVE;
                el.x1 = wire.x + pip_x;
                el.y1 = line_y;
                el.x2 = el.x1;
                el.y2 = line_0;
                arch->addDecalGraphic(active_id, el);
                el.style = GraphicElement::STYLE_HIDDEN;
                arch->addDecalGraphic(inactive_id, el);

                el.style = GraphicElement::STYLE_ACTIVE;
                el.x1 = pip_x;
                el.y1 = line_y;
                el.x2 = pip_x + arch->gridDimX - 1.;
                el.y2 = el.y1;
                arch->addDecalGraphic(active_id, el);
                el.style = GraphicElement::STYLE_HIDDEN;
                arch->addDecalGraphic(inactive_id, el);
            }
            arch->setWireDecal(wire.name, active, inactive);
            return;
        } else {
            if (split_res.at(1).substr(0, 2) == "GT") {
                snprintf(buf, sizeof(buf), "%s_active", wire.name.c_str(arch));
                active_id = arch->id(buf);
                active.decal = active_id;
                snprintf(buf, sizeof(buf), "%s_inactive", wire.name.c_str(arch));
                inactive_id = arch->id(buf);
                inactive.decal = inactive_id;
                active.x = inactive.x = 0;
                active.y = inactive.y = 0;

                float pip_y = PIP_Y(id_GT00);
                float line_x = WIRE_X(CLK_GT00_X) + wire.x;
                float line_0 = WIRE_X(0) + wire.x;
                if (split_res.at(1).at(2) == '1') {
                    pip_y = PIP_Y(id_GT10);
                    line_x = WIRE_X(CLK_GT10_X) + wire.x;
                }

                // create if absent
                if (arch->decal_graphics.count(active_id) == 0) {
                    el.type = GraphicElement::TYPE_LINE;
                    el.style = GraphicElement::STYLE_ACTIVE;
                    el.x1 = line_x;
                    el.y1 = pip_y + arch->gridDimY - 1.;
                    el.x2 = el.x1;
                    el.y2 = pip_y;
                    arch->addDecalGraphic(active_id, el);
                    el.style = GraphicElement::STYLE_HIDDEN;
                    arch->addDecalGraphic(inactive_id, el);

                    for (int i = 0; i <= arch->gridDimY - 1; ++i) {
                        el.style = GraphicElement::STYLE_ACTIVE;
                        el.x1 = line_x;
                        el.y1 = pip_y + arch->gridDimY - 1. - i;
                        el.x2 = line_0;
                        el.y2 = el.y1;
                        arch->addDecalGraphic(active_id, el);
                        el.style = GraphicElement::STYLE_HIDDEN;
                        arch->addDecalGraphic(inactive_id, el);
                    }
                }
                arch->setWireDecal(wire.name, active, inactive);
                return;
            } else {
                if (split_res.at(1).substr(0, 2) == "GB") {
                    snprintf(buf, sizeof(buf), "%s_active", wire.name.c_str(arch));
                    active_id = arch->id(buf);
                    active.decal = active_id;
                    snprintf(buf, sizeof(buf), "%s_inactive", wire.name.c_str(arch));
                    inactive_id = arch->id(buf);
                    inactive.decal = inactive_id;
                    active.x = inactive.x = 0;
                    active.y = inactive.y = 0;

                    float line_y = WIRE_Y(CLK_GBO0_Y) + arch->gridDimY - 1. - wire.y;
                    float line_0 = WIRE_Y(0) + arch->gridDimY - 1. - wire.y;
                    float pip_x = PIP_X(arch->id(split_res.at(1)));
                    if (split_res.at(1).at(2) >= '4') {
                        line_y = WIRE_Y(CLK_GBO1_Y) + arch->gridDimY - 1. - wire.y;
                    }

                    // create if absent
                    if (arch->decal_graphics.count(active_id) == 0) {
                        el.type = GraphicElement::TYPE_LINE;
                        el.style = GraphicElement::STYLE_ACTIVE;
                        el.x1 = wire.x + pip_x;
                        el.y1 = line_y;
                        el.x2 = el.x1;
                        el.y2 = line_0;
                        arch->addDecalGraphic(active_id, el);
                        el.style = GraphicElement::STYLE_INACTIVE;
                        arch->addDecalGraphic(inactive_id, el);
                    }
                    arch->setWireDecal(wire.name, active, inactive);
                    return;
                }
            }
        }
    }
    // std::cout << wire.name.str(arch) << ":" << wire.type.str(arch) << " R" << wire.y + 1 << "C" << wire.x + 1 <<
    // std::endl;
}

void gfxCreateBelDecals(Arch *arch)
{
    GraphicElement el;
    // LUTs
    el.type = GraphicElement::TYPE_BOX;
    el.style = GraphicElement::STYLE_ACTIVE;
    el.x1 = lut_x;
    el.x2 = el.x1 + lut_w;
    el.y1 = 0.;
    el.y2 = el.y1 + lut_h;
    arch->addDecalGraphic(id_DECAL_LUT_ACTIVE, el);
    arch->addDecalGraphic(id_DECAL_LUTDFF_ACTIVE, el);
    arch->addDecalGraphic(id_DECAL_LUT_UNUSED_DFF_ACTIVE, el);
    arch->addDecalGraphic(id_DECAL_ALU_ACTIVE, el);
    el.style = GraphicElement::STYLE_INACTIVE;
    arch->addDecalGraphic(id_DECAL_LUT_INACTIVE, el);
    arch->addDecalGraphic(id_DECAL_LUTDFF_INACTIVE, el);
    el.x1 = dff_x;
    el.x2 = el.x1 + dff_w;
    el.y1 = 0.;
    el.y2 = el.y1 + lut_h;
    arch->addDecalGraphic(id_DECAL_LUTDFF_INACTIVE, el);
    arch->addDecalGraphic(id_DECAL_LUT_UNUSED_DFF_ACTIVE, el);
    arch->addDecalGraphic(id_DECAL_ALU_ACTIVE, el);
    el.style = GraphicElement::STYLE_ACTIVE;
    arch->addDecalGraphic(id_DECAL_LUTDFF_ACTIVE, el);
    el.type = GraphicElement::TYPE_LOCAL_LINE;
    el.x1 = lut_x + 0.33f * lut_w;
    el.x2 = el.x1 + 0.33f * lut_w;
    el.y1 = 0.66f * lut_h;
    el.y2 = el.y1;
    arch->addDecalGraphic(id_DECAL_ALU_ACTIVE, el);
    el.y1 = 0.3f * lut_h;
    el.y2 = el.y1;
    arch->addDecalGraphic(id_DECAL_ALU_ACTIVE, el);
    el.x1 = lut_x + 0.5f * lut_w;
    el.x2 = el.x1;
    el.y1 = 0.5f * lut_h;
    el.y2 = el.y1 + 0.33f * lut_h;
    arch->addDecalGraphic(id_DECAL_ALU_ACTIVE, el);

    // LUT group
    el.type = GraphicElement::TYPE_BOX;
    el.style = GraphicElement::STYLE_FRAME;
    el.x1 = grp_lut_x;
    el.x2 = el.x1 + grp_lut_w;
    el.y1 = 0.;
    el.y2 = el.y1 + grp_lut_h;
    arch->addDecalGraphic(id_DECAL_GRP_LUT, el);

    // CRU group
    el.type = GraphicElement::TYPE_BOX;
    el.style = GraphicElement::STYLE_FRAME;
    el.x1 = cru_x;
    el.x2 = el.x1 + cru_w;
    el.y1 = cru_y;
    el.y2 = el.y1 + cru_h;
    arch->addDecalGraphic(id_DECAL_CRU, el);

    // Mux with upper 1 input
    el.type = GraphicElement::TYPE_LINE;
    el.style = GraphicElement::STYLE_INACTIVE;
    el.x1 = 0.;
    el.x2 = mux_w;
    el.y1 = 0.;
    el.y2 = mux_f;
    arch->addDecalGraphic(id_DECAL_MUXUPPER_INACTIVE, el);
    el.style = GraphicElement::STYLE_ACTIVE;
    arch->addDecalGraphic(id_DECAL_MUXUPPER_ACTIVE, el);
    el.style = GraphicElement::STYLE_INACTIVE;
    el.x1 = el.x2;
    el.y1 = el.y2;
    el.y2 = mux_h - mux_f;
    arch->addDecalGraphic(id_DECAL_MUXUPPER_INACTIVE, el);
    el.style = GraphicElement::STYLE_ACTIVE;
    arch->addDecalGraphic(id_DECAL_MUXUPPER_ACTIVE, el);
    el.style = GraphicElement::STYLE_INACTIVE;
    el.x2 = 0.;
    el.y1 = el.y2;
    el.y2 = mux_h;
    arch->addDecalGraphic(id_DECAL_MUXUPPER_INACTIVE, el);
    el.style = GraphicElement::STYLE_ACTIVE;
    arch->addDecalGraphic(id_DECAL_MUXUPPER_ACTIVE, el);
    el.style = GraphicElement::STYLE_INACTIVE;
    el.x1 = el.x2;
    el.y1 = mux_h;
    el.y2 = 0.;
    arch->addDecalGraphic(id_DECAL_MUXUPPER_INACTIVE, el);
    el.style = GraphicElement::STYLE_ACTIVE;
    arch->addDecalGraphic(id_DECAL_MUXUPPER_ACTIVE, el);
    // 1
    el.style = GraphicElement::STYLE_INACTIVE;
    el.x1 = 0.0038;
    el.x2 = 0.0118;
    el.y1 = el.y2 = 0.0598;
    arch->addDecalGraphic(id_DECAL_MUXUPPER_INACTIVE, el);
    el.style = GraphicElement::STYLE_ACTIVE;
    arch->addDecalGraphic(id_DECAL_MUXUPPER_ACTIVE, el);
    el.style = GraphicElement::STYLE_INACTIVE;
    el.x1 = (el.x1 + el.x2) / 2.;
    el.x2 = el.x1;
    el.y2 = 0.0808;
    arch->addDecalGraphic(id_DECAL_MUXUPPER_INACTIVE, el);
    el.style = GraphicElement::STYLE_ACTIVE;
    arch->addDecalGraphic(id_DECAL_MUXUPPER_ACTIVE, el);
    el.style = GraphicElement::STYLE_INACTIVE;
    el.x2 = 0.0038;
    el.y1 = el.y2;
    el.y2 = 0.0797;
    arch->addDecalGraphic(id_DECAL_MUXUPPER_INACTIVE, el);
    el.style = GraphicElement::STYLE_ACTIVE;
    arch->addDecalGraphic(id_DECAL_MUXUPPER_ACTIVE, el);

    // Mux with lower 1 input
    el.type = GraphicElement::TYPE_LINE;
    el.style = GraphicElement::STYLE_INACTIVE;
    el.x1 = 0.;
    el.x2 = mux_w;
    el.y1 = 0.;
    el.y2 = mux_f;
    arch->addDecalGraphic(id_DECAL_MUXLOWER_INACTIVE, el);
    el.style = GraphicElement::STYLE_ACTIVE;
    arch->addDecalGraphic(id_DECAL_MUXLOWER_ACTIVE, el);
    el.style = GraphicElement::STYLE_INACTIVE;
    el.x1 = el.x2;
    el.y1 = el.y2;
    el.y2 = mux_h - mux_f;
    arch->addDecalGraphic(id_DECAL_MUXLOWER_INACTIVE, el);
    el.style = GraphicElement::STYLE_ACTIVE;
    arch->addDecalGraphic(id_DECAL_MUXLOWER_ACTIVE, el);
    el.style = GraphicElement::STYLE_INACTIVE;
    el.x2 = 0.;
    el.y1 = el.y2;
    el.y2 = mux_h;
    arch->addDecalGraphic(id_DECAL_MUXLOWER_INACTIVE, el);
    el.style = GraphicElement::STYLE_ACTIVE;
    arch->addDecalGraphic(id_DECAL_MUXLOWER_ACTIVE, el);
    el.style = GraphicElement::STYLE_INACTIVE;
    el.x1 = el.x2;
    el.y1 = mux_h;
    el.y2 = 0.;
    arch->addDecalGraphic(id_DECAL_MUXLOWER_INACTIVE, el);
    el.style = GraphicElement::STYLE_ACTIVE;
    arch->addDecalGraphic(id_DECAL_MUXLOWER_ACTIVE, el);
    // 1
    el.style = GraphicElement::STYLE_INACTIVE;
    el.x1 = 0.0038;
    el.x2 = 0.0118;
    el.y1 = el.y2 = 0.0140;
    arch->addDecalGraphic(id_DECAL_MUXLOWER_INACTIVE, el);
    el.style = GraphicElement::STYLE_ACTIVE;
    arch->addDecalGraphic(id_DECAL_MUXLOWER_ACTIVE, el);
    el.style = GraphicElement::STYLE_INACTIVE;
    el.x1 = (el.x1 + el.x2) / 2.;
    el.x2 = el.x1;
    el.y2 = 0.0352;
    arch->addDecalGraphic(id_DECAL_MUXLOWER_INACTIVE, el);
    el.style = GraphicElement::STYLE_ACTIVE;
    arch->addDecalGraphic(id_DECAL_MUXLOWER_ACTIVE, el);
    el.style = GraphicElement::STYLE_INACTIVE;
    el.x2 = 0.0038;
    el.y1 = el.y2;
    el.y2 = 0.0341;
    arch->addDecalGraphic(id_DECAL_MUXLOWER_INACTIVE, el);
    el.style = GraphicElement::STYLE_ACTIVE;
    arch->addDecalGraphic(id_DECAL_MUXLOWER_ACTIVE, el);

    // IOB
    el.type = GraphicElement::TYPE_LINE;
    el.style = GraphicElement::STYLE_INACTIVE;
    el.x1 = 0.;
    el.x2 = io_w;
    el.y1 = 0.;
    el.y2 = el.y1;
    arch->addDecalGraphic(id_DECAL_IOB_INACTIVE, el);
    el.style = GraphicElement::STYLE_ACTIVE;
    arch->addDecalGraphic(id_DECAL_IOB_ACTIVE, el);
    el.style = GraphicElement::STYLE_INACTIVE;
    el.x1 = el.x2;
    el.y2 = io_h;
    arch->addDecalGraphic(id_DECAL_IOB_INACTIVE, el);
    el.style = GraphicElement::STYLE_ACTIVE;
    arch->addDecalGraphic(id_DECAL_IOB_ACTIVE, el);
    el.style = GraphicElement::STYLE_INACTIVE;
    el.x1 = 0.;
    el.y1 = el.y2;
    arch->addDecalGraphic(id_DECAL_IOB_INACTIVE, el);
    el.style = GraphicElement::STYLE_ACTIVE;
    arch->addDecalGraphic(id_DECAL_IOB_ACTIVE, el);
    el.style = GraphicElement::STYLE_INACTIVE;
    el.x2 = el.x1;
    el.y2 = 0.;
    arch->addDecalGraphic(id_DECAL_IOB_INACTIVE, el);
    el.style = GraphicElement::STYLE_ACTIVE;
    arch->addDecalGraphic(id_DECAL_IOB_ACTIVE, el);
    el.style = GraphicElement::STYLE_INACTIVE;
    el.x1 = io_w;
    el.x2 = io_w * 1.3f;
    el.y2 = el.y1 = io_h / 2.f;
    arch->addDecalGraphic(id_DECAL_IOB_INACTIVE, el);
    el.style = GraphicElement::STYLE_ACTIVE;
    arch->addDecalGraphic(id_DECAL_IOB_ACTIVE, el);

    // IOBS
    el.type = GraphicElement::TYPE_LINE;
    el.style = GraphicElement::STYLE_INACTIVE;
    el.x1 = 0.;
    el.x2 = ios_w;
    el.y1 = 0.;
    el.y2 = el.y1;
    arch->addDecalGraphic(id_DECAL_IOBS_INACTIVE, el);
    el.style = GraphicElement::STYLE_ACTIVE;
    arch->addDecalGraphic(id_DECAL_IOBS_ACTIVE, el);
    el.style = GraphicElement::STYLE_INACTIVE;
    el.x1 = el.x2;
    el.y2 = ios_h;
    arch->addDecalGraphic(id_DECAL_IOBS_INACTIVE, el);
    el.style = GraphicElement::STYLE_ACTIVE;
    arch->addDecalGraphic(id_DECAL_IOBS_ACTIVE, el);
    el.style = GraphicElement::STYLE_INACTIVE;
    el.x1 = 0.;
    el.y1 = el.y2;
    arch->addDecalGraphic(id_DECAL_IOBS_INACTIVE, el);
    el.style = GraphicElement::STYLE_ACTIVE;
    arch->addDecalGraphic(id_DECAL_IOBS_ACTIVE, el);
    el.style = GraphicElement::STYLE_INACTIVE;
    el.x2 = el.x1;
    el.y2 = 0.;
    arch->addDecalGraphic(id_DECAL_IOBS_INACTIVE, el);
    el.style = GraphicElement::STYLE_ACTIVE;
    arch->addDecalGraphic(id_DECAL_IOBS_ACTIVE, el);
    el.style = GraphicElement::STYLE_INACTIVE;
    el.x1 = ios_w;
    el.x2 = ios_w * 1.3f;
    el.y2 = el.y1 = ios_h / 2.f;
    arch->addDecalGraphic(id_DECAL_IOBS_INACTIVE, el);
    el.style = GraphicElement::STYLE_ACTIVE;
    arch->addDecalGraphic(id_DECAL_IOBS_ACTIVE, el);
}

void gfxSetBelDefaultDecal(Arch *arch, BelInfo &bel)
{
    DecalXY active, inactive;
    switch (bel.type.hash()) {
    case ID_SLICE:
        active.x = inactive.x = bel.x;
        active.y = inactive.y = arch->gridDimY - 1. - bel.y + lut_y[bel.z];
        if (bel.z < 6) {
            active.decal = id_DECAL_LUTDFF_ACTIVE;
            inactive.decal = id_DECAL_LUTDFF_INACTIVE;
        } else {
            active.decal = id_DECAL_LUT_ACTIVE;
            inactive.decal = id_DECAL_LUT_INACTIVE;
        }
        arch->setBelDecal(bel.name, active, inactive);
        break;
    case ID_MUX2_LUT5:
        active.x = inactive.x = bel.x + mux2lut5_x;
        active.y = inactive.y = arch->gridDimY - 1. - bel.y + mux2lut5_y[(bel.z - BelZ::mux_0_z) >> 1];
        active.decal = id_DECAL_MUXUPPER_ACTIVE;
        inactive.decal = id_DECAL_MUXUPPER_INACTIVE;
        arch->setBelDecal(bel.name, active, inactive);
        break;
    case ID_MUX2_LUT6:
        active.x = inactive.x = bel.x + mux2lut6_x;
        active.y = inactive.y = arch->gridDimY - 1. - bel.y + mux2lut6_y[(bel.z - BelZ::mux_0_z) / 5];
        active.decal = id_DECAL_MUXLOWER_ACTIVE;
        inactive.decal = id_DECAL_MUXLOWER_INACTIVE;
        arch->setBelDecal(bel.name, active, inactive);
        break;
    case ID_MUX2_LUT7:
        active.x = inactive.x = bel.x + mux2lut7_x;
        active.y = inactive.y = arch->gridDimY - 1. - bel.y + mux2lut7_y;
        active.decal = id_DECAL_MUXLOWER_ACTIVE;
        inactive.decal = id_DECAL_MUXLOWER_INACTIVE;
        arch->setBelDecal(bel.name, active, inactive);
        break;
    case ID_MUX2_LUT8:
        active.x = inactive.x = bel.x + mux2lut8_x;
        active.y = inactive.y = arch->gridDimY - 1. - bel.y + mux2lut8_y;
        active.decal = id_DECAL_MUXUPPER_ACTIVE;
        inactive.decal = id_DECAL_MUXUPPER_INACTIVE;
        arch->setBelDecal(bel.name, active, inactive);
        break;
    case ID_IOB:
        active.x = inactive.x = bel.x + io_x;
        active.y = inactive.y = arch->gridDimY - 1. - bel.y + io_y + bel.z * (2 * io_gap + io_h);
        active.decal = id_DECAL_IOB_ACTIVE;
        inactive.decal = id_DECAL_IOB_INACTIVE;
        arch->setBelDecal(bel.name, active, inactive);
        gfxSetIOBWireDecals(arch, bel);
        break;
    case ID_IOBS:
        active.x = inactive.x = bel.x + ios_x + (ios_w + ios_gap_x) * (bel.z % 3);
        active.y = inactive.y = arch->gridDimY - 1. - bel.y + ios_y + (ios_h + ios_gap_y) * (bel.z / 3);
        active.decal = id_DECAL_IOBS_ACTIVE;
        inactive.decal = id_DECAL_IOBS_INACTIVE;
        arch->setBelDecal(bel.name, active, inactive);
        gfxSetIOBSWireDecals(arch, bel);
        break;
    default:
        break;
    }
}

void gfxSetIOBWireDecals(Arch *arch, BelInfo &bel)
{
    DecalXY active, inactive;
    GraphicElement el;
    char buf[20];

    // set decals for I, O and OE input wires
    for (auto pi : bel.pins) {
        WireInfo &wi = arch->wire_info(pi.second.wire);
        // decal name: wire_port_z_active|inactive
        snprintf(buf, sizeof(buf), "%s_%s_%u_active", wi.type.c_str(arch), pi.first.c_str(arch), bel.z);
        IdString active_id = arch->id(buf);
        active.decal = active_id;
        snprintf(buf, sizeof(buf), "%s_%s_%u_inactive", wi.type.c_str(arch), pi.first.c_str(arch), bel.z);
        IdString inactive_id = arch->id(buf);
        inactive.decal = inactive_id;
        active.x = inactive.x = bel.x;
        active.y = inactive.y = arch->gridDimY - 1. - bel.y;
        if (arch->decal_graphics.count(active_id) == 0) {
            el.type = GraphicElement::TYPE_LOCAL_LINE;
            el.style = GraphicElement::STYLE_ACTIVE;
            el.x1 = cru_x + cru_w;
            el.y1 = pipPoint.at(wi.type).second;
            el.x2 = io_x;
            el.y2 = portPoint.at(pi.first) + io_y + bel.z * (2 * io_gap + io_h);
            arch->addDecalGraphic(active_id, el);
            el.style = GraphicElement::STYLE_INACTIVE;
            arch->addDecalGraphic(inactive_id, el);
            for (auto seg : portSign.at(pi.first)) {
                el.style = GraphicElement::STYLE_ACTIVE;
                el.x1 = std::get<0>(seg) + io_x;
                el.y1 = std::get<1>(seg) + io_y + bel.z * (2 * io_gap + io_h);
                el.x2 = std::get<2>(seg) + io_x;
                el.y2 = std::get<3>(seg) + io_y + bel.z * (2 * io_gap + io_h);
                arch->addDecalGraphic(active_id, el);
                el.style = GraphicElement::STYLE_INACTIVE;
                arch->addDecalGraphic(inactive_id, el);
            }
        }
        arch->setWireDecal(wi.name, active, inactive);
    }
}

void gfxSetIOBSWireDecals(Arch *arch, BelInfo &bel)
{
    DecalXY active, inactive;
    GraphicElement el;
    char buf[20];

    // set decals for I, O and OE input wires
    for (auto pi : bel.pins) {
        WireInfo &wi = arch->wire_info(pi.second.wire);
        // decal name: ios_wire_port_z_active|inactive
        snprintf(buf, sizeof(buf), "ios_%s_%s_%u_active", wi.type.c_str(arch), pi.first.c_str(arch), bel.z);
        IdString active_id = arch->id(buf);
        active.decal = active_id;
        snprintf(buf, sizeof(buf), "ios_%s_%s_%u_inactive", wi.type.c_str(arch), pi.first.c_str(arch), bel.z);
        IdString inactive_id = arch->id(buf);
        inactive.decal = inactive_id;
        active.x = inactive.x = bel.x;
        active.y = inactive.y = arch->gridDimY - 1. - bel.y;
        if (arch->decal_graphics.count(active_id) == 0) {
            // leftmost wires
            el.type = GraphicElement::TYPE_LOCAL_LINE;
            if (bel.z % 3 == 0) {
                el.style = GraphicElement::STYLE_ACTIVE;
                el.x1 = cru_x + cru_w;
                el.y1 = pipPoint.at(wi.type).second;
                el.x2 = ios_x;
                el.y2 = ios_scl * portPoint.at(pi.first) + ios_y + (ios_h + ios_gap_y) * (bel.z / 3);
                arch->addDecalGraphic(active_id, el);
                el.style = GraphicElement::STYLE_INACTIVE;
                arch->addDecalGraphic(inactive_id, el);
            } else {
                float col = (bel.z % 3) - 1;
                float rel_port = portPoint.at(pi.first) / io_h;
                el.style = GraphicElement::STYLE_ACTIVE;
                el.x1 = cru_x + cru_w;
                el.y1 = pipPoint.at(wi.type).second;
                el.x2 = ios_x * (0.97 - 0.02 * col);
                el.y2 = (rel_port + col) * 0.5 * ios_gap_y + ios_y + ios_h + (ios_h + ios_gap_y) * (bel.z / 3);
                arch->addDecalGraphic(active_id, el);
                el.style = GraphicElement::STYLE_INACTIVE;
                arch->addDecalGraphic(inactive_id, el);
                el.style = GraphicElement::STYLE_ACTIVE;
                el.x1 = ios_x + (ios_w + ios_gap_x) * (col + 1) - ios_gap_x + ios_w * 0.3 +
                        rel_port * (ios_gap_x - 0.3 * ios_w);
                el.y1 = el.y2;
                arch->addDecalGraphic(active_id, el);
                el.style = GraphicElement::STYLE_INACTIVE;
                arch->addDecalGraphic(inactive_id, el);
                el.style = GraphicElement::STYLE_ACTIVE;
                el.x2 = el.x1;
                el.y2 = ios_scl * portPoint.at(pi.first) + ios_y + (ios_h + ios_gap_y) * (bel.z / 3);
                arch->addDecalGraphic(active_id, el);
                el.style = GraphicElement::STYLE_INACTIVE;
                arch->addDecalGraphic(inactive_id, el);
                el.style = GraphicElement::STYLE_ACTIVE;
                el.x1 = ios_x + (ios_w + ios_gap_x) * (col + 1);
                el.y1 = el.y2;
                arch->addDecalGraphic(active_id, el);
                el.style = GraphicElement::STYLE_INACTIVE;
                arch->addDecalGraphic(inactive_id, el);
                el.style = GraphicElement::STYLE_ACTIVE;
            }
            // signs
            for (auto seg : portSign.at(pi.first)) {
                el.style = GraphicElement::STYLE_ACTIVE;
                el.x1 = ios_scl * std::get<0>(seg) + ios_x + (ios_w + ios_gap_x) * (bel.z % 3);
                el.y1 = ios_scl * std::get<1>(seg) + ios_y + (ios_h + ios_gap_y) * (bel.z / 3);
                el.x2 = ios_scl * std::get<2>(seg) + ios_x + (ios_w + ios_gap_x) * (bel.z % 3);
                el.y2 = ios_scl * std::get<3>(seg) + ios_y + (ios_h + ios_gap_y) * (bel.z / 3);
                arch->addDecalGraphic(active_id, el);
                el.style = GraphicElement::STYLE_INACTIVE;
                arch->addDecalGraphic(inactive_id, el);
            }
        }
        arch->setWireDecal(wi.name, active, inactive);
    }
}

DecalXY gfxGetLutGroupDecalXY(int x, int y, int z)
{
    DecalXY decalxy;
    decalxy.decal = id_DECAL_GRP_LUT;
    decalxy.x = x;
    decalxy.y = y + grp_lut_y[z];
    return decalxy;
}

DecalXY gfxGetCruGroupDecalXY(int x, int y)
{
    DecalXY decalxy;
    decalxy.decal = id_DECAL_CRU;
    decalxy.x = x;
    decalxy.y = y;
    return decalxy;
}

NEXTPNR_NAMESPACE_END
