/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright 2009--2026 by Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */

#pragma once


/////////////////////// Stdlib includes
#include <map>


/////////////////////// Qt includes
#include <QString>
#include <QList>


/////////////////////// pappsomspp includes


/////////////////////// libXpertMassCore includes
#include "MsXpS/libXpertMassCore/jsclassregistrar.h"
#include "MsXpS/libXpertMassCore/globals.hpp"
#include "MsXpS/libXpertMassCore/PolChemDef.hpp"
#include "MsXpS/libXpertMassCore/CleavageMotif.hpp"
#include "MsXpS/libXpertMassCore/CleavageRule.hpp"


/////////////////////// libXpertMassGui includes


/////////////////////// Local includes
#include "MsXpS/export-import-config.h"

namespace MsXpS
{
namespace libXpertMassCore
{


/*  BEGIN CLASS JS REFERENCE
 *  namespace: MsXpS::libXpertMassCore
 *  class name: CleavageAgent
 */

class DECLSPEC CleavageAgent: public QObject
{
  Q_OBJECT

  public:
  explicit CleavageAgent(QObject *parent = nullptr);
  explicit CleavageAgent(PolChemDefCstSPtr pol_chem_def_csp,
                         const QDomElement &element,
                         int version,
                         QObject *parent = nullptr);
  explicit CleavageAgent(PolChemDefCstSPtr pol_chem_def_csp,
                         const QString &name    = QString(),
                         const QString &pattern = QString(),
                         QObject *parent        = nullptr);

  // Pseudo copy-constructor that does not copy the QObject base class.
  explicit CleavageAgent(const CleavageAgent &other, QObject *parent = nullptr);

  // Prevent accidental copying
  CleavageAgent(const CleavageAgent &other) = delete;

  ~CleavageAgent();

  bool initialize(const CleavageAgent &other);

  //////////////// THE POLCHEMDEF /////////////////////
  void setPolchemDefCstSPtr(PolChemDefCstSPtr pol_chem_def_csp);
  PolChemDefCstSPtr getPolchemDefCstSPtr() const;

  //////////////// THE NAME /////////////////////
  void setName(const QString &name);
  const QString &getName() const;

  //////////////// THE PATTERN /////////////////////
  void setPattern(const QString &);
  const QString &getPattern() const;

  //////////////// THE MOTIFS /////////////////////
  const std::vector<CleavageMotifSPtr> &getMotifsCstRef() const;
  std::vector<CleavageMotifSPtr> &getMotifsRef();

  //////////////// THE RULES /////////////////////
  const std::vector<CleavageRuleSPtr> &getRulesCstRef() const;
  std::vector<CleavageRuleSPtr> &getRulesRef();
  CleavageRuleSPtr getCleavageRuleSPtrByName(const QString &name) const;

  //////////////// PARSING /////////////////////
  bool parse();

  //////////////// OPERATORS /////////////////////
  // Prevent accidental copying
  CleavageAgent &operator=(const CleavageAgent &other) = delete;
  bool operator==(const CleavageAgent &other) const;
  bool operator!=(const CleavageAgent &other) const;

  //////////////// VALIDATIONS /////////////////////
  CleavageAgentCstSPtr getFromPolChemDefByName() const;
  Enums::PolChemDefEntityStatus isKnownByNameInPolChemDef() const;

  CleavageRuleCstSPtr getCleavageRuleCstSPtrByName(const QString &name) const;
  int getCleavageRuleIndexByName(const QString &name) const;

  bool validate(ErrorList *error_list_p) const;
  bool isValid() const;

  //////////////// XML DATA LOADING WRITING /////////////////////
  bool renderXmlClsElement(const QDomElement &element, int version = 1);
  bool renderXmlClaElement(const QDomElement &element, int version = 2);

  QString
  formatXmlClaElement(int offset,
                      const QString &indent = Utils::xmlIndentationToken) const;

  //////////////// UTILS /////////////////////
  QString toString() const;

  static void registerJsConstructor(QJSEngine *engine);

  protected:
  PolChemDefCstSPtr mcsp_polChemDef = nullptr;
  QString m_name                    = "";
  QString m_pattern                 = "";

  std::vector<CleavageMotifSPtr> m_motifs;
  std::vector<CleavageRuleSPtr> m_rules;

  mutable bool m_isValid = false;
};

typedef std::shared_ptr<CleavageAgent> CleavageAgentSPtr;
typedef std::shared_ptr<const CleavageAgent> CleavageAgentCstSPtr;

/*  END CLASS JS REFERENCE
 *  namespace: MsXpS::libXpertMassCore
 *  class name: CleavageAgent
 */

} // namespace libXpertMassCore
MSXPS_REGISTER_JS_CLASS(MsXpS::libXpertMassCore, CleavageAgent)
} // namespace MsXpS
