/* -*- Mode: C++; tab-width: 4 -*- */
/* vi: set ts=4 sw=4 noexpandtab: */

/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009,2010 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * The tests in this file are based on code written by Florian Weimer:
 *
 * Copyright (C) 2009,2010 Florian Weimer <fw@deneb.enyo.de>
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

/**
 * @file diagnostics/extensions/stacktrace/stacktrace.t.cpp
 *
 * $Id: stacktrace.t.cpp 895 2011-05-26 17:19:57Z tautschn $
 *
 * @author Michael Tautschnig
 */

#include <diagnostics/unittest.hpp>

// tested component
#include <diagnostics/extensions/stacktrace/handler.hpp>
#include <diagnostics/extensions/stacktrace/posix.hpp>
#include <diagnostics/extensions/stacktrace/error.hpp>
#include <diagnostics/extensions/stacktrace/process.hpp>
#include <diagnostics/extensions/stacktrace/frame_visitor.hpp>

#include <vector>
#include <cerrno>

#define TEST_COMPONENT_NAME Stacktrace
#define TEST_COMPONENT_NAMESPACE diagnostics

DIAGNOSTICS_NAMESPACE_BEGIN;
TEST_NAMESPACE_BEGIN;
TEST_COMPONENT_TEST_NAMESPACE_BEGIN;

using namespace unittest;

void test_posix(Test_Data & test_data)
{
  using namespace ::diagnostics::stacktrace::POSIX;

  TEST_ASSERT(Exists("/"));
  TEST_ASSERT(Exists("."));
  TEST_ASSERT(Exists("test"));
  TEST_ASSERT(IsAbsolute("/"));
  TEST_ASSERT(IsExecutable("/"));
  TEST_ASSERT(IsExecutable("."));
  TEST_ASSERT(IsExecutable("test"));
  TEST_ASSERT(!Exists("/enyo does not exist"));
  TEST_ASSERT(IsAbsolute("/enyo does not exist"));
  TEST_ASSERT(!IsExecutable("/enyo does not exist"));
  TEST_ASSERT(!IsAbsolute("."));
  TEST_ASSERT(!IsAbsolute("./"));

  TEST_ASSERT_RELATION(SearchCommand("sh", "/bin:/usr/bin"), ==, "/bin/sh");
  TEST_ASSERT_RELATION(SearchCommand("sh", ".:/bin:/usr/bin"), ==, "/bin/sh");
  TEST_ASSERT_RELATION(SearchCommand("test", ".:/bin:/usr/bin"), ==,
             Cwd() + "/test");
  TEST_ASSERT_RELATION(SearchCommand("sh", ":/bin:/usr/bin"), ==, "/bin/sh");
  TEST_ASSERT_RELATION(SearchCommand("test", ":/bin:/usr/bin"), ==,
             Cwd() + "/test");
  TEST_ASSERT_RELATION(SearchCommand("test", "/enyo does not exist:"), ==,
             Cwd() + "/test");

  TEST_THROWING_BLOCK_ENTER;
  SearchCommand("sh", "/enyo does not exist:.");
  TEST_THROWING_BLOCK_EXIT2(Error, e.Code() == ENOENT);
  TEST_THROWING_BLOCK_ENTER;
  SearchCommand("Makefile", ".:/enyo does not exist");
  TEST_THROWING_BLOCK_EXIT2(Error, e.Code() == EACCES);
}

void test_posix_process(Test_Data & test_data)
{
  using namespace ::diagnostics::stacktrace::POSIX;

  ProcessResult res;
  static char *const args1[] =
    {"sh", "-c", "echo stdout; echo stderr >&2; exit 17", NULL};
  Run("/bin/sh", args1, NULL, res);
  TEST_ASSERT_RELATION(res.Output, ==, "stdout\n");
  TEST_ASSERT_RELATION(res.Error, ==, "stderr\n");
  TEST_ASSERT_RELATION(res.Status, ==, 17);

  static char *const args2[] =
    {"sh", "-c", "echo stdout-only; exit 18", NULL};
  Run("/bin/sh", args2, NULL, res);
  TEST_ASSERT_RELATION(res.Output, ==, "stdout-only\n");
  TEST_ASSERT_RELATION(res.Error, ==, "");
  TEST_ASSERT_RELATION(res.Status, ==, 18);

  static char *const args3[] =
    {"sh", "-c", "echo stderr-only >&2; exit 19", NULL};
  Run("/bin/sh", args3, NULL, res);
  TEST_ASSERT_RELATION(res.Output, ==, "");
  TEST_ASSERT_RELATION(res.Error, ==, "stderr-only\n");
  TEST_ASSERT_RELATION(res.Status, ==, 19);

  static char *const args4[] =
    {"sh", "-c",
     "sleep 1; echo sleep-stdout; echo sleep-stderr >&2; exit 20", NULL};
  Run("/bin/sh", args4, NULL, res);
  TEST_ASSERT_RELATION(res.Output, ==, "sleep-stdout\n");
  TEST_ASSERT_RELATION(res.Error, ==, "sleep-stderr\n");
  TEST_ASSERT_RELATION(res.Status, ==, 20);
}

using namespace ::diagnostics::stacktrace::Traceback;

namespace {
  struct my_frame {
    bool resolved;
    std::string function;
    std::string file;
    unsigned line;

    my_frame()
      : resolved(false), line(0)
    {
    }
  };

  struct my_visitor : FrameVisitor, SourceVisitor {
    std::vector<my_frame> loc;
    unsigned unresolved;

    my_visitor()
      : unresolved(0)
    {
    }

    bool Frame(void *, const char *dso, void *resolved)
    {
      if (!(resolved && WalkSource(dso, resolved))) {
        loc.push_back(my_frame());
      }
      return true;
    }

    void Frame(const char *func, const char *source, unsigned line)
    {
      my_frame f;
      f.resolved = true;
      f.function = func;
      f.file = source;
      f.line = line;
      loc.push_back(f);
    }
  };

  inline void
  call_my_visitor_2(my_visitor &v)
  {
    v.Walk();
  }

  inline void
  call_my_visitor_1(my_visitor &v)
  {
    call_my_visitor_2(v);
  }
}

void test_traceback(Test_Data & test_data)
{
  my_visitor v;
  call_my_visitor_1(v);
  TEST_ASSERT_RELATION(v.loc.at(0).function, ==, "call_my_visitor_2");
  TEST_ASSERT_RELATION(v.loc.at(1).function, ==, "call_my_visitor_1");
}

TEST_COMPONENT_TEST_NAMESPACE_END;
TEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;

TEST_SUITE_BEGIN;
TEST_NORMAL_CASE(&test_posix,LEVEL_PROD);
TEST_NORMAL_CASE(&test_posix_process,LEVEL_PROD);
#ifndef __FreeBSD_kernel__
TEST_NORMAL_CASE(&test_traceback,LEVEL_PROD);
#endif
TEST_SUITE_END;

STREAM_TEST_SYSTEM_MAIN;

