# SPDX-FileCopyrightText: Peter Pentchev <roam@ringlet.net>
# SPDX-License-Identifier: GPL-2.0-or-later
"""Parse the human-readable output of `debsigs --list`."""

from __future__ import annotations

import typing

import pyparsing as pyp

from testsigs import defs


if typing.TYPE_CHECKING:
    from typing import Final


_p_first = pyp.Literal("GPG signatures in ") + pyp.CharsNotIn("\n") + pyp.Char("\n")

_p_sig_type = pyp.Regex("[A-Za-z0-9_]+")

_p_hex = pyp.Word("0123456789ABCDEF")

_p_sig = (
    _p_sig_type
    + pyp.Literal(": signed by ").suppress()
    + _p_hex
    + pyp.Literal(" on ").suppress()
    + pyp.CharsNotIn("\n").suppress()
    + pyp.Char("\n").suppress()
)


@_p_sig.set_parse_action
def _parse_sig(tokens: pyp.ParseResults) -> defs.DebSig:
    """Parse a signature line."""
    match tokens.as_list():
        case [sig_type, key_id] if isinstance(sig_type, str) and isinstance(key_id, str):
            return defs.DebSig(sig_type=sig_type, key_id=key_id)

        case _:
            raise RuntimeError(repr(tokens))


_p_last = pyp.Literal(" *** NO ATTEMPT HAS BEEN MADE") + pyp.CharsNotIn("\n") + pyp.Char("\n")

_p_list = _p_first.suppress() + pyp.ZeroOrMore(_p_sig) + _p_last.suppress()

_p_list_complete = _p_list.leave_whitespace().parse_with_tabs()


def parse_list(output: str) -> list[defs.DebSig]:
    """Parse the output of `debsigs --list`, extract the info we care about."""
    res: Final = _p_list_complete.parse_string(output, parse_all=True).as_list()
    if not all(isinstance(sig, defs.DebSig) for sig in res):
        raise RuntimeError(repr(res))
    return res
