/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2016 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */


#include <string.h>
#include <stdlib.h>

#include "cdw_disc.h"
#include "cdw_drive.h"
#include "cdw_config.h"
#include "cdw_dvd_rw_mediainfo.h"
#include "cdw_cdrecord.h"
#include "cdw_xorriso.h"
#include "gettext.h"
#include "cdw_debug.h"
#include "cdw_main_window.h"
#include "cdw_widgets.h"
#include "cdw_processwin.h"
#include "cdw_fs.h"
#include "cdw_utils.h"
#include "cdw_ofs.h"

/**
   \file cdw_disc.c

   \brief Few functions that operate on variable of type cdw_disc_t.
   The variable - if correctly updated - represents most information
   about a disc in drive that cdw can get.


\verbatim
CD-R and CD-RW Disc Capacities
(capacities indicated in bytes)

(source: http://www.osta.org/technology/cdqa7.htm)

Disc    Playing   Audio       CD-ROM        CD-ROM        CD-i/XA       CD-i/XA
Size    Time                  Mode 1        Mode 2        Form 1        Form 2
                  2352        2048          2336          2048          2324     <-- bytes per sector

 8 cm  18 min   190 512 000   165 888 000   189 216 000   165 888 000   188 244 000
 8 cm  21 min   222 264 000   193 536 000   220 752 000   193 536 000   219 618 000
12 cm  63 min   666 792 000   580 608 000   662 256 000   580 608 000   658 854 000
12 cm  74 min   783 216 000   681 984 000   777 888 000   681 984 000   773 892 000
12 cm  80 min   846 720 000   737 280 000   840 960 000   737 280 000   836 640 000
\endverbatim


   Sources of information printed below are (in no particular order):
   \li http://www.mscience.com/faq502.html
   \li http://www.mscience.com/faq62.html
   \li http://www.mscience.com/faq60.html
   \li http://www.cdrlabs.com/forums/mode-versus-mode-discs-t8075.html
   \li "Upgrading and Repairing PCs" by Scott Mueller, Published by Que Publishing, 2003; ISBN 0789729741, 9780789729743
   \li http://www.cdrfaq.org/
   \li http://sony.storagesupport.com/node/6405
   \li http://www.ecma-international.org/publications/files/ECMA-ST/Ecma-130.pdf
   \li http://www.pcguide.com/ref/cd/formatXA-c.html

   Information goes as follows:
   - Red Book: defines CD-DA, audio only (no sector structure)
   - Yellow Book: defines:
       - Mode 0 (12B sync + 4B header + 2336B data (all zeros))
       - Mode 1 (12B sync + 4B header + 2048B data + 4B EDC + 8B blank + 276B data protection);
         Referred to as: "CD-ROM Mode 1", "Mode 1"
       - Mode 2 (12B sync + 4B header + 2336B data);
         Referred to as: "CD-ROM Mode 2", "Mode 2", "Mode2 Formless"

   - Yellow Book extension defines CD-ROM XA (CD-ROM/XA, eXtended Architecture):
       - Mode 1: (12B sync + 4B header + 2048B data + 4B EDC + 8B blank + 276B ECC)
       - Mode 2: for mixing sectors with and without data correction on the
         same track; introduces 8B subheader; Referred to as "CD-ROM/XA MODE 2"
       - Mode 2 Form 1: (12B sync + 4B header + 8B subheader + 2048B data + 4B EDC + 276B ECC)
         for error-sensitive data;
         Referred to as "CD-ROM/XA MODE 2 Form 1", "CD-ROM/XA Form 1", "Form 1"
       - Mode 2 Form 2: (12B sync + 4B header + 8B subheader + 2324B data + 4B EDC)
         for error-tolerant data (e.g. multimedia)
         Referred to as "CD-ROM/XA MODE 2 Form 2", "CD-ROM/XA Form 2", "Form 2"
   - Green Book: defines CD-i (CD-interactive, obsolete?, www.icdia.com);
     it also modified Yellow Book Mode 2, by adding:
       - Mode 2 Form 1 (8 bytes of subheader) + (2048 bytes of data)
       - Mode 2 Form 2 (2324 bytes of data)


   Some excerpts:
   http://www.mscience.com/faq62.html:
   "In practice, CD-ROM/XA should only be used for multimedia applications,
   because not all drives recognize the CD-ROM/XA format, while all data
   drives recognize CD-ROM Mode 1."

   http://www.cdrfaq.org/faq02.html#S2-2:
   "ECMA-119 describes ISO-9660, and ECMA-130 sounds a lot like "yellow book"
   if you say it slowly." ("yellow book" without XA extension)

   http://www.mscience.com/faq502.html
   "CD-ROM/XA added two new formats for multimedia applications, neither of
   which is the same as ISO 10149 Mode 2."
*/





extern cdw_config_t global_config; /* main cdw configuration variable */

static void cdw_disc_reset(cdw_disc_t *disc);

static bool cdw_disc_get_make_processwin(void);

static cdw_rv_t cdw_disc_get_meta_info(cdw_disc_t *disc);
static cdw_rv_t cdw_disc_get_meta_info_with_cdio(cdw_disc_t *disc);
static cdw_rv_t cdw_disc_get_meta_info_with_external_tools(cdw_disc_t *disc);

static cdw_rv_t cdw_disc_validate(cdw_disc_t *disc);
static void cdw_disc_resolve_type_label(cdw_disc_t *disc);
static void cdw_disc_resolve_write_speeds(cdw_disc_t *disc);
static void cdw_disc_resolve_capacities(cdw_disc_t *disc);

static void cdw_disc_debug_print_disc(cdw_disc_t *disc);





/* these labels are in the same order as in cdw_disc_type_t type
   declaration in cdw_disc.h */
static const char *cdw_disc_type_labels[] = {
	(char *) NULL, /* place for "unknown" */
	(char *) NULL, /* place for "none" */
	(char *) NULL, /* place for "CD-Audio" */
	"CD-R",
	"CD-RW",
	"CD-ROM",
	"DVD-R",
	"DVD-R Seq",
	"DVD-R Res",
	"DVD+R",
	"DVD-RW",
	"DVD-RW Seq",
	"DVD-RW Res",
	"DVD+RW",
	"DVD-ROM",
	"DVD+R DL" };


enum {
	CDW_DISC_NOT_READABLE_EMPTY,
	CDW_DISC_NOT_READABLE_NO_ISO9660,     /* no support for iso9660 in operating system kernel */
	CDW_DISC_NOT_READABLE_NO_UDF,         /* no support for udf in operating system kernel */
	CDW_DISC_NOT_READABLE_UNSUPPORTED_FS, /* file system not supported by cdw */
	CDW_DISC_NOT_READABLE_OTHER
};

cdw_id_clabel_t disc_not_readable_reasons[] = {
	{ CDW_DISC_NOT_READABLE_EMPTY,          (char *) NULL },
	{ CDW_DISC_NOT_READABLE_NO_ISO9660,     (char *) NULL },
	{ CDW_DISC_NOT_READABLE_NO_UDF,         (char *) NULL },
	{ CDW_DISC_NOT_READABLE_UNSUPPORTED_FS, (char *) NULL },
	{ CDW_DISC_NOT_READABLE_OTHER,          (char *) NULL },
	{ -1,                                   (char *) NULL }};



/**
   \brief Initialize cdw_disc module
*/
void cdw_disc_init(void)
{
	/* values returned by gettext() / _() are static strings
	   and they must not be free()d */

	/* 2TRANS: "unknown" as in "Disc type: unknown"; keep short */
	cdw_disc_type_labels[CDW_DISC_TYPE_UNKNOWN] = _("unknown");
	/* 2TRANS: this string means "there is no disc in drive"; keep short */
	cdw_disc_type_labels[CDW_DISC_NONE] = _("No disc");
	/* 2TRANS: this string means "regular CD disc with music"; keep short */
	cdw_disc_type_labels[CDW_CD_AUDIO] = _("Audio CD");

	/* 2TRANS: this is message printed in dialog window */
	disc_not_readable_reasons[CDW_DISC_NOT_READABLE_EMPTY].label = _("cdw can't read the disc because the disc is empty.");
	/* 2TRANS: this is message printed in dialog window */
	disc_not_readable_reasons[CDW_DISC_NOT_READABLE_NO_ISO9660].label = _("cdw can't read the disc because your operating system does not support ISO9660 file system.");
	/* 2TRANS: this is message printed in dialog window */
	disc_not_readable_reasons[CDW_DISC_NOT_READABLE_NO_UDF].label = _("cdw can't read the disc because your operating system does not support UDF file system.");
	/* 2TRANS: this is message printed in dialog window;
	   "not supported" means "cdw can't handle it" */
	disc_not_readable_reasons[CDW_DISC_NOT_READABLE_UNSUPPORTED_FS].label = _("cdw can't read the disc because file system on the disc is not supported by cdw.");
	/* 2TRANS: this is message printed in dialog window;
	   there is some generic/unknown issue/problem with
	   checking the disc */
	disc_not_readable_reasons[CDW_DISC_NOT_READABLE_OTHER].label = _("cdw can't read the disc.");



	return;
}





/**
   \brief Create new disc data structure, initialize it with default values

   \date Function's top-level comment reviewed on 2012-01-29
   \date Function's body reviewed on 2012-01-29

   Allocate new data structure of type cdw_disc_t, initialize all its
   fields with default, initial values. Return pointer to the data structure.
   The function also initializes cdw_disc_t->cdio.

   You have to call cdw_disc_delete() to properly deallocate the data
   structure.

   \return pointer to disc data structure on success
   \return NULL pointer on failure
*/
cdw_disc_t *cdw_disc_new(void)
{
	cdw_disc_t *disc = (cdw_disc_t *) malloc(sizeof (cdw_disc_t));
	if (!disc) {
		cdw_vdm ("ERROR: malloc\n");
		return (cdw_disc_t *) NULL;
	}

	cdw_disc_reset(disc);

	disc->cdio = cdw_cdio_new();
	if (!disc->cdio) {
		cdw_vdm ("ERROR: cdw_cdio_new()\n");
		free(disc);
		disc = (cdw_disc_t *) NULL;
	}

	return disc;
}





/**
   \brief Get metainformation about a disc

   \date Function's top-level comment reviewed on 2012-01-29
   \date Function's body reviewed on 2012-01-29

   Get all available metainformation about a disc (well, at least
   those pieces of information that are needed to fill fields of
   cdw_disc_t data structure). Save the information in \p disc.

   The function searches for disc in default optical disc drive.
   Then it checks if there is any disc in the drive, and whether the
   disc is unmounted. It also does few more things, so that you
   just have to make one function call to 'get' a disc.

   \p disc should be created first with cdw_disc_new().
   Don't call the function twice with the same \p disc, as this would
   cause memory leaks.

   \p disc - data structure to which to save disc metainformation.

   Function returns CDW_ERROR if there were some serious errors during
   execution of the function. Failed call to strdup() is such an error,
   but no disc in drive is not an error.

   \return CDW_ERROR if errors occur during information retrieval
   \return CDW_NO if information about disc can't be collected
   \return CDW_OK if information about disc has been collected
*/
cdw_rv_t cdw_disc_get(cdw_disc_t *disc)
{
	cdw_assert (disc, "ERROR: disc is NULL\n");
	if (!disc) {
		/* probably cdw_disc_new() returned NULL, caller
		   didn't check this, and called this function
		   with NULL parameter */
		return CDW_NO;
	}

	bool local_processwin = cdw_disc_get_make_processwin();

	cdw_rv_t retval = CDW_ERROR;
	const char *device_fullpath = cdw_drive_get_drive_fullpath();
	if (!device_fullpath) {
		cdw_vdm ("WARNING: no drive\n");
		retval = CDW_NO;
		goto cdw_get_disc_return;
	}

	disc->device_fullpath = strdup(device_fullpath);
	if (!disc->device_fullpath) {
		cdw_vdm ("ERROR: strdup()\n");
		retval = CDW_ERROR;
		goto cdw_get_disc_return;
	}

	cdw_rv_t crv = cdw_drive_disc_availability();
	if (crv != CDW_OK) {
		cdw_vdm ("WARNING: cdw_drive_availability()\n");
		retval = crv;
		goto cdw_get_disc_return;
	}

	crv = cdw_fs_check_device_mounted(disc->device_fullpath);
	if (crv == CDW_ERROR) {
		cdw_vdm ("WARNING: checking if disc is mounted ended with error\n");
		/* let's try and proceed with further checks anyway */
	} else if (crv == CDW_OK) { /* device is mounted */
		cdw_vdm ("INFO: optical disc is mounted\n");
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("Cannot check media: disc is mounted. Please unmount your optical disc."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		retval = CDW_NO;
		goto cdw_get_disc_return;
	} else { /* mounted == CDW_NO, not mounted */
		cdw_vdm ("INFO: disc is not mounted\n");
	}


	crv = cdw_disc_get_meta_info(disc);
	if (crv == CDW_ERROR) {
		cdw_vdm ("ERROR: failed to get disc meta info\n");
		cdw_disc_reset(disc);
		retval = CDW_ERROR;
		goto cdw_get_disc_return;
	} else {
		cdw_disc_resolve(disc);
		retval = CDW_OK;
		goto cdw_get_disc_return;
	}

 cdw_get_disc_return:
	if (local_processwin) {
		cdw_processwin_destroy("", false);
	}

	return retval;
}





/**
   \brief Create processwin needed during execution of cdw_disc_get()

   \date Function's top-level comment reviewed on 2012-01-29
   \date Function's body reviewed on 2012-01-29

   During execution of cdw_disc_get() and functions called by cdw_disc_get(),
   some information needs to be displayed in a process window. This function
   checks if one exists already, and if not, it creates it.
   The process window is created with default title and message, appropriate
   for process of getting disc metainformation.

   If the process window has been created by the function, the function
   returns true. It returns false otherwise.

   \return true if the function created process window
   \return false otherwise
*/
bool cdw_disc_get_make_processwin(void)
{
	if (cdw_processwin_is_active()) {
		return false;
	} else {
		/* 2TRANS: this is title of process window; reading
		   disc meta information */
		cdw_processwin_create(_("Read disc information"),
				      /* 2TRANS: this is message in
					 process window */
				      _("Reading disc information"),
				      false);
		return true;
	}
}





/**
   \brief Deallocate cdw_disc_t data structure

   \date Function's top-level comment reviewed on 2012-01-29
   \date Function's body reviewed on 2012-01-29

   Deallocate all memory used by given \p disc.
   Handle deallocating cdw_disc_t->cdio as well.
   Set the disc to NULL.

   Function accepts NULL disc (*disc == NULL). It does not accept
   NULL pointer (disc == NULL).

   \param disc - pointer to disc to deallocate
*/
void cdw_disc_delete(cdw_disc_t **disc)
{
	cdw_assert (disc, "ERROR: pointer to disc is NULL\n");
	if (!disc) {
		cdw_vdm ("ERROR: pointer to disc is NULL\n");
		return;
	}

	if (!(*disc)) {
		cdw_vdm ("WARNING: disc is NULL\n");
		return;
	}

	if ((*disc)->cdio) {
		cdw_cdio_delete(&(*disc)->cdio);
	}

	if ((*disc)->device_fullpath) {
		free((*disc)->device_fullpath);
		(*disc)->device_fullpath = (char *) NULL;
	}

	free(*disc);
	*disc = (cdw_disc_t *) NULL;

	return;
}





/**
   \brief Reset values in a disc

   \date Function's top-level comment reviewed on 2012-01-29
   \date Function's body reviewed on 2012-01-29

   Set some values of \p disc to their initial state, which corresponds
   to 'no disc in drive'. Don't touch cdw_disc_t->cdio nor any of fields
   in cdw_disc_t->cdio.

   \param disc - disc to reset
*/
void cdw_disc_reset(cdw_disc_t *disc)
{
	cdw_assert (disc, "ERROR: disc is NULL\n");

	disc->type = CDW_DISC_TYPE_UNKNOWN;
	disc->type_label[0] = '\0';

	disc->device_fullpath = (char *) NULL;

	disc->state_empty = CDW_UNKNOWN;
	disc->state_writable = CDW_UNKNOWN;
	disc->type_erasable = CDW_UNKNOWN;
	disc->type_writable = CDW_UNKNOWN;

	disc->capacity.sectors_total = 0;
	disc->capacity.sectors_used = 0;



	disc->write_speeds.n_speeds = -1;
	disc->write_speeds.drive_max_speed = -1;
	disc->write_speeds.drive_default_speed = -1;
	for (int i = 0; i < CDW_DISC_N_SPEEDS_MAX; i++) {
		disc->write_speeds.speeds[i] = 0;
	}

	disc->accepted_disc_modes = CDW_DISC_MODE_INIT;

	disc->media_info_source = CDW_TOOL_NONE;

	disc->cdrecord_info.last_sess_start = 0;
	disc->cdrecord_info.next_sess_start = 0;
	disc->cdrecord_info.has_toc = false;
	disc->cdrecord_info.phys_size = -1;
	disc->cdrecord_info.start_of_lead_out = -1;
	disc->cdrecord_info.rzone_size = 0;

	disc->dvd_rw_mediainfo_info.read_capacity = -1;
	/* 'e' = "error"; dvd+rw-mediainfo aborts reading metainfo of DVD-ROM
	   disc before getting to "status", so for DVD-ROM status stays as 'e';
	   not a big problem because dvd+rw-mediainfo pipe regex code will
	   properly recognize type of disc as DVD-ROM, and everything will be
	   handled correctly */
	disc->dvd_rw_mediainfo_info.disc_status = 'e';

	disc->dvd_rw_mediainfo_info.end_lba = -1;
	disc->dvd_rw_mediainfo_info.all_tracks_size = -1;
	disc->dvd_rw_mediainfo_info.data_tracks_size = -1;
	disc->dvd_rw_mediainfo_info.last_track_size = -1;

	disc->xorriso_info.is_written = CDW_UNKNOWN;
	disc->xorriso_info.is_blank = CDW_UNKNOWN;
	disc->xorriso_info.is_appendable = CDW_UNKNOWN;
	disc->xorriso_info.is_closed = CDW_UNKNOWN;
	disc->xorriso_info.sectors_used = -1;
	disc->xorriso_info.sectors_total = -1;

	disc->libburn_info.unformatted_size = 0;
	disc->libburn_info.formatting_status = BURN_FORMAT_IS_UNKNOWN;
	disc->libburn_info.read_capacity = -1;

	/* TODO: should we add call to cdw_cdio_reset()? */

	/* take care of other things that should be recalculated based
	   on the assignments made in this function */
	cdw_disc_resolve(disc);

	return;
}





/**
   \brief Get initial value of burning/erasing speed for given disc

   \date Function's top-level comment reviewed on 2012-01-29
   \date Function's body reviewed on 2012-01-29

   Check \p disc to see what should be the initial/default value
   of burning/erasing speed for the \p disc. The value should have
   been saved in \p disc during collecting of disc meta data.

   The value is either default speed for given combination of
   disc + drive, or (if there is no such speed saved), first value
   on list of supported burning speeds.

   'id' in function's name means 'value'.

   \param disc - disc for which to get initial write speed

   \return value of initial burning/erasing speed
*/
int cdw_disc_get_initial_write_speed_id(cdw_disc_t *disc)
{
	cdw_assert (disc, "ERROR: disc is NULL\n");
	cdw_assert (disc->write_speeds.n_speeds > 0, "ERROR: you forgot to call cdw_disc_validate()\nand now n_speeds is non-positive\n");

	if (disc->write_speeds.drive_default_speed != -1) {
		return disc->write_speeds.drive_default_speed;
	} else {
		return disc->write_speeds.speeds[0];
	}
}





/**
   \brief Get metainformation from a disc

   \date Function's top-level comment reviewed on 2012-02-02
   \date Function's body reviewed on 2012-02-02

   Get meta information from disc currently in drive.
   The function should be invoked after ensuring that an optical disc
   exists in a drive, and that the disc is not mounted.

   Meta information is collected using first with cdio calls and then
   with external tools (cdrecord, dvd+rw-mediainfo, xorriso).
   The high-level information is then resolved using cdw_disc_resolve(),
   and validated with cdw_disc_validate().

   \param disc - disc variable into which to store the information

   \return CDW_OK if media info retrieved successfully
   \return CDW_ERROR if function failed to retrieve full info
*/
cdw_rv_t cdw_disc_get_meta_info(cdw_disc_t *disc)
{
	cdw_assert (disc, "ERROR: disc is NULL\n");

	cdw_rv_t crv = cdw_disc_get_meta_info_with_cdio(disc);
	if (crv != OK) {
		cdw_vdm ("ERROR: failed to open disc with cdio calls\n");
		return CDW_ERROR;
	}

	if (disc->cdio->simple_type == CDW_DISC_SIMPLE_TYPE_CD
	    && disc->cdio->ofs->type == CDIO_FS_AUDIO) {

		/* we won't call external tools to get more meta information
		   about a disc, so for audio CDs we have to set type here */
		disc->type = CDW_CD_AUDIO;

		cdw_disc_resolve(disc);
		cdw_disc_debug_print_disc(disc);

		/* not much else to do, everything that we needed
		   to know about Audio CD has been checked with cdio */
		return CDW_OK;
	}

	crv = cdw_disc_get_meta_info_with_external_tools(disc);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to get metainfo from a disc\n");
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message dialog window,
				      program cannot get metadata from disc */
				   _("Cannot get information about disc."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		cdw_main_window_wrefresh();

		return CDW_ERROR;
	}


	if (disc->type == CDW_DVD_RW
	    || disc->type == CDW_DVD_RW_SEQ
	    || disc->type == CDW_DVD_RW_RES) {

		/* DVD-RW has some quirks and they are not fully
		   supported (this was more true in cdw v. 0.3.91,
		   situation has improved in 0.3.92). */

		if (global_config.general.show_dvd_rw_support_warning) {

			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Warning"),
					   /* 2TRANS: this is message dialog window */
					   _("This is DVD-RW disc, it is not fully supported by cdw. You may have some problems."),
					   CDW_BUTTONS_OK, CDW_COLORS_WARNING);
			cdw_main_window_wrefresh();

			/* don't annoy user with this message anymore
			   in current run of cdw; this option is
			   not saved to configuration file */
			global_config.general.show_dvd_rw_support_warning = false;
		}
	}

	/* at this point we know that we have _some_ meta data
	   in "disc" data structure */

	/* calculate some more high-level information */
	cdw_disc_resolve(disc);

	/* validate basic meta data - check for obvious errors */
	crv = cdw_disc_validate(disc);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to validate disc information\n");
		return CDW_ERROR;
	} else {
		cdw_disc_debug_print_disc(disc);
		return CDW_OK;
	}
}





/**
   \brief Get basic disc meta information using libcdio calls

   \date Function's top-level comment reviewed on 2012-02-02
   \date Function's body reviewed on 2012-02-02

   Get information such as simple disc type, number of tracks on a disc,
   file system on a disc, and some more information about the file system.

   \param disc - disc variable into which to store the information

   \return CDW_OK on success
   \return CDW_ERROR when function failed to get information (e.g. because of error)
*/
cdw_rv_t cdw_disc_get_meta_info_with_cdio(cdw_disc_t *disc)
{
	cdw_assert (disc, "ERROR: disc is NULL\n");

	cdw_rv_t crv = cdw_cdio_get(disc->cdio, disc->device_fullpath);
	if (crv == CDW_ERROR) {
		cdw_vdm ("ERROR: cdw_cdio_get()\n");
		return CDW_ERROR;
	}

	if (disc->cdio->simple_type == CDW_DISC_SIMPLE_TYPE_UNKNOWN) {
		cdw_vdm ("ERROR: failed to get meta info for disc %d, which is not CD nor DVD\n", disc->cdio->simple_type);
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message dialog window,
				      program cannot get metadata from disc */
				   _("Cannot get basic information about disc."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		cdw_main_window_wrefresh();
		return CDW_ERROR;
	}

	return CDW_OK;
}





/**
   \brief Call external tools to collect meta information about a disc

   \date Function's top-level comment reviewed on 2012-02-02
   \date Function's body reviewed on 2012-02-02

   Function checks disc type, then checks which tool should check
   disc meta information, and then calls function calling the tool
   to do the job.

   \param disc - disc variable in which to store disc meta information

   \return CDW_OK on success
   \return CDW_ERROR on errors
*/
cdw_rv_t cdw_disc_get_meta_info_with_external_tools(cdw_disc_t *disc)
{
	cdw_assert (disc, "ERROR: disc is NULL\n");

	cdw_task_t *task = cdw_task_new(CDW_TASK_MEDIA_INFO, disc);
	if (!task) {
		cdw_vdm ("ERROR: failed to create a task\n");
		return CDW_ERROR;
	}
	cdw_id_t tool_id = task->media_info.tool.id;
	cdw_assert (tool_id == CDW_TOOL_CDRECORD
		    || tool_id == CDW_TOOL_XORRISO
		    || tool_id == CDW_TOOL_DVD_RW_MEDIAINFO,
		    "ERROR: tool selector returns unsupported tool %lld\n", tool_id);
	cdw_assert (task->media_info.tool.label,
		    "ERROR: tool selector returns NULL path to tool\n");

	disc->media_info_source = tool_id;
	cdw_rv_t crv = CDW_OK;
	if (tool_id == CDW_TOOL_CDRECORD) {
		/* 2TRANS: this is message displayed in process window */
		cdw_processwin_display_main_info(_("Getting disc info with cdrecord"));
		crv = cdw_cdrecord_run_task(task, disc);

	} else if (tool_id == CDW_TOOL_DVD_RW_MEDIAINFO) {
		/* 2TRANS: this is message displayed in process window */
		cdw_processwin_display_main_info(_("Getting disc info with dvd+rw-mediainfo"));
		crv = cdw_dvd_rw_mediainfo_run_task(task, disc);

	} else if (tool_id == CDW_TOOL_XORRISO) {
		/* 2TRANS: this is message displayed in process window */
		cdw_processwin_display_main_info(_("Getting disc info with xorriso"));
		crv = cdw_xorriso_run_task(task, disc);
	} else {
		; /* covered by assert above */
	}
	cdw_rv_t tool_status = cdw_task_check_tool_status(task);
	cdw_task_delete(&task);

	if (tool_status != CDW_OK ||  crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to get media info with %s\n", cdw_ext_tools_get_tool_name(tool_id));
		return CDW_ERROR;
	} else {
		return CDW_OK;
	}
}





/**
   \brief Remove any repeating values from table of writing speeds, sort the speeds

   \date Function's top-level comment reviewed on 2012-02-02
   \date Function's body reviewed on 2012-02-02

   When parsing output of "cdrecord -prcap", cdw puts speeds available
   for a disc in table. The problem is that the speed values can repeat
   in the table (cdrecord prints them this way). This function removes
   duplicates from the table.

   There is also another problem with writing speeds table, as printed by
   "cdrecord -prcap" or dvd+rw-mediainfo: values in the table may be in
   order from highest to lowest. This function reverses the table to have
   values from lowest to highest.

   Similar problem occurs when getting write speeds table with libburn.

   The function works on disc->write_speeds.speeds[].

   Caller has to make sure that number of (non-sorted) speeds is greater
   than zero.

   \param disc - disc in which to fix write speeds table
*/
void cdw_disc_resolve_write_speeds(cdw_disc_t *disc)
{
#ifndef NDEBUG
	cdw_assert (disc, "ERROR: disc is NULL\n");
	cdw_vdm ("INFO: table of write speeds before resolving:\n");
	for (int i = 0; i < CDW_DISC_N_SPEEDS_MAX; i++) {
		if (disc->write_speeds.speeds[i] != 0) {
			cdw_vdm ("INFO:   write_speeds[%d] = '%d'\n", i, disc->write_speeds.speeds[i]);
		}
	}
#endif

	/* number of speeds in table has been limited at the level of parsing
	   cdrecord/dvd+rw-mediainfo/libburn output, so at this point we are
	   sure that there is no more than CDW_DISC_N_SPEEDS_MAX
	   speed values in write_speeds[] table

	   output from libburn or dvd+rw-mediainfo may contain duplicates:
	   example list of speeds from dvd+rw-mediainfo: 16, 16, 12, 10, 8, 4
	   example list of speeds from libburn: 16, 12, 10, 8, 4, 16, 16, 12, 10, 8, 4, 16 */

	/* algorithm of resolving any table of speeds:
	   1. sort from highest to lowest (with possible zeros at the end of result table);
	   2. remove duplicates (compress);
	   3. count non-zero speeds to get number of valid speeds: n_speeds;
	   4. sort from lowest to highest, but only first n_speeds items; */

	qsort(disc->write_speeds.speeds, CDW_DISC_N_SPEEDS_MAX, sizeof(disc->write_speeds.speeds[0]), cdw_utils_compare_ints_reverse);

	cdw_utils_compress_table_of_ints(disc->write_speeds.speeds, CDW_DISC_N_SPEEDS_MAX);

	for (int i = 0; i < CDW_DISC_N_SPEEDS_MAX; i++) {
		if (disc->write_speeds.speeds[i] == 0) {
			disc->write_speeds.n_speeds = i;
			break;
		}
	}

	qsort(disc->write_speeds.speeds, (size_t) disc->write_speeds.n_speeds, sizeof(disc->write_speeds.speeds[0]), cdw_utils_compare_ints);

#ifndef NDEBUG
	cdw_vdm ("INFO: table of write speeds after resolving (%d speeds total):\n", disc->write_speeds.n_speeds);
	for (int i = 0; i < disc->write_speeds.n_speeds; i++) {
		cdw_vdm ("INFO:   write_speeds[%d] = '%d'\n", i, disc->write_speeds.speeds[i]);
	}
#endif

	return;
}





/**
   \brief Check disc data structure for any obvious errors

   \date Function's top-level comment reviewed on 2012-02-02
   \date Function's body reviewed on 2012-02-02

   The function checks if fields of \p disc don't contain any
   obvious errors or inconsistencies.

   NOTE: the function should avoid calculating any values of \p disc
   fields, it should only check them!

   \param disc - disc to be checked

   \return CDW_ERROR on errors in disc
   \return CDW_OK if disc seems to be ok
*/
cdw_rv_t cdw_disc_validate(cdw_disc_t *disc)
{
	cdw_assert (disc, "ERROR: disc is NULL\n");

	if (disc->cdio->simple_type != CDW_DISC_SIMPLE_TYPE_CD
	    && disc->cdio->simple_type != CDW_DISC_SIMPLE_TYPE_DVD) {

		/* simple_type _has to_ be known to do anything useful */
		cdw_vdm ("ERROR: disc is neither CD nor DVD\n");
		return CDW_ERROR;
	}

	if (disc->type == CDW_CD_AUDIO
	    || disc->type == CDW_CD_ROM
	    || disc->type == CDW_DVD_ROM) {

		cdw_assert (disc->type_writable == CDW_FALSE,
			    "ERROR: read only disc not marked as non-writable\n");
	}

	if (disc->type_writable == CDW_TRUE) {
		if (disc->write_speeds.n_speeds == 0) {
			cdw_vdm ("ERROR: no writing speeds for writable disc!\n");
			return CDW_ERROR;
		} else if (disc->write_speeds.n_speeds == 1) {
			if (disc->write_speeds.speeds[0] == 0) {
				cdw_vdm ("ERROR: first and only writing speed is zero!\n");
				return CDW_ERROR;
			}
		} else {
			; /* speeds table is non-empty, so it seems to be ok */
		}
	} else {
		; /* read-only disc, not much to validate */
	}

	return CDW_OK;
}





/**
   \brief Calculate values of some fields in disc that represent
   higher-level information about disc

   \date Function's top-level comment reviewed on 2012-02-02
   \date Function's body reviewed on 2012-02-02

   The function checks some fields in \p disc that were set by external
   tools modules (let's call the fields lower-level data), and based on
   values of these fields it sets values of some fields representing
   higher-level information.

   Simple example: "if disc type is DVD-RW then disc is erasable", or
   "if disc type is CD-RW and certain values related to session information
   have certain values then disc is or is not writable".

   Rest of code in cdw can then check these higher-level fields instead
   figuring it out itself over and over again.

   Function also sets appropriate value of disc->type_label
   so nice label of disc type can also be used.

   Call this function in function getting meta information from
   a disc (cdw_disc_get_meta_info()) or in function resetting
   a disc (cdw_disc_reset()).

   \param disc - disc to be resolved

   \return CDW_OK when there were no problems
   \return CDW_NO if disc type is NONE or UNKNOWN
   \return CDW_ERROR on errors
*/
cdw_rv_t cdw_disc_resolve(cdw_disc_t *disc)
{
	cdw_assert (disc, "ERROR: disc is NULL\n");

	/* first set disc label, it may be needed later */
	cdw_disc_resolve_type_label(disc);

	/* safe defaults; these values should be assigned
	   in disc reset function, but anyway... */
	disc->type_erasable = CDW_UNKNOWN;
	disc->type_writable = CDW_UNKNOWN;
	disc->state_writable = CDW_UNKNOWN;
	/* don't modify disc->state_empty, it was already set using
	   information about file system on the disc */

	if (!disc->device_fullpath) {
		return CDW_NO;
	}

	int status = cdw_drive_status(disc->device_fullpath);
	if (status == CDS_NO_DISC || status == CDS_TRAY_OPEN) {
		/* this check is to avoid further unnecessary checks
		   and possible printing of warning messages that
		   disc type is unknown */
		return CDW_NO;
	}


	/* types: erasable and writable, this is quite easy part */
	cdw_disc_type_t t = disc->type;
	if (t == CDW_DISC_TYPE_UNKNOWN) {
		cdw_vdm ("WARNING: disc in drive is of type \"unknown\"\n");
		return CDW_NO;

	} else if (t == CDW_DISC_NONE) {
		cdw_vdm ("INFO: no disc in drive\n");
		/* TODO: should this happen at all? should we call this
		   function when there is no disc in drive? */
		return CDW_NO;

	} else if (t == CDW_CD_AUDIO
		   || t == CDW_CD_ROM
		   || t == CDW_DVD_ROM) {

		disc->type_erasable = CDW_FALSE;
		disc->type_writable = CDW_FALSE;

	} else if (t == CDW_CD_R
		   || t == CDW_DVD_R
		   || t == CDW_DVD_R_SEQ
		   || t == CDW_DVD_R_RES
		   || t == CDW_DVD_RP
		   || (t == CDW_DVD_RP_DL && global_config.general.support_dvd_rp_dl)) {

		disc->type_erasable = CDW_FALSE;
		disc->type_writable = CDW_TRUE;

	} else if (t == CDW_CD_RW
		   || t == CDW_DVD_RW
		   || t == CDW_DVD_RW_SEQ
		   || t == CDW_DVD_RW_RES
		   || t == CDW_DVD_RWP) {

		disc->type_erasable = CDW_TRUE;
		disc->type_writable = CDW_TRUE;
	} else {
		cdw_vdm ("ERROR: unknown disc type %d\n", t);
		return CDW_ERROR;
	}


	/* now more difficult part: it is tool-dependent
	   and (since tool support varies between tools
	   and tool versions) may be tricky */

	int empty = disc->state_empty;
	if (disc->media_info_source == CDW_TOOL_CDRECORD) {
		cdw_cdrecord_set_disc_states(disc);
	} else if (disc->media_info_source == CDW_TOOL_DVD_RW_MEDIAINFO) {
		cdw_dvd_rw_mediainfo_set_disc_states(disc);
	} else if (disc->media_info_source == CDW_TOOL_XORRISO) {
		cdw_xorriso_set_disc_states(disc);
	} else {
		cdw_vdm ("ERROR: unknown media info tool in disc struct: %lld\n",
			 disc->media_info_source);

		return CDW_ERROR;
	}
	if (empty != disc->state_empty) {
		cdw_vdm ("WARNING: external tool changes \"state empty\"\n");
		disc->state_empty = empty;
	}


	if (disc->state_empty == CDW_UNKNOWN) {
		cdw_vdm ("WARNING: state_empty = UNKNOWN\n");
		cdw_vdm ("WARNING: current ofs type = %d / %s\n", disc->cdio->ofs->type, disc->cdio->ofs->type_label);

		if (disc->cdio->blank) {
			disc->state_empty = CDW_TRUE;
		} else {
			disc->state_empty = CDW_FALSE;
		}
	}

	/* some common, tool independent settings;
	   these "state values" may have been set by tool-dependent
	   code, but here cover some special cases, in tool independent fashion */
	if (disc->type == CDW_DVD_RWP
	    || disc->type == CDW_DVD_RW_RES) {

		/* The two types of disc receive special treatment.
		   I'm not sure if this is 100% true, but I think that you
		   can always expand (grow) iso file system that exists on
		   'DVD+RW' and 'DVD-RW Restricted Overwrite'; search for
		   "it seems that with DVD+RW and DVD-RW Restricted"
		   comment in write_wizard.c */

		if (disc->state_writable != CDW_TRUE) {
			cdw_vdm ("WARNING: tool \"%s\" didn't set \"writable=true\" for \"%s\" disc\n",
				 cdw_ext_tools_get_tool_name(disc->media_info_source),
				 disc->type_label);
		}

		disc->state_writable = CDW_TRUE;

	} else if (disc->type == CDW_CD_ROM
		   || disc->type == CDW_CD_AUDIO
		   || disc->type == CDW_DVD_ROM) {

		if (disc->state_writable != CDW_FALSE) {
			cdw_vdm ("WARNING: tool \"%s\" didn't set \"writable=false\" for \"%s\" disc\n",
				 cdw_ext_tools_get_tool_name(disc->media_info_source),
				 disc->type_label);
		}

		disc->state_writable = CDW_FALSE;
	} else {
		; /* pass */
	}

	if (disc->type_writable == CDW_TRUE) {
		/* writable disc has some write speeds that
		   need to be taken care of */
		cdw_disc_resolve_write_speeds(disc);

		/* calculating capacity makes sense only for
		   writable discs */
		cdw_disc_resolve_capacities(disc);
	}

	return CDW_OK;
}





/**
   \brief Function sets proper disc type label

   \date Function's top-level comment reviewed on 2012-02-02
   \date Function's body reviewed on 2012-02-02

   Function checks type of given \p disc (an integer value) and
   copies proper human-readable label to \p disc->type_label field.
   Disc type must be set before calling this function.

   If a disc type is DVD+R DL, function checks if support for this
   disc type is enabled, and acts accordingly.

   \param disc - disc to be processed
*/
void cdw_disc_resolve_type_label(cdw_disc_t *disc)
{
	cdw_assert (disc, "ERROR: disc is NULL\n");

	char *l = disc->type_label;

	if (disc->type == CDW_DVD_RP_DL &&
	    ! global_config.general.support_dvd_rp_dl) {

		strncpy(l, cdw_disc_type_labels[CDW_DISC_TYPE_UNKNOWN], CDW_DISC_TYPE_LABEL_LEN);
	} else {
		strncpy(l, cdw_disc_type_labels[disc->type], CDW_DISC_TYPE_LABEL_LEN);
	}
	l[CDW_DISC_TYPE_LABEL_LEN] = '\0';

	cdw_vdm ("INFO: disc type label %d resolved as \"%s\"\n", disc->type, l);
	return;
}





/**
   \brief Function checking if given disc is readable

   \date Function's top-level comment reviewed on 2012-02-02
   \date Function's body reviewed on 2012-02-02

   Disc is readable if it is not empty and is of proper (supported) type
   and has supported file system.

   \param disc - disc to be checked

   \return (char *) NULL pointer if cdw will be able to read data or audio from the disc
   \return pointer to const char string describing reason why disc can't be read, if it can't be read by cdw
*/
const char *cdw_disc_check_if_is_readable(cdw_disc_t *disc)
{
	cdw_assert (disc, "ERROR: disc is NULL\n");

	if (disc->state_empty == CDW_TRUE) {
		cdw_vdm ("ERROR: disc is empty, shouldn't you have checked this before calling this function?\n");
		return disc_not_readable_reasons[CDW_DISC_NOT_READABLE_EMPTY].label;;
	}

	if (disc->cdio->simple_type == CDW_DISC_SIMPLE_TYPE_CD) {
		if (disc->type == CDW_CD_AUDIO) {
			/* reading Audio CDs with cdio calls */
			return (char *) NULL;
		}
		if (disc->cdio->mode == CDIO_DISC_MODE_CD_DATA /* Mode1/Mode2 Formless? */
		    || disc->cdio->mode == CDIO_DISC_MODE_CD_XA /* Mode2 Form1 / Mode2 Form2 ? */
		    /* not 100% sure what this is, but cdw doesn't support it */
		    /* ||  disc->cdio_disc_mode == CDIO_DISC_MODE_CD_MIXED */ ) {

			if (cdw_ofs_is_iso(disc->cdio->ofs->type)) {
				return (char *) NULL;
			}
		} else if (disc->cdio->mode == CDIO_DISC_MODE_CD_DA) {
			/* well, this check is redundant, covered by earlier "if" */
			return (char *) NULL;
		} else {
			;
		}

		return disc_not_readable_reasons[CDW_DISC_NOT_READABLE_OTHER].label;
	} else if (disc->cdio->simple_type == CDW_DISC_SIMPLE_TYPE_DVD) {
		if (cdw_ofs_is_iso(disc->cdio->ofs->type)) {

			return (char *) NULL;

			/* For some time I thought that in order to
			   read data from optical disc with ISO9660 file
			   system you have to have support for such file
			   system enabled in operating system kernel.
			   It turns out that this is not the case - I was
			   able to successfully read such disc with support
			   for ISO9660 completely disabled in kernel.
			   The disc can't be mounted, but it can be read
			   with read().
			   So - this piece of code below is not necessary,
			   but I will keep it for posterity. */
#if 0
			/* reading DVDs with UNIX read(), so I still need to
			   check for support of ISO9660 in kernel */

			cdw_rv_t crv = cdw_sys_check_file_system_support(CDIO_FS_ISO_9660);
			if (crv == CDW_OK) {
				return (char *) NULL;
			} else if (crv == CDW_NO) {
				return disc_not_readable_reasons[CDW_DISC_NOT_READABLE_NO_ISO9660].label;
			} else { /* CDW_CANCEL || CDW_ERROR */
				return disc_not_readable_reasons[CDW_DISC_NOT_READABLE_OTHER].label;
			}
#endif
		} else {
			cdw_vdm ("INFO: disc is not readable because file system type is not readable\n");
			return disc_not_readable_reasons[CDW_DISC_NOT_READABLE_UNSUPPORTED_FS].label;;
		}
	} else {
		cdw_assert (0, "ERROR: incorrect simple disc type %d\n", disc->cdio->simple_type);
		return disc_not_readable_reasons[CDW_DISC_NOT_READABLE_OTHER].label;
	}
}





/**
   \brief Debug function printing to stderr basic information about a disc

   \date Function's top-level comment reviewed on 2012-02-02
   \date Function's body reviewed on 2012-02-02

   \param disc - disc to print
*/
void cdw_disc_debug_print_disc(cdw_disc_t *disc)
{
	cdw_assert (disc, "ERROR: disc is NULL\n");

	cdw_vdm ("CURRENT DISC:    simple type = \"%s\"\n", disc->cdio->simple_type_label);
	cdw_vdm ("CURRENT DISC:           type = \"%s\" (%d)\n", disc->type_label, disc->type);
	cdw_vdm ("CURRENT DISC:    state empty = \"%s\"\n", cdw_utils_get_cdw_bool_type_label(disc->state_empty));
	cdw_vdm ("CURRENT DISC: state writable = \"%s\"\n", cdw_utils_get_cdw_bool_type_label(disc->state_writable));
	cdw_vdm ("CURRENT DISC:  type writable = \"%s\"\n", cdw_utils_get_cdw_bool_type_label(disc->type_writable));
	cdw_vdm ("CURRENT DISC:  type erasable = \"%s\"\n", cdw_utils_get_cdw_bool_type_label(disc->type_erasable));

	float coeff = 512.0; /* (n_sectors * 2048) / (1024.0 * 1024.0) = n_sectors / 512.0 */
	cdw_vdm ("CURRENT DISC CAPACITY:  used = %lld sectors (%.3f MB)\n", disc->capacity.sectors_used, (double) disc->capacity.sectors_used / coeff);
	cdw_vdm ("CURRENT DISC CAPACITY: total = %lld sectors (%.3f MB)\n", disc->capacity.sectors_total, (double) disc->capacity.sectors_total / coeff);

	if (disc->state_empty == CDW_TRUE) {
		return;
	}
	cdw_vdm ("CURRENT DISC:   file system: = \"%s\"\n", disc->cdio->ofs->type_label);
	if (cdw_ofs_is_iso(disc->cdio->ofs->type)) {
		cdw_vdm ("CURRENT DISC:     volume id: = \"%s\"\n", disc->cdio->ofs->volume_id);
	}

	if (disc->type_writable) {
		cdw_vdm ("CURRENT DISC: %d write speed(s):\n", disc->write_speeds.n_speeds);
		for (int i = 0; i < disc->write_speeds.n_speeds; i++) {
			cdw_vdm ("CURRENT DISC:    %d\n", disc->write_speeds.speeds[i]);
		}
	}
	return;
}





/**
   \brief Resolve capacity of a disc

   \date Function's top-level comment reviewed on 2012-02-02
   \date Function's body reviewed on 2012-02-02

   Each disc can be characterized by total capacity (amount of sectors
   that can be put on a disc), and used space (amount of sectors already
   holding data / amount of burned sectors).

   Knowing the two parameters is a useful thing, and showing the
   information to the user is important.

   Unfortunately:
    - getting the correct values is tricky,
    - getting the values is tool dependent.

   The function tries to do its best, very often using number of sectors
   from cdw_disc_t->cdw_cdio_t->cdw_ofs_t as 'used space'.

   \param disc - disc for which to resolve the values
*/
void cdw_disc_resolve_capacities(cdw_disc_t *disc)
{
	if (disc->cdrecord_info.rzone_size > 0
	    && disc->cdrecord_info.rzone_size > disc->cdrecord_info.phys_size) {

		/* sometimes rzone size is a great indicator of total
		   disc size, but, alas, only sometimes */

		cdw_vdm ("INFO: using cdrecord info (DVD rzone size)\n");
		disc->capacity.sectors_total = disc->cdrecord_info.rzone_size;
		disc->capacity.sectors_used = disc->cdio->ofs->n_sectors;
	} else if (disc->cdrecord_info.phys_size != -1) {
		cdw_vdm ("INFO: using cdrecord info (DVD phys size)\n");
		disc->capacity.sectors_total = disc->cdrecord_info.phys_size;
		disc->capacity.sectors_used = disc->cdio->ofs->n_sectors;
	} else if (disc->cdrecord_info.start_of_lead_out != -1) {
		cdw_vdm ("INFO: using cdrecord info (CD)\n");
		disc->capacity.sectors_total = disc->cdrecord_info.start_of_lead_out;
		disc->capacity.sectors_used = disc->cdio->ofs->n_sectors;
	} else if (disc->dvd_rw_mediainfo_info.end_lba != -1) {
		cdw_vdm ("INFO: using dvd+rw-mediainfo info\n");
		disc->capacity.sectors_total = disc->dvd_rw_mediainfo_info.end_lba;
		disc->capacity.sectors_used = disc->cdio->ofs->n_sectors;
	} else if (disc->xorriso_info.sectors_total != -1) {
		cdw_vdm ("INFO: using xorriso info\n");
		disc->capacity.sectors_total = disc->xorriso_info.sectors_total;
		disc->capacity.sectors_used = disc->xorriso_info.sectors_used >= 0 ? disc->xorriso_info.sectors_used : disc->cdio->ofs->n_sectors;
	} else if (disc->libburn_info.unformatted_size != 0) {
		cdw_vdm ("INFO: using libburn info\n");
		/* FIXME: are you sure that this is always 2048? */
		disc->capacity.sectors_total = disc->libburn_info.unformatted_size / 2048;
		disc->capacity.sectors_used = 0;
		if (disc->cdio->ofs->n_sectors > 0) {
			cdw_vdm ("WARNING: assuming that disc is empty while fs size != 0 (%ld)\n", disc->cdio->ofs->n_sectors);
		}
	} else {
		cdw_vdm ("WARNING: no info available\n");
	}
	return;
}





/**
   \brief Check if disc type allows for writing

   \date Function's top-level comment reviewed on 2012-02-02
   \date Function's body reviewed on 2012-02-02

   Some disc types just can't be written to...

   \param disc - disc to check

   \return true if type of given disc allows writing to it
   \return false if type of given disc doesn't allow writing to it
*/
bool cdw_disc_is_disc_type_writable(cdw_disc_t *disc)
{
	if (disc->type == CDW_DISC_TYPE_UNKNOWN
	    || disc->type == CDW_CD_AUDIO
	    || disc->type == CDW_CD_ROM
	    || disc->type == CDW_DVD_ROM) {

		return false;
	} else {
		return true;
	}
}





/**
   \brief Get number of first track on a disc

   \date Function's top-level comment reviewed on 2012-02-02
   \date Function's body reviewed on 2012-02-02

   This is just a simple accessor function.

   Get number of first track on a disc. cdio gets this information
   through dedicated function (cdio_get_first_track_num()), so maybe
   this isn't that obvious.

   \param disc - disc to check

   \return number of first track on a disc
*/
track_t cdw_disc_get_first_track(cdw_disc_t *disc)
{
	cdw_assert (disc, "ERROR: disc is NULL\n");
	return disc->cdio->first_track;
}





/**
   \brief Get number of last track on a disc

   \date Function's top-level comment reviewed on 2012-02-02
   \date Function's body reviewed on 2012-02-02

   This is just a simple accessor function.

   Get number of last track on a disc.

   \param disc - disc to check

   \return number of last track on a disc
*/
track_t cdw_disc_get_last_track(cdw_disc_t *disc)
{
	cdw_assert (disc, "ERROR: disc is NULL\n");
	return disc->cdio->last_track;
}





/**
   \brief Get total number of tracks on a disc

   \date Function's top-level comment reviewed on 2012-02-02
   \date Function's body reviewed on 2012-02-02

   This is just a simple accessor function.

   Get total number of tracks on a disc.

   \param disc - disc to check

   \return total number of tracks on a disc
*/
track_t cdw_disc_get_number_of_tracks(cdw_disc_t *disc)
{
	cdw_assert (disc, "ERROR: disc is NULL\n");
	return disc->cdio->n_tracks;
}





/*
  Capacity of Blu-ray Standard disc size, XL 4 layer: 128 001 769 472 bytes
  Capacity of uint32_t type:                            4 294 967 295
  Capacity of uint64_t type:               18 446 744 073 709 551 615

  \return zero on errors
  \return total disc capacity (in bytes) on success
*/
uint64_t cdw_disc_get_total_capacity_bytes(void)
{
	cdw_disc_t *disc = cdw_disc_new();
	if (!disc) {
		cdw_vdm ("ERROR: failed to make new disc\n");
		return 0;
	}

	cdw_rv_t crv = cdw_disc_get(disc);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to get disc info\n");
		cdw_disc_delete(&disc);
		return 0;
	}

	cdw_main_window_volume_info_view_update(disc->capacity.sectors_used, disc->capacity.sectors_total, false);

	uint64_t capacity = 0;
	if (disc->capacity.sectors_total < 0) {
		cdw_vdm ("ERROR: failed to get sectors total from disc\n");
		capacity = 0;
	} else {
		/* 2048 bytes per sector is a kind of lowest common
		   denominator. Most (but not all) discs supported by
		   cdw (CD and DVD discs) will have at least 2048
		   bytes per sector.

		   FIXME: if you ever decide to support other disc
		   types, the sector size will may be different (I
		   haven't checked this).

		   FIXME: this may be true for empty CD discs, but for
		   CD discs that already have some tracks in different
		   formats this is not exactly true. */
		capacity = ((uint64_t) disc->capacity.sectors_total) * 2048;
	}

	cdw_disc_delete(&disc);
	return capacity;
}
