/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.internal.xml;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import org.ovirt.api.metamodel.runtime.util.ArrayListWithHref;
import org.ovirt.api.metamodel.runtime.util.ListWithHref;
import org.ovirt.api.metamodel.runtime.xml.XmlReader;
import org.ovirt.engine.sdk4.internal.containers.NetworkContainer;
import org.ovirt.engine.sdk4.types.Network;

public class XmlNetworkReader {
    
    public static Network readOne(XmlReader reader) {
        // Do nothing if there aren't more tags:
        if (!reader.forward()) {
            return null;
        }
        
        // Create the object:
        NetworkContainer object = new NetworkContainer();
        
        // Process the attributes:
        for (int i = 0; i < reader.getAttributeCount(); i++) {
            String name = reader.getAttributeLocalName(i);
            String image = reader.getAttributeValue(i);
            switch (name) {
                case "href":
                object.href(image);
                break;
                case "id":
                object.id(image);
                break;
                default:
                break;
            }
        }
        
        // Process the inner elements:
        reader.next();
        while (reader.forward()) {
            String name = reader.getLocalName();
            switch (name) {
                case "cluster":
                object.cluster(XmlClusterReader.readOne(reader));
                break;
                case "comment":
                object.comment(reader.readString());
                break;
                case "data_center":
                object.dataCenter(XmlDataCenterReader.readOne(reader));
                break;
                case "description":
                object.description(reader.readString());
                break;
                case "display":
                object.display(reader.readBoolean());
                break;
                case "dns_resolver_configuration":
                object.dnsResolverConfiguration(XmlDnsResolverConfigurationReader.readOne(reader));
                break;
                case "ip":
                object.ip(XmlIpReader.readOne(reader));
                break;
                case "mtu":
                object.mtu(reader.readInteger());
                break;
                case "name":
                object.name(reader.readString());
                break;
                case "network_labels":
                object.networkLabels(XmlNetworkLabelReader.readMany(reader));
                break;
                case "permissions":
                object.permissions(XmlPermissionReader.readMany(reader));
                break;
                case "profile_required":
                object.profileRequired(reader.readBoolean());
                break;
                case "qos":
                object.qos(XmlQosReader.readOne(reader));
                break;
                case "required":
                object.required(reader.readBoolean());
                break;
                case "status":
                object.status(XmlNetworkStatusReader.readOne(reader));
                break;
                case "stp":
                object.stp(reader.readBoolean());
                break;
                case "usages":
                object.usages(XmlNetworkUsageReader.readMany(reader));
                break;
                case "vlan":
                object.vlan(XmlVlanReader.readOne(reader));
                break;
                case "vnic_profiles":
                object.vnicProfiles(XmlVnicProfileReader.readMany(reader));
                break;
                case "link":
                readLink(reader, object);
                break;
                default:
                reader.skip();
                break;
            }
        }
        
        // Discard the end tag:
        reader.next();
        
        return object;
    }
    
    public static Iterator<Network> iterateMany(final XmlReader reader) {
        return new Iterator<Network>() {
            private boolean first = true;
            
            @Override
            public void remove() {
                // Empty on purpose
            }
            
            @Override
            public boolean hasNext() {
                if (first) {
                    if (!reader.forward()) {
                        return false;
                    }
                    reader.next();
                    first = false;
                }
                if (!reader.forward()) {
                    reader.next();
                    return false;
                }
                return true;
            }
            
            @Override
            public Network next() {
                Network next = readOne(reader);
                if (next == null) {
                    throw new NoSuchElementException();
                }
                return next;
            }
        };
    }
    
    public static List<Network> readMany(XmlReader reader) {
        List<Network> list = new ArrayList<>();
        Iterator<Network> iterator = iterateMany(reader);
        while (iterator.hasNext()) {
            list.add(iterator.next());
        }
        return list;
    }
    
    private static void readLink(XmlReader reader, NetworkContainer object) {
        // Process the attributes:
        String rel = null;
        String href = null;
        for (int i = 0; i < reader.getAttributeCount(); i++) {
            String attrName = reader.getAttributeLocalName(i);
            String attrVal = reader.getAttributeValue(i);
            switch (attrName) {
                case "href":
                href = attrVal;
                break;
                case "rel":
                rel = attrVal;
                break;
                default:
                reader.skip();
                break;
            }
        }
        if (href != null) {
            ListWithHref list = new ArrayListWithHref();
            list.href(href);
            switch (rel) {
                case "networklabels":
                object.networkLabels(list);
                break;
                case "permissions":
                object.permissions(list);
                break;
                case "vnicprofiles":
                object.vnicProfiles(list);
                break;
                default:
                break;
            }
        }
        reader.skip();
    }
}

